/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mitre.mrald.query;

import java.util.List;

import org.mitre.mrald.parser.ParserElement;


/**
 * A BroadcastTargetProvider provides a list of other MRALD instances that can answer
 * a query built by the components supplied in the {@link #getTargets getTargets} method.
 *
 * @author Jeffrey Hoyt
  */
public interface BroadcastTargetProvider
{
    /**
     * This method should take in all the component parts of a query and determine
     * which other MRALD instances can answer that query.  For simpliciy, we'll assume
     * there is only one query to be built.
     *
     * @param sqlElements List of SqlElements that are parts of a single query
     * @param linkElements List of LinkElements that define the joins between tables
     *  This could just as easily be part of the List above, but they are already
     *  separated out, and special handling may be required for these.
     * @param datasource The local datasource name, so we know which local schema
     *   the query is built against
     *
     * @return A List of appropriate BroadcastQueryTargets which for the query
     * (where "the query" is the one which would result from composing a query
     * from the component parts).  The target must be fully populated as the
     * later processing will build the query from and broadcast the query based
     * on the contents of the target.
     */
    public List<BroadcastQueryTarget> getTargets( List<ParserElement> sqlElements, List <LinkElement> linkElements, String datasource );

    /**
     * Reloads the list of potential targets from source
     */
    public void reload(  );
}

