/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package test.org.mitre.lattice.graph;
import java.io.File;
import java.util.ArrayList;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.mitre.lattice.graph.SampleLattices;
import org.mitre.lattice.lattice.InvalidLatticeStructureException;
import org.mitre.lattice.lattice.LatticeFunctions;
import org.mitre.lattice.lattice.LatticeNode;
import org.mitre.lattice.lattice.LatticeTree;
import org.mitre.lattice.lattice.NodeNotFoundException;
import org.mitre.lattice.lattice.NullLatticeNodeException;
import org.mitre.mrald.util.Config;
/**
 *  Description of the Class
 *
 *@author     JCHOYT
 *@created    June 5, 2003
 */
public class LatticeFunctionsTest extends TestCase
{
    ArrayList children = new ArrayList();

    LatticeTree lattice;
    ArrayList parents = new ArrayList();


    /**
     *  The main program for the LatticeTreeTest class
     *
     *@param  args  The command line arguments
     */
    public static void main( String[] args )
    {
        junit.textui.TestRunner.run( suite() );
    }


    /**
     *  A unit test suite for JUnit
     *
     *@return    The test suite
     */
    public static Test suite()
    {
        /*
         *  the dynamic way
         */
        return new TestSuite( LatticeFunctionsTest.class );
    }


    /**
     *  A unit test for JUnit
     */
    public void testGetChildTree()
    {
        try
        {
            LatticeTree latticeNew = SampleLattices.buildVeryVeryComplexDiamondLattice();
            LatticeNode key = latticeNew.searchTree( "topRightPoint" );
            Object rootNode = LatticeFunctions.getChildTree( key );
            assertEquals( " The last Node should be the Root Node ", "Public", ( ( LatticeNode ) rootNode ).getName() );
        }
        catch ( NodeNotFoundException e )
        {
            fail( "Node not found" );
        }
        catch ( NullLatticeNodeException e )
        {
            fail( "Node is null" );
        }
        catch ( InvalidLatticeStructureException e )
        {
            fail( e.getMessage() );
        }
    }


    /**
     *  A unit test for JUnit
     */
    public void testGetDominant2()
    {

        LatticeTree latticeNew = SampleLattices.buildVeryVeryComplexDiamondLattice();
        LatticeNode root = latticeNew.getRootNode();

        assertEquals( " The root node should be ", "Public", root.getName() );
        /*ArrayList nodes =*/ LatticeFunctions.getMostDominant( root );

        assertEquals( " The max level should be ", 5, LatticeFunctions.getMaxLevel() );

        ArrayList nodes1 = LatticeFunctions.getNodesOnLevel( 3 );
        assertEquals( " The number of nodes should be 5 ", 5, nodes1.size() );

        //assertEquals( "One of the most dominant Nodes should be topPoint ", "topRightPoint", ((LatticeNode)nodes.get(0)).getName());

    }


    /**
     *  A unit test for JUnit
     */
    public void testInitialize()
    {
        try
        {

            LatticeTree newLattice = SampleLattices.buildComplexDiamondLattice();
            LatticeNode ln = newLattice.getRootNode();
            ArrayList<LatticeNode> flattenLattice = new ArrayList<LatticeNode>();

            flattenLattice = LatticeFunctions.addRoot( flattenLattice, ln );
            flattenLattice = LatticeFunctions.flattenLattice( flattenLattice, ln, 0 );

            assertEquals( "Number of elements in Flattened Array should be ", 8, flattenLattice.size() );

            LatticeFunctions.calcMaxNodesPerLevel();
            assertEquals( "Number of levels should be ", 4, LatticeFunctions.getMaxLevel() );
            assertEquals( "Number of max nodes in a level should be ", 3, LatticeFunctions.getMaxNodesPerLevel() );

            int noOfBoxes = LatticeFunctions.getNoOfGrids();
            assertEquals( "Number of grids should be ", 12, noOfBoxes );

            //double maxDistance = LatticeFunctions.getMaxDistance(noOfBoxes, LatticeFunctions.getBoxWidth(800), LatticeFunctions.getBoxHeight(600) );
            //int test = (int) maxDistance;
            //assertEquals( "Max Distance should be ", 17313, test);

            java.util.HashMap mapGenes = LatticeFunctions.getMapGeneToNode();
            assertEquals( "No of Genes should be ", 8, mapGenes.size() );
            java.util.HashMap mapNodes = LatticeFunctions.getMapNodeToGene();
            assertEquals( "No of Genes should be ", 8, mapNodes.size() );

            assertEquals( "Level of Node should be 0 ", 0, LatticeFunctions.getLevelOfNode( ln ) );

        }
        catch ( Exception e )
        {
            e.printStackTrace();
            fail( "Error e " + e.getMessage() );
        }
    }


    //Move to LatticeTreeTest
    /**
     *  A unit test for JUnit
     */
    public void testSearchTree()
    {
        try
        {

            LatticeTree latticeNew = SampleLattices.buildVeryVeryComplexDiamondLattice();
            LatticeNode test = latticeNew.searchTree( "topRightPoint" );
            assertEquals( " The searched node should be ", "topRightPoint", test.getName() );
        }
        catch ( NodeNotFoundException e )
        {
            fail( "Node not found" );
        }
    }


    /**
     *  A unit test for JUnit
     */
    public void testSerializeTree()
    {
        try
        {
            LatticeTree latticeNew = SampleLattices.buildComplexDiamondLattice();
            String filename = "test.ser";
            LatticeFunctions.serializeLatticeTree( latticeNew, filename );
            File sObj = new File( "test.ser" );
            assertTrue( "Serialized object should be relatively large", sObj.length() > 200L );
        }
        catch ( Exception e )
        {
            e.printStackTrace();
        }
    }


    /**
     *  A unit test for JUnit
     */
    public void testUnSerializeTree()
    {
        try
        {
            LatticeTree newLattice = LatticeFunctions.recoverLatticeTree( "lattice.ser" );
            LatticeNode ln = newLattice.getRootNode();
            /*
             *  test that there are 3 parents of the root node
             */
            assertEquals( 3, ln.getParents().size() );
            /*
             *  test that the first parent of the root node is the same as the root node of the reference lattice
             */
            LatticeTree refLattice = SampleLattices.buildComplexDiamondLattice();
            LatticeNode refRootNode = refLattice.getRootNode();
            LatticeNode refFirstParent = refRootNode.getParents().get( 0 );
            LatticeNode testFirstParent = ln.getParents().get( 0 );
            assertEquals( refFirstParent.getName(), testFirstParent.getName() );
        }
        catch ( Exception e )
        {
            fail( "Error e " + e.getMessage() );
            e.printStackTrace();
        }
    }


    /**
     *  The JUnit setup method
     */
    protected void setUp()
    {
        Config.setProperty( "LOGFILE", "neuro.log" );
        Config.setProperty( "LOGPATH", System.getProperty("java.io.tmpdir")  );
        Config.setProperty( "latticeProps", System.getProperty("lattice.dir")  );
        Config.setProperty( "mraldLogLevel", "2" );
        Config.setLatticeFactory( new org.mitre.mrald.util.LatticeFactory() );
        try
        {
            LatticeTree newLattice = SampleLattices.buildComplexDiamondLattice();
	    LatticeFunctions.setImplmnt();
            LatticeFunctions.serializeLatticeTree( newLattice, "lattice.ser" );
        }
        catch ( Exception e )
        {
            e.printStackTrace();
            fail( e.getMessage() );
        }
        File ser = new File( "lattice.ser" );
        assertTrue( ser.exists() );
        LatticeFunctions.setIsSet( false );
    }

    public void testGetDominant()
    {

    LatticeTree latticeNew= SampleLattices.buildVeryVeryComplexDiamondLattice();
    LatticeNode root = latticeNew.getRootNode();

    assertEquals(" The root node should be " , "Public", root.getName());
    /*ArrayList nodes =*/ LatticeFunctions.getMostDominant( root );

    assertEquals(" The max level should be " , 5, LatticeFunctions.getMaxLevel());

    ArrayList nodes1 = LatticeFunctions.getNodesOnLevel(3);
    assertEquals(" The number of nodes should be 5 " , 5, nodes1.size());

    //assertEquals( "One of the most dominant Nodes should be topPoint ", "topRightPoint", ((LatticeNode)nodes.get(0)).getName());

    }

    /**
     *  The JUnit teardown method
     */
    protected void tearDown()
    {
        File sObj = new File( "test.ser" );
        if ( sObj.exists() )
        {
            sObj.delete();
        }
        sObj = new File( "lattice.ser" );
        if ( sObj.exists() )
        {
            sObj.delete();
        }
    }
}


