/*
 *  Copyright 2008 The MITRE Corporation (http://www.mitre.org/). All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package test.org.mitre.lattice.lattice;
//// :noTabs=true:maxLineLen=0:indentSize=4:indentOnTab=true:folding=indent:
import java.io.File;
import java.util.ArrayList;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.mitre.lattice.graph.SampleLattices;
import org.mitre.lattice.lattice.InvalidLatticeStructureException;
import org.mitre.lattice.lattice.LatticeNode;
import org.mitre.lattice.lattice.LatticeTree;
import org.mitre.lattice.lattice.NodeNotFoundException;
import org.mitre.mrald.util.Config;
/**
 *  Description of the Class
 *
 *@author     Gail Hamilton
 *@created    June 5, 2003
 */
public class LatticeTreeTest extends TestCase
{
    ArrayList<LatticeNode> children = new ArrayList<LatticeNode>();

    LatticeTree lattice;
    ArrayList<LatticeNode> parents = new ArrayList<LatticeNode>();


    /**
     *  The main program for the LatticeTreeTest class
     *
     *@param  args  The command line arguments
     */
    public static void main( String[] args )
    {
        junit.textui.TestRunner.run( suite() );
    }


    /**
     *  A unit test suite for JUnit
     *
     *@return    The test suite
     */
    public static Test suite()
    {
        /*
         *  the dynamic way
         */
        return new TestSuite( LatticeTreeTest.class );
    }


    /**
     *  Description of the Method
     *
     *@param  keyNode     Description of the Parameter
     *@param  nodeType    Description of the Parameter
     *@param  checkNodes  Description of the Parameter
     *@return             Description of the Return Value
     */
    public boolean checkConsistent( LatticeNode keyNode, String nodeType, ArrayList checkNodes )
    {
        //Check to see if the parent being added is contained
        //within the Child nodes.
        boolean test = true;
        ArrayList children = keyNode.getChildren();

        if ( nodeType.equals( "children" ) )
        {
            for ( int i = 0; i < checkNodes.size(); i++ )
            {
                if ( !children.contains( checkNodes.get( i ) ) )
                {
                    test = true;
                    for ( int j = 0; j < children.size(); j++ )
                    {
                        LatticeNode child = ( LatticeNode ) children.get( j );
                        test = checkConsistent( child, "children", checkNodes );

                        //If inconsistency detected
                        if ( !test )
                        {
                            return test;
                        }
                    }
                }
                else
                {
                    test = false;
                    return test;
                }
            }

            return true;
        }
        return true;
    }


    /**
     *  A unit test for JUnit
     */
    public void testDominates()
    {
        //Using the buildDIamond case
        LatticeNode ln = lattice.getRootNode();
        ArrayList parents = ln.getParents();
        for ( int i = 0; i < parents.size(); i++ )
        {
            LatticeNode newNode = ( LatticeNode ) parents.get( i );
            assertEquals( "Should now be 1 parent", 1, newNode.countParentNodes() );
            assertEquals( "Should now be 1 parent", 1, newNode.countParentNodes() );

            ArrayList dominatesList = lattice.dominates( newNode );
            assertEquals( "Node name should now be dominated by Top Point ", "topPoint", ( ( LatticeNode ) dominatesList.get( 0 ) ).getName() );

        }

        ArrayList dominatesList = lattice.dominates( ln );
        assertEquals( "Node name should now be dominated by 4 Nodes ", 4, dominatesList.size() );
        assertEquals( "First Node name should be Public ", "Public", ( ( LatticeNode ) dominatesList.get( 0 ) ).getName() );
        assertEquals( "Second Node name should be Right Node ", "rightPoint", ( ( LatticeNode ) dominatesList.get( 1 ) ).getName() );
        assertEquals( "Third Node name should be Top Node ", "topPoint", ( ( LatticeNode ) dominatesList.get( 2 ) ).getName() );
        assertEquals( "Fourth Node name should be Left Node ", "leftPoint", ( ( LatticeNode ) dominatesList.get( 3 ) ).getName() );

    }


    /**
     *  Description of the Method
     */
    public void searchTree()
    {
    }


    /**
     *  Description of the Method
     */
    public void searchTreeString()
    {
    }


    /**
     *  Test all the add node conditions
     */
    public void testAddNode0()
    {
        /*
         *  add with child null
         */
        LatticeTree tree = new LatticeTree();
        try
        {
            tree.addNode( "newNode", parents, null );
        }
        catch ( Exception e )
        {
            return;
            // successful if exception thrown
        }
        fail( "Expected an exception to be thrown - all nodes need a child node" );
    }


    /**
     *  add with parent null
     */
    public void testAddNode1()
    {
        try
        {
        LatticeTree tree = new LatticeTree();
        boolean btest = tree.addNode( "nodeTest" );
        assertEquals( "Node was added", true, btest );
        assertEquals( "Should now be 1 nodes", 1, tree.countNodes() );
        String rootParent = tree.getRootNode().getParents().get( 0 ).getName();
        assertEquals( "root_node's parent should be named newNode, but it's named " + rootParent, "nodeTest", rootParent );
        }
        catch(InvalidLatticeStructureException e)
        {
            fail(e.getMessage());
        }
    }


    /**
     *  add intermediate node
     */
    public void testAddNode2()
    {
    }


    /**
     *  add with invalid children
     */
    public void testAddNode3()
    {
    }


    /**
     *  add with invalid parents
     */
    public void testAddNode4()
    {
    }


    /**
     *  ensure all nodes have a child node (except for the root_node)
     */
    public void testCheckTree1()
    {
        /*
         *  build a good tree
         */
        /*
         *  build a bad tree
         */
    }


    /**
     *  ensure all nodes are reachable
     */
    public void testCheckTree2()
    {
    }


    /**
     *  ensure no cyclicals
     */
    public void testCheckTree3()
    {
    }



    /**
     *  A unit test for JUnit
     */
    public void testCheckTree5()
    {
        /*
         *  fail( "need to figure out other things that make for a bad lattice and code tests to check for instances" );
         */
    }


    /**
     *  test that adding children works
     */
    public void restConsistentIsFalse()
    {
        try
        {
            boolean checkChildren = false;
            /*
             *  Attempting to add a parent that is already a child
             */
            LatticeTree newLattice = SampleLattices.buildVeryVeryComplexDiamondLattice();

//            GraphUserLattice graph = new GraphUserLattice();

            String[] addParentGroups = new String[]{"bottomRightPoint"};

            String addGroup = "midRightPoint";
            LatticeNode newNode = null;
            ArrayList<LatticeNode> addGroups = new ArrayList<LatticeNode>();

            try
            {
                newNode = newLattice.searchTree( addGroup );

                boolean parentAdded = false;

                assertEquals( "The lattice should have 2 values", 1, addParentGroups.length );

                for ( int i = 0; i < addParentGroups.length; i++ )
                {

                    if ( !addParentGroups[i].equals( "" ) )
                    {
                        LatticeNode parentKey = newLattice.searchTree( addParentGroups[i] );

                        if ( parentKey.addChild( newNode ) )
                        {
                            parentAdded = newNode.addParent( parentKey );
                        }

                        if ( parentAdded )
                        {
                            //MraldOutFile.logToFile(Config.getProperty("LOGFILE"), "Parent added " + parentKey.getName() + " to " + newNode.getName());
                        }

                        addGroups.add( parentKey );

                    }

                }

                //newLattice = (LatticeTree)graph.addParentNode(addGroup, addParentGroups, newLattice);
            }
            catch ( NodeNotFoundException e )
            {
                fail( "Node " + addGroup + " not found " );
            }

            /*boolean result =*/ newNode.checkConsistent( checkChildren, addGroups );
        }
        catch ( Exception e )
        {
            assertEquals( "The lattice should return that it is inconsistent", false, false );
        }
    }


    /**
     *  test that adding children works
     */
    public void restConsistentIsTrue()
    {
        try
        {
            /*
             *  Attempting to add a parent that is already a child
             */
            boolean isChild = true;
            LatticeTree newLattice = SampleLattices.buildVeryVeryComplexDiamondLattice();

//            GraphUserLattice graph = new GraphUserLattice();

            String[] addParentGroups = new String[]{"topRightPoint"};

            String addGroup = "midRightPoint";
            LatticeNode newNode = null;
            ArrayList<LatticeNode> addGroups = new ArrayList<LatticeNode>();

            try
            {
                newNode = newLattice.searchTree( addGroup );
                boolean parentAdded = false;
                assertEquals( "The lattice should have 2 values", 1, addParentGroups.length );
                for ( int i = 0; i < addParentGroups.length; i++ )
                {
                    if ( !addParentGroups[i].equals( "" ) )
                    {
                        LatticeNode parentKey = newLattice.searchTree( addParentGroups[i] );
                        if ( parentKey.addChild( newNode ) )
                        {
                            parentAdded = newNode.addParent( parentKey );
                        }
                        if ( parentAdded )
                        {
                            //MraldOutFile.logToFile(Config.getProperty("LOGFILE"), "Parent added " + parentKey.getName() + " to " + newNode.getName());
                        }
                        addGroups.add( parentKey );
                    }
                }
            }
            catch ( NodeNotFoundException e )
            {
                fail( "Node " + addGroup + " not found " );
            }
            boolean result = newNode.checkConsistent( isChild, addGroups );
            assertEquals( "The lattice should return that it is consistent", true, result );
        }
        catch ( Exception e )
        {
            fail( e.getMessage() );
        }
    }


    /**
     *  test that adding children works
     */
    public void testDeleteLink()
    {
        //String[] values = new String[]{"Parent:extremeMidLeftPoint~Child:bottomRightPoint~Type:DeleteLink"};
        //LatticeElement newElement = new LatticeElement();
        //newElement.process(
    }


    /**
     *  A unit test for JUnit
     */
    public void testDominates2()
    {
        lattice = SampleLattices.buildComplexDiamondLattice();

        //Using the buildComplexDiamond case
        LatticeNode ln = lattice.getRootNode();
        ArrayList parents = ln.getParents();
        for ( int i = 0; i < parents.size(); i++ )
        {
            LatticeNode newNode = ( LatticeNode ) parents.get( i );
            String nodeName = newNode.getName();

            if ( nodeName.equals( "topPoint" ) )
            {
                assertEquals( "Should now be 0 parents", 0, newNode.countParentNodes() );
                ArrayList dominatesList = lattice.dominates( newNode );
                assertEquals( "Node topPoint should now be dominated by 0 Nodes ", 0, dominatesList.size() );
                assertEquals( "Node topPoint should now have 1 child Node ", 1, newNode.countChildNodes() );

            }
            else if ( nodeName.equals( "topLeftPoint" ) )
            {
                assertEquals( "Should now be 0 parents", 0, newNode.countParentNodes() );
                ArrayList dominatesList = lattice.dominates( newNode );
                assertEquals( "Node topLeftPoint should now be dominated by 0 Nodes ", 0, dominatesList.size() );
                assertEquals( "Node topLeftPoint should now have 2 child Nodes ", 2, newNode.countChildNodes() );

            }
            else if ( nodeName.equals( "topCenterPoint" ) )
            {
                assertEquals( "Should now be 1 parents", 1, newNode.countParentNodes() );
                ArrayList dominatesList = lattice.dominates( newNode );
                assertEquals( "Node topCenterPoint should now be dominated by 1 Node ", 1, dominatesList.size() );
                assertEquals( "Node topCenterPoint should now have 3 child Nodes ", 3, newNode.countChildNodes() );

            }
            else if ( nodeName.equals( "topRightPoint" ) )
            {
                assertEquals( "Should now be 0 parents", 0, newNode.countParentNodes() );
                ArrayList dominatesList = lattice.dominates( newNode );
                assertEquals( "Node topRightPoint should now be dominated by 0 Node ", 0, dominatesList.size() );
                assertEquals( "Node topRightPoint should now have 2 child Nodes ", 2, newNode.countChildNodes() );

            }
            else if ( nodeName.equals( "leftPoint" ) )
            {
                assertEquals( "Should now be 2 parents", 2, newNode.countParentNodes() );
                ArrayList dominatesList = lattice.dominates( newNode );
                assertEquals( "Node leftPoint should now be dominated by 3 Nodes ", 3, dominatesList.size() );
                assertEquals( "Node leftPoint should now have 1 child Nodes ", 1, newNode.countChildNodes() );

            }
            else if ( nodeName.equals( "centerPoint" ) )
            {
                assertEquals( "Should now be 3 parents", 3, newNode.countParentNodes() );
                ArrayList dominatesList = lattice.dominates( newNode );
                assertEquals( "Node CenterPoint should now be dominated by 4 Nodes ", 4, dominatesList.size() );
                assertEquals( "Node centerPoint should now have 1 child Nodes ", 1, newNode.countChildNodes() );

            }
            else if ( nodeName.equals( "rightPoint" ) )
            {
                assertEquals( "Should now be 2 parents", 2, newNode.countParentNodes() );
                ArrayList dominatesList = lattice.dominates( newNode );
                assertEquals( "Node rightPoint should now be dominated by 3 Nodes ", 3, dominatesList.size() );
                assertEquals( "Node rightPoint should now have 1 child Nodes ", 1, newNode.countChildNodes() );

            }
            else
            {
                assertEquals( "Should now be 3 parents", 3, newNode.countParentNodes() );
            }
            /*
             *  root
             */
        }

        /*
         *  The actual order of nodes is not important. Test to check that they all exist
         *  but remove for recursion, As changes in the code may change the order without
         *  affecting functionality
         */
        ArrayList dominatesList = lattice.dominates( ln );
        assertEquals( "Node name should now be dominated by 8 Nodes ", 8, dominatesList.size() );
        assertEquals( "First Node name should be Public ", "Public", ( ( LatticeNode ) dominatesList.get( 0 ) ).getName() );
        assertEquals( "2nd Node name should be Right Node ", "rightPoint", ( ( LatticeNode ) dominatesList.get( 1 ) ).getName() );
        assertEquals( "3rd Node name should be Top Right Node ", "topRightPoint", ( ( LatticeNode ) dominatesList.get( 2 ) ).getName() );
        assertEquals( "4th Node name should be Top Center Node ", "topCenterPoint", ( ( LatticeNode ) dominatesList.get( 3 ) ).getName() );
        assertEquals( "5th  Node name should be Top Node ", "topPoint", ( ( LatticeNode ) dominatesList.get( 4 ) ).getName() );
        assertEquals( "6th Node name should be Left Node ", "leftPoint", ( ( LatticeNode ) dominatesList.get( 5 ) ).getName() );
        assertEquals( "7th Node name should be Top Left Node ", "topLeftPoint", ( ( LatticeNode ) dominatesList.get( 6 ) ).getName() );
        assertEquals( "8th Node name should be Center Node ", "centerPoint", ( ( LatticeNode ) dominatesList.get( 7 ) ).getName() );

    }


    /**
     *  A unit test for JUnit
     */
    public void testEquals()
    {
    }


    /**
     *  A unit test for JUnit
     */
    public void testExportTree()
    {
    }


    /**
     *  A unit test for JUnit
     */
    public void testImportTree()
    {
    }


    /**
     *  A unit test for JUnit
     */
    public void testRemoveNode()
    {
    }


    /*
     *  Check that relationships are symmetrical
     *  i.e. That if a child exists. Then this child has a parent object. And that is the same
     *  as the current object
     */
    /**
     *  A unit test for JUnit
     */
    public void testSymmetry()
    {
        //Using the buildDIamond case
        LatticeNode ln = lattice.getRootNode();
        ArrayList parents = ln.getParents();
        for ( int i = 0; i < parents.size(); i++ )
        {
            LatticeNode newNode = ( LatticeNode ) parents.get( i );
            assertEquals( "Should now be 1 parent", 1, newNode.countParentNodes() );
            assertEquals( "Should now be 1 parent", 1, newNode.countParentNodes() );
            if ( i == 0 )
            {
                assertEquals( "Node name should now be Right Point ", "rightPoint", newNode.getName() );
            }
            else if ( i == 1 )
            {
                assertEquals( "Node name should now be Left Point ", "leftPoint", newNode.getName() );
            }

            ArrayList newParents = newNode.getParents();
            if ( newParents.size() > 0 )
            {
                LatticeNode newParentNode = ( LatticeNode ) newParents.get( 0 );
                ArrayList newChildren = newParentNode.getChildren();

                assertTrue( "The current node should be in the children of the parent node ", newChildren.contains( newNode ) );
            }

        }
    }


    /**
     *  The JUnit setup method
     */
    protected void setUp()
    {
        //Config.setProperty( "LOGPATH", "mrald.log" );
        Config.setProperty( "mraldLogLevel", "2" );
        Config.setProperty( "LOGFILE", "mrald.log" );
        Config.setLatticeFactory( new org.mitre.mrald.util.LatticeFactory() );
        LatticeNode parentNode1 = Config.getLatticeFactory().createNode("parentNode1" );
        LatticeNode parentNode2 = Config.getLatticeFactory().createNode("parentNode2" );
        LatticeNode childNode1 = Config.getLatticeFactory().createNode("childNode1" );
        LatticeNode childNode2 = Config.getLatticeFactory().createNode("childNode2" );
        LatticeNode childNode3 = Config.getLatticeFactory().createNode("childNode3" );

        children.add( childNode1 );
        children.add( childNode2 );
        children.add( childNode3 );
        parents.add( parentNode1 );
        parents.add( parentNode2 );
        lattice = SampleLattices.buildDiamondLattice();
    }


    /**
     *  The JUnit teardown method
     */
    protected void tearDown()
    {
        File sObj = new File( "mrald.log" );
        if ( sObj.exists() )
        {
            sObj.delete();
        }
    }
}


