#!/usr/bin/python
"""Convert moco parameters from fsl to neuropipes standard

This module reformats the motion parameters generated by the various
analysis software to the following format
scan to scan differences
[trans_x,trans_y,trans_z,rot_x,rot_y,rot_z]
[mm,mm,mm,rad,rad,rad]
"""

import os
import sys
import yaml
import copy
import numpy as np
import traceback

def mocoreformat(inputfile,outputfile=''):
    """Reformat moco parameters

    Input/Output
    -----
    YAML formatted specification

    Input Requirements
    ------------------
    filetype : one of FSL, SPM or SIEMENS
    mocopar : list of motion parameter files

    in addition to the mandatory fields (source, name, location)
    
    Output
    ------
    in addition to the mandatory fields (source, name, location)

    
    """
    source = 'mit'
    modulename = 'mocoreformat'
    modulename = ''.join((source,'.',modulename))
    
    # read YAML input
    try:
        if type(inputfile) == type(str()):
            fh = open(inputfile,'r')
            yaml_in = yaml.load(fh)
            fh.close()
        else:
            yaml_in = inputfile
            
        # Ensure correct parameters are being sent for the correct module
        yaml_in = yaml_in[modulename]
    except:
        print("Cannot find or load input file")
        traceback.print_exc()

    # Create directory for module output
    outputlocation = os.path.join(yaml_in['location'],modulename)
    if not os.path.exists(outputlocation):
        os.mkdir(outputlocation)

    # initialize YAML output
    output             = copy.deepcopy(yaml_in)
    output['mocopar']  = []
    
    for series in range(len(yaml_in['mocopar'])):
        mocofile = yaml_in['mocopar'][series]
        mc = np.loadtxt(mocofile)

        # Process the various possible input types
        if yaml_in['filetype'] == 'FSL':
            mc = mc[:,[3,4,5,0,1,2]]
        elif  yaml_in['filetype'] == 'SPM':
            # do nothing
            pass
        elif  yaml_in['filetype'] == 'SIEMENS':
            print("Cannot read siemens moco file")
            
        # write output to outputfile
        (filepath,filename) = os.path.split(mocofile)
        outputmocofile = os.path.join(outputlocation,''.join(('mcref.',str(series),'.',filename)))
        np.savetxt(outputmocofile,mc,fmt='%.9f',delimiter=' ')
        
        output['mocopar'].insert(series,outputmocofile)

    yaml_out = {}
    yaml_out[modulename] = output

    # Generate filename and write yaml output
    if outputfile == '':
        outputfile = os.path.join(outputlocation,''.join((modulename,'.yaml')))

    # print("writing output to: %s"%outputfile)
    fh = open(outputfile,'wt')
    yaml.dump(output,fh,default_flow_style=False)
    fh.close()

    return yaml_out

if __name__=="__main__":
    try:
        mocoreformat(sys.argv[1],sys.argv[2])
    except:
        print "An unhandled exception occured, here's the traceback!"
        traceback.print_exc()

