#!/usr/bin/python
"""Convert moco parameters from fsl to neuropipes standard

This module generates files containing indices of outliers in
functional imaging run based on the intensity of the images and the
estimates of movement parameters.

"""

import os
import sys
import yaml
import numpy as np
import traceback

def mocoreformat(inputfile,outputfile=''):
    """Wrapper routine around the outlier detection algorithm

    Input
    -----
    YAML formatted input specification
    
    """
    source = 'mit'
    modulename = 'mocoreformat'

    # read YAML input
    try:
        fh = open(inputfile,'r')
        yaml_in = yaml.load(fh)
        fh.close()
        yaml_in = yaml_in[modulename]
    except:
        print("Cannot find or load input file")
        traceback.print_exc()

    # Create directory for module output
    outputlocation = os.path.join(yaml_in['location'],''.join((source,'.',modulename)))
    if not os.path.exists(outputlocation):
        os.mkdir(outputlocation)

    # initialize YAML output
    output = {'source' : source , 'name' : modulename, 'mocopar' : []}
    for series in range(len(yaml_in['mocopar'])):
        mocofile = yaml_in['mocopar'][series]
        mc = np.loadtxt(mocofile)

        # Process the various possible input types
        if yaml_in['filetype'] == 'FSL':
            mc = mc[:,[3,4,5,0,1,2]]
        elif  yaml_in['filetype'] == 'SPM':
            mc = concatenate( (np.zeros(1,6),diff(mc,n=1,axis=0)) , axis=0)
        elif  yaml_in['filetype'] == 'SIEMENS':
            print("Cannot read siemens moco file")
            
        # write output to outputfile
        (filepath,filename) = os.path.split(mocofile)
        outputmocofile = os.path.join(outputlocation,filename)
        np.savetxt(outputmocofile,mc,fmt='%.9f',delimiter=' ')
        
        output['mocopar'].insert(series,outputmocofile)

    # Generate filename and write yaml output
    if outputfile == '':
        outputfile = os.path.join(outputlocation,''.join((modulename,'.yaml')))

    # print("writing output to: %s"%outputfile)
    fh = open(outputfile,'wt')
    yaml.dump(output,fh,default_flow_style=False)
    fh.close()

    return output

if __name__=="__main__":
    try:
        mocopar(sys.argv[1],sys.argv[2])
    except:
        print "An unhandled exception occured, here's the traceback!"
        traceback.print_exc()

