/*=========================================================================

  Program:         Level-Set Segmentation for Slicer3
  Module:          LevelSetStepping.cxx
  Language:        C++
  Date:            2008-08-05 12:13:32
  Version:         1.00
  Author:	       Carlos S. Mendoza - Universidad de Sevilla, Spain
  E-mail:          carlos.sanchez.mendoza@gmail.com
  Acknowledgments: This work is funded by a scholarship for research 
                   personnel instruction from the University of 
				   Sevilla, Spain.
				   
				   This work has been developed under the supervision of
				   Mr. Steve Pieper Ph.D. and and Ron Kikinis M.D. during 
				   an internship in the Surgical Planning Laboratory in 
				   Harvard Medical School and Brigham and Women's Hospital.

=========================================================================*/

/*-------------------------------------------------------------------------

This module implements the core level-set functionality. A parameterized 
geodesic active contours level-set computation takes place, for a given 
number of iterations. The initial label image and the feature image from
the image to be segmented must be provided.

-------------------------------------------------------------------------*/


#if defined(_MSC_VER)
#pragma warning ( disable : 4786 )
#endif

#ifdef __BORLANDC__
#define ITK_LEAN_AND_MEAN
#endif

#include "LevelSetSteppingCLP.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIterator.h"
#include "itkGeodesicActiveContourLevelSetImageFilter.h"
#include "itkSignedDanielssonDistanceMapImageFilter.h" 
#include "itkBinaryThresholdImageFilter.h"
#include "itkImage.h"
#include <iostream>

int main( int argc, char *argv[] )
{	
	PARSE_ARGS;
	
	typedef float FloatPixelType;
	const unsigned int Dimension = 3;
	typedef itk::Image<FloatPixelType,Dimension> FloatImageType;
	typedef unsigned char LabelPixelType;
	typedef itk::Image<LabelPixelType,Dimension> LabelImageType;
	typedef itk::ImageFileReader<FloatImageType> FeatureReaderType;
	typedef itk::ImageFileReader<LabelImageType> LabelReaderType;
	typedef itk::ImageFileWriter<LabelImageType> LabelWriterType;
	typedef itk::SignedDanielssonDistanceMapImageFilter<LabelImageType,FloatImageType> InitialLevelSetCreatorType;
	typedef itk::GeodesicActiveContourLevelSetImageFilter<FloatImageType,FloatImageType,FloatPixelType> GeodesicActiveContourFilterType;
	typedef itk::BinaryThresholdImageFilter<FloatImageType,LabelImageType> ZeroLevelSetThresholder;
	typedef itk::ImageRegionIterator<LabelImageType> LabelIteratorType;
	
	// Both the initial label image and the feature image are sources of our pipeline
	
	FeatureReaderType::Pointer featureReader = FeatureReaderType::New();
	featureReader->SetFileName(featureImage.c_str());

	LabelReaderType::Pointer labelReader = LabelReaderType::New();
	labelReader->SetFileName(initialLabel.c_str());
	labelReader->Update();

	// We iterate the label image to find what the initial label value is
	// We want to use that value in the resulting label image

	LabelIteratorType iter1(labelReader->GetOutput(),labelReader->GetOutput()->GetLargestPossibleRegion());

	unsigned int label=0;
	for (iter1.GoToBegin();!iter1.IsAtEnd();++iter1)
	{
		if (iter1.Get() != 0) 
		{
			label=iter1.Get();
			break;
		}
	}
	
	// Once the label has been found we want to produce a distance image from the label image
	// The inside of our initial label image regions will have negative values and the outside
	// will have positive values, as the distance is computed from the closest region's border

	InitialLevelSetCreatorType::Pointer levelSetCreator = InitialLevelSetCreatorType::New();
	levelSetCreator->SetInput(labelReader->GetOutput());

	// The resulting distance image can be fed into a geodesic active contours filter

	GeodesicActiveContourFilterType::Pointer geodesicFilter = GeodesicActiveContourFilterType::New();
	geodesicFilter->SetFeatureImage(featureReader->GetOutput());
	geodesicFilter->SetInput(levelSetCreator->GetDistanceMap());
	
	// We set the initial isovalue to 1 instead of 0 so that really small regions can actually evolve at all
	// (we will have guaranteed zeros but no minus ones in these really small regions)
	
	geodesicFilter->SetIsoSurfaceValue(1);
	
	// Then the parameters for the level-set equation are all parameters
	geodesicFilter->SetPropagationScaling(alpha);
	geodesicFilter->SetCurvatureScaling(beta);
	geodesicFilter->SetAdvectionScaling(gamma);
	
	// As is the number of iterations
	
	geodesicFilter->SetNumberOfIterations(iterations);
	geodesicFilter->SetMaximumRMSError(0.00001);
	geodesicFilter->UseImageSpacingOn();

	// The resulting distance image has to be converted back to a label image
	// so we threshold it getting the border (zero level-set) and everything inside
	// The resulting value in the thresholded is the previously computed preexisting label
	
	ZeroLevelSetThresholder::Pointer thresholder = ZeroLevelSetThresholder::New();

	thresholder->SetInput(geodesicFilter->GetOutput());
	thresholder->SetLowerThreshold(-1000.0);
	thresholder->SetUpperThreshold(0.0);
	thresholder->SetOutsideValue(0);
	thresholder->SetInsideValue(label);
	
	LabelWriterType::Pointer labelWriter = LabelWriterType::New();
	labelWriter->SetFileName(outputLabel.c_str());
	labelWriter->SetInput(thresholder->GetOutput());

	// Ready to write the output

	try
	{
		labelWriter->Update();
	}
	
	catch( itk::ExceptionObject & excep )
	{
	std::cerr << "Exception caught !" << std::endl;
	std::cerr << excep << std::endl;
	return EXIT_FAILURE;
	}
	return EXIT_SUCCESS;
}
















