#include <iostream>
#include <fstream>
#include <vector>
#include <utility>
#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkVector.h"
#include "itkPluginFilterWatcher.h"
#include "itkImageToVectorImageFilter.h"
#include "itkMaskImageFilter.h"
#include "itkGradientAnisotropicDiffusionImageFilter.h"
#include "itkGradientMagnitudeImageFilter.h"
#include "itkWatershedImageFilter.h"
#include "itkInvertIntensityImageFilter.h"
#include "itkMinimumMaximumImageCalculator.h"
#include "itkSignedMaurerDistanceMapImageFilter.h"
#include "itkLabelStatisticsImageFilter.h"
#include "itkImageToListSampleFilter.h"
#include "DiffusionScalarLabelStatsCLP.h"

int main(int argc, char * argv [])
{
  PARSE_ARGS;

  bool violated=false;
  if (inputDiffusionScalarImage.size() == 0) { violated = true; std::cout << "  --inputScalarImage Required! " << std::endl; }
  if (inputLesionImage.size() == 0) { violated = true; std::cout << "  --inputLesionImage Required! " << std::endl; }
  if (inputBrainMask.size() == 0) { violated = true; std::cout << "  --inputBrainMask Required! " << std::endl; }
  if (outputStatsFile.size() == 0) { violated = true; std::cout << "  --outputStatsFile Required! "  << std::endl; }
  if (violated) exit(1);

  typedef float       PixelType;
  typedef short       MaskPixelType;
  const unsigned int          Dimension = 3;

  typedef itk::Image< PixelType,  Dimension >   DiffImageType;
  typedef itk::Image< MaskPixelType,  Dimension >   MaskImageType;
  typedef itk::ImageFileReader< DiffImageType  >  ReaderType;
  typedef itk::ImageFileReader< MaskImageType  >  MaskReaderType;

  ReaderType::Pointer inputDiffusionReader = ReaderType::New();
  inputDiffusionReader->SetFileName(inputDiffusionScalarImage.c_str());

  MaskReaderType::Pointer inputLesionReader = MaskReaderType::New();
  inputLesionReader->SetFileName(inputLesionImage.c_str());
  
  MaskReaderType::Pointer inputBrainMaskReader = MaskReaderType::New();
  inputBrainMaskReader->SetFileName(inputBrainMask.c_str());
 
  typedef itk::MaskImageFilter< MaskImageType, MaskImageType, MaskImageType > MaskFilterType;
  MaskFilterType::Pointer maskFilter = MaskFilterType::New();

  typedef itk::SignedMaurerDistanceMapImageFilter< MaskImageType, MaskImageType> DistanceMapFilterType;
  DistanceMapFilterType::Pointer distanceMapFilter = DistanceMapFilterType::New();
  distanceMapFilter->SetInsideIsPositive(false); // Inside distances are negative
  distanceMapFilter->SetUseImageSpacing(true);
  typedef DistanceMapFilterType::OutputImageType DistanceMapImageType;

  typedef itk::LabelStatisticsImageFilter< DiffImageType, MaskImageType > LabelStatsFilterType;
  LabelStatsFilterType::Pointer labelStatsFilter = LabelStatsFilterType::New();
  
  try {
    distanceMapFilter->SetInput( inputLesionReader->GetOutput() );
    distanceMapFilter->UseImageSpacingOff();
    distanceMapFilter->SquaredDistanceOff();
    maskFilter->SetInput1( distanceMapFilter->GetOutput() );
    maskFilter->SetInput2( inputBrainMaskReader->GetOutput() );
    maskFilter->SetOutsideValue(0);
    labelStatsFilter->SetInput(inputDiffusionReader->GetOutput());
    labelStatsFilter->SetLabelInput(maskFilter->GetOutput());
    labelStatsFilter->Update();  
  }
  catch (itk::ExceptionObject &excep)
  {
    std::cerr << argv[0] << ": exception caught !" << std::endl;
    return EXIT_FAILURE;
  }

  int numLabels = labelStatsFilter->GetNumberOfLabels();

  //std::cout << std::endl << "numLabels: " << numLabels << std::endl;
  
  float min = 0; float mean = 0; float max = 0; float sigma = 0;

  typedef itk::Statistics::ImageToListSampleFilter< DiffImageType, MaskImageType > ImageToListFilterType;
  ImageToListFilterType::Pointer imageToListFilter = ImageToListFilterType::New();

  // File output
  std::ofstream fout;
  fout.open (outputStatsFile.c_str());  
  fout << "Distance,Min,Mean,Max,Sigma,values..." << std::endl;

  for(int i=0;i<numLabels;++i)
    {
    // write stats for each label to a file.
    min = labelStatsFilter->GetMinimum(i);
    mean = labelStatsFilter->GetMean(i);
    max = labelStatsFilter->GetMaximum(i);
    sigma = labelStatsFilter->GetSigma(i);

    fout << i << "," << min << "," << mean << "," << max << "," << sigma << ",";

    try {
      // Get the diffusion values under this label
      imageToListFilter->SetInput(inputDiffusionReader->GetOutput());
      imageToListFilter->SetMaskImage(distanceMapFilter->GetOutput());
      imageToListFilter->SetMaskValue(i);
      imageToListFilter->Modified();
      imageToListFilter->Update();
      
      // Iterate over the diffusion values vector and print them to file.
      typedef ImageToListFilterType::ListSampleType listType;
      //listType voxelList = *imageToListFilter->GetOutput();

      //listType::Iterator listItr = voxelList.Begin();
      listType::ConstIterator listItr = imageToListFilter->GetOutput()->Begin();

      //for(listItr=voxelList.Begin();listItr!=voxelList.End();++listItr)
      for(listItr=(imageToListFilter->GetOutput())->Begin();listItr!=imageToListFilter->GetOutput()->End();++listItr)
        {
	//std::cout << "counting: " << listItr.GetMeasurementVector() << ",";
        // Print 
 	if(listItr!=(imageToListFilter->GetOutput())->Begin())
          {
          fout << ",";
          } 
        fout << listItr.GetMeasurementVector()[0];
        }
        
      }
    catch(itk::ExceptionObject &excep)
      {
      std::cerr << argv[0] << ": Exception: " << excep << std::endl;
      return EXIT_FAILURE;
      }
    fout << std::endl;
    //std::cout << "loop: " << i << ",";
    }
    fout.close();

  return EXIT_SUCCESS;
}

