#ifndef lesionSegmentationModel_h
#define lesionSegmentationModel_h

#include <vector>
//#include <vnl_vector>
#include <fstream>
#include <iostream>
#include "itkArray.h"
#include "itkVariableSizeMatrix.h"
#include "itkByteSwapper.h"
//#include "itkIO.h"
#include "itkVariableLengthVector.h"
#include <math.h>

class lesionSegmentationModel
{
private:
  typedef lesionSegmentationModel Self;
  typedef enum { readFail,writeFail } ioErr;
  typedef enum { file_signature=0x12345678,
    swapped_file_signature=0x78563412 } fileSig;
public:
  //typedef vnl_vector<double> LesionCentroidType;
  typedef itk::Array<double> DataClassMeanType;
  typedef DataClassMeanType LesionCentroidType;
  typedef itk::VariableSizeMatrix<double> DataCovarianceType;
  typedef itk::VariableLengthVector< double > TrainingVectorType;

public:
  double & getDistanceThreshold() {return m_distanceThreshold;}
  void setDistanceThreshold(double inputDistanceThreshold)
    {
    m_distanceThreshold = inputDistanceThreshold;
    }

  lesionSegmentationModel(unsigned int inputNumDistFeatures, unsigned int inputNumBayesFeatures, unsigned int inputNumFeatures) : m_Swapped(false)
    {
    m_NumFeatures = inputNumFeatures;
    m_NumDistFeatures = inputNumDistFeatures;
    m_NumBayesFeatures = inputNumBayesFeatures;
    InitializeModel();
    }

  LesionCentroidType & getLesionCentroid() {return m_lesionCentroid;}
  void setLesionCentroid(LesionCentroidType &inputCentroid)
    {
    m_lesionCentroid = inputCentroid;
    }

  DataClassMeanType & getLesionDataMean() {return m_lesionDataMean;}
  void setLesionDataMean(DataClassMeanType &inputLesionMean)
    {
    m_lesionDataMean = inputLesionMean;
    }

  DataClassMeanType & getNonLesionDataMean() {return m_nonLesionDataMean;}
  void setNonLesionDataMean(DataClassMeanType &inputNonLesionMean)
    {
    m_nonLesionDataMean = inputNonLesionMean;
    }

  DataCovarianceType & getLesionDataCovariance() {return m_lesionDataCovariance;}
  void setLesionDataCovariance(const DataCovarianceType &inputLesionCovariance)
    {
    m_lesionDataCovariance = inputLesionCovariance;
    }

  DataCovarianceType & getNonLesionDataCovariance() {return m_nonLesionDataCovariance;}
  void setNonLesionDataCovariance(const DataCovarianceType &inputNonLesionCovariance)
    {
    m_nonLesionDataCovariance = inputNonLesionCovariance;
    }

  TrainingVectorType & getTrainingMins() {return m_trainingMins;}
  void setTrainingMins(TrainingVectorType &inputTrainingMins)
    {
    m_trainingMins = inputTrainingMins;
    }

  TrainingVectorType & getTrainingMaxs() {return m_trainingMaxs;}
  void setTrainingMaxs(TrainingVectorType &inputTrainingMaxs)
    {
    m_trainingMaxs = inputTrainingMaxs;
    }

  int getNumFeatures(){return m_NumFeatures;}
  int getNumDistFeatures(){return m_NumDistFeatures;}
  int getNumBayesFeatures(){return m_NumBayesFeatures;}

  void WritemodelFilename(const std::string &filename)
    {
    ////////////////////////////////////////////////////////////////////////////

    std::ofstream output(filename.c_str()); // open setup file for reading
    if (!output.is_open())
      {
      std::cerr << "Can't write " << filename << std::endl;
      std::cerr.flush();
      }
    try
      {
      this->Write<unsigned int>(output,file_signature); //Write out the signature first
 
      this->Write<double>(output,this->getDistanceThreshold());
      this->Write(output,this->getLesionCentroid());
      this->Write(output,this->getLesionDataMean());
      this->Write(output,this->getNonLesionDataMean());
      this->Write(output,this->getLesionDataCovariance());
      this->Write(output,this->getNonLesionDataCovariance());
      this->Write(output,this->getTrainingMins());
      this->Write(output,this->getTrainingMaxs());
      }
    catch (ioErr e)
      {
      std::cerr << "Write failed for " << filename << std::endl;
      }
    output.close();
    }

  void ReadmodelFilename(const std::string &filename)
    {
    ////////////////////////////////////////////////////////////////////////////

    std::ifstream input(filename.c_str()); // open setup file for reading
    if (!input.is_open())
      {
      std::cerr << "Can't read " << filename << std::endl;
      std::cerr.flush();
      exit(-1);
      }
    try
      {
      unsigned int sig;
      this->Read<unsigned int>(input,sig);
      if(sig != file_signature && sig != swapped_file_signature)
        {
        this->m_Swapped = false;
        }
      else if(sig == swapped_file_signature)
        {
        this->m_Swapped = true;
        }
      this->Read<double>(input,this->getDistanceThreshold());
      this->Read(input,this->getLesionCentroid());
      this->Read(input,this->getLesionDataMean());
      this->Read(input,this->getNonLesionDataMean());
      this->Read(input,this->getLesionDataCovariance());
      this->Read(input,this->getNonLesionDataCovariance());
      this->Read(input,this->getTrainingMins());
      this->Read(input,this->getTrainingMaxs());
      }
    catch (ioErr e)
      {
      std::cerr << "Read failed for " << filename << std::endl;
      std::cerr << e << std::endl;
      exit(-1);
      }
    input.close();
    }
  
  void InitializeModel()
    {
    //Allocate the outter dim for all datasets
    this->m_lesionCentroid.set_size(this->getNumDistFeatures());
    this->m_lesionDataMean.SetSize(this->getNumBayesFeatures());
    this->m_nonLesionDataMean.SetSize(this->getNumBayesFeatures());
    this->m_lesionDataCovariance.SetSize(this->getNumBayesFeatures(),this->getNumBayesFeatures());
    this->m_nonLesionDataCovariance.SetSize(this->getNumBayesFeatures(),this->getNumBayesFeatures());
    this->m_trainingMins.SetSize(this->getNumFeatures());
    this->m_trainingMaxs.SetSize(this->getNumFeatures());
    }

private:
  bool m_Swapped;
  unsigned int m_NumFeatures;
  unsigned int m_NumDistFeatures;
  unsigned int m_NumBayesFeatures;

  template <class T>
    void
    Write(std::ofstream &f,T var)
      {
      if(f.bad() || f.eof())
        {
        throw writeFail;
        }
      f.write(reinterpret_cast<char *>(&var),sizeof(T));
      }
  template <class T>
    void
    Read(std::ifstream &f, T &var)
      {
      if(f.bad() || f.eof())
        {
        throw readFail;
        }
      f.read(reinterpret_cast<char *>(&var),sizeof(T));
      if(this->m_Swapped)
        {
        if(itk::ByteSwapper<T>::SystemIsBigEndian())
          {
          itk::ByteSwapper<T>::SwapFromSystemToLittleEndian(&var);
          }
        else
          {
          itk::ByteSwapper<T>::SwapFromSystemToBigEndian(&var);
          }
        }
      }
  void Write(std::ofstream &f,const DataCovarianceType &matrix)
    {
    for(int x=0;x<matrix.Rows();x++)
      {
      for(int y=0;y<matrix.Cols();y++)
        {
        this->Write<double>(f,matrix(x,y));
        }
      }
    }
  void Read(std::ifstream &f,DataCovarianceType &matrix)
    {
    for(int x=0;x<matrix.Rows();x++)
      {
      for(int y=0;y<matrix.Cols();y++)
        {
        this->Read<double>(f,matrix(x,y));
        }
      }
    }
  void Write(std::ofstream &f,const DataClassMeanType &vec)
    {
    for(int y=0;y<vec.Size();y++)
      {
      this->Write<double>(f,vec[y]);
      }
    }
  void Read(std::ifstream &f,DataClassMeanType &vec)
    {
    for(int y=0;y<vec.Size();y++)
      {
      this->Read<double>(f,vec[y]);
      }
    }
  void Write(std::ofstream &f,const TrainingVectorType &vec)
    {
    for(int y=0;y<vec.Size();y++)
      {
      this->Write<double>(f,vec[y]);
      }
    }
  void Read(std::ifstream &f,TrainingVectorType &vec)
    {
    for(int y=0;y<vec.Size();y++)
      {
      this->Read<double>(f,vec[y]);
      }
    }
  /* 
  ** The unscaled lesion centroid, the scaled lesion samples mean & covariance,
  ** and the scaled non-lesion mean and covariance
  */
  double m_distanceThreshold;
  LesionCentroidType m_lesionCentroid;
  DataClassMeanType m_lesionDataMean;
  DataClassMeanType m_nonLesionDataMean;
  DataCovarianceType m_lesionDataCovariance;
  DataCovarianceType m_nonLesionDataCovariance; 
  TrainingVectorType m_trainingMins;
  TrainingVectorType m_trainingMaxs;
};

#endif // lesionSegmentationModel_h
