/*=========================================================================

Program:   Insight Segmentation & Registration Toolkit
Module:    $RCSfile: itkHammerRegistrationFunction.txx,v $
Language:  C++
Date:      $Date: 2009/01/15 15:06:35 $
Version:   $Revision: 1.1 $

Copyright (c) Insight Software Consortium. All rights reserved.
See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for
details.

This program is developed under NIH NCBC collaboration grant
R01 EB006733, "Development and Dissemination of Robust Brain MRI
Measurement Tools". 

This software is distributed WITHOUT ANY WARRANTY; without even 
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the above copyright notices for more information.


=========================================================================*/
#ifndef __itkHammerRegistrationFunction_txx
#define __itkHammerRegistrationFunction_txx

#include "itkHammerRegistrationFunction.h"
#include "itkExceptionObject.h"
#include "vnl/vnl_math.h"
#include "itkImageRegionIterator.h"

namespace itk
{

/**
 * Default constructor
 */
template <class TFixedImage, class TMovingImage, class TDeformationField>
HammerRegistrationFunction<TFixedImage,TMovingImage,TDeformationField>
::HammerRegistrationFunction()
{

  RadiusType r;
  for(unsigned int j = 0; j < ImageDimension; j++ )
  {
    r[j] = 0;
  }
  this->SetRadius(r);

  //set up the default parameter once the registration function is intialized
  //TODO:

  for(int l=0; l<MAX_LEVEL; l++)
  {
    m_GuassianAtLevelSigma[0][l] = 0 ;
  }

  for(int s=1; s<MAX_SIGMA; s++)
  {
    for(int l=0; l<MAX_LEVEL; l++)
    {
      m_GuassianAtLevelSigma[s][l] = exp(-l*l/(2.0*s*s)) ;
    }
  }

  this->m_PointMatchingNeighborhood.empty();
  this->m_SubvolumeNeighborhood.empty();
  this->m_ModelDrivingPoint.empty();
  this->m_SubjectDrivingPoint.empty();
  this->m_ModelDrivingPointDisplacement.empty();
  this->m_InverseDisplacement.empty();
  this->m_SearchNeighbor.empty();

  this->m_GlobalAffineTransform = AffineTransformType::New();
  this->m_GlobalAffineTransform->SetIdentity();

  this->m_TimeStep = 0.0;
}


/**
 * Standard "PrintSelf" method.
 */
template <class TFixedImage, class TMovingImage, class TDeformationField>
void
HammerRegistrationFunction<TFixedImage,TMovingImage,TDeformationField>
::PrintSelf(std::ostream& os, Indent indent) const
{
  Superclass::PrintSelf(os, indent);

  os << indent << "Metric: ";
  os << m_Metric << std::endl;
  os << indent << "NumberOfPixelsProcessed: ";
  os << m_NumberOfPixelsProcessed << std::endl;
}


/**
 * Update the metric and release the per-thread-global data.
 */

template <class TFixedImage, class TMovingImage, class TDeformationField>
void
HammerRegistrationFunction<TFixedImage,TMovingImage,TDeformationField>
::ReleaseGlobalDataPointer( void *gd ) const
{
  GlobalDataStruct * globalData = (GlobalDataStruct *) gd;

  m_MetricCalculationLock.Lock();
  m_SumOfSquaredDifference += globalData->m_SumOfSquaredDifference;
  m_NumberOfPixelsProcessed += globalData->m_NumberOfPixelsProcessed;
  m_SumOfSquaredChange += globalData->m_SumOfSquaredChange;
  if ( m_NumberOfPixelsProcessed )
    {
    m_Metric = m_SumOfSquaredDifference / 
      static_cast<double>( m_NumberOfPixelsProcessed ); 
    m_RMSChange = vcl_sqrt( m_SumOfSquaredChange / 
                            static_cast<double>( m_NumberOfPixelsProcessed ) ); 
    }
  m_MetricCalculationLock.Unlock();

  delete globalData;
}


/**
 * Set the function state values before each iteration
 */
template <class TFixedImage, class TMovingImage, class TDeformationField>
void
HammerRegistrationFunction<TFixedImage,TMovingImage,TDeformationField>
::InitializeIteration()
{
  if( !this->GetMovingImage() || !this->GetFixedImage() )
    {
    itkExceptionMacro( << "MovingImage, FixedImage and/or Interpolator not set" );
    }

  // cache fixed image information
  SpacingType fixedImageSpacing    = this->GetFixedImage()->GetSpacing();
  RadiusType radius = this->GetRadius();

  // find the real radius in the widest dimension of the search area
  float physicalRadius = 0;
  for (int k = 0; k < InputImageDimension; k++)
    {
    float d = radius[k]*fixedImageSpacing[k];
    if (physicalRadius < d)
      {
      physicalRadius = d;
      }
    }

  // compute spherical neighborhood for geometrical matching
  typename FixedImageType::Pointer dummyImage = FixedImageType::New();
  RegionType dummyRegion;
  PointType dummyOrigin;
  IndexType dummyStart;
  SizeType dummySize;

  dummyImage->SetSpacing( fixedImageSpacing );
  for (int k = 0; k < InputImageDimension; k++)
    {
    dummySize[k] = radius[k]+radius[k]+1;
    dummyStart[k] = -radius[k];
    dummyOrigin[k] = 0;
    }
  dummyRegion.SetIndex( dummyStart );
  dummyRegion.SetSize( dummySize );
  dummyImage->SetRegions( dummyRegion );
  dummyImage->SetOrigin( dummyOrigin );

  float radiusSqr = physicalRadius*physicalRadius;

  itk::ImageRegionIteratorWithIndex<FixedImageType> it( dummyImage, dummyRegion );
  for (it.GoToBegin(); !it.IsAtEnd(); ++it)
    {
    IndexType dummyIdx = it.GetIndex();
    PointType point;
    dummyImage->TransformIndexToPhysicalPoint( dummyIdx, point );
    float d = 0;
    for (int k = 0; k < InputImageDimension; k++)
      {
      d += point[k]*point[k];
      }
    if (d > radiusSqr)
      {
      continue;
      }
    else
      {
      NeighborOffsetType offset;
      for (int k = 0; k < InputImageDimension; k++)
        {
        offset[k] = dummyIdx[k];
        }
      this->m_OffsetInSphericalNeighborhood.push_back(offset);
      }
    }
   
}


/**
 * Compute update at a specify neighbourhood
 */
template <class TFixedImage, class TMovingImage, class TDeformationField>
typename HammerRegistrationFunction<TFixedImage,TMovingImage,TDeformationField>
::PixelType
HammerRegistrationFunction<TFixedImage,TMovingImage,TDeformationField>
::ComputeUpdate(const NeighborhoodType &it, void * gd,
                const FloatOffsetType& itkNotUsed(offset))
{
  // Get fixed image related information
  // Note: no need to check the index is within
  // fixed image buffer. This is done by the external filter.
  const IndexType index = it.GetIndex();  

  PixelType update;
  return update;

}

/**
 * Create the neighborhood in point matching
**/
template <class TFixedImage, class TMovingImage, class TDeformationField>
void 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::CreatePointMatchingNeighbor(int Radius)
{
  CalculateNeighborhoodbyIncreasingRadius(&m_PointMatchingNeighborhood, Radius);
}

/**
 * Create the neighborhood in subvolumn deform
*/
template <class TFixedImage, class TMovingImage, class TDeformationField>
void 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::CreateSubvolumnNeighbor(int Radius)
{
  CalculateNeighborhoodbyIncreasingRadius(&m_SubvolumeNeighborhood, Radius);
}

/**
 * Create the neighbor from center to outside
**/
template <class TFixedImage, class TMovingImage, class TDeformationField>
void
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::CalculateNeighborhoodbyIncreasingRadius(VectorArrayType &Neighbor, int Radius)  throw (InvalidRequestedRegionError) 
{
  // pad input requested region by 1, so the simple edge detection
  // works on the entire image domain
  typename InputImageType::SpacingType inputSpacing = m_SubjectImage->GetSpacing();

  NeighborOffsetType dummyIdx;
  int x, y, z;
  for(int r=1;r<=Radius;r++)
  {
    for(z=-r; z<=r; z+=2*r)
    {  
      for(x=-r; x<=r; x++)
      {
        for(y=-r; y<=r; y++)
        {
          dummyIdx[0] = x/inputSpacing[0];
          dummyIdx[1] = y/inputSpacing[1];
          dummyIdx[2] = z/inputSpacing[2];
          Neighbor.push_back(dummyIdx);
        }
      }
    }

    for(x=-r; x<=r; x+=2*r)
    {
      for(z=-r+1; z<=r-1; z++) 
      {
        for(y=-r; y<=r; y++)
        {
          dummyIdx[0] = x/inputSpacing[0];
          dummyIdx[1] = y/inputSpacing[1];
          dummyIdx[2] = z/inputSpacing[2];
          Neighbor.push_back(dummyIdx);
        }
      }
    }

    for(y=-r; y<=r; y+=2*r)
    {
      for(z=-r+1; z<=r-1; z++) 
      {
        for(x=-r+1; x<=r-1; x++) 
        {
          dummyIdx[0] = x/inputSpacing[0];
          dummyIdx[1] = y/inputSpacing[1];
          dummyIdx[2] = z/inputSpacing[2];
          Neighbor.push_back(dummyIdx);
        }
      }
    }

  }   // end for(int r=1;r<=Radius;r++)

}  // end CalculateNeighborhoodbyIncreasingRadius

/**
*  Create the search neighborhood
**/
template <class TFixedImage, class TMovingImage, class TDeformationField>
void
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::CreateSearchNeighbor(int Radius)
{  
  SpacingType inputSpacing = m_FixedImage->GetSpacing();
  Size<InputImageDimension> sphereRadius;
  for (int i = 0; i < InputImageDimension; ++i)
  {
    //sphereRadius[i] = static_cast<unsigned long>( this->m_Scale/inputSpacing[i] );
    sphereRadius[i] = static_cast<unsigned long>(Radius);
  }

  // compute spherical neighborhood for geometrical attribute
  // computation
  FixedImagePointer dummyImage = FixedImageType::New();
  RegionType dummyRegion;
  PointType dummyOrigin;
  IndexType dummyStart;
  SizeType dummySize;

  dummyImage->SetSpacing( inputSpacing );
  for (int k = 0; k < InputImageDimension; k++)
  {
    dummySize[k] = sphereRadius[k]+sphereRadius[k]+1;
    dummyStart[k] = -sphereRadius[k];
    dummyOrigin[k] = 0;
  }
  dummyRegion.SetIndex( dummyStart );
  dummyRegion.SetSize( dummySize );
  dummyImage->SetRegions( dummyRegion );
  dummyImage->SetOrigin( dummyOrigin );
  
  itk::ImageRegionIteratorWithIndex<InputImageType> it( dummyImage, dummyRegion );
  for (it.GoToBegin(); !it.IsAtEnd(); ++it)
  {
    typename InputImageType::IndexType dummyIdx = it.GetIndex();
    typename InputImageType::PointType point;
    //dummyImage->TransformIndexToPhysicalPoint( dummyIdx, point );
    NeighborOffsetType offset;      
    for (int k = 0; k < InputImageDimension; k++)
    {
      offset[k] = dummyIdx[k];
    }
    this->m_SearchNeighbor.push_back(offset);                
  }  
}


/**
 * the metric function
**/
template <class TFixedImage, class TMovingImage, class TDeformationField>
float
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::SimilarityBetweenTwoImgAttribute(AttributeVectorType Template_Feature, AttributeVectorType Subject_Feature) const
{
  return Template_Feature.ComputeSimilarity(Subject_Feature);
}


/**
 * compute the magnitude of vector
**/
template <class TFixedImage, class TMovingImage, class TDeformationField>
float 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::ComputeVectorMagnitude(DeformationVectorType &Deform_Vector)
{
  return Deform_Vector.GetNorm();
}

/**
 * the core function which determine the correspondence on model driving voxel array
*/
template <class TFixedImage, class TMovingImage, class TDeformationField>
float 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::DetermineCorrespondenceOnOneDrivingVoxel(const int &DrivingPointIndex, DeformationVectorType &DeformationUpdate) const
{
  int s;
  float PointSimilarity, NeighborhoodSimilarity, MaxSimilarityDegree;
  float DisplacementMagnitude;
  float RealSize;

  AttributeVectorType TemplateFeature, SubjectFeature;
  DeformationVectorType DeformationOnParticularPoint, TentativeWarp;
  DeformationUpdate.Fill( 0 );


  IndexType ImageIndex;
  if(!m_FixedImage->TransformPhysicalPointtoIndex(m_PickedPointsOnFixedImage[DrivingPointIndex], ImageIndex))
    printf("Model driving point is out of the image boundary.\n");


  //Step 1: check whether the the input point has the inverse force
  Displacement_Magnitude = ComputeVectorMagnitude(m_InverseDisplacement[DrivingPointIndex]);
  if(Displacement_Magnitude>0 && m_IterationRatio<=ITER_THRD)
  {    
    DeformationVectorType CurrentDeformation, PreviouseDeformation;
    CurrentDeformation = m_DeformationField->GetPixel(ImageIndex);
    PreviousDeformation = m_PreviouseDeformationField->GetPixel(ImageIndex);
    for(s=0;s<ImageDimension;s++)
    {
      DeformationUpdate[s] = (PreviousDeformation[s] - CurrentDeformation[s] + m_InverseDisplacement[DrivingPointIndex][s])*(1.0+(1.0 + m_IterationRatio))/2.0;
    }
    DisplacementMagnitude = ComputeVectormagnitude(DeformationUpdate);
    if(DisplacementMagnitude > (m_SubvolumeRadius>>1))
    {
      for(s=0;s<ImageDimension;s++)
        DeformationUpdate[s] = DeformationUpdate[s]/DisplacementMagnitude*(m_SubvolumeRadius>>1);
    }
    return m_SubvolumnSimilarityThreshold + 0.1;
  }
          
  //Step 2: find the correspondence of particular point
  MaxSimilarityDegree = 0;
  TemplateFeature = m_FixedAttributeImage->GetPixel(ImageIndex);
  DeformationOnParticularPoint = m_DeformationFiled->GetPixel(ImageIndex);
  for (int SearchItor = 0; SearchItor < m_SearchNeighbor.size(); SearchItor++)
  {
    //Step 2.1: compare the similarity between two singel points
    NeighborOffsetType offset = m_SearchNeighbor[SearchItor];
    IndexType SubjIdx;
    for (s=0; s<InputImageDimension; s++)
      {
        SubjIdx[s] = ImageIndex[s] + DeformationOnParticularPoint[s] + offset[s];
      }
    if(!m_FixedImage->GetLargestPossibleRegion().IsInside(SubjIdx))
      continue;
    SubjectFeature = m_MovingAttributeImage->GetPixel(SubjIdx);
    PointSimilarity = SimilarityBetweenTwoImgAttribute(TemplateFeature, SubjectFeature);
    //Step 2.2: compare the similarity between two neighborhood
    if(Point_Similarity>m_PointMatchingThreshold || (m_IsBigVN== true && TemplateFeature[EDGE]>0 && SubjectFeature[EDGE]>0))
    {      
      for(s=0;s<InputImageDimension;s++)
      { 
        TentativeWarp[s] = offset[s];
      }
      NeighborhoodSimilarity = SubVolumnMatching(ImageIndex, TentativeWarp, m_SubvolumeNeighborhood, m_NeighborhoodStep);
      if(NeighborhoodSimilarity>MaxSimilarityDegree)
      {
        for(s=0;s<InputImageDimension;s++)
          DeformUpdate[s] = TentativeWarp[s];
        MaxSimilarityDegree = NeighborhoodSimilarity;
      }

    }
  }
  return ( MaxSimilarityDegree );
}

template <class TFixedImage, class TMovingImage, class TDeformationField>
float 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::SubVolumnMatching(IndexType &ImageIndex, DeformationVectorType TentativeWarp, VectorArrayType CertainNeighborhood, int NeighborhoodStep) const
{
  float NeighborhoodSimilarity, CurrentSimilarity, RealSize, DisplacementMagnitude;
  AttributeVectorType SubvolumnTemplateFeature, SubvolumnSubjectFeature;
  IndexType MdlIdx, SubjIdx;
  DeformationVectorType DeformationOnParticularPoint;

  NeighborhoodSimilarity = 0;
  RealSize = 0;
  for(int SubvolumnIter=0; SubvolumnIter < CertainNeighborhood.size(); SubvolumnIter += NeighborhoodStep)
  {
    DisplacementMagnitude = 0;
    for(int s=0;s<InputImageDimension;s++)
    {
      MdlIdx[s] = ImageIndex[s] + CertainNeighborhood[SubvolumnIter][s];
      DisplacementMagnitude += abs(CertainNeighborhood[SubvolumnIter][s]);
    }
    DisplacementMagnitude /= InputImageDimension;
    if(!m_FixedAttributeImage->GetLargestPossibleRegion().IsInside(MdlIdx))
    {
      continue;
    }

    SubvolumnTemplateFeature = m_FixedAttributeImage->GetPixel(MdlIdx);        
    DeformationOnParticularPoint = this->m_DeformationField->GetPixel(MdlIdx);
                                             
    for(int s=0;s<InputImageDimension;s++)
    {
      SubjIdx[s] = MdlIdx[s] + DeformationOnParticularPoint[s] + TentativeWarp[s]*m_GuassianAtLevelSigma[(int)DisplacementMagnitude][m_PointMatchRadius];
    }

    if(!m_MovingAttributeImage->GetLargestPossibleRegion().IsInside(SubjIdx))
    {
      continue;
    }

    SubvolumnSubjectFeature = m_MovingAttributeImage->GetPixel(SubjIdx);

    CurrentSimilarity = SimilarityBetweenTwoImgAttribute(SubvolumnSubjectFeature, SubvolumnTemplateFeature);
    if(CurrentSimilarity < 0.6 && m_IsYoungBrain == true && SubvolumnTemplateFeature[EDGE] == 0)
    {
      CurrentSimilarity = 0;
    }

    RealSize += 1.0;
    if( SubvolumnTemplateFeature[EDGE]>0 ) /* June 6, 2001*/
    {
      CurrentSimilarity *= 1.2 ;
      RealSize += (1.2-1.0) ;
    }
    NeighborhoodSimilarity += CurrentSimilarity;
  }
  if(RealSize>0)
  {
    NeighborhoodSimilarity /= RealSize;
  }

  return NeighborhoodSimilarity;  
}

template <class TFixedImage, class TMovingImage, class TDeformationField>
void 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::FindingInverseForceFromSubject() const
{
  typedef itk::Image<int, InputImageDimension> ModelDrivingPointImageType;
  ModelDrivingPointImageType::Pointer ModelDrivingPointImage = ModelDrivingPointImageType::New();
  
  RegionType dummyRegion;
  int PointID;
  int s, SearchItor;
  IndexType SubjIndex, MdlIndex;
  AttributeVectorType TemplateFeature, SubjectFeature;
  DeformationVectorType TentativeWarp;
  float MaxSimilarityDegree;
  int MdlPickedPointID;
  int MdlPickedPointNum;
  std::vector<float> Multiple;
  
  dummyRegion = m_FixedImage->GetRequestedRegion();
  ModelDrivingPointImage->SetRegions(dummyRegion);
  ModelDrivingPointImage->Allocate();

  typedef itk::ImageRegionIterator<ModelDrivingPointImageType> IteratorType;
  IteratorType Itor(ModelDrivingPointImage, dummyRegion);
  for(Itor.GoToBegin();!Itor.IsAtEnd();++Itor)
  {
    Itor.Set(-1);
  }

  MdlPickedPointNum = m_PickedPointsOnFixedImage.size();
  for(PointID=0;PointID<MdlPickedPointNum;PointID++)
  {
    ModelDrivingPointImage->SetPixel(m_PickedPointsOnFixedImage[PointID], PointID);
    Multiple.push_back(0);
  }

  for(PointID=0;PointID<m_PickedPointsOnMovingImage.size();PointID++)
  {    
    MaxSimilarityDegree = 0;
    
    SubjectFeature = m_MovingAttributeImage->GetPixel(m_PickedPointsOnMovingImage[PointID]);
    for (SearchItor = 0; SearchItor < m_SearchNeighbor.size(); SearchItor++)
    {
      for(s=0;s<InputImageDimension;s++)
      {
        MdlIndex[s] = SubjIndex[s] + m_SearchNeighbor[SearchItor][s];
      }
      MdlPickedPointID = ModelDrivingPointImage->GetPixel(MdlIndex);
      if( MdlPickedPointID >= 0)
      {
        TemplateFeature = m_FixedAttributeImage->GetPixel(MdlIndex);
        float PointSimilarity = SimilarityBetweenTwoImgAttribute(TemplateFeature, SubjectFeature);
        if(PointSimilarity>m_PointMatchingThreshold)
        {
          for(s=0;s<InputImageDimension;s++)
          {
            TentativeWarp[s] = -m_SearchNeighbor[SearchItor][s];
          }

          float SubvolumnSimilarity = SubVolumnMatching(MdlIndex, TentativeWarp, m_SubvolumeNeighborhood, m_NeighborhoodStep);
          if(SubvolumnSimilarity>MaxSimilarityDegree)
          {
            MaxSimilarityDegree = SubvolumnSimilarity;
            for(s=0;s<InputImageDimension;s++)
            {
              m_InverseDisplacement[MdlPickedPointID][s] += TentativeWarp[s];
            }

            Multiple[MdlPickedPointID] += 1.0;
          }
        }
      }
    }    
  }

  for(PointID=0;PointID<MdlPickedPointNum;PointID++)
  {
    if(Multiple[PointID]>0)
    {
      for(s=0;s<InputImageDimension;s++)
      {
        m_InverseDisplacement[PointID][s] /= Multiple[PointID];
      }
    }
  }
}

template <class TFixedImage, class TMovingImage, class TDeformationField>
void 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::DisseminateDeformation(const int &DrivingPointIndex, DeformationVectorType TentativeWarp, VectorArrayType CertainNeighborhood, int NeighborhoodSize, int GaussianSigma)
{
  int PointID, s;
  float CenterRate, Delta;
  IndexType CenterPointIndex, StudyPointIndex, SurroundingPointIndex;
  float Distance, Weight;
  DeformationVectorType MeanDeform, DeformationOnCertainPoint, DeformationOnCenterPoint, DeformUpdate;

  CenterRate = m_SmoothFactor; 
  Delta = 0.005*m_IterationRatio ; /* June 6, 2001*/
  if(!m_FixedImage->TransformPhysicalPointtoIndex(m_PickedPointsOnFixedImage[DrivingPointIndex], CenterPointIndex))
  {
    printf("Model driving point is out of the image boundary.\n");
  }

  for(PointID=0;PointID<NeighborhoodSize;PointID++)
  {    
    Distance = 0;
    for(s=0;s<InputImageDimension;s++)
    {
      MeanDeform[s] = 0;
      StudyPointIndex[s] = CenterPointIndex[s] + CertainNeighborhood[s];
      Distance += abs(CentainNeighborhood[s]);
    }
    Distance /= InputImageDimension;
    if(!m_FixedImage->GetLargestPossibleRegion().IsInside(StudyPointIndex))
    {
      continue;
    }

    Weight = m_GuassianAtLevelSigma[(int)Distance][GaussianSigma];
    float TempNum = 0;
    for(int n=0;n<27;n++) //need to find better way 
    {
      for(s=0;s<InputImageDimension;s++)
      {
        SurroundingPointIndex[s] = StudyPointIndex[s] + CertainNeighborhood[s];
      }

      if(!m_FixedImage->GetLargestPossibleRegion().IsInside(SurroundingPointIndex))
      {
        continue;
      }

      DeformationOnCertainPoint = m_DeformationField->GetPixel(SurroundingPointIndex);
      for(s=0;s<InputImageDimension;s++)
      {
        MeanDeform[s] += DeformationOnCertainPoint[s];
      }

      TempNum += 1.0;
    }
    if(TempNum>0)
    {
      for(s=0;s<InputImageDimension;s++)
        MeanDeform[s] /= TempNum;
    }
    DeformationOnCenterPoint = m_DeformationField->GetPixel(StudyPoint);
    for(s=0;s<InputImageDimension;s++)
    {
      MeanDeform[s] = (MeanDeform[s]-DeformationOnCenterPoint[s])*CenterRate;
      DeformUpdate[s] = (MeanDeform[s]+TentativeWarp[s]*Weight)*(m_DeformRate+Delta);
    }
    m_DeformationField->SetPixel(StudyPoint, DeformUpdate);
  }
}

template <class TFixedImage, class TMovingImage, class TDeformationField>
void 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::IdentifyDrivingVoxelsInFixedImage( )
{
  IdentifyDrivingVoxels( m_FixedAttributeImage, m_PickedPointsOnFixedImage );
}

template <class TFixedImage, class TMovingImage, class TDeformationField>
void 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::IdentifyDrivingVoxelsInMovingImage( )
{
  IdentifyDrivingVoxels( m_MovingAttributeImage, m_PickedPointsOnMovingImage );
}

template <class TFixedImage, class TMovingImage, class TDeformationField>
int 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::IdentifyDrivingVoxels( ImageAttributePointerType avPointer, std::vector<PointType> & drivingVoxels )
{
  // Generate a binary image that indicates voxels already been selected
  FixedImageType::Pointer mask = FixedImageType::New();
  mask->CopyInformation( m_FixedAttributeImage );
  mask->SetRegions( mask->GetLargestPossibleRegion() );
  mask->Allocate();
  mask->FillBuffer( 0 );

  unsigned int nVoxels = drivingVoxels.size();
  for (unsigned int k = 0; k < nVoxels; k++)
  {
    FixedImageType::IndexType idx = drivingVoxels[k];
    mask->SetPixel( idx, 1 );
  }

  itk::ImageRegionIteratorWithIndex<ImageAttributeType> 
    itAV( avPointer, avPointer->GetLargestPossibleRegion() );

  itk::ImageRegionIteratorWithIndex<FixedImageType> 
    itMask( mask, mask->GetLargestPossibleRegion() );

  for ( itAV.GoToBegin(), itMask.GoToBegin(); !itAV.IsAtEnd(); ++itAV, ++itMask )
  {
    if ( itMask.Get() == 1 )   // voxel already a driving voxel
    {
      continue;
    }
    AttributeVectorType a = itAV.Get();
    if ( a.IsQualifiedDrivingVoxel( m_DrivingVoxelQualification ) )
    {
      drivingVoxels.push_back( itAV.GetIndex() );
      nVoxels ++;
    }
  }

  return nVoxels;
}


/** Gaussian smoothing of the deformationfield */
template <class TFixedImage, class TMovingImage, class TDeformationField>
void 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::SmoothDeformationField( )
{
  typedef itk::SmoothingRecursiveGaussianImageFilter
    <DeformationFieldType, DeformationFieldType> SmootherType;
  SmootherType::Pointer smoother = SmootherType::New();

  smoother->SetInput( this->m_DeformationField );
  smoother->SetSigma( m_DeformationFieldSigma );

  smoother->Update();

  // copy things back
  itk::ImageRegionIterator<DeformationFieldType>
    it0( this->m_DeformationField, this->m_DeformationField->GetLargestPossibleRegion() );
  itk::ImageRegionConstIterator<DeformationFieldType>
    it1( smoother->GetOutput(), smoother->GetOutput()->GetLargestPossibleRegion() );

  for (it0.GoToBegin(), it1.GoToBegin(); !it0.IsAtEnd(); ++it0, ++it1)
  {
    it0.Set( it1.Get() );
  }

  return;
}



/** fit the deformation field with a global affine transform */
template <class TFixedImage, class TMovingImage, class TDeformationField>
void 
HammerRegistrationFunction<TFixedImage, TMovingImage, TDeformationField>
::FitGlobalAffineTransform( )
{
}


} // end namespace itk

#endif
