/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: itkWarpImageFilter.txx,v $
  Language:  C++
  Date:      $Date: 2008-10-07 17:31:02 $
  Version:   $Revision: 1.27 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkHammerWarpImageFilter_txx
#define __itkHammerWarpImageFilter_txx
#include "itkHammerWarpImageFilter.h"

#include "itkImageRegionIterator.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkImageRegionConstIterator.h"
#include "itkImageRegionConstIteratorWithIndex.h"
#include "itkNumericTraits.h"
#include "itkProgressReporter.h"

namespace itk
{

/**
 * Default constructor.
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
HammerWarpImageFilter<TInputImage,TOutputImage,TDeformationField>
::HammerWarpImageFilter()
{
  // Setup the number of required inputs
  this->SetNumberOfRequiredInputs( 2 );  
  
  // Setup default values
  m_OutputSpacing.Fill( 1.0 );
  m_OutputOrigin.Fill( 0.0 );
  m_OutputDirection.SetIdentity();

  m_EdgePaddingValue = NumericTraits<PixelType>::Zero;

  // Setup default interpolator
  typename DefaultInterpolatorType::Pointer interp =
    DefaultInterpolatorType::New();

  m_Interpolator = 
    static_cast<InterpolatorType*>( interp.GetPointer() );

}

/**
 * Standard PrintSelf method.
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
HammerWarpImageFilter<TInputImage,TOutputImage,TDeformationField>
::PrintSelf(std::ostream& os, Indent indent) const
{

  Superclass::PrintSelf(os, indent);

  os << indent << "OutputSpacing: " << m_OutputSpacing << std::endl;
  os << indent << "OutputOrigin: " << m_OutputOrigin << std::endl;
  os << indent << "OutputDirection: " << m_OutputDirection << std::endl;
  os << indent << "EdgePaddingValue: "
     << static_cast<typename NumericTraits<PixelType>::PrintType>(m_EdgePaddingValue)
     << std::endl;
  os << indent << "Interpolator: " << m_Interpolator.GetPointer() << std::endl;
  
}


/**
 * Set the output image spacing.
 *
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
HammerWarpImageFilter<TInputImage,TOutputImage,TDeformationField>
::SetOutputSpacing(
  const double* spacing)
{
  SpacingType s(spacing);
  this->SetOutputSpacing( s );
}


/**
 * Set the output image origin.
 *
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
HammerWarpImageFilter<TInputImage,TOutputImage,TDeformationField>
::SetOutputOrigin(
  const double* origin)
{
  PointType p(origin);
  this->SetOutputOrigin(p);
}

/**
 * Set deformation field as Inputs[1] for this ProcessObject.
 *
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
HammerWarpImageFilter<TInputImage,TOutputImage,TDeformationField>
::SetDeformationField(
 const DeformationFieldType * field )
{
  // const cast is needed because the pipeline is not const-correct.
  DeformationFieldType * input =  
       const_cast< DeformationFieldType * >( field );
  this->ProcessObject::SetNthInput( 1, input );
}


/**
 * Return a pointer to the deformation field.
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
typename HammerWarpImageFilter<TInputImage,TOutputImage,TDeformationField>
::DeformationFieldType *
HammerWarpImageFilter<TInputImage,TOutputImage,TDeformationField>
::GetDeformationField(void)
{
  return static_cast<DeformationFieldType *>
    ( this->ProcessObject::GetInput( 1 ));
}



/**
 * Compute the output for the region specified by outputRegionForThread.
 */
template <class TInputImage,class TOutputImage,class TDeformationField>
void
HammerWarpImageFilter<TInputImage,TOutputImage,TDeformationField>
::GenerateData()
{
	unsigned int s;
	InputImageConstPointer inputPtr = this->GetInput();
	this->AllocateOutputs();
	m_OutputImage = this->GetOutput(0);
	DeformationFieldPointer fieldPtr = this->GetDeformationField();

	IndexType index, TargetIndex;
	PointType point;
	DisplacementType displacement;
	PixelType interpvalue;
	InputImageType::RegionType dummyRegion = inputPtr->GetLargestPossibleRegion();
	SizeType dummySize = dummyRegion.GetSize();

	itk::ImageRegionConstIterator<DeformationFieldType> DeformFld_Iter(fieldPtr, fieldPtr->GetLargestPossibleRegion());
// 	itk::ImageRegionConstIterator<InputImageType> InputImage_Iter(inputPtr, inputPtr->GetLargestPossibleRegion());
 	
	DeformFld_Iter.GoToBegin();
// 	InputImage_Iter.GoToBegin();
// 	OutputImage_Iter.GoToBegin();
	//OutputImagePointer tempImage = OutputImageType::New();
// 	m_OutputImage->CopyInformation(m_OutputImage);
// 	m_OutputImage->SetRegions(m_OutputImage->GetLargestPossibleRegion());
// 	m_OutputImage->Allocate();
	itk::ImageRegionIterator<OutputImageType> OutputImage_Iter(m_OutputImage, m_OutputImage->GetLargestPossibleRegion());
	OutputImage_Iter.GoToBegin();
	for(unsigned int k=0;k<dummySize[2];k++)
		for(unsigned int i=0;i<dummySize[0];i++)
			for(unsigned int j=0;j<dummySize[1];j++)
			{
				displacement = DeformFld_Iter.Get();
				index = DeformFld_Iter.GetIndex();
				for(s=0;s<ImageDimension;s++)
				{
					TargetIndex[s] = index[s] + (int)(displacement[s]+0.5);
					if(TargetIndex[s]<0)
						TargetIndex[s] = 0;
					if(TargetIndex[s]>=dummySize[s])
						TargetIndex[s] = dummySize[s]-1;
				}
				interpvalue = inputPtr->GetPixel(TargetIndex);
				OutputImage_Iter.Set(interpvalue);
				++OutputImage_Iter;
				++DeformFld_Iter;				
			}
}

template <class TInputImage,class TOutputImage,class TDeformationField>
void
HammerWarpImageFilter<TInputImage,TOutputImage,TDeformationField>
::GenerateInputRequestedRegion()
{

  // call the superclass's implementation
  Superclass::GenerateInputRequestedRegion();

  // request the largest possible region for the input image
  InputImagePointer inputPtr = 
    const_cast< InputImageType * >( this->GetInput() );

  if( inputPtr )
    {
    inputPtr->SetRequestedRegionToLargestPossibleRegion();
    }

  // just propagate up the output requested region for the 
  // deformation field.
  DeformationFieldPointer fieldPtr = this->GetDeformationField();
  OutputImagePointer outputPtr = this->GetOutput();
  if( fieldPtr )
    {
    fieldPtr->SetRequestedRegion( outputPtr->GetRequestedRegion() );
    }

}


template <class TInputImage,class TOutputImage,class TDeformationField>
void
HammerWarpImageFilter<TInputImage,TOutputImage,TDeformationField>
::GenerateOutputInformation()
{
  // call the superclass's implementation of this method
  Superclass::GenerateOutputInformation();

  OutputImagePointer outputPtr = this->GetOutput();

  outputPtr->SetSpacing( m_OutputSpacing );
  outputPtr->SetOrigin( m_OutputOrigin );
  outputPtr->SetDirection( m_OutputDirection );

  DeformationFieldPointer fieldPtr = this->GetDeformationField();
  if( fieldPtr )
    {
    outputPtr->SetLargestPossibleRegion( fieldPtr->
                                         GetLargestPossibleRegion() );
    }

}


} // end namespace itk

#endif
