/*=========================================================================

  Program:   HAMMER Deformable Registration
  Module:    $RCSfile: itkHammerDeformationImageFunction.txx,v $
  Language:  C++
  Date:      $Date: 2006/02/06 22:01:57 $
  Version:   $Revision: 1.13 $

=========================================================================*/
#ifndef __itkHammerDeformationImageFunction_txx
#define __itkHammerDeformationImageFunction_txx

#include "itkHammerDeformationImageFunction.h"
#include "itkNumericTraits.h"
#include "itkConstNeighborhoodIterator.h"

namespace itk
{

/**
   * Constructor
   */
template <class TInputImage, class TCoordRep>
HammerDeformationImageFunction<TInputImage,TCoordRep>
::HammerDeformationImageFunction()
{
	int s, l;
	m_NeighborhoodRadius = 5;
	m_VoxelSimilarityThreshold = 0.6;
	m_NeighborhoodSimilarityThreshold = 0.4;
	m_Sigma = 1;
	m_TemplateDrivingPointLocationReady = false;
	m_SubvolumnRadius = 7;
	m_NeighborhoodStep = 3;
	for(l=0; l<MAX_LEVEL; l++)
		GuassianAtLevelSigma[0][l] = 0 ;

	for(s=1; s<MAX_SIGMA; s++)
		for(l=0; l<MAX_LEVEL; l++)
			GuassianAtLevelSigma[s][l] = exp(-l*l/(2.0*s*s)) ;
	this->m_OffsetInSphericalNeighborhood.empty();
}


/**
   *
   */
template <class TInputImage, class TCoordRep>
void
HammerDeformationImageFunction<TInputImage,TCoordRep>
::PrintSelf(std::ostream& os, Indent indent) const
{
  this->Superclass::PrintSelf(os,indent);
  os << indent << "NeighborhoodRadius: "  << m_NeighborhoodRadius << std::endl;
}

/**
*
*/

template <class TInputImage, class TCoordRep>
float
HammerDeformationImageFunction<TInputImage, TCoordRep>
::SimilarityBetweenTwoImgAttribute(InputAttributeType Template_Feature, InputAttributeType Subject_Feature) const
{
	float Similarity_Degree;
	Similarity_Degree = 1.0;
	for(int k=0;k<5;k++)
	{
		Similarity_Degree *= (1.0-abs(Template_Feature[k]-Subject_Feature[k])/255.0);
	}
	return Similarity_Degree;
}

template <class TInputImage, class TCoordRep>
void
HammerDeformationImageFunction<TInputImage, TCoordRep>
::GenerateSearchNeighborhood() throw (InvalidRequestedRegionError) 
{
	// pad input requested region by 1, so the simple edge detection
	// works on the entire image domain
	unsigned long radius = 1;
	typename InputImageType::SpacingType inputSpacing = m_SubjectImage->GetSpacing();
	Size<InputImageDimension> sphereRadius;
	for (int i = 0; i < InputImageDimension; ++i)
	{
		sphereRadius[i] = static_cast<unsigned long>( this->m_Scale/inputSpacing[i] );
		if (sphereRadius[i]>radius)
		{
			radius = sphereRadius[i];
		}
	}

	// compute spherical neighborhood for geometrical attribute
	// computation
	typename InputImageType::Pointer dummyImage = InputImageType::New();
	typename InputImageType::RegionType dummyRegion;
	typename InputImageType::PointType dummyOrigin;
	typename InputImageType::IndexType dummyStart;
	typename InputImageType::SizeType dummySize;

	dummyImage->SetSpacing( inputSpacing );
	for (int k = 0; k < InputImageDimension; k++)
	{
		dummySize[k] = sphereRadius[k]+sphereRadius[k]+1;
		dummyStart[k] = -sphereRadius[k];
		dummyOrigin[k] = 0;
	}
	dummyRegion.SetIndex( dummyStart );
	dummyRegion.SetSize( dummySize );
	dummyImage->SetRegions( dummyRegion );
	dummyImage->SetOrigin( dummyOrigin );

	float radiusSqr = this->m_Scale * this->m_Scale;
	itk::ImageRegionIteratorWithIndex<InputImageType> it( dummyImage, dummyRegion );
	for (it.GoToBegin(); !it.IsAtEnd(); ++it)
	{
		typename InputImageType::IndexType dummyIdx = it.GetIndex();
		typename InputImageType::PointType point;
		dummyImage->TransformIndexToPhysicalPoint( dummyIdx, point );
		float d = 0;
		for (int k = 0; k < InputImageDimension; k++)
		{
			d += point[k]*point[k];
		}
		if (d > radiusSqr)
		{
			continue;
		}
		else
		{
			NeighborOffsetType offset;
			
			for (int k = 0; k < InputImageDimension; k++)
			{
				offset[k] = dummyIdx[k];
			}
			this->m_OffsetInSphericalNeighborhood.push_back(offset);			      
		}
	}	
}

template <class TInputImage, class TCoordRep>
void
HammerDeformationImageFunction<TInputImage, TCoordRep>
::GenerateSubvolumnMatchingNeighborhood()  throw (InvalidRequestedRegionError) 
{
	// pad input requested region by 1, so the simple edge detection
	// works on the entire image domain
	typename InputImageType::SpacingType inputSpacing = m_SubjectImage->GetSpacing();

	NeighborOffsetType dummyIdx;
	int x, y, z;
	for(int r=1;r<=m_SubvolumnRadius;r++)
	{
		for(z=-r; z<=r; z+=2*r)
			for(x=-r; x<=r; x++)
				for(y=-r; y<=r; y++)
				{
					dummyIdx[0] = x/inputSpacing[0];
					dummyIdx[1] = y/inputSpacing[1];
					dummyIdx[2] = z/inputSpacing[2];
					this->m_SubvolumnMatchingNeighborhood.push_back(dummyIdx);
				}
		for(x=-r; x<=r; x+=2*r)
			for(z=-r+1; z<=r-1; z++) 
				for(y=-r; y<=r; y++)
				{
					dummyIdx[0] = x/inputSpacing[0];
					dummyIdx[1] = y/inputSpacing[1];
					dummyIdx[2] = z/inputSpacing[2];
					this->m_SubvolumnMatchingNeighborhood.push_back(dummyIdx);
				}
		for(y=-r; y<=r; y+=2*r)
			for(z=-r+1; z<=r-1; z++) 
				for(x=-r+1; x<=r-1; x++) 
				{
					dummyIdx[0] = x/inputSpacing[0];
					dummyIdx[1] = y/inputSpacing[1];
					dummyIdx[2] = z/inputSpacing[2];
					this->m_SubvolumnMatchingNeighborhood.push_back(dummyIdx);
				}
	}
}

/**
*
*/
template <class TInputImage, class TCoordRep>
void
HammerDeformationImageFunction<TInputImage, TCoordRep>
::ObtainTemplateDrivingPoint() const
{
	if(m_TemplateDrivingPointLocationReady == true)
		return;

	//m_TemplateDrivingPointLocation = LocationImageType::New();

	typename LocationImageType::RegionType LocationImageRegion;
	typename LocationImageType::PointType LocationImageOrigin;
	typename LocationImageType::IndexType LocationImageStart;
	typename LocationImageType::SizeType LocationImageSize;
	typename LocationImageType::SpacingType inputSpacing = m_SubjectImage->GetSpacing();
	m_TemplateDrivingPointLocation->SetSpacing( inputSpacing );
	LocationImageSize = m_SubjectImage->GetLargestPossibleRegion().GetSize();
	for (int k = 0; k < InputImageDimension; k++)
	{		
		LocationImageStart[k] = 0;
		LocationImageOrigin[k] = 0;
	}
	LocationImageRegion.SetIndex( LocationImageStart );
	LocationImageRegion.SetSize( LocationImageSize );
	m_TemplateDrivingPointLocation->SetRegions( LocationImageRegion );
	m_TemplateDrivingPointLocation->SetOrigin( LocationImageOrigin );
	m_TemplateDrivingPointLocation->Allocate();

	long Index = 0;
	LocationImageType::IndexType LocationIndex;
	DrivingPointIterator DrivingPoint_Iterator = m_TemplateDrivingPointArray->Begin();
	DrivingPointIterator DrivingPoint_End = m_TemplateDrivingPointArray->End();
	while(DrivingPoint_Iterator!=DrivingPoint_End)
	{
		PointDisplacementType::PointType Coordinate = DrivingPoint_Iterator.Value();
		LocationIndex[0] = (long)Coordinate[0];
		LocationIndex[1] = (long)Coordinate[1];
		LocationIndex[2] = (long)Coordinate[2];
		m_TemplateDrivingPointLocation->SetPixel(LocationIndex, Index);
		Index++;
		DrivingPoint_Iterator++;	
	}	
}



/**
 *
 */

template <class TInputImage, class TCoordRep>
typename HammerDeformationImageFunction<TInputImage,TCoordRep>
::RealType
HammerDeformationImageFunction<TInputImage,TCoordRep>
::EvaluateAtIndex(const IndexType& index) const
{
	RealType retValue;
	//typedef InputImageType::PixelType ImgAttribute;
	float Point_Similarity, Neighborhood_Similarity, MaxSimilarityDegree, Current_Similarity;
	float Displacement_Magnitude;
	int Real_Size;

	InputAttributeType Template_Feature, Subject_Feature, Subvolumn_Template_Feature, Subvolumn_Subject_Feature;
	DisplacementType Deformation_on_Particular_Point, TentativeWarp;
	retValue.Fill( 0 );

	if(m_TemplateDrivingPointLocationReady == false)
	{
		ObtainTemplateDrivingPoint();
	}
	//Step 1: check whether the the input point is the driving point
	if(m_TemplateDrivingPointLocation->GetPixel(index) < 0)
	{
		return (retValue);
	}
	//Step 2: find the correspondence of particular point
	MaxSimilarityDegree = 0;
	Template_Feature = this->GetInputImage()->GetPixel(index);
	Deformation_on_Particular_Point = m_DeformationFiled->GetPixel(index);
	for (int k = 0; k < m_OffsetInSphericalNeighborhood.size(); k++)
	{
		//Step 2.1: compare the similarity between two singel points
		NeighborOffsetType offset = m_OffsetInSphericalNeighborhood[k];
		InputImageType::IndexType SubjIdx, MdlIdx;
		for (int m=0; m < InputImageDimension; m++)
			{
				SubjIdx[m] = index[m] + Deformation_on_Particular_Point[m] + offset[m];
			}
		if(!m_SubjectImage->GetLargestPossibleRegion().IsInside(SubjIdx))
			continue;
		Subject_Feature = m_SubjectImage->GetPixel(SubjIdx);
		Point_Similarity = SimilarityBetweenTwoImgAttribute(Template_Feature, Subject_Feature);
		//Step 2.2: compare the similarity between two neighborhood
		if(Point_Similarity>m_VoxelSimilarityThreshold || (m_IsBigVN== true && Template_Feature[3]>0 && Subject_Feature[3]>0))
		{			
			for(int m=0;m<InputImageDimension;m++)
			{ 
				TentativeWarp[m] = offset[m];
			}
			
			Neighborhood_Similarity = 0;
			Real_Size = 0;
			for(int s=0;s<m_SubvolumnMatchingNeighborhood.size();s+=m_NeighborhoodStep)
			{
				Displacement_Magnitude = 0;
				for(int m=0;m<InputImageDimension;m++)
				{
					MdlIdx[m] = index[m] + m_SubvolumnMatchingNeighborhood[s][m];
					Displacement_Magnitude += m_SubvolumnMatchingNeighborhood[s][m];
				}
				Displacement_Magnitude /= 3.0;
				if(!m_SubjectImage->GetLargestPossibleRegion().IsInside(MdlIdx))
					continue;

				Subvolumn_Template_Feature = this->GetInputImage()->GetPixel(MdlIdx);				
				Deformation_on_Particular_Point = m_DeformationFiled->GetPixel(MdlIdx);

				for(int m=0;m<InputImageDimension;m++)
					SubjIdx[m] = MdlIdx[m] + Deformation_on_Particular_Point[m] + TentativeWarp[m]*GuassianAtLevelSigma[(int)Displacement_Magnitude][m_SubvolumnRadius];
				if(!m_SubjectImage->GetLargestPossibleRegion().IsInside(SubjIdx))
					continue;
				Subvolumn_Subject_Feature = m_SubjectImage->GetPixel(SubjIdx);
				Current_Similarity = SimilarityBetweenTwoImgAttribute(Subvolumn_Subject_Feature, Subvolumn_Template_Feature);
				if(Point_Similarity < 0.6 && m_IsYoungBrain == true && Subvolumn_Template_Feature[0] == 0)
					Current_Similarity = 0;
				Real_Size += 1.0;
				if( Subvolumn_Template_Feature[0]>0 ) /* June 6, 2001*/
				{
					Current_Similarity *= 1.2 ;
					Real_Size += (1.2-1.0) ;
				}
				Neighborhood_Similarity += Current_Similarity;
			}
			if(Real_Size>0)
			Neighborhood_Similarity /= Real_Size;
			if(Neighborhood_Similarity>MaxSimilarityDegree && Neighborhood_Similarity>m_NeighborhoodSimilarityThreshold)
			{
				for(int m=0;m<InputImageDimension;m++)
					retValue[m] = TentativeWarp[m];
				MaxSimilarityDegree = Neighborhood_Similarity;
			}
		}
	}
	return ( retValue );
}


} // end namespace itk

#endif
