/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: HammerTPS.txx,v $
  Language:  C++
  Date:      $Date: 2007-10-25 03:55:09 $
  Version:   $Revision: 1.20 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef _HammerTPS_txx
#define _HammerTPS_txx
#include "itkHammerTPS.h"
#include <vnl/vnl_trace.h>
#include <vnl/algo/vnl_qr.h>
#include <vnl/algo/vnl_svd.h>
namespace itk
{

template <class TScalarType, unsigned int NDimensions>
void
HammerTPS<TScalarType, NDimensions>::
ComputeG(const InputVectorType & x, GMatrixType & gmatrix) const
{
  const TScalarType r = x.GetNorm();
  gmatrix.fill( NumericTraits< TScalarType >::Zero );
  for(unsigned int i=0; i<1; i++)
    {
    gmatrix[i][i] = -r;
    }
}

template <class TScalarType, unsigned int NDimensions>
double
HammerTPS<TScalarType, NDimensions>::
ComputePointDistance(InputPointType &Point1, InputPointType &Point2)
{
	double Dist;
	for(int i=0;i<NDimensions;i++)
		Dist += sqrt((Point1[i]-Point2[i])*(Point1[i]-Point2[i]));
	return -Dist;
}


template <class TScalarType, unsigned int NDimensions>
void
HammerTPS<TScalarType, NDimensions>::
PrepareTPSKernel()
{
	int i;	
	vnl_vector<double> V_ij;
	int PointNum = this->m_SourceLandmarks->GetNumberOfPoints();
	InputPointType Point1, Point2;
	
	TPS_Psi.set_size(PointNum, PointNum);
	FixedPoint.set_size(PointNum, NDimensions+1);
	MovingPoint.set_size(PointNum, NDimensions+1);
	FixedPoint.set_column(0, 1);
	MovingPoint.set_column(0, 1);
	
	PointsIterator f_sp  = this->m_SourceLandmarks->GetPoints()->Begin();
	PointsIterator f_end = this->m_SourceLandmarks->GetPoints()->End();
	PointsIterator m_sp = this->m_TargetLandmarks->GetPoints()->Begin();
	i = 0;
	while( f_sp != f_end )
    {
		for(int k=0;k<NDimensions;k++)
		{
			FixedPoint(i, k+1) = f_sp->Value()[k];
			MovingPoint(i, k+1) = m_sp->Value()[k];
		}
		i++;		
		f_sp++;
		m_sp++;
    }
	
	for (int i=0;i<PointNum;i++)
	{		
		for (int j=i;j<PointNum;j++)
		{
			V_ij = FixedPoint.get_row(i) - FixedPoint.get_row(j);			
			TPS_Psi(i,j) = -V_ij.two_norm();
		}
	}
	for (int i=0;i<PointNum;i++)
		for (int j=0;j<i;j++)
			TPS_Psi(i,j) = TPS_Psi(j,i);

	vnl_qr<double> qr_FixedPoint(FixedPoint);
	Q1 = qr_FixedPoint.Q().extract(PointNum, 4, 0, 0);
	Q2 = qr_FixedPoint.Q().extract(PointNum, PointNum-4, 0, 4);
	R = qr_FixedPoint.R().extract(4, 4, 0, 0);
	vnl_svd<double> svd(R);
	inverse_R = svd.inverse();
}

template <class TScalarType, unsigned int NDimensions>
double
HammerTPS<TScalarType, NDimensions>::
UpdateTPSParameter()
{
	vnl_matrix<double> T, S, I, I4, Gamma;
	vnl_matrix<double> inverse_T;
	double BendingEnergy;
	int PointNum = this->m_SourceLandmarks->GetNumberOfPoints();

	I.set_size(PointNum-NDimensions-1, PointNum-NDimensions-1);
	I4.set_size(NDimensions+1, NDimensions+1);
	I.set_identity();
	I4.set_identity();
	I = I*m_Lemda;
	T = Q2.transpose()*TPS_Psi*Q2 + I;
	
	vnl_svd<double> svd(T);
	inverse_T = svd.inverse();

	Gamma = inverse_T * Q2.transpose() * MovingPoint;

	TPS_Param = Q2 * Gamma;
	//TPS_Affine = inverse_R * Q1.transpose() * (MovingCord - TPS_Psi*TPS_Param);
	//A = inv(R'*R + lamda2 * eye(length(R),length(R))) * ( R'*q1'*(y-K*q2*gamma) - R'*R);
	//d = A + eye(dim+1,dim+1);
	S = R.transpose()*R + m_Lemda2*I4;
	vnl_svd<double> svd2(S);
	TPS_Affine = svd2.inverse() * (R.transpose()*Q1.transpose()*(MovingPoint-TPS_Psi*TPS_Param) - R.transpose()*R) + I4;

	BendingEnergy = vnl_trace(TPS_Param*MovingPoint.transpose()) * m_Lemda;

	return BendingEnergy;
}
	

/**
 *
 */
template <class TScalarType, unsigned int NDimensions>
typename HammerTPS<TScalarType, NDimensions>::OutputPointType
HammerTPS<TScalarType, NDimensions>
::TransformPoint( const InputPointType& thisPoint ) const
{
	OutputPointType result;
	vnl_matrix<double> input_point, affine_part, kernel_matrix;
	vnl_vector<double> V_ij;
	vnl_matrix<double> tps_part;
	int PointNum;
	double Dist;
	MultiThreader::Pointer m_Threader = MultiThreader::New();
	int m_NumberOfThreads = m_Threader->GetNumberOfThreads();
	  //m_Threader->SetNumberOfThreads(1);
	//vcl_cerr<<this->FixedPoint;
	PointNum = this->m_SourceLandmarks->GetNumberOfPoints();
	input_point.set_size(1, NDimensions+1);  
	affine_part.set_size(1, NDimensions+1);
	kernel_matrix.set_size(1, PointNum);
	input_point(0, 0) = 1;
	for(unsigned i=0;i<NDimensions;i++)
		input_point(0,i+1) = thisPoint[i];

	typedef typename OutputPointType::ValueType ValueType;

	result.Fill( NumericTraits< ValueType >::Zero );
	//vcl_cerr<<FixedPoint;
	for(int k=0;k<PointNum;k++)
	{
		/*
		Dist = 0;
		for(int d=0;d<NDimensions;d++)
		{
			Dist += (input_point(0, d+1)-FixedPoint(k,d+1))*(input_point(0, d+1)-FixedPoint(k,d+1));
		}
		kernel_matrix(0, k) = -sqrt(Dist);
		*/
		V_ij = input_point.get_row(0) - FixedPoint.get_row(k);			
		//V_ij = input_point.get_row(0);
		Dist = V_ij.two_norm();
		kernel_matrix(0,k) = -Dist;	
		//vcl_cerr << kernel_matrix;
	}

	affine_part = input_point*TPS_Affine;
	tps_part = kernel_matrix*TPS_Param;
	
	for(unsigned int i=0;i<NDimensions;i++)
		result[i] = affine_part(0, i+1) + tps_part(0, i+1);

	return result;

}

/**
 * Default implementation of the the method. This can be overloaded
 * in transforms whose kernel produce diagonal G matrices.
 */
template <class TScalarType, unsigned int NDimensions>
void
HammerTPS<TScalarType, NDimensions>::
ComputeTPSContribution( const InputPointType  & thisPoint,
                                OutputPointType & result  ) 
{
	double dist;
	unsigned long numberOfLandmarks = this->m_SourceLandmarks->GetNumberOfPoints();

	PointsIterator sp  = this->m_SourceLandmarks->GetPoints()->Begin();

	GMatrixType Gmatrix; 

	for(unsigned int lnd=0; lnd < numberOfLandmarks; lnd++ )
	{
		//this->ComputeG( thisPoint - sp->Value(), Gmatrix );
		dist = ComputePointDistance(thisPoint, sp);
		for(unsigned int dim=0; dim < NDimensions; dim++ )
		{
			result[ dim ] += dist * TPS_Param(lnd,dim+1);    
		}
		++sp;
	}
}


} // namespace itk
#endif
