
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string>
#include <vector>

#include "itkFixedArray.h"
#include "itkMatrix.h"
#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkShrinkImageFilter.h"

#include "itkVector.h"
#include "../../../utilities/tclap/CmdLine.h"

#include "../plugin/itkHammerTissueAttributeVectorImageFilter.h"
#include "../plugin/itkHammerTissueAttributeVector.h"

using namespace TCLAP;

typedef itk::HammerTissueAttributeVector ImgAttribute;

typedef float FloatType;
typedef itk::Vector<FloatType, 3>       ITKFvector3d;
typedef itk::Vector<int, 3>             Ivector3d;
typedef itk::Matrix<FloatType, 4, 4>    Matrix;
typedef itk::Image<unsigned char, 3>    ImageType;
typedef itk::Image<ITKFvector3d, 3>     DeformationFieldType;
typedef itk::Image<ImgAttribute, 3>     AttributeImageType;

void WriteAttributeComponent( char* filename, AttributeImageType::Pointer avImg, int idx )
{  
  ImageType::Pointer cImg = ImageType::New();
  cImg->CopyInformation( avImg );
  cImg->SetRegions( cImg->GetLargestPossibleRegion() );
  cImg->Allocate();
  
  itk::ImageRegionIteratorWithIndex<ImageType> it( cImg, cImg->GetLargestPossibleRegion() );
  for (it.GoToBegin(); !it.IsAtEnd(); ++it)
    {
    ImageType::IndexType imgIdx = it.GetIndex();
    AttributeImageType::PixelType p = avImg->GetPixel( imgIdx );
    it.Set( p[idx] );
    }
  
  itk::ImageFileWriter<ImageType>::Pointer w = itk::ImageFileWriter<ImageType>::New();
  w->SetFileName( filename );
  w->SetInput( cImg );
  w->Update();
  
  return;
}


int main(int argc,char *argv[])
  
{
  float XYZres = 1. ;
  int nThreads = 1;
  std::string fixedFilename;
  std::string movingFilename;
  
  itk::OStringStream msg;
  
  try
    {
    CmdLine cl ( "HAMMER Attribute Vector Test: Heirarchical Attribute Matching Mechanism for Elastic Registration, NA-MIC Kit",
                 ' ',
                 "$Revision: 1.2 $" );
    
    msg.str ( "" ); msg << "Number of threads (default: " << nThreads << ")";
    ValueArg<int> ThreadArg ( "T", "Thread", msg.str(), false, nThreads, "int", cl );
    
    msg.str ( "" ); msg << "sample rate (default: " << XYZres << ")";
    ValueArg<float> SampleRateArg ( "s", "Sample", msg.str(), false, XYZres, "float", cl );
    
    TCLAP::UnlabeledValueArg<std::string> FixedImageArg("fixed", "Fixed image filename", 1, fixedFilename, "std::string", cl); 
    TCLAP::UnlabeledValueArg<std::string> MovingImageArg("brainSurface", "Moving image filename", 1, movingFilename, "std::string", cl);
    
    cl.parse ( argc, argv );
    nThreads = ThreadArg.getValue();
    XYZres =  SampleRateArg.getValue();
    
    fixedFilename = FixedImageArg.getValue();
    movingFilename = MovingImageArg.getValue();
    } 
  catch ( ArgException exception ) 
    {
    std::cerr << "error: " << exception.error() << " for arg " << exception.argId() << std::endl;
    exit ( EXIT_FAILURE );
  }
  
  double scale = 7*(1./XYZres) ; if(scale<3) scale=3; printf("scale=%f\n", scale) ;
  
  /***** Model image, segmented *****/
  /*Img_XY = 256 ;*/	 
  
  printf("\nmodel image : %s\n", fixedFilename.c_str()) ;  
  printf("subject image : %s\n", movingFilename.c_str()) ;  
  
  /*** Load in fixed image and compute the attribute vectors ***/
  itk::ImageFileReader<ImageType>::Pointer ImgReader = itk::ImageFileReader<ImageType>::New();
  ImgReader->SetFileName( fixedFilename.c_str() );
  try
    {
    ImgReader->Update();
    }
  catch( itk::ExceptionObject *ex )
    {
    std::cerr << ex << std::endl;
    }

  typedef itk::ShrinkImageFilter<ImageType, ImageType> DownSampleType;
  DownSampleType::Pointer downsample = DownSampleType::New();
  downsample->SetInput( ImgReader->GetOutput() );
  for (int k = 0; k < 3; k++)
    {
    downsample->SetShrinkFactor( k, static_cast<int> (XYZres) );
    }
  downsample->Update();

  ImageType::Pointer Img = downsample->GetOutput();
  Img->DisconnectPipeline();
  std::cout << "Fixed image file read in\n";

  typedef itk::HammerTissueAttributeVectorImageFilter<ImageType, AttributeImageType> AttributeFilterType;
  AttributeFilterType::Pointer modleAttributeFilter = AttributeFilterType::New();
  modleAttributeFilter->SetInput( Img );
  modleAttributeFilter->SetBGValue( 0 );
  modleAttributeFilter->SetGMValue( 150 );
  modleAttributeFilter->SetWMValue( 250 );
  modleAttributeFilter->SetVNValue( 50 );
  modleAttributeFilter->SetCSFValue( 10 );

  modleAttributeFilter->SetNumberOfThreads( nThreads );
  modleAttributeFilter->SetStrength( 1 );
  modleAttributeFilter->SetScale( scale );
  modleAttributeFilter->Update();

  AttributeImageType::Pointer fixedAVec = modleAttributeFilter->GetOutput();
  fixedAVec->DisconnectPipeline();

  WriteAttributeComponent( "fAttributeV0.mha", fixedAVec, 0 );
  WriteAttributeComponent( "fAttributeV1.mha", fixedAVec, 1 );
  WriteAttributeComponent( "fAttributeV2.mha", fixedAVec, 2 );
  WriteAttributeComponent( "fAttributeV3.mha", fixedAVec, 3 );
  WriteAttributeComponent( "fAttributeV4.mha", fixedAVec, 4 );

   /*** Load in moving image and compute the attribute vectors ***/
  ImgReader->SetFileName( movingFilename.c_str() );
  try
    {
    ImgReader->Update();
    }
  catch( itk::ExceptionObject *ex )
    {
    std::cerr << ex << std::endl;
    }

  downsample->SetInput( ImgReader->GetOutput() );
  for (int k = 0; k < 3; k++)
    {
    downsample->SetShrinkFactor( k, static_cast<int> (XYZres) );
    }
  downsample->Update();

  ImageType::Pointer mImg = downsample->GetOutput();
  mImg->DisconnectPipeline();
  std::cout << "Moving image file read in\n";

  AttributeFilterType::Pointer subjectAttributeFilter = AttributeFilterType::New();
  subjectAttributeFilter->SetInput( mImg );
  subjectAttributeFilter->SetBGValue( 0 );
  subjectAttributeFilter->SetGMValue( 150 );
  subjectAttributeFilter->SetWMValue( 250 );
  subjectAttributeFilter ->SetVNValue( 50 );
  subjectAttributeFilter->SetCSFValue( 10 );

  subjectAttributeFilter->SetNumberOfThreads( nThreads );
  subjectAttributeFilter->SetStrength( 1 );
  subjectAttributeFilter->SetScale( scale );
  subjectAttributeFilter->Update();

  AttributeImageType::Pointer movingAVec = subjectAttributeFilter->GetOutput();
  movingAVec->DisconnectPipeline();
  
  WriteAttributeComponent( "mAttributeV0.mha", movingAVec, 0 );
  WriteAttributeComponent( "mAttributeV1.mha", movingAVec, 1 );
  WriteAttributeComponent( "mAttributeV2.mha", movingAVec, 2 );
  WriteAttributeComponent( "mAttributeV3.mha", movingAVec, 3 );
  WriteAttributeComponent( "mAttributeV4.mha", movingAVec, 4 );

  return 0;
}



