
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string>
#include <vector>

#include "itkFixedArray.h"
#include "itkMatrix.h"
#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkShrinkImageFilter.h"

#include "itkVector.h"
#include "../../../utilities/tclap/CmdLine.h"

#include "../plugin/itkHammerTissueAttributeVectorImageFilter.h"
#include "../plugin/itkHammerRegistrationFunction.h"

using namespace TCLAP;

typedef itk::HammerTissueAttributeVector ImgAttribute;

typedef float FloatType;
typedef itk::Vector<FloatType, 3>       ITKFvector3d;
typedef itk::Vector<int, 3>             Ivector3d;
typedef itk::Matrix<FloatType, 4, 4>    Matrix;
typedef itk::Image<unsigned char, 3>    ImageType;
typedef itk::Image<ITKFvector3d, 3>     DeformationFieldType;
typedef itk::Image<ImgAttribute, 3>     AttributeImageType;

unsigned char Geom_UP = static_cast<unsigned char>(0.9*255) ;
unsigned char Geom_DN = static_cast<unsigned char>(0.4*255) ;
unsigned char VNvlm_UP = static_cast<unsigned char>(255/12) ;
unsigned char VNvlm_DN = 170 ; /* ~65% added on Dec 2006 */
unsigned char CSFBG_UP = static_cast<unsigned char>(255/4) ;

unsigned char Geom_DownsUp = static_cast<unsigned char>(Geom_DN/1.1) ; /*1.5*/

int main(int argc,char *argv[])
  
{
  float XYZres = 1. ;
  int nThreads = 1;
  std::string fixedFilename;
  std::string movingFilename;

  itk::OStringStream msg;

  try
    {
      CmdLine cl ( "HAMMER Attribute Vector Test: Heirarchical Attribute Matching Mechanism for Elastic Registration, NA-MIC Kit",
        ' ',
        "$Revision: 1.2 $" );

    msg.str ( "" ); msg << "Number of threads (default: " << nThreads << ")";
    ValueArg<int> ThreadArg ( "T", "Thread", msg.str(), false, nThreads, "int", cl );

    msg.str ( "" ); msg << "sample rate (default: " << XYZres << ")";
    ValueArg<float> SampleRateArg ( "s", "Sample", msg.str(), false, XYZres, "float", cl );
	    
    UnlabeledValueArg<std::string> FixedImageArg ( "fixed", "Fixed image filename", 1, fixedFilename, "string", cl );
    UnlabeledValueArg<std::string> MovingImageArg ( "moving", "Moving image filename", 1, movingFilename, "string", cl );

    cl.parse ( argc, argv );
    nThreads = ThreadArg.getValue();
    XYZres =  SampleRateArg.getValue();

    fixedFilename = FixedImageArg.getValue();
    movingFilename = MovingImageArg.getValue();
    } 
  catch ( ArgException exception ) {
  std::cerr << "error: " << exception.error() << " for arg " << exception.argId() << std::endl;
  exit ( EXIT_FAILURE );
  }

  double scale = 7*(1./XYZres) ; if(scale<3) scale=3; printf("scale=%f\n", scale) ;

  /***** Model image, segmented *****/
  /*Img_XY = 256 ;*/	 

  printf("\nmodel image : %s\n", fixedFilename.c_str()) ;  
  printf("subject image : %s\n", movingFilename.c_str()) ;  

  /*** Load in fixed image and compute the attribute vectors ***/
  itk::ImageFileReader<ImageType>::Pointer ImgReader = itk::ImageFileReader<ImageType>::New();
  ImgReader->SetFileName( fixedFilename.c_str() );
  try
    {
    ImgReader->Update();
    }
  catch( itk::ExceptionObject *ex )
    {
    std::cerr << ex << std::endl;
    }

  typedef itk::ShrinkImageFilter<ImageType, ImageType> DownSampleType;
  DownSampleType::Pointer downsample = DownSampleType::New();
  downsample->SetInput( ImgReader->GetOutput() );
  for (int k = 0; k < 3; k++)
    {
    downsample->SetShrinkFactor( k, static_cast<int> (XYZres) );
    }
  downsample->Update();

  ImageType::Pointer Img = downsample->GetOutput();
  Img->DisconnectPipeline();
  std::cout << "Fixed image file read in\n";

  typedef itk::HammerTissueAttributeVectorImageFilter<ImageType, AttributeImageType> AttributeFilterType;
  AttributeFilterType::Pointer modleAttributeFilter = AttributeFilterType::New();
  modleAttributeFilter->SetInput( Img );
  modleAttributeFilter->SetBGValue( 0 );
  modleAttributeFilter->SetGMValue( 150 );
  modleAttributeFilter->SetWMValue( 250 );
  modleAttributeFilter->SetVNValue( 50 );
  modleAttributeFilter->SetCSFValue( 10 );

  modleAttributeFilter->SetNumberOfThreads( nThreads );
  modleAttributeFilter->SetStrength( 1 );
  modleAttributeFilter->SetScale( scale );
  modleAttributeFilter->Update();

  AttributeImageType::Pointer fixedAVec = modleAttributeFilter->GetOutput();
  fixedAVec->DisconnectPipeline();

   /*** Load in moving image and compute the attribute vectors ***/
  ImgReader->SetFileName( movingFilename.c_str() );
  try
    {
    ImgReader->Update();
    }
  catch( itk::ExceptionObject *ex )
    {
    std::cerr << ex << std::endl;
    }

  downsample->SetInput( ImgReader->GetOutput() );
  for (int k = 0; k < 3; k++)
    {
    downsample->SetShrinkFactor( k, static_cast<int> (XYZres) );
    }
  downsample->Update();

  ImageType::Pointer mImg = downsample->GetOutput();
  mImg->DisconnectPipeline();
  std::cout << "Moving image file read in\n";

  AttributeFilterType::Pointer subjectAttributeFilter = AttributeFilterType::New();
  subjectAttributeFilter->SetInput( mImg );
  subjectAttributeFilter->SetBGValue( 0 );
  subjectAttributeFilter->SetGMValue( 150 );
  subjectAttributeFilter->SetWMValue( 250 );
  subjectAttributeFilter ->SetVNValue( 50 );
  subjectAttributeFilter->SetCSFValue( 10 );

  subjectAttributeFilter->SetNumberOfThreads( nThreads );
  subjectAttributeFilter->SetStrength( 1 );
  subjectAttributeFilter->SetScale( scale );
  subjectAttributeFilter->Update();

  AttributeImageType::Pointer movingAVec = subjectAttributeFilter->GetOutput();
  movingAVec->DisconnectPipeline();
  
  // identify the edge point for cost function computation
  itk::ImageRegionIteratorWithIndex<AttributeImageType> 
    itAV( fixedAVec, 
                 fixedAVec->GetLargestPossibleRegion() );
  itk::ImageRegionIteratorWithIndex<AttributeImageType> 
    itAVm( movingAVec, 
                 movingAVec->GetLargestPossibleRegion() );
  
  ImageType::Pointer fixedImageEdgePointLabel = ImageType::New();
  fixedImageEdgePointLabel->CopyInformation( Img );
  fixedImageEdgePointLabel->SetRegions( Img->GetLargestPossibleRegion() );
  fixedImageEdgePointLabel->Allocate();
  fixedImageEdgePointLabel->FillBuffer( 0 );

  ImageType::Pointer movingImageEdgePointLabel = ImageType::New();
  movingImageEdgePointLabel->CopyInformation( Img );
  movingImageEdgePointLabel->SetRegions( Img->GetLargestPossibleRegion() );
  movingImageEdgePointLabel->Allocate();
  movingImageEdgePointLabel->FillBuffer( 0 );
  
  std::vector<typename ImageType::IndexType> edgePointInFixedImage;
  std::vector<typename ImageType::IndexType> edgePointInMovingImage;
  std::vector<ImgAttribute> edgePointInFixedImageAttribute;
  std::vector<ImgAttribute> edgePointInMovingImageAttribute;
  
  for ( itAV.GoToBegin(); !itAV.IsAtEnd(); ++itAV )
    {
    ImgAttribute a = itAV.Get();
    
    if (a[0] == 0)
      {
      continue;
      }
    if (a[2]<=Geom_UP 
        && (a[2]>=Geom_DN || a[2]<=Geom_DownsUp) 
        && a[3]<=VNvlm_UP 
        && a[4]<=CSFBG_UP) 
      {
      continue;
      }
    if (a[3] > VNvlm_DN)
      {
      continue;
      }
    AttributeImageType::IndexType idx = itAV.GetIndex();    
    edgePointInFixedImage.push_back( idx );
    edgePointInFixedImageAttribute.push_back( a );

    fixedImageEdgePointLabel->SetPixel( idx, 1 );
    
    }

  for ( itAVm.GoToBegin(); !itAVm.IsAtEnd(); ++itAVm )
    {
    ImgAttribute a = itAVm.Get();
    
    if (a[0] == 0)
      {
      continue;
      }
    if (a[2]<=Geom_UP 
        && (a[2]>=Geom_DN || a[2]<=Geom_DownsUp) 
        && a[3]<=VNvlm_UP 
        && a[4]<=CSFBG_UP) 
      {
      continue;
      }
    if (a[3] > VNvlm_DN)
      {
      continue;
      }
    AttributeImageType::IndexType idx = itAV.GetIndex();    
    edgePointInMovingImage.push_back( idx );
    edgePointInMovingImageAttribute.push_back( a );

    movingImageEdgePointLabel->SetPixel( idx, 1 );
    
    }

  itk::ImageFileWriter<ImageType>::Pointer w = itk::ImageFileWriter<ImageType>::New();
  w->SetFileName( "FixedImageEdgePointsPicked.mha");
  w->SetInput( fixedImageEdgePointLabel );
  w->Update();

  w->SetFileName( "MovingImageEdgePointsPicked.mha");
  w->SetInput( movingImageEdgePointLabel );
  w->Update();

  std::cout << "Picked edge points in fixed image:\n"
    << edgePointInFixedImage.size() << std::endl;
  std::cout << "Picked edge points in moving image:\n"
    << edgePointInMovingImage.size() << std::endl;

  DeformationFieldType::Pointer deformationField = DeformationFieldType::New();
  deformationField->CopyInformation( Img );
  deformationField->SetRegions( Img->GetLargestPossibleRegion() );
  deformationField->Allocate();
  DeformationFieldType::PixelType zeroDeformation;
  zeroDeformation.Fill( 0.0 );
  deformationField->FillBuffer( zeroDeformation );

  typedef itk::HammerRegistrationFunction<ImageType, ImageType, DeformationFieldType> HammerFunctionType;
  HammerFunctionType::Pointer hammerFunction = HammerFunctionType::New();
  hammerFunction->SetFixedImage( Img );
  hammerFunction->SetMovingImage( mImg );
  hammerFunction->SetDeformationField( deformationField );

  hammerFunction->SetFixedAttributeImage ( fixedAVec );
  hammerFunction->SetMovingAttributeImage ( movingAVec );

  hammerFunction->SetPickedPointsOnFixedImage( edgePointInFixedImage );
  hammerFunction->SetPickedPointsOnMovingImage( edgePointInMovingImage );

  hammerFunction->FindingInverseForceFromSubject();


  HammerFunctionType::RadiusType radius;
  radius.Fill( 7 );
  hammerFunction->InitializeIteration();
  return 0;
}



