/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: GenDeformationField.cxx,v $
  Language:  C++
  Date:      $Date: 2008/09/16 13:13:25 $
  Version:   $Revision: 1.1 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even 
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
  PURPOSE.  See the above copyright notices for more information.

  =========================================================================*/
#if defined(_MSC_VER)
#pragma warning ( disable : 4786 )
#endif

#include "itkImageFileReader.h" 
#include "itkImageFileWriter.h" 

#include "itkImageRegionIterator.h"
#include "itkImageRandomIteratorWithIndex.h"

#include "itkDemonsRegistrationFilter.h"
#include "itkHistogramMatchingImageFilter.h"
#include "itkCastImageFilter.h"
#include "itkWarpImageFilter.h"
#include "itkLinearInterpolateImageFunction.h"
#include "itkNearestNeighborInterpolateImageFunction.h"

#include "itkThresholdImageFilter.h"
#include "itkExtractImageFilter.h"
#include "itkRawImageIO.h"
#include "itkGradientAnisotropicDiffusionImageFilter.h"
#include "itkGradientVectorFlowImageFilter.h"

#include "itkGradientRecursiveGaussianImageFilter.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkCovariantVector.h"
#include "itkGradientImageFilter.h"
#include "itkGradientToMagnitudeImageFilter.h"
#include "itkDerivativeImageFilter.h"
#include "itkGradientVectorFlowImageFilter.h"
#include "itkLaplacianImageFilter.h"

#include "itkOrientedImage.h"
#include "itkIndex.h"

#include "itkCannyEdgeDetectionImageFilter.h"
#include "itkSimpleFilterWatcher.h"
#include "itkRescaleIntensityImageFilter.h"

#include "itkSignedDanielssonDistanceMapImageFilter.h"
#include "itkSignedMaurerDistanceMapImageFilter.h" 

#include "itkPointSet.h"
#include "itkBSplineScatteredDataPointSetToImageFilter.h"

#include "itkImageToHistogramGenerator.h"
#include "itkHistogram.h"
#include "itkScalarImageToListAdaptor.h"
#include "itkListSampleToHistogramGenerator.h"
#include "itkListSampleToHistogramFilter.h"

#include <iostream>
#include <sstream>
#include <string>

static int iterationnumber = 0;

const unsigned int Dimension = 3;
typedef double PixelType;
typedef itk::OrientedImage< unsigned char, Dimension >  LabelImageType;
typedef itk::Vector<double, Dimension> VectorType;
typedef itk::OrientedImage<VectorType, Dimension> VectorImageType;

typedef itk::ImageFileReader< LabelImageType >  LabelReaderType;
typedef itk::ImageFileReader< VectorImageType >  VectorReaderType;
typedef itk::ImageFileWriter< LabelImageType >  LabelWriterType;

std::string itoa(int num)
{
  std::stringstream converter;
  converter << num;
  return converter.str();
}

// a templated image writing function
template <typename W, typename I> 
int _OutputImage(typename W::Pointer writer, typename I::Pointer image, const char* filename)
{
  writer->SetFileName(filename);
  writer->SetInput( image );
      
  try 
    { 
    writer->Update(); 
    } 
  catch( itk::ExceptionObject & err ) 
    { 
    std::cout << "ExceptionObject caught !" << std::endl; 
    std::cout << err << std::endl; 
    return EXIT_FAILURE;
    } 
  return EXIT_SUCCESS;
}

template <typename W, typename I> 
int OutputImage(typename W::Pointer writer, typename I::Pointer  image, std::string filename)
{
  int Good = _OutputImage<W, I>(writer, image, filename.c_str());
  return (Good);
}

void GenerateGrid( LabelImageType::Pointer img, int dim, int blocksize )
{
  LabelImageType::SpacingType spacing = img->GetSpacing();
  LabelImageType::RegionType region = img->GetLargestPossibleRegion();  

  LabelImageType::IndexType startIndex;

  img->FillBuffer( 0 );
  for (int k = 0; k < img->GetImageDimension(); k++)
  {
    int isize = static_cast<int> (floor( static_cast<float>(region.GetSize(k))/static_cast<float>(blocksize) -0.5 )) ;
    startIndex[k] = static_cast<int>( static_cast<float>(region.GetSize(k)-isize*blocksize)/2.0 );
  }


  itk::ImageRegionIteratorWithIndex<LabelImageType> it(img, img->GetLargestPossibleRegion() );
  for (it.GoToBegin(); !it.IsAtEnd(); ++it)
  {
    LabelImageType::IndexType idx = it.GetIndex();
    for (int k = 0; k < img->GetImageDimension(); k++)
    {
      if (k == dim)
      {
        continue;
      }
      if ( (idx[k]-startIndex[k])%blocksize == 0 )
      {
        it.Set( 255 );
        break;
      }
    }    
  }
}

int main( int argc, char *argv[] )
{
  if ( argc < 3 )
    {
      std::cerr << "Missing Parameters " << std::endl;
      std::cerr << "Usage: " << argv[0];
      std::cerr << " inputVectors outImage spacing\n";
      return 1;
    }

  int blocksize = atoi(argv[3]);

  std::string prefix = argv[2];
  std::string outFileName;

  VectorReaderType::Pointer vectorReader = VectorReaderType::New();
  vectorReader->SetFileName( argv[1] );
  vectorReader->Update();

  itk::ImageRegionIteratorWithIndex<VectorImageType> itv( vectorReader->GetOutput(),
    vectorReader->GetOutput()->GetLargestPossibleRegion() );
  for (itv.GoToBegin(); !itv.IsAtEnd(); ++itv)
  {
    //std::cout << itv.GetIndex() << ": " << itv.Get() << std::endl;;
  }

  LabelImageType::Pointer warpField = LabelImageType::New();
  warpField->CopyInformation( vectorReader->GetOutput() );
  warpField->SetRegions( warpField->GetLargestPossibleRegion() );
  warpField->Allocate();

  warpField->FillBuffer( 0 );

  
  typedef itk::WarpImageFilter<
    LabelImageType, 
    LabelImageType,
    VectorImageType  >     WarperType;
  
  typedef itk::NearestNeighborInterpolateImageFunction<
    LabelImageType,
    double          >  InterpolatorType;

  WarperType::Pointer warper = WarperType::New();
  InterpolatorType::Pointer interpolator = InterpolatorType::New();
  warper->SetInput( warpField );
  warper->SetInterpolator( interpolator );
  warper->SetOutputSpacing( warpField->GetSpacing() );
  warper->SetOutputOrigin(warpField ->GetOrigin() );
  warper->SetOutputDirection( warpField->GetDirection() );
  warper->SetDeformationField( vectorReader->GetOutput() );

  LabelWriterType::Pointer volWriter = LabelWriterType::New();

  outFileName = prefix+"axi.mha";
  GenerateGrid(warpField, 2, blocksize);
  warper->Update();
  OutputImage<LabelWriterType, LabelImageType>(volWriter, warper->GetOutput(), outFileName.c_str());

  outFileName = prefix+"sag.mha";
  warpField->FillBuffer (0);
  GenerateGrid(warpField, 0, blocksize);

  WarperType::Pointer warper2 = WarperType::New();
  warper2->SetInput( warpField );
  warper2->SetInterpolator( interpolator );
  warper2->SetOutputSpacing( warpField->GetSpacing() );
  warper2->SetOutputOrigin(warpField ->GetOrigin() );
  warper2->SetOutputDirection( warpField->GetDirection() );
  warper2->SetDeformationField( vectorReader->GetOutput() );

  warper2->Update();
  OutputImage<LabelWriterType, LabelImageType>(volWriter, warper2->GetOutput(), outFileName.c_str());

  outFileName = prefix+"cor.mha";
  warpField->FillBuffer (0);
  GenerateGrid(warpField, 1, blocksize);
  WarperType::Pointer warper3 = WarperType::New();
  warper3->SetInput( warpField );
  warper3->SetInterpolator( interpolator );
  warper3->SetOutputSpacing( warpField->GetSpacing() );
  warper3->SetOutputOrigin(warpField ->GetOrigin() );
  warper3->SetOutputDirection( warpField->GetDirection() );
  warper3->SetDeformationField( vectorReader->GetOutput() );


  warper3->Update();
  OutputImage<LabelWriterType, LabelImageType>(volWriter, warper3->GetOutput(), outFileName.c_str());

  return 0;
}
