#include <vcl_cstdio.h>

#include "itkRescaleIntensityImageFilter.h"
#include "itkShiftScaleImageFilter.h"

#include <3d_image/3d_image_io.h>

bool load_img8 (const vcl_string& name, Image8Type::Pointer& image)
{  
  typedef itk::ImageFileReader<Image8Type > ReaderType;
  ReaderType::Pointer reader = ReaderType::New();
  reader->SetFileName (name.c_str());    
  vcl_printf ("    load_img8 (%s): \n", name.c_str());
  try {
    reader->Update();
  }
  catch (itk::ExceptionObject &ex) {
    vcl_cout << ex << vcl_endl;
    return false;
  }
  image = reader->GetOutput();
  ///vcl_printf ("\t done.\n");
  return true;
}

bool load_img_f16 (const vcl_string& name, ImageType::Pointer& image)
{
  typedef itk::ImageFileReader<ImageType> ReaderType;
  ReaderType::Pointer reader = ReaderType::New();
  reader->SetFileName (name.c_str());    
  vcl_printf ("    load_img_f16 (%s): \n", name.c_str());
  try {
    reader->Update();
  }
  catch (itk::ExceptionObject &ex) {
    vcl_cout << ex << vcl_endl;
    return false;
  }
  image = reader->GetOutput();
  ///vcl_printf ("\t done.\n");
  return true;
}

bool load_img_double (const vcl_string& name, ImageDType::Pointer& image)
{
  typedef itk::ImageFileReader<ImageDType> ReaderType;
  ReaderType::Pointer reader = ReaderType::New();
  reader->SetFileName (name.c_str());    
  vcl_printf ("    load_img_double (%s): \n", name.c_str());
  try {
    reader->Update();
  }
  catch (itk::ExceptionObject &ex) {
    vcl_cout << ex << vcl_endl;
    return false;
  }
  image = reader->GetOutput();
  ///vcl_printf ("\t done.\n");
  return true;
}

bool save_img_f16 (const vcl_string& name, const ImageType::Pointer& image)
{
  vcl_printf ("    save_img_f16(): %s\n", name.c_str());

  typedef itk::ImageFileWriter<ImageType> WriterType;
  WriterType::Pointer writer = WriterType::New();  
  writer -> SetFileName (name.c_str());
  writer -> SetInput (image);
  try { 
    writer->Update();
  } 
	catch( itk::ExceptionObject & err ) {
    vcl_cout << "      ExceptionObject caught !" << vcl_endl; 
    vcl_cout << err << vcl_endl; 
    return false;
  } 
  ///vcl_printf ("\t done.\n\n");
  return true;
}

bool save_img8 (const vcl_string& name, const ImageType::Pointer& image)
{
  vcl_printf ("    save_img8(): %s\n", name.c_str());

  //Convert into unsigned char image type.
  ///vcl_printf ("\tConvert and cast into unsigned char image...\n");
  typedef itk::RescaleIntensityImageFilter<ImageType, itk::OrientedImage<unsigned char,3> >  RescaleFilterType;
  RescaleFilterType::Pointer rescaleFilter = RescaleFilterType::New();
  rescaleFilter->SetInput (image);
  rescaleFilter->SetOutputMinimum (0);
  rescaleFilter->SetOutputMaximum (255);
  rescaleFilter->Update();
  itk::OrientedImage<unsigned char,3>::Pointer img_out = rescaleFilter->GetOutput();

  typedef itk::ImageFileWriter<itk::OrientedImage<unsigned char,3> > WriterType;
  WriterType::Pointer writer = WriterType::New();  
  writer -> SetFileName(name.c_str());
  writer -> SetInput(img_out);
  try { 
    writer->Update();
  } 
	catch( itk::ExceptionObject & err ) {
    vcl_cout << "      ExceptionObject caught !" << vcl_endl; 
    vcl_cout << err << vcl_endl; 
    return false;
  } 
  ///vcl_printf ("\t done.\n\n");
  return true;
}

bool save_01_img8 (const vcl_string& name, const ImageType::Pointer& image)
{
  vcl_printf ("    save_01_img8(): %s\n", name.c_str());

  //Convert into unsigned char image type.
  ///vcl_printf ("\tConvert and cast into unsigned char image...\n");
  typedef itk::ShiftScaleImageFilter<ImageType, itk::OrientedImage<unsigned char,3> >  ShiftscaleFilterType;
  ShiftscaleFilterType::Pointer shiftScaleFilter = ShiftscaleFilterType::New();
  shiftScaleFilter->SetInput (image);
  shiftScaleFilter->SetScale (255);
  shiftScaleFilter->SetShift (0);
  shiftScaleFilter->Update();
  itk::OrientedImage<unsigned char,3>::Pointer img_out = shiftScaleFilter->GetOutput();

  typedef itk::ImageFileWriter<itk::OrientedImage<unsigned char,3> > WriterType;
  WriterType::Pointer writer = WriterType::New();  
  writer -> SetFileName(name.c_str());
  writer -> SetInput(img_out);
  try { 
    writer->Update();
  } 
	catch( itk::ExceptionObject & err ) {
    vcl_cout << "      ExceptionObject caught !" << vcl_endl; 
    vcl_cout << err << vcl_endl; 
    return false;
  } 
  ///vcl_printf ("\t done.\n\n");
  return true;
}

//Write an image to file.
bool save_image_double (const char* file, const ImageDType::Pointer& image)
{
  vcl_printf ("    save_image_double() to %s...\n", file);
  typedef itk::ImageFileWriter<ImageDType> WriterType;
  WriterType::Pointer writer = WriterType::New();
  writer->SetFileName (file);
  writer->SetInput (image);
  try { 
    writer->Update(); 
  } 
  catch( itk::ExceptionObject & err ) { 
    vcl_cerr << "ExceptionObject caught !" << std::endl; 
    vcl_cerr << err << std::endl; 
    return false;
  } 
  return true;
}


bool save_mem_fun_u (const vcl_string& name, 
                     const vcl_vector<ImageType::Pointer>& mem_fun_u)
{
  const int n_class = mem_fun_u.size();

  for (int k = 0; k < n_class; k++) {
    vcl_stringstream filename;  
    filename << name << k << ".mhd";
    if (save_img8 (filename.str(), mem_fun_u[k]) == false)
      return false;
  }

  return true;
}

int save_bin_pt_off (const ImageType::Pointer& image,
                     const char* filename)
{
	FILE* fp;
  if ((fp = fopen(filename, "w")) == NULL) {
    ///printf (vcl_cout, "  can't open PLY2 file %s to write.\n", file);
    return false; 
  }

  fprintf (fp, "OFF\n");

  //Count number of non-zero points.
  int total_pts = 0;
  typedef itk::ImageRegionConstIterator< ImageType > ConstIteratorType;
  ConstIteratorType it (image, image->GetRequestedRegion());
  for (it.GoToBegin(); !it.IsAtEnd(); ++it) {
    float pixel = it.Get();
    if (pixel != 0)
      total_pts++;
  }

  fprintf (fp, "%d 0 0\n", total_pts);

  //Points
  typedef itk::ImageRegionConstIteratorWithIndex< ImageType > ConstIndexIteratorType;
  ConstIndexIteratorType iit (image, image->GetRequestedRegion());
  for (iit.GoToBegin(); !iit.IsAtEnd(); ++iit) {
    float pixel = iit.Get();
    if (pixel != 0) {
      ImageType::IndexType idx = iit.GetIndex();
      fprintf (fp, "%f ", float(idx[0]));
      fprintf (fp, "%f ", float(idx[1]));
      fprintf (fp, "%f ", float(idx[2]));
      fprintf (fp, "\n");
    }
  }

  printf ("  %s totally %d points saved ...\n", 
          filename, total_pts);
  return total_pts;
}