#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <math.h>
#include <matrixSHEN.h>
#include <mvcd.h>

#define ALLPLN 1658
#define X_PLN  11
#define Y_PLN  22
#define Z_PLN  33
#define FAIL   0
#define OOKK   1
#define NNO    0
#define YYES   1

typedef struct {
  Ivector3d        pos;
  unsigned char    value;
} VOXEL;

typedef struct {
  char        t1[200], t2[200], pd[200], flair[200], mask[200];
} FILEGROUP;

typedef struct {
  unsigned char        ***t1, ***t2, ***pd, ***flair, ***mask;
} VOLUMEGROUP;

int   GetLineNumber(char filename[100]);
void  show_usage(int argc, char **argv);
void  NormalizeVector(float *vec, int vec_len);
void  ReadImg(char filename[80], unsigned short ***vol, Ivector3d dim);
void  ReadImgUC(char filename[80], unsigned char ***vol, Ivector3d dim);
void  WriteImgUC(char filename[80], unsigned char ***data, Ivector3d dim);
void  ReadImg_FLOAT(char filename[80], float ***vol, Ivector3d dim);
void  ReadFeatureVector(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim);
void  ReadFeatureVector_FLOAT(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim);
void  PrintVector(float *vec, int vec_len);
void  PrintVectorINT(int *vec, int vec_len);
void  calculate_hood_byIncreasingRadius(Ivector3d *Hood, int HoodSize, int *pixelNumInHood, float xres,float yres,float zres);
void  region_growing(unsigned char ***in, Ivector3d dim, int nbrSize, int GreySelected, int RegionPointNum, int GrowingOnPlan);
void  NeighboringSimilarPointsSearch(long int *PointNum, int ***Detected_Region, int order, unsigned char ***src, 
				     unsigned char ***status, int i, int j, int k, int GreySelected, Ivector3d dim, 
				     Ivector3d *Stack, unsigned char ***VoxelInStack, int nbrSize, int GrowingOnPlan);
void  calculate_hood_byIncreasingRadius_simple(Ivector3d *Hood, int HoodSize, int *pixelNumInHood);
int   GetMaxLabel(int *label, int label_num);
int   GetImgZDim(char file[1000], Ivector3d dim);
int   IsBoundaryVoxel(unsigned char ***img, Ivector3d dim, Ivector3d pos);
void  ClosingOperationOnBinary3DImage(unsigned char ***in, unsigned char ***out, int Radius, Ivector3d dim, float Z_THICKINK);
void  OpeningOperationOnBinary3DImage(unsigned char ***in, unsigned char ***out, int Radius, Ivector3d dim, float Z_THICKINK);
int   IsEmptyImg(unsigned char ***img, unsigned char ***dist_img, Ivector3d dim, int dist_threshold);
void  InitializeImg(unsigned char ***img, Ivector3d dim);
void  GetMaskImg(unsigned char ***img, unsigned char ***mask, Ivector3d dim);
void  FillConcaveRegionsInBinary3DImage(unsigned char ***in, unsigned char ***out, int Radius, float level, Ivector3d dim, float Z_THICKINK);
void  ComputeDistImg(unsigned char ***img, unsigned char ***dist, Ivector3d dim, Ivector3d cen, Fvector3d res);


void show_usage(int argc, char **argv)
{
   printf("USAGE: %s input.img output.img\n\
\t -d <int>,<int>              : image dimension in XY (default: 256,256)\n\
\t -v <float>,<float>,<float>  : image resolution (default: 0.9375,0.9375,3.0)\n\
\t -r <float>,<float>          : intensity ratio threshold for CSF and Eye separately (default: 0.1,0.98)\n\
\t -R <float>                  : ratio of distance to the center of image (default: 0.25)\n\
\t -i <int>                    : iteration number (default: 20)\n\
", argv[0]);
   exit(1);
}

int main(int argc, char **argv)
{
  FILE          *fp;
  int           i, j, k, ii, jj, kk, l, h, c, threshold, histo[256], max_label, ind1, ind2, label_num, label_count;
  int           sum, pixelNumIn_MyHood, total_volume, Radius, MyHoodSize, *labels, *CSFcount, max_CSFcount;
  int           t, half, CSF_threshold, EYE_threshold, partial_volume, r, iter, max_dist, dist_threshold;
  Ivector3d     dim, *MyHood, pos, cen;
  Fvector3d     res;
  extern char   *optarg;
  unsigned char ***in, ***mask_img, ***closed_img, ***tmp_img, ***diff_img, ***opened_img, ***dist;
  float         ratio, ratio_threshold, tmp, smooth_factor, EYE_ratio_threshold, CSF_ratio_threshold, CSF_ratio, tmp_ratio;
  float         totalx, totaly, totalz, ratio_distance;
  VOXEL         *voxel_list;
  char          cmd_create_mask[1000], cmd_closing[1000], cmd_diff[1000], cmd_opening[1000], cmd_mask_again[1000], cmd_create_result[1000];
  
  /* read in arguments */
  if (argc < 3) show_usage(argc, argv);
  
  dim.x = dim.y = 256; dim.z = 124;
  CSF_ratio_threshold = 0.1;  EYE_ratio_threshold = 0.98;
  res.x = res.y = 0.9375; res.z = 3.0;
  iter = 20;
  ratio_distance = 0.25;
  while((c=getopt(argc-2,argv+2,"d:v:r:i:R:")) != -1) {
    switch(c) {
    case 'd':
      sscanf(optarg, "%d,%d", &dim.x, &dim.y);
      break;

    case 'v':
      sscanf(optarg, "%f,%f,%f", &res.x, &res.y, &res.z);
      break;

    case 'r':
      sscanf(optarg, "%f,%f", &CSF_ratio_threshold, &EYE_ratio_threshold);
      break;

    case 'i':
      iter = atoi(optarg);
      break;

    case 'R':
      ratio_distance = atof(optarg);
      break;

    default:
      break;
    }
  }

  dim.z = GetImgZDim(argv[1], dim);
  printf("image dimension: (%d, %d, %d)\n(CSF, EYE) intensity threshold: (%f, %f)\n", dim.x, dim.y, dim.z, CSF_ratio_threshold, EYE_ratio_threshold);
  printf("image resolution: (%f, %f, %f)\n", res.x, res.y, res.z);
  printf("ratio of distance: %f\n", ratio_distance);
  printf("iter number: %d\n", iter);

  in         = UCalloc3d(dim.x, dim.y, dim.z);
  diff_img   = UCalloc3d(dim.x, dim.y, dim.z);
  mask_img   = UCalloc3d(dim.x, dim.y, dim.z);
  closed_img = UCalloc3d(dim.x, dim.y, dim.z);
  opened_img = UCalloc3d(dim.x, dim.y, dim.z);
  tmp_img    = UCalloc3d(dim.x, dim.y, dim.z);
  dist       = UCalloc3d(dim.x, dim.y, dim.z);
  ReadImgUC(argv[1], in, dim);
  printf("%s read ok\n", argv[1]);

  // compute histogram and image center
  cen.x = cen.y = cen.z = 0;
  totalx = totaly = totalz = 0.0;
  for (i=0; i<256; i++) histo[i] = 0;
  total_volume = 0;
  for  (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
   	    if (in[k][i][j] != 0) {
          histo[in[k][i][j]]++;
          total_volume++;
          totalx += i;
          totaly += j;
          totalz += k;
        }
//  PrintVectorINT(histo, 256);
  cen.x = (int)(totalx / total_volume);
  cen.y = (int)(totaly / total_volume);
  cen.z = (int)(totalz / total_volume);
  printf("image center: (%d, %d, %d)\n", cen.x, cen.y, cen.z);

  ComputeDistImg(in, dist, dim, cen, res);
//  WriteImgUC("dist_img", dist, dim);
  max_dist = 0;
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
        if (dist[k][i][j] > max_dist) max_dist = dist[k][i][j];
  dist_threshold = (int)(ratio_distance*max_dist);
  printf("(max dist, distance threshold): (%d, %d)\n", max_dist, dist_threshold);//exit(1);

  partial_volume = 0;  CSF_threshold = 0;
  for (i=1; i<256; i++) {
    partial_volume += histo[i];
    tmp_ratio = (float)partial_volume / total_volume;
    if (tmp_ratio>CSF_ratio_threshold) break;
  }
  CSF_threshold = i;

  partial_volume = 0;  EYE_threshold = 0;
  for (i=1; i<256; i++) {
    partial_volume += histo[i];
    tmp_ratio = (float)partial_volume / total_volume;
    if (tmp_ratio>EYE_ratio_threshold) break;
  }
  EYE_threshold = i;
  printf("(CSF, EYE) threshold: (%d, %d)\n", CSF_threshold, EYE_threshold);

  for (r=0; r<iter; r++) {
    printf("iteration: %d\n", r);
    GetMaskImg(in, mask_img, dim);
//    WriteImgUC("mask_img", mask_img, dim);
    ClosingOperationOnBinary3DImage(mask_img, closed_img, 3, dim, res.z);
//    WriteImgUC("closed_img", closed_img, dim);
    
    InitializeImg(diff_img, dim);
    for  (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
        for (j=0; j<dim.y; j++)
          if ((mask_img[k][i][j] == 255) && (closed_img[k][i][j] == 0)) diff_img[k][i][j] = 255;
//    WriteImgUC("diff_img", diff_img, dim);
    
    if (IsEmptyImg(diff_img, dist, dim, dist_threshold)) break;

    InitializeImg(tmp_img, dim);
    for  (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
        for (j=0; j<dim.y; j++) tmp_img[k][i][j] = in[k][i][j];
    for  (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
        for (j=0; j<dim.y; j++) 
          if ((in[k][i][j] > EYE_threshold) && (diff_img[k][i][j] != 0) && (dist[k][i][j] > dist_threshold)) tmp_img[k][i][j] = 0;
//    WriteImgUC("tmp_img", tmp_img, dim);

//    GetMaskImg(tmp_img, mask_img, dim);
//    WriteImgUC("mask1_img", mask_img, dim);
          
//    OpeningOperationOnBinary3DImage(mask_img, opened_img, 2, dim, res.z);
//    WriteImgUC("opened_img", opened_img, dim);
    
//    for  (k=0; k<dim.z; k++)
//      for (i=0; i<dim.x; i++)
//        for (j=0; j<dim.y; j++)
//          if ((in[k][i][j] != 0) && (opened_img[k][i][j] == 0)) in[k][i][j] = 0;
//    WriteImgUC("in1", in, dim);exit(1);
    for  (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
        for (j=0; j<dim.y; j++) in[k][i][j] = tmp_img[k][i][j];
  }

  WriteImgUC(argv[2], in, dim);

  UCfree3d(in,       dim.z, dim.x);
  UCfree3d(diff_img, dim.z, dim.x);
  UCfree3d(mask_img, dim.z, dim.x);
  UCfree3d(closed_img, dim.z, dim.x);
  UCfree3d(opened_img, dim.z, dim.x);
  UCfree3d(tmp_img, dim.z, dim.x);

  return 0;
}

void GetMaskImg(unsigned char ***img, unsigned char ***mask, Ivector3d dim)
{
  int        i, j, k;
  unsigned char ***tmp;
  
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) mask[k][i][j] = 0;

  tmp = UCalloc3d(dim.x, dim.y, dim.z);
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) 
        if (img[k][i][j] != 0) //tmp[k][i][j] = 255;
          mask[k][i][j] = 255;
        
//  FillConcaveRegionsInBinary3DImage(tmp, mask, 3, 0.3, dim, 3.0);
  UCfree3d(tmp, dim.z, dim.x);
}

void ComputeDistImg(unsigned char ***img, unsigned char ***dist, Ivector3d dim, Ivector3d cen, Fvector3d res)
{
  int        i, j, k, l, x, y, z, t, r, half;
  float      d;
  Fvector3d  pt, delta;

  printf("img center: (%d, %d, %d)\n", cen.x, cen.y, cen.z);
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) 
        if (img[k][i][j] != 0) {
          delta.x = (i - cen.x)*res.x;
          delta.y = (j - cen.y)*res.y;
          delta.z = (k - cen.z)*res.z;
          d = sqrt(delta.x*delta.x + delta.y*delta.y + delta.z*delta.z);
          dist[k][i][j] = (int)d;
      }
}

void InitializeImg(unsigned char ***img, Ivector3d dim)
{
  int        i, j, k;
  
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) img[k][i][j] = 0;
}

int IsEmptyImg(unsigned char ***img, unsigned char ***dist_img, Ivector3d dim, int dist_threshold)
{
  int        i, j, k;
  
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
        if ((img[k][i][j] != 0) && (dist_img[k][i][j] > dist_threshold)) return 0;
        
  return 1;
}

int IsBoundaryVoxel(unsigned char ***img, Ivector3d dim, Ivector3d pos)
{
  int        i, j, k, count;

  count = 0;
  if (img[pos.z][pos.x][pos.y] != 0)
    for (k=pos.z-1; k<=pos.z+1; k++)
      for (i=pos.x-1; i<=pos.x+1; i++)
	for (j=pos.y-1; j<=pos.y+1; j++)
	  if ((i>0) && (j>0) && (k>0) && (i<dim.x) && (j<dim.y) && (k<dim.z) && (img[k][i][j]==0)) count++;

  if ((count > 6) && (img[pos.z][pos.x][pos.y] != 0)) return 1;

  return 0;
}

int GetMaxLabel(int *label, int label_num)
{
  int l, max_label;

  max_label = -1;
  for (l=0; l<label_num; l++)
    if (label[l] > max_label) max_label = label[l];

  return max_label;
}

void calculate_hood_byIncreasingRadius_simple(Ivector3d *Hood, int HoodSize, int *pixelNumInHood)
{
  int        i, j, k, l ;
  int        x, y, z, t, r  ;
  int        half ; 
  t = 0;  
  Hood[0].x = Hood[0].y = Hood[0].z = 0;
  t++;
  half = HoodSize/2;

  for (k=-half; k<=half; k++)
    for (i=-half; i<=half; i++)
      for (j=-half; j<=half; j++)
	if((i*i+j*j+k*k)<=half*half) {
	  Hood[t].x = i;	      Hood[t].y = j;	      Hood[t].z = k;
	  t++;
	}
  (*pixelNumInHood) = t;
//  printf("total %d in the search area!\n", t);
}

void calculate_hood_byIncreasingRadius(Ivector3d *Hood, int HoodSize, int *pixelNumInHood, float xres,float yres,float zres)
{
  int        i, j, k, l ;
  int        x, y, z, t, r  ;
  int        half ; 

  t = 0;  
  Hood[0].x = Hood[0].y = Hood[0].z = 0;
  t++;
  half = HoodSize/2;
  for(r=1; r<=half; r++) {
      for(z=-r; z<=r; z+=2*r)
	for(x=-r; x<=r; x++)
	  for(y=-r; y<=r; y++)
	    if((x*x+y*y+z*z)<=r*r) {
	      Hood[t].x = x/xres;	      Hood[t].y = y/yres;	      Hood[t].z = z/zres;
	      t++;
	    }
      for(x=-r; x<=r; x+=2*r)
	for(z=-r+1; z<=r-1; z++) 
	  for(y=-r; y<=r; y++)
	    if((x*x+y*y+z*z)<=r*r) {
	      Hood[t].x = x/xres;	      Hood[t].y = y/yres;	      Hood[t].z = z/zres;
	      t++;
	    }
      for(y=-r; y<=r; y+=2*r)
	for(z=-r+1; z<=r-1; z++) 
	  for(x=-r+1; x<=r-1; x++) 
	    if((x*x+y*y+z*z)<=r*r) {
	      Hood[t].x = x/xres;	      Hood[t].y = y/yres;	      Hood[t].z = z/zres;
	      t++ ;
	    }
  }
  (*pixelNumInHood) = t;
//  printf("total %d in the search area!\n", t);
}

void ReadFeatureVector(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim)
{
  int                   i, j, k;
  unsigned short        ***img;

  img = USalloc3d(dim.x, dim.y, dim.z);
  ReadImg(filename, img, dim);
  for (i=0; i<vec_len; i++) vec[i] = (float)img[pos[i].z][pos[i].x][pos[i].y];

  USfree3d(img, dim.z, dim.x);
}

void ReadFeatureVector_FLOAT(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim)
{
  int          i, j, k;
  float        ***img;

  img = Falloc3d(dim.x, dim.y, dim.z);
  ReadImg_FLOAT(filename, img, dim);
  for (i=0; i<vec_len; i++) {
    vec[i] = img[pos[i].z][pos[i].x][pos[i].y];
  }

  Ffree3d(img, dim.z, dim.x);
}

void ReadImg(char filename[80], unsigned short ***vol, Ivector3d dim)
{
   FILE   *fp;
   int    i, j, k;

   fp = fopen(filename, "r");
   for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
         fread(vol[k][i], 2, dim.y, fp);
   fclose(fp);
}

void ReadImgUC(char filename[80], unsigned char ***vol, Ivector3d dim)
{
   FILE   *fp;
   int    i, j, k;

   fp = fopen(filename, "r");
   for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
         fread(vol[k][i], 1, dim.y, fp);
   fclose(fp);
}

void WriteImgUC(char filename[80], unsigned char ***data, Ivector3d dim)
{
  FILE  *fp;
  int   i, k ;

  fp=fopen(filename,"w");
  for(k=0;k<dim.z;k++)
    for(i=0;i<dim.x;i++) fwrite(data[k][i],1,dim.y,fp);
  fclose(fp);
}

void ReadImg_FLOAT(char filename[80], float ***vol, Ivector3d dim)
{
   FILE   *fp;
   int    i, j, k;

   fp = fopen(filename, "r");
   for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
         fread(vol[k][i], sizeof(float), dim.y, fp);
   fclose(fp);
}

void NormalizeVector(float *vec, int vec_len)
{
  int        i;
  float      sum;

  sum = 0.0;
  for (i=0; i<vec_len; i++) sum += vec[i]*vec[i];
  sum = sqrt(sum);
  for (i=0; i<vec_len; i++) vec[i] /= sum;
}

int GetLineNumber(char filename[100])
{
  FILE        *fp;
  char        ch;
  int         count;

  count = 0;
  fp = fopen(filename, "r");
  while (feof(fp) == 0) {
    ch = fgetc(fp);
    if (ch == '\n') count++;
  }
  fclose(fp);

  return count;
}

void PrintVector(float *vec, int vec_len)
{
  int        i;
  
  for (i=0; i<vec_len; i++) printf("%f ", vec[i]);
  printf("\n");
}

void PrintVectorINT(int *vec, int vec_len)
{
  int        i;
  
  for (i=0; i<vec_len; i++) printf("%d ", vec[i]);
  printf("\n");
}

void region_growing(unsigned char ***in, Ivector3d dim, int nbrSize, int GreySelected, int RegionPointNum, int GrowingOnPlan)
{
  unsigned char         ***status, ***VoxelInStack; 
  int                   ***Detected_Region, order;
  int                   i, j, k, ii, jj, kk;
  long int              PointNum;
  FILE                  *fp;
  int                   *PointNumInCertainOrder;
  int                   ActualUsedRegionPointNum;
  Ivector3d             *Stack;

  printf("\n\n nbrSize=%d\n GreySelected=%d\n RegionPointNum=%d\n \n", nbrSize, GreySelected, RegionPointNum) ;

  /* apply for memory */ 
  status = UCalloc3d(dim.x, dim.y, dim.z);
  Detected_Region = Ialloc3d(dim.x, dim.y, dim.z);
  PointNumInCertainOrder = Ialloc1d(dim.x*dim.y*dim.z);

  /* reset ... */
  for(i=0; i<dim.x; i++)
    for(j=0; j<dim.y; j++)
      for(k=0; k<dim.z; k++) status[k][i][j] = FAIL;
  
  /* Global variables: for growing */	
  VoxelInStack = UCalloc3d(dim.x, dim.y, dim.z);
  Stack = Ivector3dalloc1d(dim.x*dim.y*dim.z);

  /* status for representing whether current voxel already in stack? */
  for(kk=0; kk<dim.z; kk++)
    for(ii=0; ii<dim.x; ii++)
      for(jj=0; jj<dim.y; jj++) VoxelInStack[kk][ii][jj] = NNO;

  /* region growing ... */
  printf("Region growing ...\n") ;
  order = 1;
  for(k=0; k<dim.z; k++) /* It's important to use this order, z-axis first*/
    for(i=0; i<dim.x; i++)
      for(j=0; j<dim.y; j++)
	if(in[k][i][j]==GreySelected && status[k][i][j]==FAIL) {
	  PointNum = 0;
	  NeighboringSimilarPointsSearch(&PointNum, Detected_Region, order, in, status, i, j, k, GreySelected, dim, 
					 Stack, VoxelInStack, nbrSize, GrowingOnPlan);

	  if( PointNum > RegionPointNum ) printf("PointNum=%d\n", PointNum);
	  PointNumInCertainOrder[order] = PointNum;
	  order ++;
	}

  /* remove small connected regions */
  for(i=0; i<dim.x; i++)
    for(j=0; j<dim.y; j++)
      for(k=0; k<dim.z; k++) {
	
	order = Detected_Region[k][i][j];
	ActualUsedRegionPointNum = RegionPointNum;
	if(order>=1 && PointNumInCertainOrder[order]<ActualUsedRegionPointNum) in[k][i][j] = 255-GreySelected;
      }

  /* free */
  UCfree3d(status, dim.z, dim.x);
  Ifree3d(Detected_Region, dim.z, dim.x);
  free(PointNumInCertainOrder);

  /* free: Global variables */
  free(Stack);
  UCfree3d(VoxelInStack, dim.z, dim.x);
}

void NeighboringSimilarPointsSearch(long int *PointNum, int ***Detected_Region, int order, unsigned char ***src, 
				    unsigned char ***status, int i, int j, int k, int GreySelected, Ivector3d dim, 
				    Ivector3d *Stack, unsigned char ***VoxelInStack, int nbrSize, int GrowingOnPlan)
{
  int        kk,ii,jj, x, y; 
  int        k_plus_nbrSize, k_minus_nbrSize, i_plus_nbrSize, i_minus_nbrSize, j_plus_nbrSize, j_minus_nbrSize;

  /* stack */
  long int   Pointer ;

  /* add the first searched point */
  Pointer = 0; 
  Stack[Pointer].x = i; 
  Stack[Pointer].y = j; 
  Stack[Pointer].z = k; 

  do{
    i = Stack[Pointer].x;
    j = Stack[Pointer].y;
    k = Stack[Pointer].z;
    Pointer --;

    status[k][i][j] = OOKK;
    (*PointNum) ++;
    Detected_Region[k][i][j] = order;

    k_plus_nbrSize =k+nbrSize;
    k_minus_nbrSize=k-nbrSize;
    i_plus_nbrSize =i+nbrSize;
    i_minus_nbrSize=i-nbrSize;
    j_plus_nbrSize =j+nbrSize;
    j_minus_nbrSize=j-nbrSize;

    if (k_plus_nbrSize>=dim.z) k_plus_nbrSize=dim.z-1;
    if (k_minus_nbrSize<0)     k_minus_nbrSize=0;
    if (j_plus_nbrSize>=dim.y) j_plus_nbrSize=dim.y-1;
    if (j_minus_nbrSize<0)     j_minus_nbrSize=0;
    if (i_plus_nbrSize>=dim.x) i_plus_nbrSize=dim.x-1;
    if (i_minus_nbrSize<0)     i_minus_nbrSize=0;

    /* now we can check immediately adjacent points to see if they too could be added to the track */
    for(kk=k_minus_nbrSize; kk<=k_plus_nbrSize; kk++)
      for(ii=i_minus_nbrSize; ii<=i_plus_nbrSize; ii++)
	for(jj=j_minus_nbrSize; jj<=j_plus_nbrSize; jj++) {

	  if( !(kk==k && jj==j && ii==i) && 
	      ((GrowingOnPlan!=X_PLN || ii==i) && (GrowingOnPlan!=Y_PLN || jj==j) && (GrowingOnPlan!=Z_PLN || kk==k)) && 
	      (kk==k&&jj==j || kk==k&&ii==i || jj==j&&ii==i) && 
	      src[kk][ii][jj]==GreySelected && status[kk][ii][jj]==FAIL && VoxelInStack[kk][ii][jj]==NNO) {
	    Pointer ++ ;
	    Stack[Pointer].x = ii ;
	    Stack[Pointer].y = jj; 
	    Stack[Pointer].z = kk; 
	    
	    VoxelInStack[kk][ii][jj]= YYES ;
	  }
	}
  }while(Pointer>=0) ;
}

int GetImgZDim(char file[1000], Ivector3d dim)
{
  FILE        *fp;

  fp = fopen(file, "r");
  fseek(fp,0,SEEK_END);
  dim.z = (ftell(fp))/(dim.x*dim.y);
  fclose(fp);

  return dim.z;
}


void ClosingOperationOnBinary3DImage(unsigned char ***in, unsigned char ***out, int Radius, Ivector3d dim, float Z_THICKINK)
{
  unsigned char         ***Temp_Img;
  int                   i, j, k ;
  int                   l, m, n, index ;
  double                TotalWhite ;
  int                   NumInSphere, RealNum ;
  int                   x_Start, x_End, y_Start, y_End, z_Start, z_End ;
  int                   X_Start, X_End, Y_Start, Y_End, Z_Start, Z_End ;
  FILE                  *fp;
  Ivector3d             *MyHood, BBStart, BBEnd, SStart, SEnd;
  int                   MyHoodSize, pixelNumIn_MyHood ;

  InitializeImg(out, dim);
  NumInSphere = (2*Radius+1)*(2*Radius+1)*(2*Radius/Z_THICKINK+1);
  printf("Closing, radius =%d\t", Radius) ;
  fflush(stdout);

  /* spherical mask */
  MyHoodSize = Radius*2+1 ;
  MyHood     = Ivector3dalloc1d(MyHoodSize*MyHoodSize*MyHoodSize) ;
  calculate_hood_byIncreasingRadius(MyHood, MyHoodSize, &pixelNumIn_MyHood, 1., 1., Z_THICKINK) ;

  /* get bounding box */
  BBStart.x = BBStart.y = BBStart.z = 1000;
  BBEnd.x = BBEnd.y = BBEnd.z = -1000;

  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
	if (in[k][i][j] != 0) {

	  if (i<BBStart.x) BBStart.x = i;
	  if (j<BBStart.y) BBStart.y = j;
	  if (k<BBStart.z) BBStart.z = k;
	  if (i>BBEnd.x) BBEnd.x = i;
	  if (j>BBEnd.y) BBEnd.y = j;
	  if (k>BBEnd.z) BBEnd.z = k;
	}

//  printf("bounding box: (%d, %d, %d) --> (%d, %d, %d)\n", BBStart.x, BBStart.y, BBStart.z, BBEnd.x, BBEnd.y, BBEnd.z);

  /* find mask ... */
  SStart.x = BBStart.x - Radius ;  if (SStart.x<0)      SStart.x =0;
  SEnd.x   = BBEnd.x   + Radius ;  if (SEnd.x>dim.x-1)  SEnd.x   =dim.x-1 ;
  SStart.y = BBStart.y - Radius ;  if (SStart.y<0)      SStart.y =0;
  SEnd.y   = BBEnd.y   + Radius ;  if (SEnd.y>dim.y-1)  SEnd.y   =dim.y-1 ;
  SStart.z = BBStart.z - Radius ;  if (SStart.z<0)      SStart.z =0;
  SEnd.z   = BBEnd.z   + Radius ;  if (SEnd.z>dim.z-1)  SEnd.z   =dim.z-1 ;

  for(k=SStart.z; k<=SEnd.z; k++)
    for(i=SStart.x; i<=SEnd.x; i++)
      for(j=SStart.y; j<=SEnd.y; j++)
	if(in[k][i][j]==255) {
	  TotalWhite = 0 ;
	  RealNum = 0 ;
	  for( index=0; index<pixelNumIn_MyHood ; index++) {

	      l = MyHood[index].x ;
	      m = MyHood[index].y ;
	      n = MyHood[index].z ;	  
	      if( l*l+m*m+n*n<Radius*Radius && i+l>=0 && i+l<dim.x && j+m>=0 && j+m<dim.y && k+n>=0 && k+n<dim.z ) {
 		    TotalWhite += in[k+n][i+l][j+m] ;
		    RealNum ++ ;
		  
		    /* for speeding up*/
		    if( in[k+n][i+l][j+m]!=255 ) index = pixelNumIn_MyHood ;
		  }
	  }
	  TotalWhite /= RealNum;
	  
	  if( TotalWhite==255 ) out[k][i][j] = 255 ;
	  else out[k][i][j] = 0 ;
	}
	else out[k][i][j] = 0 ;

  printf("closed\n");
  /* free */
  free(MyHood) ;
}


void OpeningOperationOnBinary3DImage(unsigned char ***in, unsigned char ***out, int Radius, Ivector3d dim, float Z_THICKINK)
{
  unsigned char         ***Temp_Img;
  int                   i, j, k;
  int                   l, m, n, index;
  double                TotalWhite;
  int                   NumInSphere, RealNum;
  double                r;
  int                   x_Start, x_End, y_Start, y_End, z_Start, z_End;
  int                   X_Start, X_End, Y_Start, Y_End, Z_Start, Z_End;
  FILE                  *fp;
  Ivector3d             *MyHood, BBStart, BBEnd, SStart, SEnd;
  int                   MyHoodSize, pixelNumIn_MyHood;

  InitializeImg(out, dim);
  NumInSphere = (2*Radius+1)*(2*Radius+1)*(2*Radius/Z_THICKINK+1) ;
  printf("opening, radius =%d\t", Radius) ;
  fflush(stdout);

  /* spherical mask */
  MyHoodSize = Radius*2+1 ;
  MyHood     = Ivector3dalloc1d(MyHoodSize*MyHoodSize*MyHoodSize) ;
  calculate_hood_byIncreasingRadius(MyHood, MyHoodSize, &pixelNumIn_MyHood, 1., 1., Z_THICKINK) ;

  /* apply for space for temporary image result */
  Temp_Img = UCalloc3d(dim.x, dim.y, dim.z);
  /* reset */
  for(k=0;k<dim.z;k++)
    for(i=0;i<dim.x;i++)
      for(j=0;j<dim.y;j++) Temp_Img[k][i][j] = 255;

  /* get bounding box */
  BBStart.x = BBStart.y = BBStart.z = 1000;
  BBEnd.x = BBEnd.y = BBEnd.z = -1000;

  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
 	    if (in[k][i][j] != 0) {

	     if (i<BBStart.x) BBStart.x = i;
	     if (j<BBStart.y) BBStart.y = j;
	     if (k<BBStart.z) BBStart.z = k;
	     if (i>BBEnd.x) BBEnd.x = i;
	     if (j>BBEnd.y) BBEnd.y = j;
	     if (k>BBEnd.z) BBEnd.z = k;
	  }

  /* find mask ... */
  SStart.x = BBStart.x-2*Radius ;  if(SStart.x<0)      SStart.x=0;
  SEnd.x   = BBEnd.x+2*Radius ;    if(SEnd.x>dim.x-1)  SEnd.x  =dim.x-1;
  SStart.y = BBStart.y-2*Radius ;  if(SStart.y<0)      SStart.y=0;
  SEnd.y   = BBEnd.y+2*Radius ;    if(SEnd.y>dim.y-1)  SEnd.y  =dim.y-1;
  SStart.z = BBStart.z-2*Radius ;  if(SStart.z<0)      SStart.z=0;
  SEnd.z   = BBEnd.z+2*Radius ;    if(SEnd.z>dim.z-1)  SEnd.z  =dim.z-1;

  for(k=SStart.z; k<=SEnd.z; k++)
    for(i=SStart.x; i<=SEnd.x; i++)
      for(j=SStart.y; j<=SEnd.y; j++)
	    if( in[k][i][j]==0 ) {

	      TotalWhite = 0 ;
	      RealNum = 0 ;
	      for( index=0; index<pixelNumIn_MyHood ; index++) {

	        l = MyHood[index].x;
	        m = MyHood[index].y;
	        n = MyHood[index].z;	  
	        if( l*l+m*m+n*n<Radius*Radius &&  i+l>=0 && i+l<dim.x && j+m>=0 && j+m<dim.y && k+n>=0 && k+n<dim.z ) {
	      
 	          TotalWhite += in[k+n][i+l][j+m];
	          RealNum ++;
	      
	          /* for speeding up*/
	          if( in[k+n][i+l][j+m]!=0 ) index = pixelNumIn_MyHood;
	        }
	      }
	      TotalWhite /= 255.0;
	  
	      if( TotalWhite==0 ) Temp_Img[k][i][j] = 255;
	      else Temp_Img[k][i][j] = 0;
	    } 
	    else Temp_Img[k][i][j] = 0;
  
        /* operation ... */
       for(k=0; k<dim.z; k++)
         for(i=0;i<dim.x;i++)
           for(j=0;j<dim.y;j++) out[k][i][j] = 255 - Temp_Img[k][i][j] ;
  
  printf("opened\n");
  /* free */
  free(MyHood);
}

void FillConcaveRegionsInBinary3DImage(unsigned char ***in, unsigned char ***out, int Radius, float level, Ivector3d dim, float Z_THICKINK)
{
  unsigned char        ***Temp_Img;
  int                  i, j, k, l, m, n ;
  double               TotalWhite ;
  int                  NumInSphere, threshold ;
  int                  x_Start, x_End, y_Start, y_End, z_Start, z_End ;
  int                  X_Start, X_End, Y_Start, Y_End, Z_Start, Z_End ;
  FILE                 *fp;
 
  InitializeImg(out, dim);
  NumInSphere = (2*Radius+1)*(2*Radius+1)*(2*Radius/Z_THICKINK+1);
  threshold = NumInSphere*level;
  printf("Fill holes, radius =%d...\t", Radius);
  fflush(stdout);

  /* apply for space for temporary image result */
  Temp_Img = UCalloc3d(dim.x,dim.y,dim.z);

  /* get bounding box */
  X_Start = Y_Start = Z_Start = 1000;
  X_End = Y_End = Z_End = -1000;

  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
	if (in[k][i][j] != 0) {

	  if (i<X_Start) X_Start = i;
	  if (j<Y_Start) Y_Start = j;
	  if (k<Z_Start) Z_Start = k;
	  if (i>X_End) X_End = i;
	  if (j>Y_End) Y_End = j;
	  if (k>Z_End) Z_End = k;
	}

  /* action ... */
  x_Start = X_Start-Radius ;  if(x_Start<0)      x_Start=0;
  x_End   = X_End+Radius ;    if(x_End>dim.x-1 ) x_End  =dim.x-1 ;
  y_Start = Y_Start-Radius ;  if(y_Start<0)      y_Start=0;
  y_End   = Y_End+Radius ;    if(y_End>dim.y-1 ) y_End  =dim.y-1 ;
  z_Start = Z_Start-Radius ;  if(z_Start<0)      z_Start=0;
  z_End   = Z_End+Radius ;    if(z_End>dim.z-1 ) z_End  =dim.z-1 ;

  for(k=z_Start; k<=z_End; k++)
    for(i=x_Start; i<=x_End; i++)
      for(j=y_Start; j<=y_End; j++)
	if( in[k][i][j]==0 ) {

	  TotalWhite = 0 ;
	  for(l=-Radius; l<=Radius; l++)
	    for(m=-Radius; m<=Radius; m++)
	      for(n=-Radius/Z_THICKINK; n<=Radius/Z_THICKINK; n++)
		if(i+l>=0 && i+l<dim.x && j+m>=0 && j+m<dim.y && k+n>=0 && k+n<dim.z) TotalWhite += in[k+n][i+l][j+m];
	  TotalWhite /= 255.0 ;
	  
	  if(TotalWhite>threshold) Temp_Img[k][i][j] = 255 ;
	}
	else Temp_Img[k][i][j] = 255 ;

  /* transfer ... */
  for(k=0; k<dim.z; k++)
    for(i=0; i<dim.x; i++)
      for(j=0; j<dim.y; j++) out[k][i][j] = Temp_Img[k][i][j];

  printf("Hole filled!\n");
  /* free */
  UCfree3d(Temp_Img, dim.z, dim.x);
}
