
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string>
#include <vector>

#include "itkFixedArray.h"
#include "itkMatrix.h"
#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkShrinkImageFilter.h"
#include "itkWarpImageFilter.h"
#include "itkVectorLinearInterpolateImageFunction.h" 
#include "itkNearestNeighborInterpolateImageFunction.h" 
#include "itkHistogramMatchingImageFilter.h"
#include "itkVector.h"
//#include "../tclap/CmdLine.h"

#include "../src/itkHammerTissueAttributeVectorImageFilter.h"
#include "../src/itkHammerDeformableRegistrationImageFilter.h"
#include "../src/itkHammerIntensityAttributeVectorImageFilter.h"
//#include "../src/itkHammerWarpImageFilter.h"
#include "time.h"

//using namespace TCLAP;

typedef itk::HammerTissueAttributeVector ImgAttribute;

typedef float FloatType;
typedef itk::Vector<FloatType, 3>       ITKFvector3d;
typedef itk::Vector<int, 3>             Ivector3d;
typedef itk::Matrix<FloatType, 4, 4>    Matrix;
typedef itk::OrientedImage<unsigned char, 3>    ImageType;
typedef itk::ImageFileReader<ImageType> ReaderType;
typedef itk::ImageFileWriter<ImageType> WriterType;

typedef itk::HistogramMatchingImageFilter<ImageType, ImageType> MatchingFilterType;

typedef itk::OrientedImage<ITKFvector3d, 3>     DeformationFieldType;
typedef itk::OrientedImage<ImgAttribute, 3>     AttributeImageType;
typedef itk::HammerTissueAttributeVectorImageFilter<ImageType, AttributeImageType> AttributeFilterType;

unsigned char Geom_UP = static_cast<unsigned char>(0.9*255) ;
unsigned char Geom_DN = static_cast<unsigned char>(0.4*255) ;
unsigned char VNvlm_UP = static_cast<unsigned char>(255/12) ;
unsigned char VNvlm_DN = 250 ; /* ~65% added on Dec 2006 */
unsigned char CSFBG_UP = static_cast<unsigned char>(255/4) ;

unsigned char Geom_DownsUp = static_cast<unsigned char>(Geom_DN/1.1) ; /*1.5*/

void
splitString (const std::string &text,
             const std::string &separators,
             std::vector<std::string> &words)
{
  const std::string::size_type n = text.length();
  std::string::size_type start = text.find_first_not_of(separators);
  while (start < n)
    {
    std::string::size_type stop = text.find_first_of(separators, start);
    if (stop > n) stop = n;
    words.push_back(text.substr(start, stop - start));
    start = text.find_first_not_of(separators, stop+1);
    }
}

void DownSample( ImageType::Pointer image,
                ImageType::Pointer oImage,
                float downSampleFactor)
{

  oImage->CopyInformation( image );

  ImageType::RegionType region = oImage->GetLargestPossibleRegion();
  ImageType::SpacingType spacing = oImage->GetSpacing();

  for (int k = 0; k < 3; k++)
  {
    float extent = static_cast<float>(region.GetSize(k))*spacing[k];
    spacing[k] *= downSampleFactor;
    int nPixel = static_cast<int>(extent/spacing[k]);
    region.SetSize(k, nPixel);
  }

  oImage->SetSpacing( spacing );
  oImage->SetRegions( region );
  oImage->Allocate();
  oImage->FillBuffer( 0 );

  itk::NearestNeighborInterpolateImageFunction< ImageType, double >::Pointer interpolator = 
    itk::NearestNeighborInterpolateImageFunction< ImageType, double >::New();

  interpolator->SetInputImage( image );
  
  itk::ImageRegionIteratorWithIndex<ImageType> itOut( oImage, region );
  for (itOut.GoToBegin(); !itOut.IsAtEnd(); ++itOut)
  {
    ImageType::IndexType idx = itOut.GetIndex();
    ImageType::PointType pt;
    
    oImage->TransformIndexToPhysicalPoint( idx, pt );
    itk::ContinuousIndex<double, 3> cIdx;
    image->TransformPhysicalPointToContinuousIndex( pt, cIdx );

    if( image->GetLargestPossibleRegion().IsInside(cIdx) )
    {
      itOut.Set( static_cast<ImageType::PixelType> (interpolator->Evaluate(pt)) );
    }
  }
}

int main(int argc,char *argv[])  
{
	float XYZres = 1. ;
	int nThreads = 1;
	std::string fixedFilename;
	std::string movingFilename;
	std::string resampledFilename;
	std::string deformationFieldname;
	std::string moving_cbq_Filename;
	std::string resampled_cbq_Filename;
    std::string iterationasstring = "50,50,50";
	std::string modalitystring = "0";
	std::string softstring = "0";
	std::string tpsstring = "0";

	unsigned char modalitytype = 0;
	unsigned char softmode = 0;
	unsigned char tpsmode = 0;

	itk::OStringStream msg;

    unsigned int iterations[3];
    iterations[0] = 20; iterations[1] = 20; iterations[2] = 20;

	try
	{
		CmdLine cl ( "HAMMER Registration Test: Heirarchical Attribute Matching Mechanism for Elastic Registration, NA-MIC Kit",
			' ',
			"$Revision: 1.2 $" );

		// parse registration parameters from command line arguments
		msg.str ( "" ); msg << "Number of threads (default: " << nThreads << ")";
		ValueArg<int> ThreadArg ( "T", "Thread", msg.str(), false, nThreads, "int", cl );
		msg.str ( "" ); msg << "sample rate (default: " << XYZres << ")";
		ValueArg<float> SampleRateArg ( "s", "Sample", msg.str(), false, XYZres, "float", cl );

		UnlabeledValueArg<std::string> FixedImageArg ( "fixed", "Fixed image filename", 1, fixedFilename, "string", cl );
		UnlabeledValueArg<std::string> MovingImageArg ( "moving", "Moving image filename", 1, movingFilename, "string", cl );
		UnlabeledValueArg<std::string> MovingCbqImageArg ( "moving_cbq", "Moving intensity filename", 1, moving_cbq_Filename, "string", cl);
		UnlabeledValueArg<std::string> ResampleImageArg ( "resampled", "Resampled image filename", 1, resampledFilename, "string", cl );
		UnlabeledValueArg<std::string> ResampleCbqImageArg ("resampled_cbq", "Resampled intensity image filename", 1, resampled_cbq_Filename, "string", cl);
		UnlabeledValueArg<std::string> DeformationFieldArg ( "deformation", "Deformation field filename", 1, deformationFieldname, "string", cl );
		UnlabeledValueArg<std::string> ModalityArg("i", "the modality indicator", 1, modalitystring, "string", cl);
		UnlabeledValueArg<std::string> SoftArg("s", "the soft correspondence indicator", 1, softstring, "string", cl);
		UnlabeledValueArg<std::string> TPSArg("t", "the TPS indicator", 1, tpsstring, "string", cl);
		printf("begin parsing the command.\n");
		cl.parse ( argc, argv );

		// parse registration parameters from command line arguments
		nThreads = ThreadArg.getValue();
		XYZres =  SampleRateArg.getValue();

		fixedFilename = FixedImageArg.getValue();
		movingFilename = MovingImageArg.getValue();
		moving_cbq_Filename = MovingCbqImageArg.getValue();
		resampledFilename = ResampleImageArg.getValue();
		resampled_cbq_Filename = ResampleCbqImageArg.getValue();
		deformationFieldname = DeformationFieldArg.getValue();
		modalitystring = ModalityArg.getValue();
		softstring = SoftArg.getValue();
		tpsstring = TPSArg.getValue();
	} 
	catch ( ArgException exception ) {
		std::cerr << "error: " << exception.error() << " for arg " << exception.argId() << std::endl;
		exit ( EXIT_FAILURE );
	}


	printf("\nmodel image : %s\n", fixedFilename.c_str()) ;  
	printf("subject image : %s\n", movingFilename.c_str()) ;  
	printf("subject intensity image : %s\n", moving_cbq_Filename.c_str());
	printf("warped image : %s\n", resampledFilename.c_str());
	printf("warped intensity image %s: \n", resampled_cbq_Filename.c_str());
	printf("deforamtion filed %s:\n", deformationFieldname.c_str());
	if(!strcmp(modalitystring.c_str(), "0"))
	{
		printf("Tissue segmented hammer.\n");
		modalitytype = 0;
	}
	else
	{
		printf("Intensity hammer.\n");
		modalitytype = 1;
	}
	if(!strcmp(softstring.c_str(), "0"))
	{
		printf("does not use soft correspondence.\n");
		softmode = 0;
	}
	else
	{
		printf("use soft correspondence.\n");
		softmode = 1;
	}
	if(!strcmp(tpsstring.c_str(),"0"))
	{
		printf("does not using TPS.\n");
		tpsmode = 0;
	}
	else
	{
		tpsmode = 1;
		printf("using TPS.\n");
	}
	/*** Load in fixed image and compute the attribute vectors ***/
	ReaderType::Pointer ImgReader = ReaderType::New();
	ImgReader->SetFileName( fixedFilename.c_str() );
	try
	{
		ImgReader->Update();
	}
	catch( itk::ExceptionObject *ex )
	{
		std::cerr << ex << std::endl;
	}
	ImageType::Pointer fImg0 = ImgReader->GetOutput();
	fImg0->DisconnectPipeline();
	std::cout << "Fixed image file read in\n";

	/*** Load in moving image and compute the attribute vectors ***/
	ImgReader->SetFileName( movingFilename.c_str() );
	try
	{
		ImgReader->Update();
	}
	catch( itk::ExceptionObject *ex )
	{
		std::cerr << ex << std::endl;
	}
	ImageType::Pointer mImg0 = ImgReader->GetOutput();
	mImg0->DisconnectPipeline();
	std::cout << "Moving image file read in\n";


	if(modalitytype == 1)
	{
		//do histogram matching;
		printf("Performing histogram matching...\n");
		MatchingFilterType::Pointer matcher = MatchingFilterType::New();
		matcher->SetInput(mImg0);
		matcher->SetReferenceImage(fImg0);
		matcher->SetNumberOfHistogramLevels(256);
		matcher->SetNumberOfMatchPoints(7);
		matcher->ThresholdAtMeanIntensityOn();
		try
		{
			matcher->Update();
		}
		catch(itk::ExceptionObject *ex)
		{
			std::cerr << ex << std::endl;
		}
		mImg0 = matcher->GetOutput();
		mImg0->DisconnectPipeline();
	}
	
	/*** Load in moving intensity image  ***/
	ImgReader->SetFileName( moving_cbq_Filename.c_str() );
	try
	{
		ImgReader->Update();
	}
	catch( itk::ExceptionObject *ex )
	{
		std::cerr << ex << std::endl;
	}
	ImageType::Pointer mCbqImg0 = ImgReader->GetOutput();
	mCbqImg0->DisconnectPipeline();
	std::cout << "Moving intensity image file read in\n";

	typedef itk::HammerDeformableRegistrationImageFilter<
		ImageType,
		DeformationFieldType> RegistrationFilterType;
	RegistrationFilterType::Pointer hammer = RegistrationFilterType::New();

	// hammer only need to know the fixed image, moving image, and 
	// how attribute vectors are computed (also similarity computation)
	// to perform registration

	hammer->SetFixedImage( fImg0 );
	hammer->SetMovingImage( mImg0 );

	// ----------------------------------------------------------
	// define an attribute vector filter object and set it to the 
	// hammer registration filter so the registration know   
	// 1. How to compute attribute vectors
	// 2. How to compute similarity between attribute vectors
	AttributeFilterType::Pointer avFilter = AttributeFilterType::New();
	// hammer->SetAttributeVectorFilter( avFilter );
	// ----------------------------------------------------------


	// ----------------------------------------------------------
	// the following lines set parameters for HAMMER registration
	// all these parameters should have default values
	hammer->SetIntensitymode(modalitytype);
	hammer->SetTPSmode(tpsmode);
	hammer->SetSoftmode(softmode);
	hammer->SetupBasicParameters();
	hammer->SetIterations( iterations[0], iterations[1], iterations[2] );	
	hammer->SetDeformRate(0.05);
	if(modalitytype == 0)
	{
		hammer->SetPointMatching_Initial(0.8);
		hammer->SetSubvolumnSimilarityThreshold(0.6);
	}
	else
	{
		hammer->SetPointMatching_Initial(0.6);
		hammer->SetSubvolumnSimilarityThreshold(0.75);
	}
	hammer->SetSearchRadius(12);
	// hammer->SetNumberOfMaximumDrivingVoxels( 5000 );
	// hammer->SetSigma ( 2 ); 
	// ----------------------------------------------------------

 	clock_t Start = clock();
 	hammer->Update();
 	clock_t End = clock();

	std::cout<<"Execution time = "<<(End-Start)/CLOCKS_PER_SEC<<"s"<<std::endl;

	typedef itk::WarpImageFilter<
		ImageType, 
		ImageType,
		DeformationFieldType  >     WarperType;

	typedef itk::NearestNeighborInterpolateImageFunction<
		ImageType,
		double >  NearestInterpolatorType;

	typedef itk::LinearInterpolateImageFunction<
		ImageType,
		double >  LinearInterpolatorType;

	WarperType::Pointer warper = WarperType::New();
	WarperType::Pointer warper2 = WarperType::New();
	NearestInterpolatorType::Pointer interpolator = NearestInterpolatorType::New();
	LinearInterpolatorType::Pointer linearinterpolator = LinearInterpolatorType::New();

	warper->SetInput( mImg0 );
	warper->SetInterpolator( interpolator );
	warper->SetOutputDirection( fImg0->GetDirection() );
	warper->SetOutputSpacing( fImg0->GetSpacing() );
	warper->SetOutputOrigin( fImg0->GetOrigin() );
	if(modalitytype == 0)
		warper->SetInterpolator( interpolator );
	else
		warper->SetInterpolator( linearinterpolator );

	//warper->SetOutputStartIndex( fImg->GetLargestPossibleRegion().GetIndex() );
	//warper->SetOutputSize( fImg->GetLargestPossibleRegion().GetSize() );

    warper->SetDeformationField(hammer->GetOutput());


	WriterType::Pointer      writer =  WriterType::New();
	writer->SetFileName( resampledFilename.c_str() );
	writer->SetInput( warper->GetOutput()   );
	writer->Update();

	warper2->SetInput(mCbqImg0);
	if(modalitytype==0)
		warper2->SetInterpolator( linearinterpolator );
	else
		warper2->SetInterpolator( interpolator );
	warper2->SetOutputDirection( mCbqImg0->GetDirection() );
	warper2->SetOutputSpacing( mCbqImg0->GetSpacing() );
	warper2->SetOutputOrigin( mCbqImg0->GetOrigin() );
	warper2->SetDeformationField(hammer->GetOutput());
	writer->SetFileName( resampled_cbq_Filename.c_str() );
	writer->SetInput( warper2->GetOutput()   );
	writer->Update();

	typedef itk::ImageFileWriter< DeformationFieldType > FieldWriterType;
	FieldWriterType::Pointer fieldWriter = FieldWriterType::New();

	fieldWriter->SetFileName( deformationFieldname.c_str() );
	fieldWriter->SetInput( hammer->GetOutput() );
	fieldWriter->Update();

	itk::VectorLinearInterpolateImageFunction<DeformationFieldType, double>::Pointer
		dfInterpolator = itk::VectorLinearInterpolateImageFunction<DeformationFieldType, double>::New();

	dfInterpolator->SetInputImage( hammer->GetOutput() );

	itk::Image<float, 3>::Pointer magField = itk::Image<float, 3>::New();
	magField->CopyInformation( fImg0 );
	magField->SetRegions( magField->GetLargestPossibleRegion() );
	magField->Allocate();

	DeformationFieldType::Pointer fullResDeformationField = DeformationFieldType::New();
	fullResDeformationField->CopyInformation( fImg0 );
	fullResDeformationField->SetRegions( fullResDeformationField->GetLargestPossibleRegion() );
	fullResDeformationField->Allocate();

	itk::ImageRegionIteratorWithIndex<DeformationFieldType> 
		itFld( fullResDeformationField, fullResDeformationField->GetLargestPossibleRegion() );

	for (itFld.GoToBegin(); !itFld.IsAtEnd(); ++itFld)
	{
		DeformationFieldType::IndexType idx = itFld.GetIndex();
		DeformationFieldType::PointType pt;
		fullResDeformationField->TransformIndexToPhysicalPoint( idx, pt );

		itk::ContinuousIndex<double, 3> cIdx;
		hammer->GetOutput()->TransformPhysicalPointToContinuousIndex( pt, cIdx );

		if( hammer->GetOutput()->GetLargestPossibleRegion().IsInside(cIdx) )
		{
			DeformationFieldType::PixelType p = dfInterpolator->Evaluate( pt );
			itFld.Set( p );
			magField->SetPixel( idx, p.GetNorm() );
		}
	}

	deformationFieldname = "FullRes" + deformationFieldname;
	fieldWriter->SetFileName( deformationFieldname.c_str() );
	fieldWriter->SetInput( fullResDeformationField );
	fieldWriter->Update();

	itk::ImageFileWriter< itk::Image<float, 3> >::Pointer fWriter = 
		itk::ImageFileWriter< itk::Image<float, 3> >::New();

	deformationFieldname = "Mag" + deformationFieldname;
	fWriter->SetFileName( deformationFieldname.c_str() );
	fWriter->SetInput( magField );
	fWriter->Update();

	return 0;
}



