/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: itkHammerIntensityAttributeVectorImageFilter.txx,v $
  Language:  C++
  Date:      $Date: 2009/01/14 21:46:50 $
  Version:   $Revision: 1.6 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for
  details.

  This program is developed under NIH NCBC collaboration grant
  R01 EB006733, "Development and Dissemination of Robust Brain MRI
  Measurement Tools". 

  This software is distributed WITHOUT ANY WARRANTY; without even 
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
  PURPOSE.  See the above copyright notices for more information.

  =========================================================================*/
#ifndef __itkHammerIntensityAttributeVectorImageFilter_txx
#define __itkHammerIntensityAttributeVectorImageFilter_txx
#include "itkHammerIntensityAttributeVectorImageFilter.h"
#include "itkRescaleIntensityImageFilter.h";
#include "itkCastImageFilter.h";
#include "itkImageRegionIterator.h"
#include "itkNeighborhoodAlgorithm.h"
//#include "itkCannyEdgeStrengthImageFilter.h"
#include "itkCannyEdgeDetectionImageFilter2.h"
#include "itkZeroFluxNeumannBoundaryCondition.h"
#include "itkProgressReporter.h"

namespace itk
{

  //
  // Constructor
  //
  template <class TInputImage, class TOutputImage>
  HammerIntensityAttributeVectorImageFilter<TInputImage, TOutputImage>
  ::HammerIntensityAttributeVectorImageFilter()
  { 
    this->m_UseImageSpacing   = true;
    this->m_Scale = 5;
    this->m_Strength = 1;
    this->m_Sigma = 1;

    this->m_OffsetInSphericalNeighborhood.clear();
    m_variance = 2.0;
    m_upperThreshold = 200.0;
    m_lowerThreshold = 0.0;


#if defined(ITK_IMAGE_BEHAVES_AS_ORIENTED_IMAGE)
    this->m_UseImageDirection = true;
#else
    this->m_UseImageDirection = false;
#endif
  }


  //
  // Destructor
  //
  template <class TInputImage, class TOutputImage>
  HammerIntensityAttributeVectorImageFilter<TInputImage, TOutputImage>
  ::~HammerIntensityAttributeVectorImageFilter()
  {
  }


  template <class TInputImage, class TOutputImage>
  void
  HammerIntensityAttributeVectorImageFilter<TInputImage, TOutputImage>
  ::GenerateInputRequestedRegion() throw (InvalidRequestedRegionError)
  {
    // call the superclass' implementation of this method
    Superclass::GenerateInputRequestedRegion();

    // get pointers to the input and output
    InputImagePointer  inputPtr = 
      const_cast< InputImageType * >( this->GetInput());
    OutputImagePointer outputPtr = this->GetOutput();

    if ( !inputPtr || !outputPtr )
      {
	return;
      }

    // pad input requested region by 1, so the simple edge detection
    // works on the entire image domain
    unsigned long radius = 1;
    typename InputImageType::SpacingType inputSpacing = this->GetInput()->GetSpacing();
    Size<InputImageDimension> sphereRadius;
    for (int i = 0; i < InputImageDimension; ++i)
      {
	sphereRadius[i] = static_cast<unsigned long>( this->m_Scale/inputSpacing[i] );
	if (sphereRadius[i]>radius)
	  {
	    radius = sphereRadius[i];
	  }
      }

    // compute spherical neighborhood for geometrical attribute
    // computation
    typename InputImageType::Pointer dummyImage = InputImageType::New();
    typename InputImageType::RegionType dummyRegion;
    typename InputImageType::PointType dummyOrigin;
    typename InputImageType::IndexType dummyStart;
    typename InputImageType::SizeType dummySize;

    dummyImage->SetSpacing( inputSpacing );
    for (int k = 0; k < InputImageDimension; k++)
      {
	dummySize[k] = sphereRadius[k]+sphereRadius[k]+1;
	dummyStart[k] = -sphereRadius[k];
	dummyOrigin[k] = 0;
      }
    dummyRegion.SetIndex( dummyStart );
    dummyRegion.SetSize( dummySize );
    dummyImage->SetRegions( dummyRegion );
    dummyImage->SetOrigin( dummyOrigin );

    float radiusSqr = this->m_Scale * this->m_Scale;
    itk::ImageRegionIteratorWithIndex<InputImageType> it( dummyImage, dummyRegion );
    for (it.GoToBegin(); !it.IsAtEnd(); ++it)
      {
	typename InputImageType::IndexType dummyIdx = it.GetIndex();
	typename InputImageType::PointType point;
	dummyImage->TransformIndexToPhysicalPoint( dummyIdx, point );
	float d = 0;
	for (int k = 0; k < InputImageDimension; k++)
	  {
	    d += point[k]*point[k];
	  }
	if (d > radiusSqr)
	  {
	    continue;
	  }
	else
	  {
	    NeighborOffsetType offset;
	    for (int k = 0; k < InputImageDimension; k++)
	      {
		offset[k] = dummyIdx[k];
	      }
	    this->m_OffsetInSphericalNeighborhood.push_back(offset);
	  }
      }

    // get a copy of the input requested region (should equal the output
    // requested region)
    typename TInputImage::RegionType inputRequestedRegion;
    inputRequestedRegion = inputPtr->GetRequestedRegion();

    // pad the input requested region by the operator radius
    inputRequestedRegion.PadByRadius( radius );

    // crop the input requested region at the input's largest possible region
    if ( inputRequestedRegion.Crop(inputPtr->GetLargestPossibleRegion()) )
      {
	inputPtr->SetRequestedRegion( inputRequestedRegion );
	return;
      }
    else
      {
	// Couldn't crop the region (requested region is outside the largest
	// possible region).  Throw an exception.

	// store what we tried to request (prior to trying to crop)
	inputPtr->SetRequestedRegion( inputRequestedRegion );

	// build an exception
	InvalidRequestedRegionError e(__FILE__, __LINE__);
	e.SetLocation(ITK_LOCATION);
	e.SetDescription("Requested region is (at least partially) outside the largest possible region.");
	e.SetDataObject(inputPtr);
	throw e;
      }
  }

  template <class TInputImage, class TOutputImage>
  void
  HammerIntensityAttributeVectorImageFilter<TInputImage, TOutputImage>
  ::CreateN1Neighbor()
  {
    m_N1Neighborhood.resize(6);
    for(int k=0;k<6;k++)
      for(int s=0;s<InputImageDimension;s++)
	m_N1Neighborhood[k][s] = 0;
    m_N1Neighborhood[0][0] = -1;
    m_N1Neighborhood[1][0] = 1;
    m_N1Neighborhood[2][1] = -1;
    m_N1Neighborhood[3][1] = 1;
    m_N1Neighborhood[4][2] = -1;
    m_N1Neighborhood[5][2] = 1;
  }

  template <class TInputImage, class TOutputImage>
  void
  HammerIntensityAttributeVectorImageFilter<TInputImage, TOutputImage>
  ::CreateFeatureNeighbor(int Radius)
  {
    //InputSpacingType spacing = this->GetInput()->GetSpacing();
    float rad_sqrd=Radius * Radius;
    int i, j, k;
    NeighborOffsetType offset;

    m_FeatureNeighborhood.clear();
    for(i=-Radius;i<=Radius;i++)
      for(j=-Radius;j<=Radius;j++)
	for(k=-Radius;k<=Radius;k++)
	  if((i*i+j*j+k*k)<=rad_sqrd)
	    {
	      offset[0] = static_cast<long int>( i );
	      offset[1] = static_cast<long int>( j );
	      offset[2] = static_cast<long int>( float(k)/1.5 );
	      m_FeatureNeighborhood.push_back(offset);
	    }
  }

  template <class TInputImage, class TOutputImage>
  void
  HammerIntensityAttributeVectorImageFilter<TInputImage, TOutputImage>
  ::GenerateData()
  {
    this->AllocateOutputs();

    typename TOutputImage::PixelType attributeVector;
    typename TOutputImage::PixelType MaxAttribute;
    InputIndexType idx, cur_idx;
    // Get the input and output
    OutputImageType *       outputImage = this->GetOutput();
    const InputImageType *  inputImage  = this->GetInput();
    typename InputImageType::SpacingType inputSpacing = inputImage->GetSpacing();
    InputRegionType dummyRegion = inputImage->GetLargestPossibleRegion();
    //Create the neighbor
    CreateN1Neighbor();
    CreateFeatureNeighbor(static_cast<int>(m_Scale));

    // Compute Canny Edge Strength
    typedef itk::OrientedImage<double, InputImageDimension>    RealImageType;
    typedef itk::CastImageFilter< InputImageType, RealImageType> CastToRealFilterType;
    typedef itk::RescaleIntensityImageFilter<RealImageType, InputImageType > RescaleFilterType;
    //typedef itk::CannyEdgeStrengthImageFilter<RealImageType, RealImageType> CannyEdgeDetectionFilterType;
    typedef itk::CannyEdgeDetectionImageFilter2<RealImageType, RealImageType> CannyEdgeDetectionFilterType;
    typedef itk::ImageFileWriter< InputImageType >  WriterType;
	
    typename CastToRealFilterType::Pointer toReal = CastToRealFilterType::New();
    typename RescaleFilterType::Pointer rescale = RescaleFilterType::New();
    typename CannyEdgeDetectionFilterType::Pointer canny = CannyEdgeDetectionFilterType::New();
    typename WriterType::Pointer writer = WriterType::New();

    rescale->SetOutputMinimum(   0 );
    rescale->SetOutputMaximum( 255 );

    toReal->SetInput(inputImage);
    // 
    // 	RealImageType::Pointer m_Test = RealImageType::New();
    // 	toReal->Update();
    // 	m_Test = toReal->GetOutput();

    canny->SetInput( toReal->GetOutput() );
    canny->SetVariance( 3 );
    canny->SetUpperThreshold( 0 );
    canny->SetLowerThreshold( 0 );

    try 
      {
	canny->Update();
      }
    catch( itk::ExceptionObject & err ) 
      { 
	std::cout << "ExceptionObject caught in canny computation!" << std::endl; 
	std::cout << err << std::endl; 
      } 
	
    ImageRegionConstIteratorWithIndex<InputImageType> src_it( inputImage, inputImage->GetLargestPossibleRegion() );
    ImageRegionIteratorWithIndex<RealImageType> real_canny_it(canny->GetOutput(), canny->GetOutput()->GetLargestPossibleRegion());

    for(src_it.GoToBegin(),real_canny_it.GoToBegin();!src_it.IsAtEnd();++src_it,++real_canny_it)
      {
	if(src_it.Get()==0)
	  real_canny_it.Set(0);
      }

    rescale->SetInput( canny->GetOutput() );

    writer->SetFileName("CannyEdge.hdr");
    writer->SetInput( rescale->GetOutput() );

    try 
      {
	writer->Update();
      }
    catch( itk::ExceptionObject & err ) 
      { 
	std::cout << "ExceptionObject caught in canny computation!" << std::endl; 
	std::cout << err << std::endl; 
      } 
    InputImagePointer m_CannyEdge = InputImageType::New();
    m_CannyEdge = rescale->GetOutput();


    //Initialize
    ImageRegionIteratorWithIndex<OutputImageType> it( outputImage, outputImage->GetLargestPossibleRegion() );
    ImageRegionConstIteratorWithIndex<InputImageType> source( inputImage, inputImage->GetLargestPossibleRegion() );
    ImageRegionConstIteratorWithIndex<InputImageType> canny_it( m_CannyEdge, m_CannyEdge->GetLargestPossibleRegion() );
    attributeVector.Fill( 0 );
    float edge_num = 0;
    /*
      [0] variance of intensity(Edge)
      [1] the intensity (Tiss)
      [2] mean (Goem)
      [3] contrast (VNvlm)
      [4] canny (CSFBG)
    */
    float EdgeHist[256];
    for(int t=0;t<256;t++)
      EdgeHist[t] = 0;
    for(it.GoToBegin(),source.GoToBegin(),canny_it.GoToBegin();!it.IsAtEnd();++it,++source,++canny_it)
      {
	attributeVector[1] = source.Get();

	if(attributeVector[1] == 0)
	  attributeVector[4] = 0;
	else
	  {
	    attributeVector[4] = canny_it.Get();
	    EdgeHist[canny_it.Get()] += 1.0;
	    if(canny_it.Get()!=0)
	      edge_num+=1.0;
	  }		
	it.Set(attributeVector);	
      }
    printf("edge_num=%d\n", static_cast<int>(edge_num));
    float subTotal = 0;
    for(int t=1;t<255;t++)
      {
	subTotal += EdgeHist[t] ;
	if( subTotal/edge_num>=m_PercentOfLowCannyEdge/*0.25*/ ) /* threshold for getting strong edges */
	  {
	    m_CannyMinimalRequiredEdgeValue = t ;
	    break ;
	  }
      }
    printf("m_CannyMinimalRequiredEdgeValue = %d\n", m_CannyMinimalRequiredEdgeValue);

    //calculat the image feature
    float total_num, total_intensity, mean_value, variance_value;
    float Cluster_Low, Cluster_High, Num_Cluster_Low, Num_Cluster_High, contrast;
    for(it.GoToBegin();!it.IsAtEnd();++it)		                   
      {
	attributeVector = it.Get();
	idx = it.GetIndex();
	if(attributeVector.GetTissueType()==0)
	  continue;		
	//step 1 calculat the intensity mean and variance
	total_num = 0;
	total_intensity = 0;
	for(unsigned int t=0;t<m_FeatureNeighborhood.size();t++)
	  {
	    for(int s=0;s<InputImageDimension;s++)
	      cur_idx[s] = idx[s] + (int)m_FeatureNeighborhood[t][s];
	    if ( !dummyRegion.IsInside(cur_idx) )
	      {
		continue;
	      }
	    total_intensity += inputImage->GetPixel(cur_idx);
	    total_num += 1.0;
	  }
	if(total_num>0)
	  mean_value = total_intensity/total_num;
	else
	  mean_value = 0;
	if(mean_value>=255)
	  attributeVector[2] = 255;
	else
	  attributeVector[2] = (unsigned char)(mean_value);

	total_num = 0;
	total_intensity = 0;
	for(unsigned int t=0;t<m_FeatureNeighborhood.size();t++)
	  {
	    for(int s=0;s<InputImageDimension;s++)
	      cur_idx[s] = idx[s] + (int)m_FeatureNeighborhood[t][s];
	    if ( !dummyRegion.IsInside(cur_idx) )
	      {
		continue;
	      }
	    float diff = inputImage->GetPixel(cur_idx) - mean_value;
	    total_intensity += (diff*diff);
	    total_num += 1.0;
	  }
	if(total_num>0)
	  variance_value = total_intensity/total_num;
	else
	  variance_value = total_intensity/total_num;
	if(variance_value>=255)
	  attributeVector[0] = 255;
	else
	  attributeVector[0] = (unsigned char)(variance_value);
	//printf("%d %d\t", attributeVector[2], attributeVector[0]);

	//step 2 get the contrast
	Cluster_Low = 0 ;
	Cluster_High = 0 ;
	Num_Cluster_Low = 0 ; 
	Num_Cluster_High = 0;
	total_num = 0;
	total_intensity = 0;
	for(unsigned int t=0;t<m_FeatureNeighborhood.size();t++)
	  {
	    for(int s=0;s<InputImageDimension;s++)
	      cur_idx[s] = idx[s] + (int)m_FeatureNeighborhood[t][s];
	    if ( !dummyRegion.IsInside(cur_idx) )
	      {
		continue;
	      }
			
	    float curr_value = inputImage->GetPixel(cur_idx);
	    if(curr_value>mean_value)
	      {
		Num_Cluster_High ++ ;
		Cluster_High += curr_value ;
	      }
	    else
	      {
		Num_Cluster_Low ++ ;
		Cluster_Low += curr_value ;
	      }
	  }
	if( Num_Cluster_High>0 )
	  Cluster_High /= Num_Cluster_High ;
	if( Num_Cluster_Low>0 )
	  Cluster_Low /= Num_Cluster_Low ;
	contrast = Cluster_High - Cluster_Low ;
	if( contrast>=255 )  
	  attributeVector[3] = 255 ;
	else
	  attributeVector[3] = (unsigned char)contrast ;
	it.Set(attributeVector);
      }//end of it	
    //step3 nomalization
    for(int t=0;t<5;t++)
      MaxAttribute[t] = 1;
    for(it.GoToBegin();!it.IsAtEnd();++it)
      {
	attributeVector = it.Get();
	for(int t=0;t<5;t++)
	  {
	    if(attributeVector[t]>MaxAttribute[t])
	      MaxAttribute[t] = attributeVector[t];
	  }
      }
    /*
      [0] variance of intensity(Edge)
      [1] the intensity (Tiss)
      [2] mean (Goem)
      [3] contrast (VNvlm)
      [4] canny (CSFBG)
    */
    printf("MaxAttribute.Tiss=%d\n",  MaxAttribute[1]) ; 
    printf("MaxAttribute.Edge=%d\n",  MaxAttribute[0]) ; 
    printf("MaxAttribute.Geom=%d\n",  MaxAttribute[2]) ; 
    printf("MaxAttribute.VNvlm=%d\n", MaxAttribute[3]) ; 
    printf("MaxAttribute.CSFBG=%d\n", MaxAttribute[4]) ; 
    for(it.GoToBegin();!it.IsAtEnd();++it)
      {
	attributeVector = it.Get();
	for(int t=0;t<5;t++)
	  {
	    float tmp = attributeVector[t]*255.0/MaxAttribute[t];
	    if(tmp>255)
	      attributeVector[t] = 255;
	    else
	      attributeVector[t] = (unsigned char)tmp;
	  }
	it.Set(attributeVector);
      }
  }

  /**
   * Standard "PrintSelf" method
   */
  template <class TInputImage, class TOutputImage>
  void
  HammerIntensityAttributeVectorImageFilter<TInputImage, TOutputImage>
  ::PrintSelf(std::ostream& os, Indent indent) const
  {
    Superclass::PrintSelf( os, indent );

    os << indent << "UseImageSpacing: " 
       << (this->m_UseImageSpacing ? "On" : "Off") << std::endl;
    os << indent << "UseImageDirection = " 
       << (this->m_UseImageDirection ? "On" : "Off") << std::endl;
  }

} // end namespace itk


#endif
