/*=========================================================================

Program:   Insight Segmentation & Registration Toolkit
Module:    $RCSfile: itkAttributeVectorImageFilter.txx,v $
Language:  C++
Date:      $Date: 2009/01/14 21:46:50 $
Version:   $Revision: 1.6 $

Copyright (c) Insight Software Consortium. All rights reserved.
See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for
details.

This program is developed under NIH NCBC collaboration grant
R01 EB006733, "Development and Dissemination of Robust Brain MRI
Measurement Tools". 

This software is distributed WITHOUT ANY WARRANTY; without even 
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkAttributeVectorImageFilter_txx
#define __itkAttributeVectorImageFilter_txx
#include "itkAttributeVectorImageFilter.h"

#include "itkImageRegionIterator.h"
#include "itkNeighborhoodAlgorithm.h"
#include "itkZeroFluxNeumannBoundaryCondition.h"
#include "itkProgressReporter.h"

namespace itk
{

//
// Constructor
//
template <class TInputImage, class TOutputImage>
AttributeVectorImageFilter<TInputImage, TOutputImage>
::AttributeVectorImageFilter()
{ 
  this->m_UseImageSpacing   = true;
  this->m_Scale = 5;
  this->m_Strength = 1;
  this->m_GMValue = 150;
  this->m_WMValue = 250;
  this->m_CSFValue = 10;
  this->m_VNValue = 50;
  this->m_BGValue = 0;

  this->m_OffsetInSphericalNeighborhood.clear();

#if defined(ITK_IMAGE_BEHAVES_AS_ORIENTED_IMAGE)
  this->m_UseImageDirection = true;
#else
  this->m_UseImageDirection = false;
#endif
}


//
// Destructor
//
template <class TInputImage, class TOutputImage>
AttributeVectorImageFilter<TInputImage, TOutputImage>
::~AttributeVectorImageFilter()
{
}


template <class TInputImage, class TOutputImage>
void
AttributeVectorImageFilter<TInputImage, TOutputImage>
::GenerateInputRequestedRegion() throw (InvalidRequestedRegionError)
{
  // call the superclass' implementation of this method
  Superclass::GenerateInputRequestedRegion();

  // get pointers to the input and output
  InputImagePointer  inputPtr = 
    const_cast< InputImageType * >( this->GetInput());
  OutputImagePointer outputPtr = this->GetOutput();

  if ( !inputPtr || !outputPtr )
  {
    return;
  }

  // pad input requested region by 1, so the simple edge detection
  // works on the entire image domain
  unsigned long radius = 1;
  typename InputImageType::SpacingType inputSpacing = this->GetInput()->GetSpacing();
  Size<InputImageDimension> sphereRadius;
  for (int i = 0; i < InputImageDimension; ++i)
  {
    sphereRadius[i] = static_cast<unsigned long>( this->m_Scale/inputSpacing[i] );
    if (sphereRadius[i]>radius)
    {
      radius = sphereRadius[i];
    }
  }

  // compute spherical neighborhood for geometrical attribute
  // computation
  typename InputImageType::Pointer dummyImage = InputImageType::New();
  typename InputImageType::RegionType dummyRegion;
  typename InputImageType::PointType dummyOrigin;
  typename InputImageType::IndexType dummyStart;
  typename InputImageType::SizeType dummySize;

  dummyImage->SetSpacing( inputSpacing );
  for (int k = 0; k < InputImageDimension; k++)
  {
    dummySize[k] = sphereRadius[k]+sphereRadius[k]+1;
    dummyStart[k] = -sphereRadius[k];
    dummyOrigin[k] = 0;
  }
  dummyRegion.SetIndex( dummyStart );
  dummyRegion.SetSize( dummySize );
  dummyImage->SetRegions( dummyRegion );
  dummyImage->SetOrigin( dummyOrigin );

  float radiusSqr = this->m_Scale * this->m_Scale;
  itk::ImageRegionIteratorWithIndex<InputImageType> it( dummyImage, dummyRegion );
  for (it.GoToBegin(); !it.IsAtEnd(); ++it)
  {
    typename InputImageType::IndexType dummyIdx = it.GetIndex();
    typename InputImageType::PointType point;
    dummyImage->TransformIndexToPhysicalPoint( dummyIdx, point );
    float d = 0;
    for (int k = 0; k < InputImageDimension; k++)
    {
      d += point[k]*point[k];
    }
    if (d > radiusSqr)
    {
      continue;
    }
    else
    {
      NeighborOffsetType offset;
      for (int k = 0; k < InputImageDimension; k++)
      {
        offset[k] = dummyIdx[k];
      }
      this->m_OffsetInSphericalNeighborhood.push_back(offset);
    }
  }

  // get a copy of the input requested region (should equal the output
  // requested region)
  typename TInputImage::RegionType inputRequestedRegion;
  inputRequestedRegion = inputPtr->GetRequestedRegion();

  // pad the input requested region by the operator radius
  inputRequestedRegion.PadByRadius( radius );

  // crop the input requested region at the input's largest possible region
  if ( inputRequestedRegion.Crop(inputPtr->GetLargestPossibleRegion()) )
  {
    inputPtr->SetRequestedRegion( inputRequestedRegion );
    return;
  }
  else
  {
    // Couldn't crop the region (requested region is outside the largest
    // possible region).  Throw an exception.

    // store what we tried to request (prior to trying to crop)
    inputPtr->SetRequestedRegion( inputRequestedRegion );

    // build an exception
    InvalidRequestedRegionError e(__FILE__, __LINE__);
    e.SetLocation(ITK_LOCATION);
    e.SetDescription("Requested region is (at least partially) outside the largest possible region.");
    e.SetDataObject(inputPtr);
    throw e;
  }
}


template <class TInputImage, class TOutputImage>
void
AttributeVectorImageFilter<TInputImage, TOutputImage>
::ThreadedGenerateData(const OutputImageRegionType& outputRegionForThread,
           int threadId)
{
  unsigned int i;
  typename TOutputImage::PixelType attributeVector;
  ZeroFluxNeumannBoundaryCondition<InputImageType> nbc;

  NeighborhoodIteratorType nit;
  NeighborhoodIteratorType sit;
  ImageRegionIterator<OutputImageType> it;

  // Get the input and output
  OutputImageType *       outputImage = this->GetOutput();
  const InputImageType *  inputImage  = this->GetInput();
  typename InputImageType::SpacingType inputSpacing = inputImage->GetSpacing();

  // Initiate iterator radius to 1 for simple edge computation
  Size<InputImageDimension> radius;
  Size<InputImageDimension> sphereRadius; 
  for (i = 0; i < InputImageDimension; ++i)    
  {
    radius[i]  = 1;
    sphereRadius[i] = static_cast<size_t>( this->m_Scale/inputSpacing[i] );
  }

  this->m_Scale = this->m_Scale*this->m_Scale;
  // Find the data-set boundary "faces"
  typename NeighborhoodAlgorithm::ImageBoundaryFacesCalculator<InputImageType>::FaceListType faceList;
  NeighborhoodAlgorithm::ImageBoundaryFacesCalculator<InputImageType> bC;
  faceList = bC(inputImage, outputRegionForThread, radius);

  typename NeighborhoodAlgorithm::ImageBoundaryFacesCalculator<InputImageType>::FaceListType::iterator fit;
  fit = faceList.begin();


  // support progress methods/callbacks
  ProgressReporter progress(this, threadId, outputRegionForThread.GetNumberOfPixels());

  // Process non-boundary face and then each of the boundary faces.  
  // These are N-d regions which border the edge of the buffer.
  for (fit=faceList.begin(); fit != faceList.end(); ++fit)
  {
    nit = NeighborhoodIteratorType(radius, inputImage, *fit);
    sit = NeighborhoodIteratorType(sphereRadius, inputImage, *fit);

    it = ImageRegionIterator<OutputImageType>(outputImage, *fit);
    nit.OverrideBoundaryCondition(&nbc);
    nit.GoToBegin();

    sit.OverrideBoundaryCondition(&nbc);
    sit.GoToBegin();

    const unsigned int neighborhoodSize = nit.Size();

    while ( ! nit.IsAtEnd() )
    {
      typename InputImageType::PixelType centerPixel = nit.GetCenterPixel();

      attributeVector.Fill( 0 );
      attributeVector[1] = centerPixel;

      // the center voxel is a background pixel 
      if ( centerPixel != this->m_WMValue && centerPixel != this->m_GMValue && centerPixel != this->m_VNValue && centerPixel != this->m_CSFValue )
      {
        it.Set( attributeVector );
        ++sit;
        ++nit;
        ++it;
        progress.CompletedPixel();
        continue;
      }

      if (nit.GetIndex()[0] == 5 && nit.GetIndex()[1] == 5)
      {
        std::cout << nit.GetIndex() << std::endl;
      }
      // process the 3X3X3 neighborhood and find the edge type;
      unsigned long WMCounter = 0;
      unsigned long GMCounter = 0;
      unsigned long VNCounter = 0;
      unsigned long CSFCounter = 0;

      for (unsigned int i = 0; i < neighborhoodSize; ++i)
      {
        unsigned char p = nit.GetPixel(i);
        if (p == this->m_WMValue)
        {
          WMCounter ++;
        }
        if (p == this->m_GMValue)
        {
          GMCounter ++;
        }
        if (p == this->m_VNValue)
        {
          VNCounter ++;
        }
        if (p == this->m_CSFValue || p == this->m_BGValue)
        {
          CSFCounter ++;
        }
      }

      if ( centerPixel == this->m_WMValue && GMCounter > VNCounter )
      {
        attributeVector[0] = this->m_WMGMEDGE;
      }
      else if ( centerPixel == this->m_WMValue && VNCounter > 0 )
      {
        attributeVector[0] = this->m_WMVNEDGE;        
      }
      else if ( centerPixel == this->m_GMValue && VNCounter >= this->m_Strength )
      {
        attributeVector[0] = this->m_GMVNEDGE; 
      }
      else if ( centerPixel == this->m_GMValue && CSFCounter >= this->m_Strength )
      {
        attributeVector[0] = this->m_GMCSFBGEDGE; 
      }

      // process the larger neighborhood to calculate the geometry
      // this is called the bubble in the original program
      unsigned long NumDown = 0;
      unsigned long CSFBGvalue = 0;
      VNCounter = 0;
      unsigned long nVoxelsInSpehericalNeighborhood = this->m_OffsetInSphericalNeighborhood.size();
      for (unsigned int i = 0; i < nVoxelsInSpehericalNeighborhood; ++i)
      {
        NeighborOffsetType offset = this->m_OffsetInSphericalNeighborhood[i];
        unsigned char p = sit.GetPixel(offset);

        if (p != this->m_WMValue)
        {
          NumDown ++;
        }
        if (p == this->m_VNValue)
        {
          VNCounter ++;          
        }
        if (p == this->m_CSFValue || p == this->m_BGValue)
        {
          CSFBGvalue ++;
        }
      }

      float value = ((float)NumDown*255*1.2/(float)nVoxelsInSpehericalNeighborhood) ;
      if( value>255 ) value = 255 ;
      attributeVector[2] = static_cast<unsigned char>( value );

      value = ((float)VNCounter*255*1.2/(float)nVoxelsInSpehericalNeighborhood);
      if( value>255 ) value = 255 ;
      attributeVector[3] = static_cast<unsigned char>( value );

      value = ((float)CSFBGvalue*255*1.2/(float)nVoxelsInSpehericalNeighborhood);
      if( value>255 ) value = 255 ;
      attributeVector[4] = static_cast<unsigned char>( value );

      it.Set( attributeVector );

      ++sit;
      ++nit;
      ++it;
      progress.CompletedPixel();
    }
  }
}

/**
  * Standard "PrintSelf" method
  */
template <class TInputImage, class TOutputImage>
void
AttributeVectorImageFilter<TInputImage, TOutputImage>
::PrintSelf(std::ostream& os, Indent indent) const
{
  Superclass::PrintSelf( os, indent );

  os << indent << "UseImageSpacing: " 
     << (this->m_UseImageSpacing ? "On" : "Off") << std::endl;
  os << indent << "UseImageDirection = " 
     << (this->m_UseImageDirection ? "On" : "Off") << std::endl;
}

} // end namespace itk


#endif
