
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string>
#include <vector>

#include "itkFixedArray.h"
#include "itkMatrix.h"
#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkShrinkImageFilter.h"

#include "itkVector.h"
#include "../../../utilities/tclap/CmdLine.h"

#include "../plugin/itkHammerTissueAttributeVectorImageFilter.h"

using namespace TCLAP;

typedef itk::HammerTissueAttributeVector ImgAttribute;

typedef float FloatType;
typedef itk::Vector<FloatType, 3>       ITKFvector3d;
typedef itk::Vector<int, 3>             Ivector3d;
typedef itk::Matrix<FloatType, 4, 4>    Matrix;
typedef itk::Image<unsigned char, 3>    ImageType;
typedef itk::Image<ITKFvector3d, 3>     DeformationFieldType;
typedef itk::Image<ImgAttribute, 3>     AttributeImageType;

int main(int argc,char *argv[])
  
{
  float XYZres = 1. ;
  int nThreads = 1;
  std::string fixedFilename;

  itk::OStringStream msg;

  try
    {
      CmdLine cl ( "HAMMER Attribute Vector Test: Heirarchical Attribute Matching Mechanism for Elastic Registration, NA-MIC Kit",
        ' ',
        "$Revision: 1.2 $" );

    msg.str ( "" ); msg << "Number of threads (default: " << nThreads << ")";
    ValueArg<int> ThreadArg ( "T", "Thread", msg.str(), false, nThreads, "int", cl );

    msg.str ( "" ); msg << "sample rate (default: " << XYZres << ")";
    ValueArg<float> SampleRateArg ( "s", "Sample", msg.str(), false, XYZres, "float", cl );
	    
    UnlabeledValueArg<std::string> FixedImageArg ( "fixed", "Fixed image filename", 1, fixedFilename, "string", cl );

    cl.parse ( argc, argv );
    nThreads = ThreadArg.getValue();
    XYZres =  SampleRateArg.getValue();

    fixedFilename = FixedImageArg.getValue();
    } 
  catch ( ArgException exception ) {
  std::cerr << "error: " << exception.error() << " for arg " << exception.argId() << std::endl;
  exit ( EXIT_FAILURE );
  }

  double scale = 7*(1./XYZres) ; if(scale<3) scale=3; printf("scale=%f\n", scale) ;

  /***** Model image, segmented *****/
  /*Img_XY = 256 ;*/	 

  printf("\nmodel image : %s\n", fixedFilename.c_str()) ;  

  /*** Load in fixed image and compute the attribute vectors ***/
  itk::ImageFileReader<ImageType>::Pointer ImgReader = itk::ImageFileReader<ImageType>::New();
  ImgReader->SetFileName( fixedFilename.c_str() );
  try
    {
    ImgReader->Update();
    }
  catch( itk::ExceptionObject *ex )
    {
    std::cerr << ex << std::endl;
    }

  typedef itk::ShrinkImageFilter<ImageType, ImageType> DownSampleType;
  DownSampleType::Pointer downsample = DownSampleType::New();
  downsample->SetInput( ImgReader->GetOutput() );
  for (int k = 0; k < 3; k++)
    {
    downsample->SetShrinkFactor( k, static_cast<int> (XYZres) );
    }
  downsample->Update();

  ImageType::Pointer Img = downsample->GetOutput();
  Img->DisconnectPipeline();
  std::cout << "Fixed image file read in\n";

  typedef itk::HammerTissueAttributeVectorImageFilter<ImageType, AttributeImageType> AttributeFilterType;
  AttributeFilterType::Pointer modleAttributeFilter = AttributeFilterType::New();
  modleAttributeFilter->SetInput( Img );
  modleAttributeFilter->SetBGValue( 0 );
  modleAttributeFilter->SetGMValue( 150 );
  modleAttributeFilter->SetWMValue( 250 );
  modleAttributeFilter->SetVNValue( 50 );
  modleAttributeFilter->SetCSFValue( 10 );

  modleAttributeFilter->SetNumberOfThreads( nThreads );
  modleAttributeFilter->SetStrength( 1 );
  modleAttributeFilter->SetScale( scale );
  modleAttributeFilter->Update();

  AttributeImageType::Pointer fixedAVec = modleAttributeFilter->GetOutput();
  fixedAVec->DisconnectPipeline();

  // identify the edge point for cost function computation
  itk::ImageRegionIteratorWithIndex<AttributeImageType> 
    itAV( fixedAVec, 
                 fixedAVec->GetLargestPossibleRegion() );
  
  ImageType::Pointer fixedImageEdgePointLabel = ImageType::New();
  fixedImageEdgePointLabel->CopyInformation( Img );
  fixedImageEdgePointLabel->SetRegions( Img->GetLargestPossibleRegion() );
  fixedImageEdgePointLabel->Allocate();
  fixedImageEdgePointLabel->FillBuffer( 0 );

  std::vector<ImageType::PointType> edgePointInFixedImage;
  std::vector<ImgAttribute> edgePointInFixedImageAttribute;
  
  for ( int k = 3; k <= 6; k++)
    {
    float r = static_cast<float>(k)/6.0;
    unsigned char Geom_UP = static_cast<unsigned char>(0.9*255) ;
    unsigned char Geom_DN = static_cast<unsigned char>(0.4*255) ;
    unsigned char VNvlm_UP = static_cast<unsigned char>(255/12) ;
    unsigned char VNvlm_DN = 170 ; /* ~65% added on Dec 2006 */
    unsigned char CSFBG_UP = static_cast<unsigned char>(255/4) ;
    if (k > 3)
      {
      Geom_UP = static_cast<unsigned char>(0.9*(1-r)*255) ;
      Geom_DN = static_cast<unsigned char>(0.4*2.0*r*255) ;
      VNvlm_UP = static_cast<unsigned char>(255/4) ;
      float s = (170.0 +(255.0-170.0)*r*1.5); /* ~65% added on Dec 2006 */
      if ( s > 255 )
        {
        VNvlm_DN = 255;
        }
      else 
        {
        VNvlm_DN = static_cast<unsigned char>( s );
        }
      CSFBG_UP = static_cast<unsigned char>(255/4) ;
      }
    unsigned char Geom_DownsUp = static_cast<unsigned char>(Geom_DN/1.1) ; /*1.5*/

    std::cout << "r: " << r << std::endl;

    std::cout << "Geom_UP: " << static_cast<int>(Geom_UP) << std::endl;
    std::cout << "Geom_DN: " << static_cast<int>(Geom_DN) << std::endl;
    std::cout << "VNvlm_UP: " << static_cast<int>( VNvlm_UP) << std::endl;
    std::cout << "VNvlm_DN: " << static_cast<int>(VNvlm_DN) << std::endl;
    std::cout << "CSFBG_UP: " << static_cast<int>(CSFBG_UP) << std::endl;
    std::cout << "Geom_DownsUp: " << static_cast<int>(Geom_DownsUp) << std::endl;

    for ( itAV.GoToBegin(); !itAV.IsAtEnd(); ++itAV )
      {
      ImgAttribute a = itAV.Get();
      if (a[0] == 0)
        {
        continue;
        }
      
      if (a[2]<=Geom_UP 
          && (a[2]>=Geom_DN || a[2]<=Geom_DownsUp) 
          && a[3]<=VNvlm_UP 
          && a[4]<=CSFBG_UP) 
        {
        continue;
        }
      if (a[3] > VNvlm_DN)
        {
        continue;
        }
      AttributeImageType::IndexType idx = itAV.GetIndex();
      AttributeImageType::PointType point;
      fixedAVec->TransformIndexToPhysicalPoint( idx, point );
      
      edgePointInFixedImage.push_back( point );
      edgePointInFixedImageAttribute.push_back( a );
      if (fixedImageEdgePointLabel->GetPixel(idx) == 0)
        {
        fixedImageEdgePointLabel->SetPixel( idx, k-2 );
        }
      }
    std::cout << "Picked edge points in fixed image:\n"
              << edgePointInFixedImage.size() << std::endl;
    edgePointInFixedImage.clear();
    edgePointInFixedImageAttribute.clear();

    }

  itk::ImageFileWriter<ImageType>::Pointer w = itk::ImageFileWriter<ImageType>::New();
  w->SetFileName( "FixedImageEdgePointsPicked.mhd");
  w->SetInput( fixedImageEdgePointLabel );
  w->Update();

  return 0;
}



