
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string>
#include <vector>

#include "itkFixedArray.h"
#include "itkMatrix.h"
#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkShrinkImageFilter.h"

#include "itkConstNeighborhoodIterator.h"
#include "itkNeighborhoodAlgorithm.h"
#include "itkVector.h"
#include "itkOffset.h"
#include "itkSize.h"

#include "../../../utilities/tclap/CmdLine.h"

#include "../plugin/itkHammerTissueAttributeVectorImageFilter.h"

using namespace TCLAP;

typedef itk::HammerTissueAttributeVector ImgAttribute;

typedef float FloatType;
typedef itk::Vector<FloatType, 3>       ITKFvector3d;
typedef itk::Vector<int, 3>             Ivector3d;
typedef itk::Matrix<FloatType, 4, 4>    Matrix;
typedef itk::Image<unsigned char, 3>    ImageType;
typedef itk::Image<ITKFvector3d, 3>     DeformationFieldType;
typedef itk::Image<ImgAttribute, 3>     AttributeImageType;

unsigned char Geom_UP = static_cast<unsigned char>(0.9*255) ;
unsigned char Geom_DN = static_cast<unsigned char>(0.4*255) ;
unsigned char VNvlm_UP = static_cast<unsigned char>(255/12) ;
unsigned char VNvlm_DN = 170 ; /* ~65% added on Dec 2006 */
unsigned char CSFBG_UP = static_cast<unsigned char>(255/4) ;

unsigned char Geom_DownsUp = static_cast<unsigned char>(Geom_DN/1.1) ; /*1.5*/

int main(int argc,char *argv[])  
{
	float XYZres = 1. ;
	int nThreads = 1;
	std::string fixedFilename;
	std::string movingFilename;
	float MatchBallRadius = 9; // mm

	itk::OStringStream msg;

	try
	{
		CmdLine cl ( "HAMMER Attribute Vector Test: Heirarchical Attribute Matching Mechanism for Elastic Registration, NA-MIC Kit",
			' ',
			"$Revision: 1.2 $" );

		msg.str ( "" ); msg << "Number of threads (default: " << nThreads << ")";
		ValueArg<int> ThreadArg ( "T", "Thread", msg.str(), false, nThreads, "int", cl );

		msg.str ( "" ); msg << "sample rate (default: " << XYZres << ")";
		ValueArg<float> SampleRateArg ( "s", "Sample", msg.str(), false, XYZres, "float", cl );

		msg.str ( "" ); msg << "Matching neighborhood radius (default: " << MatchBallRadius << ")";
		ValueArg<float> RadiusArg ( "r", "radius", msg.str(), false, MatchBallRadius, "float", cl );

		UnlabeledValueArg<std::string> FixedImageArg ( "fixed", "Fixed image filename", 1, fixedFilename, "string", cl );
		UnlabeledValueArg<std::string> MovingImageArg ( "moving", "Moving image filename", 1, movingFilename, "string", cl );

		cl.parse ( argc, argv );
		nThreads = ThreadArg.getValue();
		XYZres =  SampleRateArg.getValue();
		MatchBallRadius = RadiusArg.getValue();
		fixedFilename = FixedImageArg.getValue();
		movingFilename = MovingImageArg.getValue();
	} 
	catch ( ArgException exception ) {
		std::cerr << "error: " << exception.error() << " for arg " << exception.argId() << std::endl;
		exit ( EXIT_FAILURE );
	}

	double scale = 7*(1./XYZres) ; if(scale<3) scale=3; printf("scale=%f\n", scale) ;

	/***** Model image, segmented *****/
	/*Img_XY = 256 ;*/	 

	printf("\nmodel image : %s\n", fixedFilename.c_str()) ;  
	printf("subject image : %s\n", movingFilename.c_str()) ;  

	/*** Load in fixed image and compute the attribute vectors ***/
	itk::ImageFileReader<ImageType>::Pointer ImgReader = itk::ImageFileReader<ImageType>::New();
	ImgReader->SetFileName( fixedFilename.c_str() );
	try
	{
		ImgReader->Update();
	}
	catch( itk::ExceptionObject *ex )
	{
		std::cerr << ex << std::endl;
	}

	typedef itk::ShrinkImageFilter<ImageType, ImageType> DownSampleType;
	DownSampleType::Pointer downsample = DownSampleType::New();
	downsample->SetInput( ImgReader->GetOutput() );
	for (int k = 0; k < 3; k++)
	{
		downsample->SetShrinkFactor( k, static_cast<int> (XYZres) );
	}
	downsample->Update();

	ImageType::Pointer Img = downsample->GetOutput();
	Img->DisconnectPipeline();
	std::cout << "Fixed image file read in\n";

	typedef itk::HammerTissueAttributeVectorImageFilter<ImageType, AttributeImageType> AttributeFilterType;
	AttributeFilterType::Pointer modleAttributeFilter = AttributeFilterType::New();
	modleAttributeFilter->SetInput( Img );
	modleAttributeFilter->SetBGValue( 0 );
	modleAttributeFilter->SetGMValue( 150 );
	modleAttributeFilter->SetWMValue( 250 );
	modleAttributeFilter->SetVNValue( 50 );
	modleAttributeFilter->SetCSFValue( 10 );

	modleAttributeFilter->SetNumberOfThreads( nThreads );
	modleAttributeFilter->SetStrength( 1 );
	modleAttributeFilter->SetScale( scale );
	modleAttributeFilter->Update();

	AttributeImageType::Pointer fixedAVec = modleAttributeFilter->GetOutput();
	fixedAVec->DisconnectPipeline();

	/*** Load in moving image and compute the attribute vectors ***/
	ImgReader->SetFileName( movingFilename.c_str() );
	try
	{
		ImgReader->Update();
	}
	catch( itk::ExceptionObject *ex )
	{
		std::cerr << ex << std::endl;
	}

	downsample->SetInput( ImgReader->GetOutput() );
	for (int k = 0; k < 3; k++)
	{
		downsample->SetShrinkFactor( k, static_cast<int> (XYZres) );
	}
	downsample->Update();

	ImageType::Pointer mImg = downsample->GetOutput();
	mImg->DisconnectPipeline();
	std::cout << "Moving image file read in\n";

	AttributeFilterType::Pointer subjectAttributeFilter = AttributeFilterType::New();
	subjectAttributeFilter->SetInput( mImg );
	subjectAttributeFilter->SetBGValue( 0 );
	subjectAttributeFilter->SetGMValue( 150 );
	subjectAttributeFilter->SetWMValue( 250 );
	subjectAttributeFilter ->SetVNValue( 50 );
	subjectAttributeFilter->SetCSFValue( 10 );

	subjectAttributeFilter->SetNumberOfThreads( nThreads );
	subjectAttributeFilter->SetStrength( 1 );
	subjectAttributeFilter->SetScale( scale );
	subjectAttributeFilter->Update();

	AttributeImageType::Pointer movingAVec = subjectAttributeFilter->GetOutput();
	movingAVec->DisconnectPipeline();

	AttributeImageType::IndexType idxInFixedImage;
	idxInFixedImage[0] = 110/XYZres;
	idxInFixedImage[1] = 82/XYZres;
	idxInFixedImage[2] = 66/XYZres;

	ImgAttribute fixedImageAttributeAtPickedPoint = fixedAVec->GetPixel( idxInFixedImage );
	for (int k = 0; k < 5; k++)
	{
		std::cout << static_cast<int>(fixedImageAttributeAtPickedPoint[k]) << "   ";
	}
	std::cout << "\n";

	// 
	typedef itk::Image<float, 3> FloatImageType;
	// prepare a ball neighborhood
	typedef itk::Offset<3> NeighborOffsetType;
	std::vector<NeighborOffsetType> nbrOffsets;

	unsigned long radius = 1;
	ImageType::SpacingType inputSpacing = Img->GetSpacing();
	itk::Size<3> sphereRadius;
	for (int i = 0; i < 3; ++i)
	{
		sphereRadius[i] = static_cast<unsigned long>( MatchBallRadius/inputSpacing[i] );
		if (sphereRadius[i]>radius)
		{
			radius = sphereRadius[i];
		}
	}

	// compute spherical neighborhood for geometrical attribute
	// computation
	ImageType::Pointer dummyImage = ImageType::New();
	ImageType::RegionType dummyRegion;
	ImageType::PointType dummyOrigin;
	ImageType::IndexType dummyStart;
	ImageType::SizeType dummySize;

	dummyImage->SetSpacing( inputSpacing );
	for (int k = 0; k < 3; k++)
	{
		dummySize[k] = sphereRadius[k]+sphereRadius[k]+1;
		dummyStart[k] = -sphereRadius[k];
		dummyOrigin[k] = 0;
	}
	dummyRegion.SetIndex( dummyStart );
	dummyRegion.SetSize( dummySize );
	dummyImage->SetRegions( dummyRegion );
	dummyImage->SetOrigin( dummyOrigin );

	float radiusSqr = MatchBallRadius*MatchBallRadius;
	itk::ImageRegionIteratorWithIndex<ImageType> it( dummyImage, dummyRegion );
	for (it.GoToBegin(); !it.IsAtEnd(); ++it)
	{
		ImageType::IndexType dummyIdx = it.GetIndex();
		ImageType::PointType point;
		dummyImage->TransformIndexToPhysicalPoint( dummyIdx, point );
		float d = 0;
		for (int k = 0; k < 3; k++)
		{
			d += point[k]*point[k];
		}
		if (d > radiusSqr)
		{
			continue;
		}
		else
		{
			NeighborOffsetType offset;
			for (int k = 0; k < 3; k++)
			{
				offset[k] = dummyIdx[k];
			}
			nbrOffsets.push_back(offset);
			std::cout << offset << std::endl;
		}
	}

	std::cout << nbrOffsets.size() << std::endl;

	FloatImageType::Pointer sim = FloatImageType::New();
	sim->CopyInformation( Img );
	sim->SetRegions( sim->GetLargestPossibleRegion() );
	sim->Allocate();

	itk::ImageRegionIteratorWithIndex<FloatImageType>
		itsim( sim, sim->GetLargestPossibleRegion() );
	for (itsim.GoToBegin(); !itsim.IsAtEnd(); ++itsim)
	{
		FloatImageType::IndexType idx = itsim.GetIndex();
		ImgAttribute mAVc = movingAVec->GetPixel( idx );

		// the following "if" make computation much faster, but make
		// the resulting similarity map patchy
		//if (mAVc[0] == 0)
		//{
		//  itsim.Set( 0 );
		//  continue;
		//}

		float simvalue = 0;
		unsigned int nVoxelInNbr = 0;
		for (unsigned int k = 0; k < nbrOffsets.size(); k++)
		{
			NeighborOffsetType offset = nbrOffsets[k];
			FloatImageType::IndexType fIdx;
			FloatImageType::IndexType mIdx;

			for (int m=0; m < 3; m++)
			{
				fIdx[m] = idxInFixedImage[m]+offset[m];
				mIdx[m] = idx[m]+offset[m];
			}

			if (!Img->GetLargestPossibleRegion().IsInside(fIdx) || !mImg->GetLargestPossibleRegion().IsInside(mIdx))
			{
				continue;
			} 

			ImgAttribute fAV = fixedAVec->GetPixel( fIdx );
			ImgAttribute mAV = movingAVec->GetPixel( mIdx );

      simvalue += fAV.ComputeSimilarity( mAV );
			nVoxelInNbr ++;
		}
		simvalue /= static_cast<float>( nVoxelInNbr );
		itsim.Set( simvalue );
	}

	itk::ImageFileWriter<FloatImageType>::Pointer fw = itk::ImageFileWriter<FloatImageType>::New();
	fw->SetInput( sim );
	fw->SetFileName( "Similarity.mhd" );
	fw->Update();

	return 0;
}



