/*=========================================================================

Program:   AFCM
Module:    $RCSfile: afcm_exec.cxx,v $
Language:  C++
Date:      $Date: 2009/03/11 18:53:56 $
Version:   $Revision: 1.1 $

Copyright (c) General Electric Global Research. All rights reserved.

This software is distributed WITHOUT ANY WARRANTY; without even 
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the above copyright notices for more information.

=========================================================================*/

#include <vcl_cstdio.h>
#include <vcl_vector.h>
#include <vcl_iostream.h>
#include <vcl_sstream.h>

#include "itkBinaryThresholdImageFilter.h"
#include "itkConnectedComponentImageFilter.h"
#include "itkRelabelComponentImageFilter.h"

#include <AFCM/afcm.h>
#include <AFCM/afcm_grid.h>
#include <AFCM/afcm_util.h>
#include <AFCM/afcm_im_util.h>
#include <AFCM/afcm_io.h>

#include "cmd_opt_proc.h"

char* input_im_arg = NULL;
char* output_im_arg = "output";
int   perform_smooth_arg = 0;
int   n_class_arg = 3;
int   n_bin_arg = 200;
float bg_thresh_arg = 0.01f; //1e-3f
float l_th_arg = 25.0f;
float u_th_arg = 1500.0f;
char* input_mask_arg = NULL;
int   gain_fit_arg = 0;
float gain_th_arg = 0.8f; //0.5f;
float gain_min_arg = 0.8f; //0.5f
int   n_grid_arg = 3; //

static OPTTABLE optab [] = {
  {"i", _STRING, &input_im_arg, "input image"},
  {"o", _STRING, &output_im_arg, "output image base"},
  {"s", _INT, &perform_smooth_arg, "perform smoothing"},
  {"nc", _INT, &n_class_arg, "number of classes"},
  {"nb", _INT, &n_bin_arg, "number of bins for mixed model fitting"},
  {"bt", _FLOAT, &bg_thresh_arg, "background threshold of intensity"},
  {"lt", _FLOAT, &l_th_arg, "lower threshold of intensity"},
  {"ut", _FLOAT, &u_th_arg, "upper threshold of intensity"},
  {"m", _STRING, &input_mask_arg, "input mask image"},
  {"g", _INT, &gain_fit_arg, "perform gain field fitting\n\
        0: no gain field correction, 1: linear fitting, 2: quadratic fitting.\n\
        3: grid division for linear fitting, 4: grid division for quadratic fitting."},
  {"gt", _FLOAT, &gain_th_arg, "intensity threshold for gain fitting"},  
  {"gm", _FLOAT, &gain_min_arg, "minimum value of fitting"},
  {"ng", _INT, &n_grid_arg, "number of grid in one direction in division"},
};
static int optab_size = 12;

int main (int argc, char* argv[])
{
  //Parse the command line parameters.
  //Exit program in case the parameter is incomplete.
  if (opt_parse_args (argc, argv, optab, optab_size) == false) {
    vcl_printf ("  No command line option specified.\n");
    vcl_printf ("  Use %s -h for more help.\n", argv[0]);
    return EXIT_FAILURE;
  }

  //Check for required command line parameters.
  if (input_im_arg == NULL) {
    vcl_printf ("Error: input image (-i) required.\n");
    return EXIT_FAILURE;
  }
  if (output_im_arg == NULL) {
    vcl_printf ("Error: output image base (-o) required.\n");
    return EXIT_FAILURE;
  }
         
  //Read in the input image file to image img_y[].
  //-i: input_image_name.mhd.
  ImageType::Pointer img_y;
  if (load_img_f16 (vcl_string(input_im_arg), img_y) == false) {
    vcl_printf ("read %s error, exit.\n", input_im_arg);
    return EXIT_FAILURE;
  }
    
  //If -s 0, skip in the case not performing smoothing.
  if (perform_smooth_arg != 0) {
    //-s 1: Perform initial smoothing.
    GradientAnisotropicSmooth (img_y);    
    img_y->Update();

    //-s 2: Perform initial smoothing, save file and quit.
    if (perform_smooth_arg == 2) {
      save_img_f16 (vcl_string(output_im_arg), img_y);
      return EXIT_SUCCESS;
    }
  }
  
  //-m : read in the input mask image
  Image8Type::Pointer img_mask;
  if (input_mask_arg != NULL) {
    if (load_img8 (vcl_string(input_mask_arg), img_mask) == false) {
      vcl_printf ("read mask %s error, exit.\n", input_mask_arg);
      return EXIT_FAILURE;
    }

    //mask the input image img_y[] by the mask img_mask
    compute_mask_img (img_y, img_mask, img_y);
  }
  
  //Initialize the image of gain field g[] to 1.
  ImageType::Pointer gain_field_g = ImageType::New();  
  gain_field_g->CopyInformation (img_y);
  gain_field_g->SetRegions (img_y->GetLargestPossibleRegion());
  ///gain_field_g->SetSpacing (img_y->GetSpacing());
  ///gain_field_g->SetOrigin (img_y->GetOrigin());
  gain_field_g->Allocate();
  gain_field_g->FillBuffer (1.0f);
  
  //Initialize the images of the membership functions u1[], u2[], u3[]
  //and a updated storage u1n[], u2n[], u3n[].
  vcl_vector<ImageType::Pointer> mem_fun_u (n_class_arg);
  vcl_vector<ImageType::Pointer> mem_fun_un (n_class_arg);
  for (int k = 0; k < n_class_arg; k++) {
    mem_fun_u[k] = ImageType::New();
    mem_fun_u[k] -> CopyInformation( img_y );
    mem_fun_u[k] -> SetRegions (img_y->GetLargestPossibleRegion());
    mem_fun_u[k] -> Allocate();
    mem_fun_u[k]->FillBuffer (0.0f);
    mem_fun_un[k] = ImageType::New();
    mem_fun_un[k] -> CopyInformation( img_y );
    mem_fun_un[k] -> SetRegions (img_y->GetLargestPossibleRegion());
    mem_fun_un[k] -> Allocate();
    mem_fun_un[k]->FillBuffer (0.0f);
  }
  
  //the centroid for different classes v1, v2, v3.
  vcl_vector<float> centroid_v;

  //Initializtion parameters.
  //-c 3: number of classes.
  //-b 200: number of bin
  //-l 25: lower threshold
  //-u 150: upper threshold

  //AFCM iteration parameters.
  //-bt: background threshold.
  const float conv_thresh = 0.01f;

  if (gain_fit_arg == 0 || gain_fit_arg == 1 || gain_fit_arg == 2) {
    //-g 0,1,2: the original AFCM with or without gain field correction.
    //   0: the original AFCM without gain field correction.
    //   1: linear regression fitting
    //   2: quadratic regression fitting
    afcm_segmentation (img_y, n_class_arg, n_bin_arg, l_th_arg, u_th_arg,
                       bg_thresh_arg, gain_fit_arg, 
                       gain_th_arg, gain_min_arg,
                       conv_thresh, gain_field_g,
                       mem_fun_u, mem_fun_un, centroid_v);
  }
  else {
    //-g 3,4: space division for gain field regression fitting.
    //   3: grid division and linear regression
    //   4: grid division and quadratic regression
    //-gs : grid size.
    afcm_segmentation_grid (img_y, n_class_arg, n_bin_arg, l_th_arg, u_th_arg,
                            bg_thresh_arg, gain_fit_arg, 
                            gain_th_arg, gain_min_arg,
                            conv_thresh, n_grid_arg,
                            gain_field_g,
                            mem_fun_u, mem_fun_un, centroid_v);
    
    //Save the gain corrected img_y[].
    vcl_stringstream filename;  
    filename << output_im_arg << "_gc.mhd";
    save_img_f16 (filename.str(), img_y);
    
    //Save the final gain field file for debugging.  
    //save_img8 ("gain_field_g.mhd", gain_field_g);
    save_01_img8 ("gain_field_g.mhd", gain_field_g);
  }
  
  //Output results:  
  vcl_printf ("\nFinal centroids:");
  vcl_printf ("    C0 %f,   C1 %f,   C2 %f.\n\n", 
              centroid_v[0], centroid_v[1], centroid_v[2]);

  //Save the three mem_fun_un[] images to files.
  save_mem_fun_u (output_im_arg, mem_fun_un);

  return EXIT_SUCCESS;
}


//===================================================================

  //print pixel value of (149, 138, 106) = 1378.
  /*ImageType::IndexType pixelIndex;
  pixelIndex[0] = 149;
  pixelIndex[1] = 138;
  pixelIndex[2] = 106;
  ImageType::PixelType pixelValue = img_y->GetPixel (pixelIndex);
  printf ("\timg(149, 138, 106) = %f.\n", pixelValue);
  //print pixel value of (149, 138, 106) = ?.
  pixelIndex[0] = 149;
  pixelIndex[1] = 138;
  pixelIndex[2] = 106;
  pixelValue = img_y->GetPixel (pixelIndex);
  printf ("\timgs(149, 138, 106) = %f.\n", pixelValue);*/
  //Simple Test Thresholding:

  /*vnl_vector<double> thresholdLow (n_class_arg);
  vnl_vector<double> thresholdHigh (n_class_arg);

  for (int i = 1; i < n_class_arg; i++) {
    thresholdHigh[i-1] = (centroid_v[i-1]+centroid_v[i])/2; 
    thresholdLow[i] = thresholdHigh[i-1];
  }
  thresholdLow[0] = centroid_v[0] + centroid_v[0] - thresholdHigh[0];
  thresholdHigh[n_class_arg-1] = centroid_v[n_class_arg-1] + centroid_v[n_class_arg-1] - thresholdLow[n_class_arg-1];

  typedef itk::BinaryThresholdImageFilter<ImageType, ImageType > BinThresholdType;
  BinThresholdType::Pointer thresholder = BinThresholdType::New();

  thresholder -> SetInput (img_y);
  for (int k = 0; k < n_class_arg; k++) {
    thresholder -> SetOutsideValue (0);
    thresholder -> SetInsideValue (1);
    thresholder -> SetLowerThreshold (thresholdLow[k]);
    thresholder -> SetUpperThreshold (thresholdHigh[k]);
    thresholder -> Update();
    mem_fun_u[k] = thresholder->GetOutput();
    mem_fun_u[k]->Update();

    //-s 1:
    if (perform_smooth_arg.getValue())
      BinaryMedianFilter (mem_fun_u[k], 1);

    //-o output: output image filename base.
    vcl_stringstream a;
    a << output_im_arg.getValue() << k << ".mhd";
    save_img8 (a.str(), mem_fun_u[k]);
  }

  typedef itk::Image<unsigned char,3> LabelType;
  typedef itk::RelabelComponentImageFilter<ImageType, ImageType > RelabelType;
  typedef itk::ConnectedComponentImageFilter<ImageType, ImageType > ConnectedComponentType;

  ConnectedComponentType::Pointer connected = ConnectedComponentType::New();
  RelabelType::Pointer labeler = RelabelType::New();

  connected->SetInput(mem_fun_u[2]);
  connected->Update();

  labeler -> SetInput( connected -> GetOutput() );
  labeler -> SetNumberOfObjectsToPrint( 1 );

  labeler -> Update();

  thresholder -> SetInput( labeler -> GetOutput() );
  thresholder -> SetLowerThreshold (1);
  thresholder -> SetUpperThreshold (1);
  thresholder -> SetOutsideValue (0);
  thresholder -> SetInsideValue (1);
  thresholder -> Update();

  ///mem_fun_u[2] = thresholder -> GetOutput();
  ///save_img8 ("Member2_label.mhd", mem_fun_u[2]);*/
