//:  Command line parser.
//   Author: MingChing Chang 050223

#include <vcl_cstdio.h>
#include <vcl_cstdlib.h>
#include <vcl_string.h>

#include "cmd_opt_proc.h"

static char *typestrings[] = {
  "void", "char", "string", "short", "int", "float", "double",
};

bool opt_parse_args (const int argc, char *argv[], 
                     OPTTABLE *optab, int optab_size)
{
  OPTTABLE  *op = optab;
  void      *val_addr = 0;
  OPT_TYPE  val_type;
  int       n, found;
  float     x;
  char      *astr, *astr1;

  if (argc == 1)
    return false;

  for (int ap=1; ap<argc; ap++) {
    astr = argv[ap];
    if (*astr == '-') {
      astr1 = astr+1;
      if (!strcmp(astr1, "?") || !strcmp(astr1, "h") ||
          !strcmp(astr1, "H") || !strcmp(astr1, "help")) {
         opt_usage (argc, argv, optab, optab_size, NULL);
      }

      for (op=optab, found=0; op->flag_string != 0 && ! found; op++) {
        if (!strcmp (astr+1, op->flag_string)) {
          val_addr = op->value_addr;
          val_type = op->value_type;
          found = 1;
          break;
        }
      }
      if (!found) {
        vcl_fprintf(stderr, "Unknown flag \"%s\"\n", argv[ap]);
        opt_usage (argc, argv, optab, optab_size, NULL);
      }
      else if(val_addr != 0) {
        astr = argv[++ap];
        switch (val_type) {
        case _VOID:
          /* void does not take any values */
          opt_usage (argc, argv, optab, optab_size, astr1);
          break;
        case _CHAR:
          /* One character argument is expected. */
          /* It is passed unchanged to the value address */
          if (*(astr+1) != 0)
            opt_usage (argc, argv, optab, optab_size, astr1);
          else
            *((char*)op->value_addr) = *astr;
          break;
        case _STRING:
          *((char **)op->value_addr) = astr;
          break;
        case _BOOL:
          if (sscanf (argv[ap], "%d", &n) == 1)
            *((bool*)op->value_addr) = (n!=0);
          else
            opt_usage (argc, argv, optab, optab_size, astr1);
          break;
        case _SHORT:
          if (sscanf (argv[ap], "%d", &n) == 1)
            *((short*)op->value_addr) = n;
          else
            opt_usage (argc, argv, optab, optab_size, astr1);
          break;
        case _INT:
          if (sscanf (argv[ap], "%d", &n) == 1)
            *((int*)op->value_addr) = n;
          else
            opt_usage (argc, argv, optab, optab_size, astr1);
          break;
        case _FLOAT:
          if (sscanf (argv[ap], "%f", &x) == 1)
            *((float*)(op->value_addr)) = x;
          else
            opt_usage (argc, argv, optab, optab_size, astr1);
          break;
        case _DOUBLE:
          if (sscanf (argv[ap], "%f", &x) == 1)
            *((double*)op->value_addr) = x;
          else
            opt_usage (argc, argv, optab, optab_size, astr1);
          break;
        }
      }
      val_addr = 0;
    }
  }

  return true;
}

#define LINELENGTH  79
#define INDENT      4 //8

#ifdef unix
#define DIRDEL '/'
#else
#define DIRDEL '\\'
#endif

int opt_usage (const int argc, char *argv[], 
               OPTTABLE *optab, int optab_size, 
               const char* astr1)
{
  char* usage_string = "Usage: ";
  char* exe_name = argv[0];
  char* short_pname = 0;
  int blen = 0;
  int maxblen = 0;
  int offset = strlen(usage_string) + 1 + strlen(exe_name);
  OPTTABLE  *op = optab;
  char buf[111];
  int i;
    
  short_pname = strrchr(exe_name, DIRDEL);
  if (short_pname == 0)
    short_pname = exe_name;
  else
    short_pname++;
  vcl_fprintf(stderr, "%s%s ", usage_string, short_pname);

  int col = offset;
  ///for (op=optab; op->flag_string != 0; op++) {
  for (op=optab, i=0; i<optab_size; op++, i++) {
    if (op->value_addr) {
      char *typestring = typestrings[op->value_type];
      sprintf (buf, "-%s <%s>  ", op->flag_string, typestring);
    }
    else
      sprintf (buf, "-%s  ", op->flag_string);

    blen = strlen (buf);
    if (blen > maxblen)
      maxblen = blen;
    if (col+blen >= LINELENGTH) {
      vcl_fprintf (stderr, "\n");
      for (int i=0; i<INDENT; i++)
        vcl_fprintf (stderr, " ");
      col = INDENT;
    }
    vcl_fprintf (stderr, buf);
    col += blen;
  }

  vcl_fprintf(stderr, "\n  where:\n");

  for (op=optab, i=0; i<optab_size; op++, i++) {
    //only show the specific command opt specified in astr1.
    if (astr1 != NULL) {
      if (strcmp (astr1, op->flag_string))
        continue; //skip the irrelevant cases.
    }

    if (op->value_addr) {
      char *typestring = typestrings[op->value_type];

      char def_value_buf[128];
      switch (op->value_type) {
      case _VOID:
        sprintf (def_value_buf, "");
      break;
      case _CHAR:
        sprintf (def_value_buf, "%c", *((char*)(op->value_addr)));
      break;
      case _STRING:
        sprintf (def_value_buf, "%s", *((char**)(op->value_addr)));
      break;
      case _BOOL:
        sprintf (def_value_buf, "%d", *((bool*)(op->value_addr)));
      break;
      case _SHORT:
        sprintf (def_value_buf, "%d", *((short*)(op->value_addr)));
      break;
      case _INT:
        sprintf (def_value_buf, "%d", *((int*)(op->value_addr)));
      break;
      case _FLOAT:
        sprintf (def_value_buf, "%.3f", *((float*)(op->value_addr)));
      break;
      case _DOUBLE:
        sprintf (def_value_buf, "%.3f", *((double*)(op->value_addr)));
      break;
      }

      ///sprintf (buf, "-%s <%s>  ", op->flag_string, typestring);
      sprintf (buf, "-%s %s  ", op->flag_string, def_value_buf);
    }
    else
      sprintf (buf, "-%s  ", op->flag_string);

    blen = strlen (buf);
    vcl_fprintf(stderr, "    %s", buf);

    for (int i=blen; i<maxblen; i++)
      vcl_fprintf (stderr, " ");

    if (op->comment) {
      vcl_fprintf (stderr, ": %s\n", op->comment);
      //vcl_fprintf (stderr, " -- %s\n", op->comment);
    }
  }

  //Exit the command line program!
  exit(1);
}

#ifdef OPTTEST
/******************************************************************/

int aa_p, bb_p, cc_p, dd_p, ss_p, vv_p, str_p; 
float aa;
int bb;
char  cc;
double  dd;
short ss;
char  *str;

OPTTABLE fake[] =
{
  {"a",     FLOAT,  &aa_p,   &aa, "No comment for aa"},
  {"bb",    INT,    &bb_p,   &bb, "No comment for bb"},
  {"ccc",   CHAR,   &cc_p,   &cc, "No comment for ccc"},
  {"dd",    DOUBLE, 0,   &dd, "No comment for dd"},
  {"s",     SHORT,  &ss_p,   &ss, "No comment for s"},
  {"v",     VOID,   &vv_p,   0,   "No comment for vv"},
  {"str",   STRING, &str_p,  &str,"No comment for str"},

  {0,0,0,0,0}
};

void main (int argc, char *argv[])
{
  opt_parse_args (argc, argv, fake);
    
  printf ("aa_p=%d, bb_p=%d, cc_p=%d, dd_p=%d, ss_p=%d, vv_p=%d, str_p=%d\n",
    aa_p, bb_p, cc_p, dd_p, ss_p, vv_p, str_p);
    
  printf ("aa=%f, bb = %d, cc=%c, dd=%lf, ss=%d\n",
    aa, bb, cc, dd, ss);
  printf ("str: \"%s\"\n", str);
}
#endif
