#include        <stddef.h>
#include        <stdlib.h>
#include        <stdio.h>
#include        <string.h>
#include        <math.h>
#include        <float.h>

#include        "affine.h"
#include        <pgstd.h>
#define errorReport pgError
#define DEBUG 0

/************************************************************************/
static  double*     allocLinD       (long               n);
static  float*      allocLinF       (long               n);
static  float*      allocVolF       (long               nx,
                                     long               ny,
                                     long               nz);
static  int         directBsplineMirror
                                    (float              *inPtr,
                                     float              *outPtr,
                                     long               nx,
                                     long               ny,
                                     long               nz,
                                     int                order);
static  long        fold            (long               i,
                                     long               n);
static  void        freeLinD        (double             *line);
static  void        freeLinF        (float              *line);
static  void        freeVolF        (float              *volume);
static  int         gaussj          (double             a[][3],
                                     double             b[][3]);
static  void        getxF2D         (float              *volumeSrc,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *rowDst,
                                     long               n);
static  void        getxF2F         (float              *volumeSrc,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     float              *rowDst,
                                     long               n);
static  void        getyF2D         (float              *volumeSrc,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *columnDst,
                                     long               n);
static  void        getzF2D         (float              *volumeSrc,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *pillarDst,
                                     long               n);
static  void        iirConvolveMirror
                                    (double             *inPtr,
                                     double             *outPtr,
                                     long               n,
                                     double             gain,
                                     double             *realPoles,
                                     long               np);
static  double      interpolWeight  (int                order,
                                     long               i,
                                     double             x);
static  double      interpolWeight0 (long               i,
                                     double             x);
static  double      interpolWeight1 (long               i,
                                     double             x);
static  double      interpolWeight2 (long               i,
                                     double             x);
static  double      interpolWeight3 (long               i,
                                     double             x);
static  double      interpolWeight4 (long               i,
                                     double             x);
static  double      interpolWeight5 (long               i,
                                     double             x);
static  double      interpolWeight6 (long               i,
                                     double             x);
static  double      interpolWeight7 (long               i,
                                     double             x);
static  int         invertTrsf      (struct trsfStruct  *dirTrsf,
                                     struct trsfStruct  *invTrsf);
static  void        putxD2F         (float              *volumeDst,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *rowSrc,
                                     long               n);
static  void        putxF2F         (float              *volumeDst,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     float              *rowSrc,
                                     long               n);
static  void        putyD2F         (float              *volumeDst,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *columnSrc,
                                     long               n);
static  void        putzD2F         (float              *volumeDst,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *pillarSrc,
                                     long               n);

/************************************************************************/
static  double*     allocLinD       (long               n) {

        return((double *)malloc((size_t)n * sizeof(double)));
} /* End of allocLinD */

/************************************************************************/
static  float*      allocLinF       (long               n) {

        return((float *)malloc((size_t)n * sizeof(float)));
} /* End of allocLinF */

/************************************************************************/
static  float*      allocVolF       (long               nx,
                                     long               ny,
                                     long               nz) {

        return((float *)malloc((size_t)(nx * ny * nz) * sizeof(float)));
} /* End of allocVolF */

/************************************************************************/
static  int         directBsplineMirror
                                    (float              *inPtr,
                                     float              *outPtr,
                                     long               nx,
                                     long               ny,
                                     long               nz,
                                     int                order) {

        double              *data;
        double              realPoles[3];
        double              gain;
        float               *p;
        long                np;
        long                x, y, z;

        p = allocLinF(nx);
        if (p == (float *)NULL) {
          errorReport("ERROR - Not enough memory in directBsplineMirror");
          return(ERROR);
        }
        for (z = 0L; (z < nz); z++)
          for (y = 0L; (y < ny); y++) {
            getxF2F(inPtr, 0L, y, z, nx, ny, p, nx);
            putxF2F(outPtr, 0L, y, z, nx, ny, p, nx);
          }
        freeLinF(p);
        switch (order) {
          case 0:
          case 1:
            return(!ERROR);
          case 2:
            np = 1L;
            realPoles[0] = sqrt(8.0) - 3.0;
            gain = 24.0 - sqrt(512.0);
            break;
          case 3:
            np = 1L;
            realPoles[0] = sqrt(3.0) - 2.0;
            gain = 12.0 - sqrt(108.0);
            break;
          case 4:
            np = 2L;
            realPoles[0] = sqrt(664.0 - sqrt(438976.0)) + sqrt(304.0) - 19.0;
            realPoles[1] = sqrt(664.0 + sqrt(438976.0)) - sqrt(304.0) - 19.0;
            gain = (1.0 - realPoles[0]) * (1.0 - realPoles[1]);
            gain *= gain;
            break;
          case 5:
            np = 2L;
            realPoles[0] = 0.5 * (sqrt(270.0 - sqrt(70980.0)) + sqrt(105.0) - 13.0);
            realPoles[1] = 0.5 * (sqrt(270.0 + sqrt(70980.0)) - sqrt(105.0) - 13.0);
            gain = (1.0 - realPoles[0]) * (1.0 - realPoles[1]);
            gain *= gain;
            break;
          case 6:
            np = 3L;
            realPoles[0] = -0.4882945893030449;
            realPoles[1] = -0.08167927107623751;
            realPoles[2] = -0.001414151808325818;
            gain = 2.598975999348578;
            break;
          case 7:
            np = 3L;
            realPoles[0] = -0.5352804307964378;
            realPoles[1] = -0.1225546151923267;
            realPoles[2] = -0.009148694809608277;
            gain = 3.024828203644183;
            break;
          default:
            errorReport("ERROR - Unknown order");
            return(ERROR);
        }
        if (nx > 1L) {
          data = allocLinD(nx);
          if (data == (double *)NULL) {
            errorReport("ERROR - Unable to allocate x data");
            return(ERROR);
          }
          for (z = 0L; (z < nz); z++)
            for (y = 0L; (y < ny); y++) {
              getxF2D(outPtr, 0L, y, z, nx, ny, data, nx);
              iirConvolveMirror(data, data, nx, gain, realPoles, np);
              putxD2F(outPtr, 0L, y, z, nx, ny, data, nx);
            }
          freeLinD(data);
        }
        if (ny > 1L) {
          data = allocLinD(ny);
          if (data == (double *)NULL) {
            errorReport("ERROR - Unable to allocate y data");
            return(ERROR);
          }
          for (z = 0L; (z < nz); z++)
            for (x = 0L; (x < nx); x++) {
              getyF2D(outPtr, x, 0L, z, nx, ny, data, ny);
              iirConvolveMirror(data, data, ny, gain, realPoles, np);
              putyD2F(outPtr, x, 0L, z, nx, ny, data, ny);
            }
          freeLinD(data);
        }
        if (nz > 1L) {
          data = allocLinD(nz);
          if (data == (double *)NULL) {
            errorReport("ERROR - Unable to allocate z data");
            return(ERROR);
          }
          for (y = 0L; (y < ny); y++)
            for (x = 0L; (x < nx); x++) {
              getzF2D(outPtr, x, y, 0L, nx, ny, data, nz);
              iirConvolveMirror(data, data, nz, gain, realPoles, np);
              putzD2F(outPtr, x, y, 0L, nx, ny, data, nz);
            }
          freeLinD(data);
        }
        return(!ERROR);
} /* End of directBsplineMirror */

/************************************************************************/
static  long        fold            (long               i,
                                     long               n) {

        long                n2;

        if (i < 0L)
          i = -i;
        if (i < n)
          return(i);
        if (n == 1L)
          return(0L);
        n2 = 2L * (n - 1L);
        i -= n2 * (i / n2);
        return((i < n) ? (i) : (n2 - i));
} /* End of fold */

/************************************************************************/
static  void        freeLinD        (double             *line) {

        free(line);
} /* End of freeLinD */

/************************************************************************/
static  void        freeLinF        (float              *line) {

        free(line);
} /* End of freeLinF */

/************************************************************************/
static  void        freeVolF        (float              *volume) {

        free(volume);
} /* End of freeVolF */

/************************************************************************/
static  int         gaussj          (double             a[][3],
                                     double             b[][3]) {

        int                 indxc[3], indxr[3];
        int                 ipiv[3];
        double              big, dum, pivinv, swap;
        int                 i, icol, irow, j, k, l, ll;
        int                 n = 3, m = 3;

        for (j = 0; (j < n); j++)
          ipiv[j] = 0;
        for (i = 0; (i < n); i++) {
          big = 0.0;
          for (j = 0; (j < n); j++)
            if (ipiv[j] != 1)
              for (k = 0; (k < n); k++) {
                if (ipiv[k] == 0) {
                  if (fabs(a[j][k]) >= big) {
                    big = fabs(a[j][k]);
                    irow = j;
                    icol = k;
                  }
                }
                else if (ipiv[k] > 1) {
                  errorReport("ERROR - Singular matrix in Gauss inversion");
                  return(ERROR);
                }
              }
          ++(ipiv[icol]);
          if (irow != icol) {
            for (l = 0; (l < n); l++) {
              swap = a[irow][l];
              a[irow][l] = a[icol][l];
              a[icol][l] = swap;
            }
            for (l = 0; (l < m); l++) {
              swap = b[irow][l];
              b[irow][l] = b[icol][l];
              b[icol][l] = swap;
            }
          }
          indxr[i] = irow;
          indxc[i] = icol;
          if ((a[icol][icol] * a[icol][icol]) == 0.0) {
            errorReport("ERROR - Singular matrix in Gauss inversion");
            return(ERROR);
          }
          pivinv = 1.0 / a[icol][icol];
          a[icol][icol] = 1.0;
          for (l = 0; (l < n); l++)
            a[icol][l] *= pivinv;
          for (l = 0; (l < m); l++)
            b[icol][l] *= pivinv;
          for (ll = 0; (ll < n); ll++)
            if (ll != icol) {
              dum = a[ll][icol];
              a[ll][icol] = 0.0;
              for (l = 0; (l < n); l++)
                a[ll][l] -= a[icol][l] * dum;
              for (l = 0; (l < m); l++)
                b[ll][l] -= b[icol][l] * dum;
            }
        }
        for (l = n - 1; (l >= 0); l--) {
          if (indxr[l] != indxc[l])
            for (k = 0; (k < n); k++) {
              swap = a[k][indxr[l]];
              a[k][indxr[l]] = a[k][indxc[l]];
              a[k][indxc[l]] = swap;
            }
        }
        return(!ERROR);
} /* End of gaussj */

/************************************************************************/
static  void        getxF2D         (float              *volumeSrc,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *rowDst,
                                     long               n) {

        double              *end;

        volumeSrc += (ptrdiff_t)(x + nx * (y + ny * z));
        end = rowDst + (ptrdiff_t)n;
        while (rowDst < end)
          *rowDst++ = (double)*volumeSrc++;
} /* End of getxF2D */

/************************************************************************/
static  void        getxF2F         (float              *volumeSrc,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     float              *rowDst,
                                     long               n) {

        volumeSrc += (ptrdiff_t)(x + nx * (y + ny * z));
        rowDst = (float *)memcpy(rowDst, volumeSrc, (size_t)n * sizeof(float));
} /* End of getxF2F */

/************************************************************************/
static  void        getyF2D         (float              *volumeSrc,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *columnDst,
                                     long               n) {

        double              *end;

        volumeSrc += (ptrdiff_t)(x + nx * (y + ny * z));
        for (end = columnDst + (ptrdiff_t)n; (columnDst < end); volumeSrc += (ptrdiff_t)nx)
          *columnDst++ = (double)*volumeSrc;
} /* End of getyF2D */

/************************************************************************/
static  void        getzF2D         (float              *volumeSrc,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *pillarDst,
                                     long               n) {

        double              *end;
        long                nxny = nx * ny;

        volumeSrc += (ptrdiff_t)(x + nx * y + nxny * z);
        for (end = pillarDst + (ptrdiff_t)n; (pillarDst < end); volumeSrc += (ptrdiff_t)nxny)
          *pillarDst++ = (double)*volumeSrc;
} /* End of getzF2D */

/************************************************************************/
static  void        iirConvolveMirror
                                    (double             *inPtr,
                                     double             *outPtr,
                                     long               n,
                                     double             gain,
                                     double             *realPoles,
                                     long               np) {

        double              *p, *q, *r;
        double              tolerance = log10((double)FLT_EPSILON);
        double              x0, pole;
        long                n2;
        long                i, j, k;

        p = inPtr;
        q = outPtr;
        r = outPtr + (ptrdiff_t)n;
        while (q < r)
          *q++ = *p++ * gain;
        if (n == 1L)
          return;
        n2 = 2L * (n - 1L);
        i = np;
        while (i-- > 0L) {
          pole = *realPoles++;
          j = (long)ceil(tolerance / log10(fabs(pole)));
          k = j - n2 * (j / n2);
          j -= k;
          if (k < n) {
            p = outPtr + (ptrdiff_t)k;
            x0 = *p;
          }
          else {
            k = n2 - k;
            p = outPtr + (ptrdiff_t)k;
            x0 = *p;
            while (++p < r)
              x0 = pole * x0 + *p;
            p--;
          }
          while (--p >= outPtr)
            x0 = pole * x0 + *p;
          while (j > 0L) {
            p++;
            while (++p < r)
              x0 = pole * x0 + *p;
            p--;
            while (--p >= outPtr)
              x0 = pole * x0 + *p;
            j -= n2;
          }
          q = p++;
          *p++ = x0;
          x0 = *(q++ + (ptrdiff_t)n);
          while (p < r)
            *p++ += *q++ * pole;
          *q = (2.0 * *q - x0) / (1.0 - pole * pole);
          p--;
          while (--q >= outPtr)
            *q += *p-- * pole;
        }
} /* End of iirConvolveMirror */

/************************************************************************/
static  double      interpolWeight  (int                order,
                                     long               i,
                                     double             x) {

        switch (order) {
          case 0:
            return(interpolWeight0(i, x));
          case 1:
            return(interpolWeight1(i, x));
          case 2:
            return(interpolWeight2(i, x));
          case 3:
            return(interpolWeight3(i, x));
          case 4:
            return(interpolWeight4(i, x));
          case 5:
            return(interpolWeight5(i, x));
          case 6:
            return(interpolWeight6(i, x));
          case 7:
            return(interpolWeight7(i, x));
          default:
            errorReport("ERROR - Unknown interpolation specification");
            return(0.0);
        }
} /* End of interpolWeight */

/************************************************************************/
static  double      interpolWeight0 (long               i,
                                     double             x) {

        double              f;

        f = fabs(x - (double)i);
        return((f > 0.5) ? (0.0) : (1.0));
} /* End of interpolWeight0 */

/************************************************************************/
static  double      interpolWeight1 (long               i,
                                     double             x) {

        double              f;

        f = fabs(x - (double)i);
        return((f > 1.0) ? (0.0) : (1.0 - f));
} /* End of interpolWeight1 */

/************************************************************************/
static  double      interpolWeight2 (long               i,
                                     double             x) {

        double              f;

        f = fabs(x - (double)i);
        if (f < 0.5)
          return(0.75 - f * f);
        if (f < 1.5) {
          f = 1.5 - f;
          return(0.5 * f * f);
        }
        return(0.0);
} /* End of interpolWeight2 */

/************************************************************************/
static  double      interpolWeight3 (long               i,
                                     double             x) {

        double              f;

        f = fabs(x - (double)i);
        if (f < 1.0)
          return((f * f * (f - 2.0) * 3.0 + 4.0) * (1.0 / 6.0));
        if (f < 2.0) {
          f = 2.0 - f;
          return(f * f * f * (1.0 / 6.0));
        }
        return(0.0);
} /* End of interpolWeight3 */

/************************************************************************/
static  double      interpolWeight4 (long               i,
                                     double             x) {

        double              f;

        f = fabs(x - (double)i);
        if (f < 0.5) {
          f *= f;
          return(f * (f * 0.25 - 0.625) + 115.0 / 192.0);
        }
        if (f < 1.5)
          return(f * (f * (f * (5.0 / 6.0 - f * (1.0 / 6.0)) - 1.25) + 5.0 / 24.0) + 55.0 / 96.0);
        if (f < 2.5) {
          f -= 2.5;
          f *= f;
          return(f * f / 24.0);
        }
        return(0.0);
} /* End of interpolWeight4 */

/************************************************************************/
static  double      interpolWeight5 (long               i,
                                     double             x) {

        double              f;

        f = fabs(x - (double)i);
        if (f < 1.0)
          return(f * f * (f * f * (0.25 - f * (1.0 / 12.0)) - 0.5) + 11.0 / 20.0);
        if (f < 2.0)
          return(f * (f * (f * (f * (f * (1.0 / 24.0) - 0.375) + 1.25) - 1.75) + 0.625)
            + 17.0 / 40.0);
        if (f < 3.0)
          return(f * (f * (f * (f * (0.125 - f * (1.0 / 120.0)) - 0.75) + 2.25) - 3.375)
            + 81.0 / 40.0);
        return(0.0);
} /* End of interpolWeight5 */

/************************************************************************/
static  double      interpolWeight6 (long               i,
                                     double             x) {

        double              f;

        f = fabs(x - (double)i);
        if (f < 0.5) {
          f *= f;
          return(f * (f * (7.0 / 48.0 - f * (1.0 / 36.0)) - 77.0 / 192.0) + 5887.0 / 11520.0);
        }
        if (f < 1.5)
          return(f * (f * (f * (f * (f * (f * (1.0 / 48.0) - 7.0 / 48.0) + 21.0 / 64.0)
            - 35.0 / 288.0) - 91.0 / 256.0) - 7.0 / 768.0) + 7861.0 / 15360.0);
        if (f < 2.5)
          return(f * (f * (f * (f * (f * (7.0 / 60.0 - f * (1.0 / 120.0)) - 21.0 / 32.0)
            + 133.0 / 72.0) - 329.0 / 128.0) + 1267.0 / 960.0) + 1379.0 / 7680.0);
        if (f < 3.5)
          return(f * (f * (f * (f * (f * (f * (1.0 / 720.0) - 7.0 / 240.0) + 49.0 / 192.0)
            - 343.0 / 288.0) + 2401.0 / 768.0) - 16807.0 / 3840.0) + 117649.0 / 46080.0);
        return(0.0);
} /* End of interpolWeight6 */

/************************************************************************/
static  double      interpolWeight7 (long               i,
                                     double             x) {

        double              f;

        f = fabs(x - (double)i);
        if (f < 1.0)
          return(f * f * (f * f * (f * f * (f * (1.0 / 144.0) - 1.0 / 36.0) + 1.0 / 9.0)
            - 1.0 / 3.0) + 151.0 / 315.0);
        if (f < 2.0)
          return(f * (f * (f * (f * (f * (f * (0.05 - f * (1.0 / 240.0)) - 7.0 / 30.0) + 0.5)
            - 7.0 / 18.0) - 0.1) - 7.0 / 90.0) + 103.0 / 210.0);
        if (f < 3.0)
          return(f * (f * (f * (f * (f * (f * (f * (1.0 / 720.0) - 1.0 / 36.0) + 7.0 / 30.0)
            - 19.0 / 18.0) + 49.0 / 18.0) - 23.0 / 6.0) + 217.0 / 90.0) - 139.0 / 630.0);
        if (f < 4.0)
          return(f * (f * (f * (f * (f * (f * (1.0 / 180.0 - f * (1.0 / 5040.0)) - 1.0 / 15.0)
            + 4.0 / 9.0) - 16.0 / 9.0) + 64.0 / 15.0) - 256.0 / 45.0) + 1024.0 / 315.0);
        return(0.0);
} /* End of interpolWeight7 */

/************************************************************************/
static  int         invertTrsf      (struct trsfStruct  *dirTrsf,
                                     struct trsfStruct  *invTrsf) {

        double              dirSkew[3][3];
        int                 i, j;

        for (j = 0; (j < 3); j++) {
          for (i = 0; (i < 3); i++) {
            dirSkew[i][j] = dirTrsf->skew[i][j];
            invTrsf->skew[i][j] = 0.0;
          }
          invTrsf->skew[j][j] = 1.0;
        }
        if (gaussj(dirSkew, invTrsf->skew) == ERROR) {
          errorReport("ERROR - Non reversible transformation");
          return(ERROR);
        }
        for (j = 0; (j < 3); j++) {
          invTrsf->dx[j] = 0.0;
          for (i = 0; (i < 3); i++)
            invTrsf->dx[j] -= invTrsf->skew[j][i] * dirTrsf->dx[i];
        }
        return(!ERROR);
} /* End of invertTrsf */

/************************************************************************/
static  void        putxD2F         (float              *volumeDst,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *rowSrc,
                                     long               n) {

        double              *end;

        volumeDst += (ptrdiff_t)(x + nx * (y + ny * z));
        end = rowSrc + (ptrdiff_t)n;
        while (rowSrc < end)
          *volumeDst++ = (float)*rowSrc++;
} /* End of putxD2F */

/************************************************************************/
static  void        putxF2F         (float              *volumeDst,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     float              *rowSrc,
                                     long               n) {

        volumeDst += (ptrdiff_t)(x + nx * (y + ny * z));
        volumeDst = (float *)memcpy(volumeDst, rowSrc, (size_t)n * sizeof(float));
} /* End of putxF2F */

/************************************************************************/
static  void        putyD2F         (float              *volumeDst,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *columnSrc,
                                     long               n) {

        double              *end;

        volumeDst += (ptrdiff_t)(x + nx * (y + ny * z));
        for (end = columnSrc + (ptrdiff_t)n; (columnSrc < end); volumeDst += (ptrdiff_t)nx)
          *volumeDst = (float)*columnSrc++;
} /* End of putyD2F */

/************************************************************************/
static  void        putzD2F         (float              *volumeDst,
                                     long               x,
                                     long               y,
                                     long               z,
                                     long               nx,
                                     long               ny,
                                     double             *pillarSrc,
                                     long               n) {

        double              *end;
        long                nxny = nx * ny;

        volumeDst += (ptrdiff_t)(x + nx * y + nxny * z);
        for (end = pillarSrc + (ptrdiff_t)n; (pillarSrc < end); volumeDst += (ptrdiff_t)nxny)
          *volumeDst = (float)*pillarSrc++;
} /* End of putzD2F */

/************************************************************************/
/* +===========================+                                        */
/* | FUNCTION: affineTransform |                                        */
/* +===========================+                                        */
/*----------------------------------------------------------------------*/
/*                                                                      */
/* Objective:   Affine transformation of a volume                       */
/*              Resampling of a fitted spline model                     */
/*                                                                      */
/* Description: - M. Unser, A. Aldroubi and M. Eden, "B-Spline Signal   */
/*              Processing: Part I-Theory," IEEE Transactions on Signal */
/*              Processing, vol. 41, no. 2, pp. 821-832, February 1993. */
/*              - M. Unser, A. Aldroubi and M. Eden, "B-Spline Signal   */
/*              Processing: Part II-Efficient Design and Applications," */
/*              IEEE Transactions on Signal Processing, vol. 41, no. 2, */
/*              pp. 834-848, February 1993.                             */
/*                                                                      */
/* Conventions: "transform" is a matrix of homogenous coordinates       */
/*              "origin" is relative to the top-upper-left corner       */
/*              "inPtr" indexing scheme: first x, then y, last z        */
/*              "outPtr" should be already allocated                    */
/*              "interpolationOrder" belongs to [0..7]                  */
/*                0 nearest neighbor                                    */
/*                1 trilinear                                           */
/*                3 tricubic                                            */
/*                                                                      */
/* Comment:     Even in 2D, all entries of "transform" are considered   */
/*              There is no antialiasing filter                         */
/*                                                                      */
/************************************************************************/
int                 affineTransform (double             transform[][4],
                                     double             origin[],
                                     float              *inPtr,
                                     float              *outPtr,
                                     long               nx,
                                     long               ny,
                                     long               nz,
                                     int                interpolationOrder,
                                     float              background,
				     double             voxel[3]) {

        double              *wtx, *wty, *wtz;
        float               *splinePtr;
        float               *p, *pi, *pj;
        long                *fdx, *fdy, *fdz;
        double              weightX[9], weightY[9], weightZ[9];
        long                foldX[9], foldY[9], foldZ[9];
        struct trsfStruct   trsf, invTrsf;
        double              a11, a21, a31;
        double              xO, yO, zO;
        double              xz, yz, zz;
        double              xy, yy, zy;
        double              xIn, yIn, zIn;
        double              xOut, yOut, zOut;
        double              q, qi, qj;
	double              X,Y,Z;
        long                half, width;
        long                nxy = nx * ny;
        long                x, y, z;
        long                i, j, k;

        if ((transform[3][0] != 0.0) || (transform[3][1] != 0.0) || (transform[3][2] != 0.0)
          || (transform[3][3] != 1.0)){
          errorReport("ERROR - Non-homogeneous transformation");
          return(ERROR);
        }
        trsf.dx[0] = transform[0][3];
        trsf.dx[1] = transform[1][3];
        trsf.dx[2] = transform[2][3];
        trsf.skew[0][0] = transform[0][0];
        trsf.skew[0][1] = transform[0][1];
        trsf.skew[0][2] = transform[0][2];
        trsf.skew[1][0] = transform[1][0];
        trsf.skew[1][1] = transform[1][1];
        trsf.skew[1][2] = transform[1][2];
        trsf.skew[2][0] = transform[2][0];
        trsf.skew[2][1] = transform[2][1];
        trsf.skew[2][2] = transform[2][2];
        if (invertTrsf(&trsf, &invTrsf) == ERROR) {
          errorReport("ERROR - Unable to compute the backward transformation");
          return(ERROR);
        }
	/* Convert origin from image space to distance space */
	//        xO = origin[0]*voxel[0];
	//        yO = origin[1]*voxel[1];
	//        zO = origin[2]*voxel[2];

	/* Convert image size from image space to distances */
	X = (double)nx * voxel[0];
	Y = (double)ny * voxel[1];
	Z = (double)nz * voxel[2];

	/* modified by Zhiqiang Lao on 3/28/2007 */
	/* so that origin is on center of distance space instead of image space */
	xO = X/2.0;
	yO = Y/2.0;
	zO = Z/2.0;

        a11 = invTrsf.skew[0][0];
        a21 = invTrsf.skew[1][0];
        a31 = invTrsf.skew[2][0];
        half = (long)interpolationOrder / 2L + 1L;
        width = 2L * half + 1L;
        switch (interpolationOrder) {
          case 0:
            for (zOut = -zO; (zOut < (Z - zO)); zOut += voxel[2]) {
              xz = invTrsf.skew[0][2] * zOut + invTrsf.dx[0] + xO;
              yz = invTrsf.skew[1][2] * zOut + invTrsf.dx[1] + yO;
              zz = invTrsf.skew[2][2] * zOut + invTrsf.dx[2] + zO;
              for (yOut = -yO; (yOut < (Y - yO)); yOut += voxel[1]) {
                xy = xz + invTrsf.skew[0][1] * yOut;
                yy = yz + invTrsf.skew[1][1] * yOut;
                zy = zz + invTrsf.skew[2][1] * yOut;
                for (xOut = -xO; (xOut < (X - xO)); xOut += voxel[0]) {
                  xIn = xy + a11 * xOut;
                  yIn = yy + a21 * xOut;
                  zIn = zy + a31 * xOut;
		  /* Convert back to image array coordinates */
                  x = (long)floor(xIn/voxel[0] + 0.5);
                  y = (long)floor(yIn/voxel[1] + 0.5);
                  z = (long)floor(zIn/voxel[2] + 0.5);
		  if (DEBUG) printf("(%d,%d,%d) -> (%f,%f,%f)\n",x,y,z,
				    xOut,yOut,zOut);
                  if ((0L <= x) && (x < nx) && (0L <= y) && (y < ny) &&
		      (0L <= z) && (z < nz))
                    *outPtr++ = *(inPtr + (ptrdiff_t)(z * nxy + y * nx + x));
                  else
                    *outPtr++ = background;
                }
              }
            }
            break;
          case 1:
            for (zOut = -zO; (zOut < (Z - zO)); zOut += voxel[2]) {
              xz = invTrsf.skew[0][2] * zOut + invTrsf.dx[0] + xO;
              yz = invTrsf.skew[1][2] * zOut + invTrsf.dx[1] + yO;
              zz = invTrsf.skew[2][2] * zOut + invTrsf.dx[2] + zO;
              for (yOut = -yO; (yOut < (Y - yO)); yOut += voxel[1]) {
                xy = xz + invTrsf.skew[0][1] * yOut;
                yy = yz + invTrsf.skew[1][1] * yOut;
                zy = zz + invTrsf.skew[2][1] * yOut;
                for (xOut = -xO; (xOut < (X - xO)); xOut += voxel[0]) {
                  xIn = (xy + a11 * xOut)/voxel[0];
                  yIn = (yy + a21 * xOut)/voxel[1];
                  zIn = (zy + a31 * xOut)/voxel[2];
                  x = (long)floor(xIn + 0.5);
                  y = (long)floor(yIn + 0.5);
                  z = (long)floor(zIn + 0.5);
		  if (DEBUG) printf("(%f,%f,%f) -> (%f,%f,%f)\n",x,y,z,
				    xOut,yOut,zOut);
                  if ((0L <= x) && (x < nx) && (0L <= y) && (y < ny) && (0L <= z) && (z < nz)) {
                    wtz = weightZ;
                    fdz = foldZ;
                    for (k = z - half; (k <= (z + half)); k++) {
                      *wtz++ = (nz == 1L) ? (1.0) : (interpolWeight1(k, zIn));
                      *fdz++ = (nz == 1L) ? (0L) : (fold(k, nz) * nxy);
                    }
                    wty = weightY;
                    fdy = foldY;
                    for (j = y - half; (j <= (y + half)); j++) {
                      *wty++ = interpolWeight1(j, yIn);
                      *fdy++ = fold(j, ny) * nx;
                    }
                    wtx = weightX;
                    fdx = foldX;
                    for (i = x - half; (i <= (x + half)); i++) {
                      *wtx++ = interpolWeight1(i, xIn);
                      *fdx++ = fold(i, nx);
                    }
                    q = 0.0;
                    wtz = weightZ;
                    fdz = foldZ;
                    for (k = (nz == 1L) ? (1L) : (width); (k-- > 0L);) {
                      wty = weightY;
                      fdy = foldY;
                      pj = inPtr + (ptrdiff_t)(*fdz++);
                      qj = 0.0;
                      for (j = width; (j-- > 0L);) {
                        wtx = weightX;
                        fdx = foldX;
                        pi = pj + (ptrdiff_t)(*fdy++);
                        qi = 0.0;
                        for (i = width; (i-- > 0L);)
                          qi += *wtx++ * (double)*(pi + (ptrdiff_t)(*fdx++));
                        qj += qi * *wty++;
                      }
                      q += qj * *wtz++;
                    }
                  }
                  else
                    q = background;
                  *outPtr++ = (float)q;
                }
              }
            }
            break;
          case 2:
          case 3:
          case 4:
          case 5:
          case 6:
          case 7:
            splinePtr = allocVolF(nx, ny, nz);
            if (splinePtr == (float *)NULL) {
              errorReport("ERROR - Not enough memory for B-spline coefficients");
              return(ERROR);
            }
            p = allocLinF(nx);
            if (p == (float *)NULL) {
              freeVolF(splinePtr);
              errorReport("ERROR - Not enough memory in affineTransform");
              return(ERROR);
            }
            for (z = 0L; (z < nz); z++)
              for (y = 0L; (y < ny); y++) {
                getxF2F(inPtr, 0L, y, z, nx, ny, p, nx);
                putxF2F(splinePtr, 0L, y, z, nx, ny, p, nx);
              }
            freeLinF(p);
            if (directBsplineMirror(splinePtr, splinePtr, nx, ny, nz, interpolationOrder)
              == ERROR) {
              freeVolF(splinePtr);
              errorReport("ERROR - Unable to compute B-spline coefficients");
              return(ERROR);
            }
            for (zOut = -zO; (zOut < (Z - zO)); zOut += voxel[2]) {
              xz = invTrsf.skew[0][2] * zOut + invTrsf.dx[0] + xO;
              yz = invTrsf.skew[1][2] * zOut + invTrsf.dx[1] + yO;
              zz = invTrsf.skew[2][2] * zOut + invTrsf.dx[2] + zO;
              for (yOut = -yO; (yOut < (Y - yO)); yOut += voxel[1]) {
                xy = xz + invTrsf.skew[0][1] * yOut;
                yy = yz + invTrsf.skew[1][1] * yOut;
                zy = zz + invTrsf.skew[2][1] * yOut;
                for (xOut = -xO; (xOut < (X - xO)); xOut += voxel[0]) {
                  xIn = (xy + a11 * xOut)/voxel[0];
                  yIn = (yy + a21 * xOut)/voxel[1];
                  zIn = (zy + a31 * xOut)/voxel[2];
                  x = (long)floor(xIn + 0.5);
                  y = (long)floor(yIn + 0.5);
                  z = (long)floor(zIn + 0.5);
		  if (DEBUG) printf("(%f,%f,%f) -> (%f,%f,%f)\n",x,y,z,
				    xOut,yOut,zOut);
                  if ((0L <= x) && (x < nx) && (0L <= y) && (y < ny) && (0L <= z) && (z < nz)) {
                    wtz = weightZ;
                    fdz = foldZ;
                    for (k = z - half; (k <= (z + half)); k++) {
                      *wtz++ = (nz == 1L) ? (1.0) : (interpolWeight(interpolationOrder, k,
                        zIn));
                      *fdz++ = (nz == 1L) ? (0L) : (fold(k, nz) * nxy);
                    }
                    wty = weightY;
                    fdy = foldY;
                    for (j = y - half; (j <= (y + half)); j++) {
                      *wty++ = interpolWeight(interpolationOrder, j, yIn);
                      *fdy++ = fold(j, ny) * nx;
                    }
                    wtx = weightX;
                    fdx = foldX;
                    for (i = x - half; (i <= (x + half)); i++) {
                      *wtx++ = interpolWeight(interpolationOrder, i, xIn);
                      *fdx++ = fold(i, nx);
                    }
                    q = 0.0;
                    wtz = weightZ;
                    fdz = foldZ;
                    for (k = (nz == 1L) ? (1L) : (width); (k-- > 0L);) {
                      wty = weightY;
                      fdy = foldY;
                      pj = splinePtr + (ptrdiff_t)(*fdz++);
                      qj = 0.0;
                      for (j = width; (j-- > 0L);) {
                        wtx = weightX;
                        fdx = foldX;
                        pi = pj + (ptrdiff_t)(*fdy++);
                        qi = 0.0;
                        for (i = width; (i-- > 0L);)
                          qi += *wtx++ * (double)*(pi + (ptrdiff_t)(*fdx++));
                        qj += qi * *wty++;
                      }
                      q += qj * *wtz++;
                    }
                  }
                  else
                    q = background;
                  *outPtr++ = (float)q;
                }
              }
            }
            freeVolF(splinePtr);
            break;
          default:
            errorReport("ERROR - Unknown interpolation specification");
            return(ERROR);
        }
        return(!ERROR);
} /* End of affineTransform */
