#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <math.h>
#include <strings.h>
#include <mvcd.h>
#include <matrixSHEN.h>  /*by SHEN*/
#include <cres.h>

#define ALLPLN 1658
#define X_PLN  11
#define Y_PLN  22
#define Z_PLN  33
#define FAIL   0
#define OOKK   1
#define NNO    0
#define YYES   1

void show_usage(int argc, char **argv);
void calculate_hood_byIncreasingRadius(Ivector3d *Hood, int HoodSize, int *pixelNumInHood, float xres,float yres,float zres);
void OpeningOperationOnBinary3DImage(int argc, char **argv, int Radius, Ivector3d dim, float Z_THICKINK);
void ClosingOperationOnBinary3DImage(int argc, char **argv, int Radius, Ivector3d dim, float Z_THICKINK);
void NeighboringSimilarPointsSearch(long int *PointNum, int ***Detected_Region, int order, unsigned char ***src, 
				    unsigned char ***status, int i, int j, int k, int GreySelected, Ivector3d dim, 
				    Ivector3d *Stack, unsigned char ***VoxelInStack, int nbrSize, int GrowingOnPlan);
void region_growing(int argc, char **argv, Ivector3d dim, int nbrSize, int GreySelected, int RegionPointNum, int GrowingOnPlan);
void FillConcaveRegionsInBinary3DImage(int argc, char **argv, int Radius, float level, Ivector3d dim, float Z_THICKINK);
void FillConcaveRegionsInBinary3DImageBy2dViews(int argc, char **argv, int Radius, float level, Ivector3d dim, float Z_THICKINK);
void RemovingNoisesInBinary3DImage(int argc, char **argv, int Radius, float ThLevel, Ivector3d dim, float Zthickness);

void show_usage(int argc, char **argv)
{
  printf("USAGE: %s <input_file> <output_file>\n\
\t -t <int>               : type of operation, 0: Opening; \n\
                                                     1: Closing; \n\
                                                     2: region growing; \n\
                                                     3: filling hole in 3D; \n\
                                                     4: filling hole in 2D views\n\
                                                     5: remove noise\n\
\t -r <int>               : radius for opening or closing (default: 3)\n\
\t -d <int>,<int>         : image XY dimension (default: 256,256)\n\
\t -v <float>             : Z thickness (default: 1.5)\n\
\t -n <int>               : volume threshold for region growing (default: 300)\n\
\t -m <int>               : neighbor size for region growing (default: 3)\n\
\t -g <int>               : growing for which plane (default: 1658)\n\
                                X_PLN: 11\n\
                                Y_PLN: 22\n\
                                Z_PLN: 33\n\
                                ALLPLN: 1658\n\
", argv[0]);
  exit(1);
}

void NeighboringSimilarPointsSearch(long int *PointNum, int ***Detected_Region, int order, unsigned char ***src, 
				    unsigned char ***status, int i, int j, int k, int GreySelected, Ivector3d dim, 
				    Ivector3d *Stack, unsigned char ***VoxelInStack, int nbrSize, int GrowingOnPlan)
{
  int        kk,ii,jj, x, y; 
  int        k_plus_nbrSize, k_minus_nbrSize, i_plus_nbrSize, i_minus_nbrSize, j_plus_nbrSize, j_minus_nbrSize;

  /* stack */
  long int   Pointer ;

  /* add the first searched point */
  Pointer = 0; 
  Stack[Pointer].x = i; 
  Stack[Pointer].y = j; 
  Stack[Pointer].z = k; 

  do{
    i = Stack[Pointer].x;
    j = Stack[Pointer].y;
    k = Stack[Pointer].z;
    Pointer --;

    status[k][i][j] = OOKK;
    (*PointNum) ++;
    Detected_Region[k][i][j] = order;

    k_plus_nbrSize =k+nbrSize;
    k_minus_nbrSize=k-nbrSize;
    i_plus_nbrSize =i+nbrSize;
    i_minus_nbrSize=i-nbrSize;
    j_plus_nbrSize =j+nbrSize;
    j_minus_nbrSize=j-nbrSize;

    if (k_plus_nbrSize>=dim.z) k_plus_nbrSize=dim.z-1;
    if (k_minus_nbrSize<0)     k_minus_nbrSize=0;
    if (j_plus_nbrSize>=dim.y) j_plus_nbrSize=dim.y-1;
    if (j_minus_nbrSize<0)     j_minus_nbrSize=0;
    if (i_plus_nbrSize>=dim.x) i_plus_nbrSize=dim.x-1;
    if (i_minus_nbrSize<0)     i_minus_nbrSize=0;

    /* now we can check immediately adjacent points to see if they too could be added to the track */
    for(kk=k_minus_nbrSize; kk<=k_plus_nbrSize; kk++)
      for(ii=i_minus_nbrSize; ii<=i_plus_nbrSize; ii++)
	for(jj=j_minus_nbrSize; jj<=j_plus_nbrSize; jj++) {

	  if( !(kk==k && jj==j && ii==i) && 
	      ((GrowingOnPlan!=X_PLN || ii==i) && (GrowingOnPlan!=Y_PLN || jj==j) && (GrowingOnPlan!=Z_PLN || kk==k)) && 
	      (kk==k&&jj==j || kk==k&&ii==i || jj==j&&ii==i) && 
	      src[kk][ii][jj]==GreySelected && status[kk][ii][jj]==FAIL && VoxelInStack[kk][ii][jj]==NNO) {
	    Pointer ++ ;
	    Stack[Pointer].x = ii ;
	    Stack[Pointer].y = jj; 
	    Stack[Pointer].z = kk; 
	    
	    VoxelInStack[kk][ii][jj]= YYES ;
	  }
	}
  }while(Pointer>=0) ;
}

void calculate_hood_byIncreasingRadius(Ivector3d *Hood, int HoodSize, int *pixelNumInHood, 
				       float xres,float yres,float zres)
{
  int        i, j, k, l ;
  int        x, y, z, t, r  ;
  int        half ; 

  t = 0;  
  Hood[0].x = Hood[0].y = Hood[0].z = 0;
  t++;
  half = HoodSize/2;
  for(r=1; r<=half; r++) {
      for(z=-r; z<=r; z+=2*r)
	for(x=-r; x<=r; x++)
	  for(y=-r; y<=r; y++)
	    if((x*x+y*y+z*z)<=r*r) {
	      Hood[t].x = x/xres;
	      Hood[t].y = y/yres;
	      Hood[t].z = z/zres;
	      t++;
	    }
      for(x=-r; x<=r; x+=2*r)
	for(z=-r+1; z<=r-1; z++) 
	  for(y=-r; y<=r; y++)
	    if((x*x+y*y+z*z)<=r*r) {
	      Hood[t].x = x/xres;
	      Hood[t].y = y/yres;
	      Hood[t].z = z/zres;
	      
	      t++;
	    }
      for(y=-r; y<=r; y+=2*r)
	for(z=-r+1; z<=r-1; z++) 
	  for(x=-r+1; x<=r-1; x++) 
	    if((x*x+y*y+z*z)<=r*r) {
	      Hood[t].x = x/xres ;
	      Hood[t].y = y/yres ;
	      Hood[t].z = z/zres ;
	      t++ ;
	    }
  }
  (*pixelNumInHood) = t ;
  printf("total %d in the search area!\n", t) ;
}

void OpeningOperationOnBinary3DImage(int argc, char **argv, int Radius, Ivector3d dim, float Z_THICKINK)
{
  unsigned char         ***in, ***out, ***Temp_Img;
  int                   i, j, k;
  int                   l, m, n, index;
  double                TotalWhite;
  int                   NumInSphere, RealNum;
  double                r;
  int                   x_Start, x_End, y_Start, y_End, z_Start, z_End;
  int                   X_Start, X_End, Y_Start, Y_End, Z_Start, Z_End;
  FILE                  *fp;
  Ivector3d             *MyHood;
  int                   MyHoodSize, pixelNumIn_MyHood;

  printf("Opening operation\n");
  /* read image */
  fp=myopen(argv[1], "r");
  fseek(fp, 0, SEEK_END);
  dim.z = ftell(fp)/(dim.x*dim.y);
  rewind(fp);

  in  = UCalloc3d(dim.x, dim.y, dim.z);
  out = UCalloc3d(dim.x, dim.y, dim.z);

  for(k=0; k<dim.z; k++)
    for(i=0; i<dim.x; i++) fread(in[k][i], 1, dim.y, fp);
  fclose(fp);

  NumInSphere = (2*Radius+1)*(2*Radius+1)*(2*Radius/Z_THICKINK+1) ;
  printf("\n\nopening operation on 3D binary image, using radius =%d\n\n", Radius) ;

  /* spherical mask */
  MyHoodSize = Radius*2+1 ;
  MyHood     = Ivector3dalloc1d(MyHoodSize*MyHoodSize*MyHoodSize) ;
  calculate_hood_byIncreasingRadius(MyHood, MyHoodSize, &pixelNumIn_MyHood, 1., 1., Z_THICKINK) ;

  /* apply for space for temporary image result */
  Temp_Img = UCalloc3d(dim.x, dim.y, dim.z);
  /* reset */
  for(k=0;k<dim.z;k++)
    for(i=0;i<dim.x;i++)
      for(j=0;j<dim.y;j++) Temp_Img[k][i][j] = 255;

  /* get bounding box */
  X_Start = Y_Start = Z_Start = 1000;
  X_End = Y_End = Z_End = -1000;

  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
	if (in[k][i][j] != 0) {

	  if (i<X_Start) X_Start = i;
	  if (j<Y_Start) Y_Start = j;
	  if (k<Z_Start) Z_Start = k;
	  if (i>X_End) X_End = i;
	  if (j>Y_End) Y_End = j;
	  if (k>Z_End) Z_End = k;
	}

  /* find mask ... */
  x_Start = X_Start-2*Radius ;  if(x_Start<0)      x_Start=0;
  x_End   = X_End+2*Radius ;    if(x_End>dim.x-1)  x_End  =dim.x-1;
  y_Start = Y_Start-2*Radius ;  if(y_Start<0)      y_Start=0;
  y_End   = Y_End+2*Radius ;    if(y_End>dim.y-1)  y_End  =dim.y-1;
  z_Start = Z_Start-2*Radius ;  if(z_Start<0)      z_Start=0;
  z_End   = Z_End+2*Radius ;    if(z_End>dim.z-1)  z_End  =dim.z-1;

  for(k=z_Start; k<=z_End; k++)
    for(i=x_Start; i<=x_End; i++)
      for(j=y_Start; j<=y_End; j++)
	if( in[k][i][j]==0 ) {

	  TotalWhite = 0 ;
	  RealNum = 0 ;
	  for( index=0; index<pixelNumIn_MyHood ; index++) {

	    l = MyHood[index].x;
	    m = MyHood[index].y;
	    n = MyHood[index].z;	  
	    if( l*l+m*m+n*n<Radius*Radius &&  
		i+l>=0 && i+l<dim.x && 
		j+m>=0 && j+m<dim.y && 
		k+n>=0 && k+n<dim.z ) {
	      
	      TotalWhite += in[k+n][i+l][j+m];
	      RealNum ++;
	      
	      /* for speeding up*/
	      if( in[k+n][i+l][j+m]!=0 ) index = pixelNumIn_MyHood;
	    }
	  }
	  TotalWhite /= 255.0;
	  
	  if( TotalWhite==0 ) Temp_Img[k][i][j] = 255;
	  else Temp_Img[k][i][j] = 0;
	} 
	else Temp_Img[k][i][j] = 0;
  
  /* operation ... */
  for(k=0; k<dim.z; k++)
    for(i=0;i<dim.x;i++)
      for(j=0;j<dim.y;j++) out[k][i][j] = 255 - Temp_Img[k][i][j] ;
  
  fp = fopen(argv[2], "w");
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) fwrite(out[k][i], 1, dim.y, fp);
  fclose(fp);

  /* free */
  UCfree3d(in, dim.z, dim.x);
  UCfree3d(out, dim.z, dim.x);
  UCfree3d(Temp_Img, dim.z, dim.x);
  free(MyHood);
}


void ClosingOperationOnBinary3DImage(int argc, char **argv, int Radius, Ivector3d dim, float Z_THICKINK)
{
  unsigned char         ***Temp_Img, ***in, ***out;
  int                   i, j, k ;
  int                   l, m, n, index ;
  double                TotalWhite ;
  int                   NumInSphere, RealNum ;
  int                   x_Start, x_End, y_Start, y_End, z_Start, z_End ;
  int                   X_Start, X_End, Y_Start, Y_End, Z_Start, Z_End ;
  FILE                  *fp;
  Ivector3d             *MyHood ;
  int                   MyHoodSize, pixelNumIn_MyHood ;

  printf("Closing operation\n");
  /* read image */
  fp=myopen(argv[1], "r");
  fseek(fp, 0, SEEK_END);
  dim.z = ftell(fp)/(dim.x*dim.y);
  rewind(fp);

  in  = UCalloc3d(dim.x, dim.y, dim.z);
  out = UCalloc3d(dim.x, dim.y, dim.z);

  for(k=0; k<dim.z; k++)
    for(i=0; i<dim.x; i++) fread(in[k][i], 1, dim.y, fp);
  fclose(fp);
  printf("%s read ok\n", argv[1]);

  NumInSphere = (2*Radius+1)*(2*Radius+1)*(2*Radius/Z_THICKINK+1);
  printf("\n\nClosing operation on 3D binary image, using radius =%d\n\n", Radius) ;

  /* spherical mask */
  MyHoodSize = Radius*2+1 ;
  MyHood     = Ivector3dalloc1d(MyHoodSize*MyHoodSize*MyHoodSize) ;
  calculate_hood_byIncreasingRadius(MyHood, MyHoodSize, &pixelNumIn_MyHood, 1., 1., Z_THICKINK) ;

  /* get bounding box */
  X_Start = Y_Start = Z_Start = 1000;
  X_End = Y_End = Z_End = -1000;

  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
	if (in[k][i][j] != 0) {

	  if (i<X_Start) X_Start = i;
	  if (j<Y_Start) Y_Start = j;
	  if (k<Z_Start) Z_Start = k;
	  if (i>X_End) X_End = i;
	  if (j>Y_End) Y_End = j;
	  if (k>Z_End) Z_End = k;
	}

  printf("bounding box: (%d, %d, %d) --> (%d, %d, %d)\n",
	 X_Start, Y_Start, Z_Start, X_End, Y_End, Z_End);

  /* find mask ... */
  x_Start = X_Start-Radius ;  if( x_Start<0 )           x_Start=0;
  x_End   = X_End+Radius ;    if( x_End>dim.x-1 )  x_End  =dim.x-1 ;
  y_Start = Y_Start-Radius ;  if( y_Start<0 )           y_Start=0;
  y_End   = Y_End+Radius ;    if( y_End>dim.y-1 )  y_End  =dim.y-1 ;
  z_Start = Z_Start-Radius ;  if( z_Start<0 )           z_Start=0;
  z_End   = Z_End+Radius ;    if( z_End>dim.z-1 )      z_End  =dim.z-1 ;

  for(k=z_Start; k<=z_End; k++)
    for(i=x_Start; i<=x_End; i++)
      for(j=y_Start; j<=y_End; j++)
	if(in[k][i][j]==255)
	{
	  TotalWhite = 0 ;
	  RealNum = 0 ;
	  for( index=0; index<pixelNumIn_MyHood ; index++) {

	      l = MyHood[index].x ;
	      m = MyHood[index].y ;
	      n = MyHood[index].z ;	  
	      if( l*l+m*m+n*n<Radius*Radius &&  
		  i+l>=0 && i+l<dim.x && 
		  j+m>=0 && j+m<dim.y && 
		  k+n>=0 && k+n<dim.z ) {

		  TotalWhite += in[k+n][i+l][j+m] ;
		  RealNum ++ ;
		  
		  /* for speeding up*/
		  if( in[k+n][i+l][j+m]!=255 ) index = pixelNumIn_MyHood ;
		}
	    }
	  TotalWhite /= RealNum;
	  
	  if( TotalWhite==255 ) out[k][i][j] = 255 ;
	  else out[k][i][j] = 0 ;
	}
	else out[k][i][j] = 0 ;

  fp = fopen(argv[2], "w");
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) fwrite(out[k][i], 1, dim.y, fp);
  fclose(fp);

  /* free */
  UCfree3d(in, dim.z, dim.x) ;
  UCfree3d(out, dim.z, dim.x) ;
  free(MyHood) ;
}

void region_growing(int argc, char **argv, Ivector3d dim, int nbrSize, int GreySelected, 
		    int RegionPointNum, int GrowingOnPlan)
{
  unsigned char         ***status, ***in, ***VoxelInStack, ***Result_Detected_Region; 
  int                   ***Detected_Region, order;
  int                   i,j,k, ii,jj,kk;
  long int              PointNum;
  FILE                  *fp;
  int                   *PointNumInCertainOrder;
  int                   ActualUsedRegionPointNum;
  Ivector3d             *Stack;

  printf("\n\n nbrSize=%d\n GreySelected=%d\n RegionPointNum=%d\n \n", nbrSize, GreySelected, RegionPointNum) ;

  /* read image */
  fp=myopen(argv[1], "r");
  fseek(fp, 0, SEEK_END);
  dim.z = ftell(fp)/(dim.x*dim.y);
  rewind(fp);

  in  = UCalloc3d(dim.x, dim.y, dim.z);

  for(k=0; k<dim.z; k++)
    for(i=0; i<dim.x; i++) fread(in[k][i], 1, dim.y, fp);
  fclose(fp);

  /* apply for memory */ 
  status = UCalloc3d(dim.x, dim.y, dim.z);
  Detected_Region = Ialloc3d(dim.x, dim.y, dim.z);
  PointNumInCertainOrder = Ialloc1d(dim.x*dim.y*dim.z);

  /* reset ... */
  for(i=0; i<dim.x; i++)
    for(j=0; j<dim.y; j++)
      for(k=0; k<dim.z; k++) status[k][i][j] = FAIL;
  
  /* Global variables: for growing */	
  VoxelInStack = UCalloc3d(dim.x, dim.y, dim.z);
  Stack = Ivector3dalloc1d(dim.x*dim.y*dim.z);

  /* status for representing whether current voxel already in stack? */
  for(kk=0; kk<dim.z; kk++)
    for(ii=0; ii<dim.x; ii++)
      for(jj=0; jj<dim.y; jj++) VoxelInStack[kk][ii][jj] = NNO;

  /* region growing ... */
  printf("Region growing ...\n") ;
  order = 1;
  for(k=0; k<dim.z; k++) /* It's important to use this order, z-axis first*/
    for(i=0; i<dim.x; i++)
      for(j=0; j<dim.y; j++)
	if(in[k][i][j]==GreySelected && status[k][i][j]==FAIL) {
	  PointNum = 0;
	  NeighboringSimilarPointsSearch(&PointNum, Detected_Region, order, in, status, i, j, k, GreySelected, dim, 
					 Stack, VoxelInStack, nbrSize, GrowingOnPlan);

	  if( PointNum > RegionPointNum ) printf("PointNum=%d\n", PointNum);
	  PointNumInCertainOrder[order] = PointNum;
	  order ++;
	}

  /* remove small connected regions */
  for(i=0; i<dim.x; i++)
    for(j=0; j<dim.y; j++)
      for(k=0; k<dim.z; k++) {
	
	order = Detected_Region[k][i][j];
	ActualUsedRegionPointNum = RegionPointNum;
	if(order>=1 && PointNumInCertainOrder[order]<ActualUsedRegionPointNum) in[k][i][j] = 255-GreySelected;
      }

  Result_Detected_Region = UCalloc3d(dim.x, dim.y, dim.z);
  for(k=0; k<dim.z; k++) /* It's important to use this order, z-axis first*/
    for(i=0; i<dim.x; i++)
      for(j=0; j<dim.y; j++) {

	order = Detected_Region[k][i][j];
	if (order>=1 && PointNumInCertainOrder[order]>RegionPointNum)
	  Result_Detected_Region[k][i][j] = (unsigned char)Detected_Region[k][i][j];
      }

  /* save result */
  fp = fopen(argv[2], "w");
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) fwrite(/*in*/Result_Detected_Region[k][i], 1, dim.y, fp);
  fclose(fp);
      
  /* free */
  UCfree3d(in, dim.z, dim.x);
  UCfree3d(status, dim.z, dim.x);
  UCfree3d(Result_Detected_Region, dim.z, dim.x);
  Ifree3d(Detected_Region, dim.z, dim.x);
  free(PointNumInCertainOrder);

  /* free: Global variables */
  free(Stack);
  UCfree3d(VoxelInStack, dim.z, dim.x);
}


void FillConcaveRegionsInBinary3DImage(int argc, char **argv, int Radius, float level, Ivector3d dim, float Z_THICKINK)
{
  unsigned char        ***Temp_Img, ***in, ***out;
  int                  i, j, k, l, m, n ;
  double               TotalWhite ;
  int                  NumInSphere, threshold ;
  int                  x_Start, x_End, y_Start, y_End, z_Start, z_End ;
  int                  X_Start, X_End, Y_Start, Y_End, Z_Start, Z_End ;
  FILE                 *fp;
 
  printf("Closing operation 3D\n");

  /* read image */
  fp=myopen(argv[1], "r");
  fseek(fp, 0, SEEK_END);
  dim.z = ftell(fp)/(dim.x*dim.y);
  rewind(fp);

  in  = UCalloc3d(dim.x, dim.y, dim.z);
  out = UCalloc3d(dim.x, dim.y, dim.z);

  for(k=0; k<dim.z; k++)
    for(i=0; i<dim.x; i++) fread(in[k][i], 1, dim.y, fp);
  fclose(fp);

  NumInSphere = (2*Radius+1)*(2*Radius+1)*(2*Radius/Z_THICKINK+1);
  threshold = NumInSphere*level;
  printf("\n\nFill holes, bays in the binary image, using radius =%d\n\n", Radius);

  /* apply for space for temporary image result */
  Temp_Img = UCalloc3d(dim.x,dim.y,dim.z);

  /* get bounding box */
  X_Start = Y_Start = Z_Start = 1000;
  X_End = Y_End = Z_End = -1000;

  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
	if (in[k][i][j] != 0) {

	  if (i<X_Start) X_Start = i;
	  if (j<Y_Start) Y_Start = j;
	  if (k<Z_Start) Z_Start = k;
	  if (i>X_End) X_End = i;
	  if (j>Y_End) Y_End = j;
	  if (k>Z_End) Z_End = k;
	}

  /* action ... */
  x_Start = X_Start-Radius ;  if(x_Start<0)      x_Start=0;
  x_End   = X_End+Radius ;    if(x_End>dim.x-1 ) x_End  =dim.x-1 ;
  y_Start = Y_Start-Radius ;  if(y_Start<0)      y_Start=0;
  y_End   = Y_End+Radius ;    if(y_End>dim.y-1 ) y_End  =dim.y-1 ;
  z_Start = Z_Start-Radius ;  if(z_Start<0)      z_Start=0;
  z_End   = Z_End+Radius ;    if(z_End>dim.z-1 ) z_End  =dim.z-1 ;

  for(k=z_Start; k<=z_End; k++)
    for(i=x_Start; i<=x_End; i++)
      for(j=y_Start; j<=y_End; j++)
	if( in[k][i][j]==0 ) {

	  TotalWhite = 0 ;
	  for(l=-Radius; l<=Radius; l++)
	    for(m=-Radius; m<=Radius; m++)
	      for(n=-Radius/Z_THICKINK; n<=Radius/Z_THICKINK; n++)
		if(i+l>=0 && i+l<dim.x && j+m>=0 && j+m<dim.y && k+n>=0 && k+n<dim.z) TotalWhite += in[k+n][i+l][j+m];
	  TotalWhite /= 255.0 ;
	  
	  if(TotalWhite>threshold) Temp_Img[k][i][j] = 255 ;
	}
	else Temp_Img[k][i][j] = 255 ;

  /* transfer ... */
  for(k=0; k<dim.z; k++)
    for(i=0; i<dim.x; i++)
      for(j=0; j<dim.y; j++) out[k][i][j] = Temp_Img[k][i][j];

  fp = fopen(argv[2], "w");
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) fwrite(out[k][i], 1, dim.y, fp);
  fclose(fp);

  printf("Hole filled!\n");
  /* free */
  UCfree3d(Temp_Img, dim.z, dim.x);
  UCfree3d(in, dim.z, dim.x);
  UCfree3d(out, dim.z, dim.x);
}


void FillConcaveRegionsInBinary3DImageBy2dViews(int argc, char **argv, int Radius, float level, Ivector3d dim, float Z_THICKINK)
{
  unsigned char         ***Temp_Img, ***in;
  int                   i, j, k, l, m, n;
  double                TotalWhite;
  int                   NumInSphere, threshold;
  int                   x_Start, x_End, y_Start, y_End, z_Start, z_End;
  int                   X_Start, X_End, Y_Start, Y_End, Z_Start, Z_End;
  FILE                  *fp;

  printf("closing according to 2d views\n");

  /* read image */
  fp=fopen(argv[1], "r");
  fseek(fp, 0, SEEK_END);
  dim.z = ftell(fp)/(dim.x*dim.y);
  rewind(fp);

  in  = UCalloc3d(dim.x, dim.y, dim.z);

  for(k=0; k<dim.z; k++)
    for(i=0; i<dim.x; i++) fread(in[k][i], 1, dim.y, fp);
  fclose(fp);

  NumInSphere = (2*Radius+1)*(2*Radius+1) ;
  threshold = NumInSphere*level ;
  printf("\n\nFill holes, bays in the binary image, using radius =%d at 2D planes!\n\n", Radius) ;

  /* apply for space for temporary image result */
  Temp_Img = UCalloc3d(dim.x,dim.y,dim.z);

  /* get bounding box */
  X_Start = Y_Start = Z_Start = 1000;
  X_End = Y_End = Z_End = -1000;

  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
	if (in[k][i][j] != 0) {

	  if (i<X_Start) X_Start = i;
	  if (j<Y_Start) Y_Start = j;
	  if (k<Z_Start) Z_Start = k;
	  if (i>X_End) X_End = i;
	  if (j>Y_End) Y_End = j;
	  if (k>Z_End) Z_End = k;
	}

  /* action ... */
  x_Start = X_Start-Radius ;  if(x_Start<0)      x_Start=0;
  x_End   = X_End+Radius ;    if(x_End>dim.x-1)  x_End  =dim.x-1;
  y_Start = Y_Start-Radius ;  if(y_Start<0)      y_Start=0;
  y_End   = Y_End+Radius ;    if(y_End>dim.y-1)  y_End  =dim.y-1;
  z_Start = Z_Start-Radius ;  if(z_Start<0)      z_Start=0;
  z_End   = Z_End+Radius ;    if(z_End>dim.z-1)  z_End  =dim.z-1;

  /* Z dimension */
  for(k=z_Start; k<=z_End; k++)
    for(i=x_Start; i<=x_End; i++)
      for(j=y_Start; j<=y_End; j++)
	if(in[k][i][j]==0) {
	  TotalWhite = 0;
	  for(l=-Radius; l<=Radius; l++)
	    for(m=-Radius; m<=Radius; m++)
	      if(i+l>=0 && i+l<dim.x && j+m>=0 && j+m<dim.y) TotalWhite += in[k][i+l][j+m];
	  TotalWhite /= 255.0 ;
	  
	  if(TotalWhite>threshold) Temp_Img[k][i][j] = 255 ;
	}
	else Temp_Img[k][i][j] = 255 ;
  
  /* transfer ... */
  for(k=0;k<dim.z;k++)
    for(i=0;i<dim.x;i++)
      for(j=0;j<dim.y;j++) in[k][i][j] = Temp_Img[k][i][j] ;
  
  /* x dimension */
  for(k=z_Start; k<=z_End; k++)
    for(i=x_Start; i<=x_End; i++)
      for(j=y_Start; j<=y_End; j++)
	if(in[k][i][j]==0) {
	  TotalWhite = 0;
	  for(m=-Radius; m<=Radius; m++)
	    for(n=-Radius/Z_THICKINK; n<=Radius/Z_THICKINK; n++)
	      if(j+m>=0 && j+m<dim.y && k+n>=0 && k+n<dim.z) TotalWhite += in[k+n][i][j+m];
	  TotalWhite /= 255.0;
	  
	  if(TotalWhite>threshold) Temp_Img[k][i][j] = 255;
	}
	else Temp_Img[k][i][j] = 255;
  
  /* transfer ... */
  for(k=0;k<dim.z;k++)
     for(i=0;i<dim.x;i++)
       for(j=0;j<dim.y;j++) in[k][i][j] = Temp_Img[k][i][j] ;
    
  /* y dimension */
  for(k=z_Start; k<=z_End; k++)
    for(i=x_Start; i<=x_End; i++)
      for(j=y_Start; j<=y_End; j++)
	if(in[k][i][j]==0) {
	  TotalWhite = 0;
	  for(l=-Radius; l<=Radius; l++)
	    for(n=-Radius/Z_THICKINK; n<=Radius/Z_THICKINK; n++)
	      if(i+l>=0 && i+l<dim.x && k+n>=0 && k+n<dim.z) TotalWhite += in[k+n][i+l][j];
	  TotalWhite /= 255.0;
	  
	  if(TotalWhite>threshold) Temp_Img[k][i][j] = 255;
	}
	else Temp_Img[k][i][j] = 255;
  
  /* transfer ... */
  for(k=0;k<dim.z;k++)
    for(i=0;i<dim.x;i++)
      for(j=0;j<dim.y;j++) in[k][i][j] = Temp_Img[k][i][j];

  printf("Hole filled!\n");
  /* free */
  UCfree3d(Temp_Img, dim.z, dim.x);
  UCfree3d(in, dim.z, dim.x);
}

void RemovingNoisesInBinary3DImage(int argc, char **argv, int Radius, float ThLevel, Ivector3d dim, float Zthickness)
{
  unsigned char ***in, ***out;
  int           i, j, k ;
  int           l, m, n ;
  double        TotalWhite ;
  int           NumInSphere, threshold ;
  int           x_Start, x_End, y_Start, y_End, z_Start, z_End ;
  int           X_Start, X_End, Y_Start, Y_End, Z_Start, Z_End ;
  FILE          *fp;
 
  /* read image */
  fp=myopen(argv[1], "r");
  fseek(fp, 0, SEEK_END);
  dim.z = ftell(fp)/(dim.x*dim.y);
  rewind(fp);

  in = UCalloc3d(dim.x, dim.y, dim.z);
  out = UCalloc3d(dim.x, dim.y, dim.z);

  for(k=0; k<dim.z; k++)
    for(i=0; i<dim.x; i++) fread(in[k][i], 1, dim.y, fp);
  fclose(fp);
  
  NumInSphere = (2*Radius+1)*(2*Radius+1)*(2*Radius/Zthickness+1) ;
  threshold = NumInSphere*ThLevel ;
  printf("\n\nSmoothing binary image, using radius =%d\n\n", Radius) ;

  /* get bounding box */
  X_Start = Y_Start = Z_Start = 1000;
  X_End = Y_End = Z_End = -1000;

  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
	if (in[k][i][j] != 0) {

	  if (i<X_Start) X_Start = i;
	  if (j<Y_Start) Y_Start = j;
	  if (k<Z_Start) Z_Start = k;
	  if (i>X_End) X_End = i;
	  if (j>Y_End) Y_End = j;
	  if (k>Z_End) Z_End = k;
	}

  /* action ... */
  x_Start = X_Start-Radius;  if( x_Start<0 )      x_Start=0;
  x_End   = X_End+Radius;    if( x_End>dim.x-1 )  x_End  =dim.x-1;
  y_Start = Y_Start-Radius;  if( y_Start<0 )      y_Start=0;
  y_End   = Y_End+Radius;    if( y_End>dim.y-1 )  y_End  =dim.y-1;
  z_Start = Z_Start-Radius;  if( z_Start<0 )      z_Start=0;
  z_End   = Z_End+Radius;    if( z_End>dim.z-1 )  z_End  =dim.z-1;

  for(k=z_Start; k<=z_End; k++)
    for(i=x_Start; i<=x_End; i++)
      for(j=y_Start; j<=y_End; j++)
	if(in[k][i][j]==255)
	{
	  TotalWhite = 0;
	  for(l=-Radius; l<=Radius; l++)
	    for(m=-Radius; m<=Radius; m++)
	      for(n=-Radius/Zthickness; n<=Radius/Zthickness; n++)
		if(i+l>=0 && i+l<dim.x && j+m>=0 && j+m<dim.y && k+n>=0 && k+n<dim.z) TotalWhite += in[k+n][i+l][j+m];
	  TotalWhite /= 255.0;

	  if(TotalWhite<threshold) out[k][i][j] = 0;
	  else out[k][i][j] = 255;
	}
	else out[k][i][j] = 0;

  fp = fopen(argv[2], "w");
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) fwrite(out[k][i], 1, dim.y, fp);
  fclose(fp);

  /* free */
  UCfree3d(in, dim.z, dim.x);
  UCfree3d(out, dim.z, dim.x);
}

int main(int argc,char *argv[])
{
  int               c, num;
  FILE              *fp;
  int               i, j, k, op, GreySelected, RegionPointNum, GrowingOnPlan, nbrSize, radius;
  float             RemovingLevel, Zthickness, level;
  Ivector3d         dim;

  num = 3;
  if(argc<num)    show_usage(argc, argv);

  GreySelected = 255;  
  RegionPointNum = 300;  
  GrowingOnPlan = 1658;
  nbrSize = 3;
  RemovingLevel = 0.3;
  radius = 3;
  Zthickness = 1.5;
  dim.x = dim.y = 256;
  level = 0.3;
  /* parameters */
  while((c=getopt(argc-2,argv+2,"t:r:d:v:n:m:g:l:")) != -1) {
    switch(c) {

    case 't':
      op = atoi(optarg);
      break;

    case 'r':
      radius=atoi(optarg);
      break;
      
    case 'd':
      sscanf(optarg, "%d,%d", &dim.x, &dim.y);

    case 'v':
      Zthickness = atof(optarg);
      break;

    case 'n':
      RegionPointNum = atoi(optarg);
      break;

    case 'm':
      nbrSize = atoi(optarg);
      break;

    case 'g':
      GrowingOnPlan = atoi(optarg);
      break;

    case 'l':
      level = atof(optarg);
      break;

    default:
      break;
    }
  }
  
  printf("operation = %d\n", op);
  printf("radius = %d\n", radius);
  printf("image XY dimension: (%d, %d)", dim.x, dim.y);
  printf("image voxel Zthickness: %f\n", Zthickness);
  printf("level = %f\n", level);

  if (op == 0) OpeningOperationOnBinary3DImage(argc, argv, radius, dim, Zthickness);
  if (op == 1) ClosingOperationOnBinary3DImage(argc, argv, radius, dim, Zthickness);
  if (op == 2) region_growing(argc, argv, dim, nbrSize, GreySelected, RegionPointNum, GrowingOnPlan);
  if (op == 3) FillConcaveRegionsInBinary3DImage(argc, argv, radius, level, dim, Zthickness);
  if (op == 4) FillConcaveRegionsInBinary3DImageBy2dViews(argc, argv, radius, level, dim, Zthickness);
  if (op == 5) RemovingNoisesInBinary3DImage(argc, argv, radius, level, dim, Zthickness);

  return 0;
}
