package bl.coe.BigSparseMath;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Random;
import java.util.Vector;

/**
 * The Class LabelPairs.
 * 
 * Provides a mechanism to load a 2xN vector of index locations and labels. 
 *  
 * @author Bennett Landman, bennett.landman@vanderbilt.edu
 */
public class LabelPairs {

	/** The index. */
	int []index;
	
	/** The labels. */
	int []labels; 

	/**
	 * Instantiates a new label pairs.
	 * 
	 * @param textFilename the text filename
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public LabelPairs(String textFilename) throws IOException {	
		BufferedReader inFp = new BufferedReader((new FileReader(textFilename)));
		int countLines=0;
		String line;
		while(null!=(line=inFp.readLine())){
			if(line.trim().length()>0) {
				countLines++;
			}
		}
		inFp.close();
		index = new int[countLines];
		labels = new int[countLines];
		inFp = new BufferedReader((new FileReader(textFilename)));
		
		countLines=0;
		while(null!=(line=inFp.readLine())) {
			if(line.length()>0) {
				String []numbers = line.trim().split(BigMathPreferences.WHITE_SPACE_REGEX);
//				System.out.println(countLines+":"+numbers[0]+":\t:"+numbers[1]+"\\"+line);
				index[countLines]=(int)Float.valueOf(numbers[0]).intValue();
				labels[countLines]=(int)Float.valueOf(numbers[1]).intValue();
				countLines++;
			}
		}
		inFp.close();
	}
	
	/**
	 * Write to a tab delimited text file with one pair per line.
	 * 
	 * @param io the io
	 * 
	 * @return the output stream
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public OutputStream writeToText(OutputStream io) throws IOException {
		BufferedWriter fp = new BufferedWriter((new OutputStreamWriter(io)));
		for(int i=0;i<index.length;i++)
			fp.write(index[i]+"\t"+labels[i]+"\n");
		fp.flush();
		return io;
	}

	/**
	 * Instantiates a new label pairs.
	 * 
	 * @param ind the ind
	 * @param lab the lab
	 */
	public LabelPairs(int []ind, int []lab) {		
		index=ind.clone();
		labels=lab.clone();
	}
	
	/**
	 * Gets the labeled column indexes.
	 * 
	 * @return the labeled column indexes
	 */
	public BigVector getLabeledColumnIndexes() {
		return new BigDenseVector(index);
	}

	/**
	 * Gets the column label values.
	 * 
	 * @return the column label values
	 */
	public int[] getColumnLabelValues() {
		return labels.clone();
	}
	
	/**
	 * Gets the number of label pairs.
	 * 
	 * @return the number of label pairs
	 */
	public int getNumberOfLabelPairs() {
		return labels.length;
	}
	
	/**
	 * Creates the randomized disjoint subsets.
	 * 
	 * @param NLabelsPerSubset the n labels per subset
	 * @param nSubsets the n subsets
	 * 
	 * @return the label pairs[]
	 */
	public LabelPairs []createRandomizedDisjointSubsets(int NLabelsPerSubset, int nSubsets, long seedRandom) {
		int N = index.length/NLabelsPerSubset;
		if(N>nSubsets)
			N = nSubsets;
		
		if(N<1)
			return null;
		LabelPairs []result = new LabelPairs [N];
		ArrayList<Integer> order = new ArrayList<Integer>(index.length); 
		for(int i=0;i<index.length;i++) {
			order.set(i,i);
		}
		Random rnd = new Random(seedRandom);
		Collections.shuffle(order,rnd);
		int []sortOrder = new int[NLabelsPerSubset];
		int curIdx=0;
		int []ind = new int[NLabelsPerSubset];
		int []lab = new int[NLabelsPerSubset];
		for(int i=0;i<N;i++) {
			for(int j=0;j<NLabelsPerSubset;j++) {
				sortOrder[j] = order.get(curIdx);			
				curIdx++;
			}
			Arrays.sort(sortOrder);
			for(int j=0;j<NLabelsPerSubset;j++) {				
				ind[j]=index[sortOrder[j]];
				lab[j]=labels[sortOrder[j]];				
			}
			result[i]=new LabelPairs(ind,lab);
		}		
		return result;
	}
	
	/**
	 * Creates the randomized overlapping subsets.
	 * 
	 * @param NLabelsPerSubset the n labels per subset
	 * @param nSubsets the n subsets
	 * 
	 * @return the label pairs[]
	 */
	public LabelPairs []createRandomizedOverlappingSubsets(int NLabelsPerSubset, int nSubsets, long seedRandom) {
		int N = nSubsets;		
		
		if(N<1)
			return null;
		LabelPairs []result = new LabelPairs [N];

		
		int []ind = new int[NLabelsPerSubset];
		int []lab = new int[NLabelsPerSubset];
		int []sortOrder = new int[NLabelsPerSubset];
		Random rnd = new Random(seedRandom);
		for(int i=0;i<N;i++) {			
			Vector<Integer> order = new Vector<Integer>(index.length);			
			order.setSize(index.length);
			for(int k=0;k<index.length;k++) {
				order.set(k,k);
			}
			Collections.shuffle(order,rnd);
			for(int j=0;j<NLabelsPerSubset;j++) {
				sortOrder[j] = order.get(order.get(j));							
			}
			Arrays.sort(sortOrder);
			for(int j=0;j<NLabelsPerSubset;j++) {				
				ind[j]=index[sortOrder[j]];
				lab[j]=labels[sortOrder[j]];			
			}
		
			result[i]=new LabelPairs(ind,lab);
		}		
		return result;
	}


}
