package bl.coe.BigSparseMath;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Vector;

/**
 * The Class BigDenseVector.
 * 
 * Provides a mechanism for storing dense vectors in a manner compatible with that of BigVector. 
 * 
 * @author Bennett Landman, bennett.landman@vanderbilt.edu
 */
public class BigDenseVector implements BigVector {

	/** The data -- stored internally*/
	protected float []data;

	/**
	 * Instantiates a new big dense vector of a given length.
	 * 
	 * @param length the number of elements
	 */
	BigDenseVector(int length) {
		data = new float[length];		
	}

	/**
	 * Instantiates a new big dense vector by copying an existing array.
	 * 
	 * @param vals the values to be stored
	 */
	public BigDenseVector(float []vals) {
		data = vals.clone();
	}

	/**
	 * Instantiates a new big dense vector by copying an existing sparse vector 
	 * 
	 * @param vec the sparse vector to be copied
	 */
	public BigDenseVector(BigSparseVector vec) {
		data = new float[vec.getLength()];
		for(int i=0;i<vec.countNonEmpty();i++){
			data[vec.getIndexAt(i)] = vec.getValueAt(i);
		}
	}

	/**
	 * Instantiates a new big dense vector based on an integer array. 
	 * Note that data are stored as floating point internally.
	 * 
	 * @param index the integer array to copy.
	 */
	public BigDenseVector(int[] index) {
		data = new float[index.length];
		for(int i=0;i<index.length;i++)
			data[i]=index[i];
	}

	/** 
	 * Counts the number of non-0 elements in the vector. 
	 * 
	 * @return the number of non-zero elements
	 * 
	 * @see bl.coe.BigSparseMath.BigVector#countNonEmpty()
	 */
	public int countNonEmpty() {
		int cnt=0;
		for(int i=0;i<data.length;i++)
			if(data[i]!=0)
				cnt++;
		return cnt;
	}

	/**
	 * Get the value at the indicated position (0-indexed)
	 * 
	 * @param c the index position
	 * @return the value at position c
	 * 
	 * @see bl.coe.BigSparseMath.BigVector#get(int)
	 */
	public float get(int c) {
		return data[c];
	}

	/**
	 * Get the total number of elements in the vector. 
	 * 
	 * @return number of elements. 
	 * 
	 * @see bl.coe.BigSparseMath.BigVector#getLength()
	 */
	public int getLength() {	
		return data.length;
	}

	/** 
	 * Set the value of the vector.
	 * 
	 * @param c the position to set
	 * @param v the value to set at position c
	 * 	
	 * @see bl.coe.BigSparseMath.BigVector#set(int, float)
	 */
	public void set(int c, float v) {
		data[c]=v;
	}

	/**
	 * Gets a direct (not-copied, not-safe) copy of the data.
	 * 
	 * @return the data 
	 */
	public float[] getDataDirect() {
		return data;
	}

	/**
	 * Load full vector text file. The format is one entry per line of text file.
	 * 
	 * @param file the file
	 * 
	 * @return the big dense vector
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	static public BigDenseVector loadFullVectorTextFile(String file) throws IOException {
		return loadFullVectorTextFile(new File(file));		
	}
	
	/**
	 * Load full vector text file. The format is one entry per line of text file.
	 * 
	 * @param file the file
	 * 
	 * @return the big dense vector
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	static public BigDenseVector loadFullVectorTextFile(File file) throws IOException {
		return loadFullVectorTextFile(new FileInputStream(file));		
	}
	
	/**
	 * Load full vector text file. The format is one entry per line of text file.
	 * 
	 * @param file the file
	 * 
	 * @return the big dense vector
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	static public BigDenseVector loadFullVectorTextFile(InputStream file) throws IOException {
		BufferedReader inFp = new BufferedReader((new InputStreamReader(file)));		
			String line=null;		
		Vector<Float> values = new Vector<Float>();
		int idx=0;
		while(null!=(line=inFp.readLine())) { 
			String []numbers= line.split("[ \t,;:a-zA-Z]+");
			for(int i=0;i<numbers.length;i++) {
				values.add(new Float(Float.valueOf(numbers[i])));
				
			}
		}
		idx=values.size();
		inFp.close();
		int []indexes = new int[idx];
		float []vals= new float[idx];
		for(int i=0;i<values.size();i++) {
			indexes[i] = values.get(i).intValue();
			vals[i]=values.get(i).floatValue();
		}
		return new BigDenseVector(vals);
	}

	/**
	 * Write text file. The format is one entry per line of text file.
	 * 
	 * @param string the string
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public void writeTextFile(String string) throws IOException {
		BufferedWriter outFp = new BufferedWriter(new FileWriter(new File(string)));
		for(int i=0;i<data.length;i++){
			outFp.write(data[i]+"\n");
		}
		outFp.close();
	}

	@Override
	public int getLastIndex() {		
		return data.length;
	}

}
