package bl.coe.BigSparseMath;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;

/**
 * The Class LabeledBigSparseMatrix.
 * 
 * Represents a sparse matrix with labeled columns. Intended to 
 * be used in classification of rows. 
 * 
 * @author Bennett Landman, bennett.landman@vanderbilt.edu
 */
public class LabeledBigSparseMatrix extends BigSparseMatrix {

	/** The Constant UUID. */
	private static final String UUID = "LBSM(v1.0)"+0+"-2ec97b4b-0fc3-4e0b-a483-c02c71ec3919";

	/** The col label. */
	protected int colLabel[];

	/**
	 * Instantiates a new labeled big sparse matrix.
	 * 
	 * @param n the n
	 * @param m the m
	 */
	public LabeledBigSparseMatrix(int n, int m) {
		super(n, m);
		nullLabels();
	}

	/**
	 * Null labels.
	 */
	public void nullLabels() {
		colLabel = new int[Mcols];
		for(int i=0;i<Mcols;i++)
			colLabel[i]=-1;

	}

	/* (non-Javadoc)
	 * @see bl.coe.BigSparseMath.BigSparseMatrix#loadFullMatrixTextFile(java.io.File)
	 */
	public void loadFullMatrixTextFile(File file) throws IOException {
		super.loadFullMatrixTextFile(file);
		nullLabels();
	}
	/* (non-Javadoc)
	 * @see bl.coe.BigSparseMath.BigSparseMatrix#extractSubMatrixByRows(int[])
	 */
	public LabeledBigSparseMatrix extractSubMatrixByRows(int []rowSubset) {
		LabeledBigSparseMatrix subMat = new LabeledBigSparseMatrix(rowSubset.length,Mcols);
		subMat.copySubMatrixByRows(this, rowSubset);			
		subMat.setColLabels(colLabel,null);
		return subMat;				
	}

	/* (non-Javadoc)
	 * @see bl.coe.BigSparseMath.BigSparseMatrix#extractSubMatrixByCols(int[])
	 */
	public LabeledBigSparseMatrix extractSubMatrixByCols(int []colSubset) {
		LabeledBigSparseMatrix subMat = new LabeledBigSparseMatrix(Nrows,colSubset.length);
		subMat.copySubMatrixByCols(this, colSubset);			
		subMat.setColLabels(colLabel,colSubset);
		return subMat;				
	}

	/* (non-Javadoc)
	 * @see bl.coe.BigSparseMath.BigSparseMatrix#extractMatrix(int[], int[])
	 */
	public LabeledBigSparseMatrix extractMatrix(int []rowSubset, int []colSubset) {
		LabeledBigSparseMatrix subMat = new LabeledBigSparseMatrix(rowSubset.length,colSubset.length);
		subMat.copySubMatrix(this, rowSubset, colSubset);			
		subMat.setColLabels(colLabel,colSubset);
		return subMat;		
	}

	/**
	 * Sets the col labels.
	 * 
	 * @param oldColLabel the old col label
	 * @param colSubset the col subset
	 */
	protected void setColLabels(int[] oldColLabel, int[] colSubset) {
		if(colSubset==null)
			colLabel = oldColLabel.clone();
		else {
			for(int i=0;i<colSubset.length;i++)
				colLabel[i]=oldColLabel[colSubset[i]];
		}					
	}
	
	/**
	 * Sets the all col labels.
	 * 
	 * @param labels the new all col labels
	 */
	public void setAllColLabels(int []labels) {
		if(labels.length!=colLabel.length)
			throw new RuntimeException("Invalid number of column labels.");
		for(int i=0;i<colLabel.length;i++)
			colLabel[i]=labels[i];
	}

	/**
	 * Parses the full matrix text file.
	 * 
	 * @param file the file
	 * @param labelFile the label file
	 * @param N the n
	 * @param M the m
	 * 
	 * @return the labeled big sparse matrix
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public static LabeledBigSparseMatrix parseFullMatrixTextFile(File file, File labelFile, int N, int M) throws IOException {
		LabeledBigSparseMatrix mat = new LabeledBigSparseMatrix(N,M);
		mat.loadFullMatrixTextFile(file,labelFile);
		return mat;
	}

	/**
	 * Load full matrix text file.
	 * 
	 * @param file the file
	 * @param labelFile the label file
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public void loadFullMatrixTextFile(File file, File labelFile) throws IOException {
		loadFullMatrixTextFile(file);
		loadLabels(new FileInputStream(labelFile));
	}

	/**
	 * Instantiates a new labeled big sparse matrix.
	 * 
	 * @param BSMfile the bS mfile
	 * @param labelFile the label file
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public LabeledBigSparseMatrix(InputStream BSMfile, InputStream labelFile) throws IOException {
		super(BSMfile);
		loadLabels(labelFile);
	}
	
	/**
	 * Instantiates a new labeled big sparse matrix.
	 * 
	 * @param bsm the bsm
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public LabeledBigSparseMatrix(BigSparseMatrix bsm) throws IOException {
		super(bsm);
		nullLabels();
	}

	/**
	 * Instantiates a new labeled big sparse matrix.
	 * 
	 * @param BSMfile the bS mfile
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public LabeledBigSparseMatrix(String BSMfile) throws IOException {
		this(new File(BSMfile));
	}
	
	/**
	 * Instantiates a new labeled big sparse matrix.
	 * 
	 * @param BSMfile the bS mfile
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public LabeledBigSparseMatrix(File BSMfile) throws IOException {
		this(new FileInputStream(BSMfile));
	}
	
	/**
	 * Instantiates a new labeled big sparse matrix.
	 * 
	 * @param BSMfile the bS mfile
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public LabeledBigSparseMatrix(InputStream BSMfile) throws IOException {
		super(BSMfile);
		nullLabels();
	}

	/**
	 * Load text labels.
	 * 
	 * @param labelFile the label file
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public void loadTextLabels(InputStream labelFile) throws IOException {
		BigDenseVector labels = BigDenseVector.loadFullVectorTextFile(labelFile);
		if(colLabel.length!=labels.getLength())
			throw new IOException("Not a valid labelFile. Mismatch in number of columns.");
		for(int i=0;i<colLabel.length;i++) {
			colLabel[i]=(int)labels.get(i);
		}
	}

	/**
	 * Load labels.
	 * 
	 * @param labelFile the label file
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public void loadLabels(InputStream labelFile) throws IOException {
		DataInputStream inFp = new DataInputStream(new BufferedInputStream((labelFile)));
		byte []uuid = UUID.getBytes();
		byte []bytes = new byte[uuid.length];
		inFp.readFully(bytes);
		if(!Arrays.equals(uuid, bytes)) {
			inFp.close();
			throw new IOException("Not a valid labelFile. UUID does not match.");
		}			
		if(inFp.readShort()!=1) {
			inFp.close();
			throw new IOException("Not a valid labelFile. Likely endianess mismatch.");
		}
		int colsVerify = inFp.readInt();
		if(Mcols!=colsVerify) {
			inFp.close();
			throw new IOException("Not a valid labelFile. Mismatch in number of columns.");
		}
		for(int c=0;c<Mcols;c++) {				
			colLabel[c] = inFp.readInt();
		}			
		inFp.close();

	}

	/**
	 * Write.
	 * 
	 * @param BSMfile the bS mfile
	 * @param labelFile the label file
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public void write(OutputStream BSMfile, OutputStream labelFile) throws IOException {
		write(BSMfile);
		writeLabels(labelFile);		
	}

	/**
	 * Write labels.
	 * 
	 * @param labelFile the label file
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	private void writeLabels(OutputStream labelFile) throws IOException {
		DataOutputStream outFp = new DataOutputStream(new BufferedOutputStream((labelFile)));
		byte []uuid = UUID.getBytes();
		outFp.write(uuid);
		outFp.writeShort(1);
		outFp.writeInt(Mcols);
		for(int c=0;c<Mcols;c++) {				
			outFp.writeInt(colLabel[c]);
		}			
		outFp.close();

	}

	/**
	 * Write.
	 * 
	 * @param BSMFile the bSM file
	 * @param labelFile the label file
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public void write(String BSMFile, String labelFile) throws IOException {
		write(new File(BSMFile), new File(labelFile));
	}

	/**
	 * Write.
	 * 
	 * @param BSMFile the bSM file
	 * @param labelFile the label file
	 * 
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	public void write(File BSMFile, File labelFile) throws IOException {
		write(new FileOutputStream(BSMFile), new FileOutputStream(labelFile));
	}

	/**
	 * Gets the label for column.
	 * 
	 * @param indexAt the index at
	 * 
	 * @return the label for column
	 */
	public int getLabelForColumn(int indexAt) {
		return colLabel[indexAt];
	}

}
