package bl.coe;

import java.io.File;
import java.io.IOException;

import bl.coe.BigSparseMath.BigSparseMatrix;
import bl.coe.BigSparseMath.BigSparseMatrixTools;
import bl.coe.BigSparseMath.BigSparseVector;

/**
 * The Class ConvertUMDTextMatrixToBSM.
 * 
 * <p> <b><i>Program Usage:</i> bl.coe.ConvertUMDTextMatrixToBSM [outFile.bsm] Nrows 
 * Mcols [set diagonal elements=# or 0 (no change)] [inFile1.txt] ... [inFileN.txt] </b>
 * 
 * <p> Convert a text file in the UMD line-by-line sparse matrix format into the BSM (Binary Sparse Matrix)
 * format. The text file has up to Nrows lines of text divided among any number of files. 
 * Each row of text starts with the 0-indexed integer index of the row. 
 * Following the row index is up to Mcols pairs of numbers. The first number of the pair is the -indexed
 * column index and the second number of the pair is the value for that location. Numeric entries are separated 
 * by one or more white space or delimiters. See the preferences class (below) for the permitted 
 * delimiting characters. 
 * 
 * <p> The matrix is saved in a big-endian binary format. See the BigSparseMatrix class below.
 * 
 * @see bl.coe.BigSparseMath.BigMathPreferences
 * @see bl.coe.BigSparseMath.BigSparseMatrix
 * 
 * @author Bennett Landman, bennett.landman@vanderbilt.edu
 */
public class ConvertUMDTextMatrixToBSM {

	/**
	 * The main method.
	 * 
	 * @param args the arguments
	 */
	public static void main(String []args){
		System.out.println("bl.coe.ConvertUMDTextMatrixToBSM <outFile.bsm> Nrows Mcols [set diagonal elements=# or 0 (no change)] <inFile1.txt> ... <inFileN.txt> ");
		if(args.length<5)
			return;
		int N = Integer.valueOf(args[1]);
		int M = Integer.valueOf(args[2]);
		System.out.println("Matrix Size: "+N+"x"+M);
		float diag = Float.valueOf(args[3]);
		if(diag==0) {
			System.out.println("Leaving diagonal elements as is.");
		} else {
			System.out.println("Setting diagonal elements to : "+diag);
		}

		BigSparseMatrix mat;
		try {
			System.out.println("Loading text matrix (this might take a while)...");
			mat = new BigSparseMatrix(N, M);
			long[] progress=null;
			for(int i=4;i<args.length;i++) {
				System.out.println("Importing file: "+args[i]);
				progress=mat.loadUMDMatrixTextFile(new File(args[i]),diag,progress);
			}
			if(diag!=0) {
				System.out.println("Adding diagonal elements on empty rows");
				int empty=0;
				for(int r=0;r<mat.getNRows();r++) {
					BigSparseVector row = mat.getRowDirect(r);
					if(row==null) {
						empty++;
						row = new BigSparseVector(1);
						row.set(r, diag);
						mat.setRow(r, row);
					}
				}
				System.out.println("Added diagonal elements for "+empty+" rows");
			}

			BigSparseMatrixTools.printStat(mat);
			System.out.println("Save BSM matrix (this might take a while)..."+args[0]);
			mat.write(args[0]);

		} catch (IOException e) {
			System.out.println("Failed: "+e.getMessage());
			e.printStackTrace();
		}
		System.out.println("Done");

	}
}
