package bl.coe;

import java.io.IOException;

import bl.coe.BigSparseMath.BigDenseVector;
import bl.coe.BigSparseMath.BigSparseMatrix;
import bl.coe.BigSparseMath.BigSparseMatrixTools;

/**
 * The Class ExtractBSMSubset.
 * 
 * <p> <b><i>Program Usage:</i> bl.coe.ExtractBSMSubset [inFile.bsm] [indexes.txt] [r|c] [outFile.bsm] </b>
 * 
 * <p> Create a new matrix based on an existing matrix by either extracting a subset of rows or columns. The inFile is 
 * a BSM file. The indexes are a single BigDenseVector text file. r indicates row subset. c indicates column subset
 * The resulting matrix is saved to outFile.bsm
 * 
 * <p> The matrix is saved in a big-endian binary format. See the BigSparseMatrix class below.
 * 
 * @see bl.coe.BigSparseMath.BigMathPreferences
 * @see bl.coe.BigSparseMath.BigSparseMatrix
 * @see bl.coe.BigSparseMath.BigDenseVector
 * 
 * @author Bennett Landman, bennett.landman@vanderbilt.edu
 * 
 */
public class ExtractBSMSubset {
	
	/**
	 * The main method.
	 * 
	 * @param args the arguments
	 */
	public static void main(String []args){
		System.out.println("bl.coe.ExtractBSMSubset <inFile.bsm> <indexes.txt> [r|c] <outFile.bsm>");
		if(args.length!=4)
			return;
		BigSparseMatrix inMat =null;
		long tic=System.currentTimeMillis();
		try {
			System.out.println("Loading matrix (this might take a while)...");
			inMat= new BigSparseMatrix(args[0]);
		} catch (IOException e) {
			System.out.println("Failed: "+e.getMessage());
			e.printStackTrace();
			return;
		}
		long toc=System.currentTimeMillis()-tic;
		System.out.println("Loading time:       "+toc/1000.f+" s");
		
		System.out.println("Loading vector");
		tic=System.currentTimeMillis();
		BigDenseVector vec =null;
		try {
			vec = BigDenseVector.loadFullVectorTextFile(args[1]);
		} catch (IOException e) {
			System.out.println("Failed: "+e.getMessage());
			e.printStackTrace();
			return;
		} 
		
		BigSparseMatrix subMat=null;
		if(args[2].toLowerCase().startsWith("r")) {
		System.out.println("Creating subMatrix by rows");
		subMat = inMat.extractSubMatrixByRowsUnsafe(vec);
		} else {
			System.out.println("Creating subMatrix by columns");
			subMat = inMat.extractSubMatrixByCols(vec);
			
		}
		BigSparseMatrixTools.printStat(subMat);
		toc=System.currentTimeMillis()-tic;
		System.out.println("Processsing time:       "+toc/1000.f+" s");
		
		System.out.println("Ok.");
		
		
		tic=System.currentTimeMillis();
		
		try {
			
			System.out.println("Save BSM submatrix (this might take a while)...");
			subMat.write(args[3]);
			
		} catch (IOException e) {
			System.out.println("Failed: "+e.getMessage());
			e.printStackTrace();
		}
		toc=System.currentTimeMillis()-tic;
		System.out.println("Saving time:       "+toc/1000.f+" s");
		System.out.println("Done");
		
	}

}
