static const char rcsid[] = "$Id: fmriqa_count.cpp,v 1.20 2009-02-17 18:34:35 gadde Exp $";

/*
 * fmriqa_count.cpp --
 * 
 *  calculates some QA measures and writes the output
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>
#include <string.h>

#include <vector>

#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

#define CONVERTTEMPLATE(inbuf, fromtype, bufsize, retbuf, totype) {	\
    fromtype * buf = NULL;						\
    fromtype * endbuf = (fromtype *)((char *)inbuf + (bufsize));	\
    size_t retsize = sizeof(totype)*((bufsize)/sizeof(*buf));		\
    totype * newbuf = NULL;						\
    newbuf = (totype *)malloc(retsize);					\
    (retbuf) = newbuf;							\
    if ((newbuf) == NULL) {						\
	fprintf(stderr, "Error allocating %lld bytes\n", (long long int)retsize);	\
    }									\
    for (buf = (fromtype *)(inbuf); buf < (endbuf); newbuf++, buf++) {	\
	*(newbuf) = (totype)*buf;					\
    }									\
}

static float *
convertBufToFloat(const void * inbuf, size_t bufsize, const char * elemtype)
{
    float * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "float32") == 0) {
	retbuf = (float *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, float);
    }
    return retbuf;
}

enum { AGGR_NONE=0, AGGR_AND, AGGR_OR };
enum { GRAN_NONE=0, GRAN_TIMESERIES, GRAN_VOLUME, GRAN_SLICE, GRAN_VOXEL };

int
main(int argc, char *argv[])
{
    const char * inputfile = NULL;
    int dimnum;
    int msbfirst = 1;
    const char * opt_granularity = "timeseries";
    const char * opt_aggregate = "and";
    int granularity = GRAN_NONE;
    int aggregate = AGGR_NONE;
    double opt_gt = -HUGE_VAL;
    double opt_ge = -HUGE_VAL;
    double opt_lt = HUGE_VAL;
    double opt_le = HUGE_VAL;
    const char * opt_select[4] = {":", ":", ":", ":"};
    int opt_histogram = 0;
    char * opt_histobounds = NULL;
    int opt_numbuckets = -1;
    double opt_bucketwidth = -HUGE_VAL;
    int opt_version = 0;
    double * histobounds = NULL;
    int numhistobounds = 0;
    struct bxhdataread bdr;
    
    const int numopts = 17;
    opt_data opts[17] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  fmriqa_count inputfile\n\n"
	  "This program outputs histograms or counts of voxels in a "
	  "BXH- or XCEDE-wrapped dataset that match the given conditions.  "
	  "Output can be per-slice, per-volume, or for entire dataset "
	  "(see --granularity).  "
	  "Histogram output requires the --histogram option.  "
	  "If histogram output is not chosen, output is as if there were "
	  "one histogram 'bucket'.  "
	  "Conditions are specified as command-line options, "
	  "described below.  "
	  "Default is to 'and' all conditions (but see --aggregate).  "
	  "Default condition for 'and' aggregate is to match all voxels."
	  "Default condition for 'or' aggregate is to match no voxels.  "
	  "Thus, with no options, this program prints out the number of "
	  "voxels in the data."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_granularity, 1, "granularity",
	  "Print counts at this granularity.  Acceptable values are "
	  "'timeseries' (default), 'volume', 'slice', and 'voxel'." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_aggregate, 1, "aggregate",
	  "Conditions are aggregated by this operator, either "
	  "'and' (default) or 'or'." },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_gt, 1, "gt",
	  "Match those voxels greater than this value." },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_ge, 1, "ge",
	  "Match those voxels greater than or equal to this value." },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_lt, 1, "lt",
	  "Match those voxels less than this value." },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_le, 1, "le",
	  "Match those voxels less than or equal to this value." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[3], 1, "timeselect",
	  "Match only those timepoints in this comma-separated list "
	  "of timepoints (first timepoint is 0).  "
	  "Any timepoint can be a contiguous range, specified as two "
	  "numbers separated by a colon, i.e. 'START:END'.  "
	  "An empty END implies the last timepoint.  "
	  "The default step of 1 (one) in ranges can be changed using "
	  "'START:STEP:END', which is equivalent to "
	  "'START,START+STEP,START+(2*STEP),...,END'." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[0], 1, "xselect",
	  "Just like timeselect, but for the 'x' coordinate." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[1], 1, "yselect",
	  "Just like timeselect, but for the 'y' coordinate." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[2], 1, "zselect",
	  "Just like timeselect, but for the 'z' coordinate." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_histogram, 1, "histogram",
	  "Specifies that output should be histogram.  "
	  "See --histobuckets to specify number of buckets, "
	  "--histobucketwidth to specify width of buckets, "
	  "or --histobounds to specify bucket boundaries." },
	{ OPT_FLAGS_FULL, OPT_VAL_INT, &opt_numbuckets, 1, "histobuckets",
	  "Valid only with --histogram option.  "
	  "Constructs this many evenly-spaced histogram buckets.  "
	  "This option is incompatible with --histobounds or "
	  "--histobucketwidth." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_histobounds, 1, "histobounds",
	  "Valid only with --histogram option.  "
	  "By default, histogram bucket boundaries are in "
	  "multiples of standard deviations.  "
	  "This option specifies alternate boundaries for (N + 1) buckets as "
	  "a space-separated list of N floating point numbers.  "
	  "For example, --histobounds \"0.0 5.0 10.0\" will separate "
	  "voxels with values -infinity < x < 0.0, 0.0 <= x < 5.0, "
          "5.0 <= x < 10.0, and 10.0 <= x < infinity.  "
	  "This option is incompatible with --histobucketwidth or "
	  "--histobucketsize." },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_bucketwidth, 1, "histobucketwidth",
	  "Valid only with --histogram option.  "
	  "Constructs evenly-spaced histogram buckets with this width.  "
	  "This option is incompatible with --histobounds or "
	  "--histobuckets." }
    };
    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc != 2) {
	fprintf(stderr, "Usage: %s [opts] xmlfile\n", argv[0]);
	fprintf(stderr, "Use the --help option for more help.\n");
	return -1;
    }

    if (strcmp(opt_aggregate, "and") == 0) {
	aggregate = AGGR_AND;
    } else if (strcmp(opt_aggregate, "or") == 0) {
	aggregate = AGGR_OR;
    } else {
	fprintf(stderr, "Bad --aggregate value '%s'\n", opt_aggregate);
	return -1;
    }
    if (strcmp(opt_granularity, "timeseries") == 0) {
	granularity = GRAN_TIMESERIES;
    } else if (strcmp(opt_granularity, "volume") == 0) {
	granularity = GRAN_VOLUME;
    } else if (strcmp(opt_granularity, "slice") == 0) {
	granularity = GRAN_SLICE;
    } else if (strcmp(opt_granularity, "voxel") == 0) {
	granularity = GRAN_VOXEL;
    } else {
	fprintf(stderr, "Bad --granularity value '%s'\n", opt_granularity);
	return -1;
    }

    if ((opt_numbuckets != -1 || opt_bucketwidth != -HUGE_VAL || opt_histobounds) &&
	!opt_histogram) {
	fprintf(stderr, "--histobuckets, --histobucketwidth, and --histobounds require --histogram!\n");
	return -1;
    }
    if ((!!(opt_numbuckets != -1) +
	 !!(opt_bucketwidth != -HUGE_VAL) +
	 !!(opt_histobounds)) > 1) {
	fprintf(stderr, "Only one of --histobuckets, --histobucketwidth, or --histobounds may be specified!\n");
	return -1;
    }
    if (opt_numbuckets == 0) {
	fprintf(stderr, "Number of buckets in --histobuckets must be greater than 0!\n");
	return -1;
    }
    if (opt_bucketwidth != -HUGE_VAL && opt_bucketwidth <= 0) {
	fprintf(stderr, "Bucket width in --histobucketwidth must be greater than 0!\n");
	return -1;
    }

    /* parse opt_histobounds */
    if (opt_histobounds) {
	double bound;
	char * endptr = NULL;
	char * curptr = opt_histobounds;
	bound = strtod(curptr, &endptr);
	while (endptr != curptr) {
	    histobounds = (double *)realloc(histobounds, sizeof(double)*(numhistobounds+1));
	    histobounds[numhistobounds] = bound;
	    numhistobounds++;
	    curptr = endptr;
	    if (*curptr != '\0') {
		if (*curptr != ',') {
		    fprintf(stderr, "Garbage at end of --histobounds: \"%s\"\n", curptr);
		    return -1;
		}
		curptr++;
	    }
	    bound = strtod(curptr, &endptr);
	}
	if (*endptr != '\0') {
	    fprintf(stderr, "Garbage at end of --histobounds: \"%s\"\n", endptr);
	    return -1;
	}
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    inputfile = argv[1];

    memset(&bdr, '\0', sizeof(bdr));

    if (bxh_dataReadFileStart(inputfile, "image", NULL, 0, NULL, opt_select, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", argv[1]);
	return -1;
    }
    if (bxh_dataReadFinish(&bdr, "float") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	return -1;
    }
    float * dataptr = (float *)bdr.dataptr;

    /* fake data to look 4-D (if it isn't already) */
    for (dimnum = bdr.datarec->numdims; dimnum < 4; dimnum++) {
	bdr.dimsizes = (size_t *)realloc(bdr.dimsizes, sizeof(size_t)*(dimnum+1));
	bdr.pagesizes = (size_t *)realloc(bdr.pagesizes, sizeof(size_t)*(dimnum+1));
	bdr.dimsizes[dimnum] = 1;
	bdr.pagesizes[dimnum] = bdr.pagesizes[dimnum-1];
    }
    for (dimnum = bdr.datarec->numdims - 1; dimnum >= 4; dimnum--) {
	bdr.dimsizes[dimnum-1] *= bdr.dimsizes[dimnum];
    }
#define C2I(x,y,z,t) ((x) + (y)*bdr.pagesizes[0] + (z)*bdr.pagesizes[1] + (t)*bdr.pagesizes[2])
	
    /*** Match voxels ***/
    size_t x, y, z, t;
    size_t * buckets = NULL;
    size_t lo, mid, hi;
    size_t lastmid = 0;
    int ind;
    float max = dataptr[0];
    float min = dataptr[0];
    double mean = dataptr[0];
    double stddev = 0;

    if (opt_histogram) {
	if (numhistobounds == 0) {
	    size_t xyzt;
	    float val;
	    float oldmean;
	    double varsum = 0;
	    /* need to find boundaries of buckets by going through data */
	    /* we start at index 1 because we've already used index 0 */
	    if (opt_aggregate[0] == 'a') {
		/* AND */
		for (xyzt = 1; xyzt < bdr.pagesizes[3]; xyzt++) {
		    val = dataptr[xyzt];
		    if ((opt_gt == -HUGE_VAL || val >  opt_gt) &&
			(opt_ge == -HUGE_VAL || val >= opt_ge) &&
			(opt_lt ==  HUGE_VAL || val <  opt_lt) &&
			(opt_le ==  HUGE_VAL || val <= opt_le)) {
			oldmean = (float)mean;
			mean += (val - oldmean)/ (xyzt + 1);
			varsum += (val - oldmean) * (val - mean);
			if (val < min) min = val;
			if (val > max) max = val;
		    }
		}
	    } else {
		/* OR */
		for (xyzt = 1; xyzt < bdr.pagesizes[3]; xyzt++) {
		    val = dataptr[xyzt];
		    if ((opt_gt != -HUGE_VAL && val >  opt_gt) ||
			(opt_ge != -HUGE_VAL && val >= opt_ge) ||
			(opt_lt !=  HUGE_VAL && val <  opt_lt) ||
			(opt_le !=  HUGE_VAL && val <= opt_le)) {
			oldmean = (float)mean;
			mean += (val - oldmean)/ (xyzt + 1);
			varsum += (val - oldmean) * (val - mean);
			if (val < min) min = val;
			if (val > max) max = val;
		    }
		}
	    }
	    stddev = sqrt(varsum / bdr.pagesizes[3]);
	    if (opt_numbuckets > 1) {
		/* equal size buckets from min to max */
		double step = (max - min) / opt_numbuckets;
		double ind;
		for (ind = 1; ind < opt_numbuckets; ind++) {
		    histobounds = (double *)realloc(histobounds, sizeof(double)*(numhistobounds+1));
		    histobounds[numhistobounds] = min + (ind * step);
		    numhistobounds++;
		}
	    } else if (opt_bucketwidth == 0) {
		/* if only one bucket, no histobounds */
	    } else {
		double bound;
		if (opt_bucketwidth == -HUGE_VAL) {
		    /* buckets of size stddev */
		    opt_bucketwidth = stddev;
		    bound = mean - (((int)((mean - min) / opt_bucketwidth)) * opt_bucketwidth);
		    if (bound <= min) bound += opt_bucketwidth;
		} else {
		    bound = min + opt_bucketwidth;
		}
		for (/*null*/; bound < max; bound += opt_bucketwidth) {
		    histobounds = (double *)realloc(histobounds, sizeof(double)*(numhistobounds+1));
		    histobounds[numhistobounds] = bound;
		    numhistobounds++;
		}
	    }
	}
    }
    opt_numbuckets = numhistobounds + 1;
    buckets = (size_t *)malloc(sizeof(size_t)*opt_numbuckets);
    memset(buckets, '\0', sizeof(size_t)*opt_numbuckets);
    fprintf(stdout, "# Voxels");
    if (opt_aggregate[0] == 'a' &&
	opt_gt == -HUGE_VAL && opt_ge == -HUGE_VAL &&
	opt_lt ==  HUGE_VAL && opt_le ==  HUGE_VAL) {
	fprintf(stdout, " matching any conditions (should match all voxels)");
    } else if (opt_aggregate[0] == 'o' &&
	       opt_gt == -HUGE_VAL && opt_ge == -HUGE_VAL &&
	       opt_lt ==  HUGE_VAL && opt_le ==  HUGE_VAL) {
	fprintf(stdout, " matching empty condition set (should match no voxels)");
    } else {
	int printop = 0;
	if (opt_gt != -HUGE_VAL) {
	    fprintf(stdout, " greater than %g", opt_gt);
	    printop = 1;
	}
	if (opt_ge != -HUGE_VAL) {
	    if (printop) {
		fprintf(stdout, " %s", opt_aggregate);
	    }
	    fprintf(stdout, " greater than or equal %g", opt_ge);
	    printop = 1;
	}
	if (opt_lt != HUGE_VAL) {
	    if (printop) {
		fprintf(stdout, " %s", opt_aggregate);
	    }
	    fprintf(stdout, " less than %g", opt_lt);
	    printop = 1;
	}
	if (opt_le != HUGE_VAL) {
	    if (printop) {
		fprintf(stdout, " %s", opt_aggregate);
	    }
	    fprintf(stdout, " less than or equal %g", opt_le);
	    printop = 1;
	}
    }
    fprintf(stdout, ":\n");
    fprintf(stdout, "#");
    if (granularity == GRAN_SLICE) {
	fprintf(stdout, "TIME SLICE");
    } else if (granularity == GRAN_VOLUME) {
	fprintf(stdout, "TIME");
    } else if (granularity == GRAN_VOXEL) {
	fprintf(stdout, "TIME Z Y X");
    }
    if (opt_histogram) {
	if (opt_numbuckets == 1) {
	    fprintf(stdout, " %g<=x<=%g", min, max);
	} else {
	    if (histobounds[0] <= min) {
		fprintf(stdout, " -infinity<x<%g", histobounds[0]);
	    } else {
		fprintf(stdout, " %g<=x<%g", min, histobounds[0]);
	    }
	    for (ind = 1; ind < opt_numbuckets - 1; ind++) {
		fprintf(stdout, " %g<=x<%g", histobounds[ind-1], histobounds[ind]);
	    }
	    ind--;
	    if (ind < opt_numbuckets - 1) {
		if (histobounds[ind] > max) {
		    fprintf(stdout, " %g<=x<infinity", histobounds[ind]);
		} else {
		    fprintf(stdout, " %g<=x<=%g", histobounds[ind], max);
		}
	    }
	}
    } else if (granularity != GRAN_VOXEL) {
	fprintf(stdout, " NUMVOXELS");
    }
    fprintf(stdout, "\n");
    for (t = 0; t < bdr.dimsizes[3]; t++) {
	for (z = 0; z < bdr.dimsizes[2]; z++) {
	    for (y = 0; y < bdr.dimsizes[1]; y++) {
		for (x = 0; x < bdr.dimsizes[0]; x++) {
		    size_t ind = C2I(x,y,z,t);
		    float val = dataptr[ind];
		    if ((opt_aggregate[0] == 'a' &&
			 ((opt_gt == -HUGE_VAL || val >  opt_gt) &&
			  (opt_ge == -HUGE_VAL || val >= opt_ge) &&
			  (opt_lt ==  HUGE_VAL || val <  opt_lt) &&
			  (opt_le ==  HUGE_VAL || val <= opt_le))) ||
			(opt_aggregate[0] == 'o' &&
			 ((opt_gt != -HUGE_VAL && val >  opt_gt) ||
			  (opt_ge != -HUGE_VAL && val >= opt_ge) ||
			  (opt_lt !=  HUGE_VAL && val <  opt_lt) ||
			  (opt_le !=  HUGE_VAL && val <= opt_le)))) {
			if (granularity == GRAN_VOXEL) {
			    fprintf(stdout, "%lu %lu %lu %lu %g\n",
				    (unsigned long)t, (unsigned long)z,
				    (unsigned long)y, (unsigned long)x,
				    (double)val);
			} else if (opt_numbuckets > 1) {
			    /* search for correct bucket */
			    lo = 0;
			    hi = opt_numbuckets;
			    mid = lastmid; /* in case data is likely to be clustered */
			    while (lo <= hi) {
				if (mid == (size_t)(opt_numbuckets - 1) ||
				    val < histobounds[mid]) {
				    if (mid == 0 || val >= histobounds[mid-1])
					break;
				    hi = mid;
				} else {
				    if (mid == (size_t)(opt_numbuckets - 2)) {
					mid = (size_t)(opt_numbuckets - 1);
					break;
				    }
				    lo = mid + 1;
				}
				mid = (lo + hi) / 2;
			    }
			    buckets[mid]++;
			    lastmid = mid;
			} else {
			    buckets[0]++;
			}
		    }
		}
	    }
	    if (granularity == GRAN_SLICE) {
		fprintf(stdout, "%lu %lu",
			(unsigned long)t, (unsigned long)z);
		for (ind = 0; ind < opt_numbuckets; ind++) {
		    fprintf(stdout, " %lu", (unsigned long)buckets[ind]);
		}
		fprintf(stdout, "\n");
		memset(buckets, '\0', sizeof(size_t)*opt_numbuckets);
	    }
	}
	if (granularity == GRAN_VOLUME) {
	    fprintf(stdout, "%lu", (unsigned long)t);
	    for (ind = 0; ind < opt_numbuckets; ind++) {
		fprintf(stdout, " %lu", (unsigned long)buckets[ind]);
	    }
	    fprintf(stdout, "\n");
	    memset(buckets, '\0', sizeof(size_t)*opt_numbuckets);
	}
    }
    if (granularity == GRAN_TIMESERIES) {
	for (ind = 0; ind < opt_numbuckets; ind++) {
	    fprintf(stdout, " %lu", (unsigned long)buckets[ind]);
	}
	fprintf(stdout, "\n");
	memset(buckets, '\0', sizeof(size_t)*opt_numbuckets);
    }

    bxh_datareaddata_free(&bdr);
    free(buckets);
    free(histobounds);
    return 0;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.19  2008/04/02 18:16:26  gadde
 * Fix opt_histobounds reading.
 *
 * Revision 1.18  2008/04/02 18:05:20  gadde
 * Only output infinity/-infinity in ranges if histogram bounds are outside
 * the actual min/max of the data.
 *
 * Revision 1.17  2008/03/07 23:05:07  gadde
 * Stop using off_t for signed data
 *
 * Revision 1.16  2007/11/12 18:21:24  gadde
 * Histogram updates.
 *
 * Revision 1.15  2006/11/10 19:11:11  gadde
 * Work on data that's not 4D and doesn't have xyzt order.
 *
 * Revision 1.14  2006/06/01 20:16:48  gadde
 * const fixes
 *
 * Revision 1.13  2005/09/20 18:37:51  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.12  2005/09/19 16:31:52  gadde
 * Documentation and help message updates.
 *
 * Revision 1.11  2005/09/14 14:49:23  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.10  2004/06/15 16:16:10  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.9  2004/05/18 15:00:39  gadde
 * Make "coords" just another granularity (voxel).
 *
 * Revision 1.8  2004/05/14 18:12:16  gadde
 * Add another granularity (coords).
 * Also be more robust with less-than-4-D data.
 *
 * Revision 1.7  2004/05/12 21:58:46  gadde
 * Allow data that is not 4-D.
 *
 * Revision 1.6  2004/05/10 18:34:04  gadde
 * Change output format slightly.
 *
 * Revision 1.5  2004/05/07 18:49:12  gadde
 * Add some options.
 *
 * Revision 1.4  2004/05/06 20:55:21  gadde
 * opts crashes if option name is NULL.
 *
 * Revision 1.3  2004/05/03 19:41:16  gadde
 * Add histogram option.
 *
 * Revision 1.2  2004/04/27 18:46:34  gadde
 * Fix comment.
 *
 * Revision 1.1  2004/04/27 18:46:07  gadde
 * *** empty log message ***
 *
 *
 */
