static const char rcsid[] = "$Id: fmriqa_minmax.cpp,v 1.10 2009-02-17 14:18:30 gadde Exp $";

/*
 * fmriqa_minmax.cpp --
 * 
 *  calculates and displays minimum and maximum values
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>
#include <string.h>

#include <vector>

#include "bxh_utils.h"
#include "opts.h"

#ifdef WIN32
#include <float.h>
#define finite _finite
#endif

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

#define CONVERTTEMPLATE(inbuf, fromtype, bufsize, retbuf, totype) {	\
    fromtype * buf = NULL;						\
    fromtype * endbuf = (fromtype *)((char *)inbuf + (bufsize));	\
    size_t retsize = sizeof(totype)*((bufsize)/sizeof(*buf));		\
    totype * newbuf = NULL;						\
    newbuf = (totype *)malloc(retsize);					\
    (retbuf) = newbuf;							\
    if ((newbuf) == NULL) {						\
	fprintf(stderr, "Error allocating %lld bytes\n", (long long int)retsize);	\
    }									\
    for (buf = (fromtype *)(inbuf); buf < (endbuf); newbuf++, buf++) {	\
	*(newbuf) = (totype)*buf;					\
    }									\
}

static float *
convertBufToFloat(const void * inbuf, size_t bufsize, const char * elemtype)
{
    float * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "float32") == 0) {
	retbuf = (float *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, float);
    }
    return retbuf;
}

int
main(int argc, char *argv[])
{
    int dimnum;
    int msbfirst = 1;
    int filenum = 0;
    int firstvoxel = 1;
    int hasnonfinite = 0;
    float maxval = 0;
    float minval = 0;

    if (argc < 2 || strcmp(argv[1], "--help") == 0) {
	fprintf(stderr, "Usage:\n  %s xmlfile(s)...\n\nThis program merely computes the minimum and maximum values in the input\nBXH- or XCEDE-wrapped dataset and writes them to standard output.\n", argv[0]);
	return -1;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    for (filenum = 1; filenum < argc; filenum++) {
	const char * inputfile = argv[filenum];
	struct bxhdataread bdr;
	float * dataptr = NULL;

	memset(&bdr, '\0', sizeof(bdr));
	if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, NULL, NULL, &bdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	    return -1;
	}
	if (bxh_dataReadFinish(&bdr, "float") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	    return -1;
	}

	dataptr = (float *)bdr.dataptr;

	/*** Calculate maxval ***/
	size_t voxelnum = 0;
	if (firstvoxel) {
	    maxval = dataptr[0];
	    minval = dataptr[0];
	    firstvoxel = 0;
	}
	for (voxelnum = 0; voxelnum < bdr.pagesizes[bdr.datarec->numdims-1]; voxelnum++) {
	    if (!finite(dataptr[voxelnum])) {
		if (!hasnonfinite) {
		    hasnonfinite = 1;
		    fprintf(stderr, "Warning: data contains non-finite values\n");
		}
		continue;
	    }
	    if (dataptr[voxelnum] > maxval) {
		maxval = dataptr[voxelnum];
	    }
	    if (dataptr[voxelnum] < minval) {
		minval = dataptr[voxelnum];
	    }
	}
	bxh_datareaddata_free(&bdr);
    }

    /* write out results */
    fprintf(stdout, "min=%g, max=%g\n", minval, maxval);
    return 0;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.9  2008/02/12 19:50:34  gadde
 * Better error reporting.
 *
 * Revision 1.8  2007/07/02 18:21:06  gadde
 * Use _finite on win32
 *
 * Revision 1.7  2007/05/15 13:29:38  gadde
 * Use finite() instead of isfinite(), for better cross-platform compatibility.
 *
 * Revision 1.6  2007/05/11 15:46:03  gadde
 * Warn if there are non-finite (i.e. Inf, NaN, etc.) values in the input(s).
 *
 * Revision 1.5  2005/09/20 18:37:52  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.4  2005/09/19 16:31:53  gadde
 * Documentation and help message updates.
 *
 * Revision 1.3  2004/06/15 16:16:10  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.2  2004/04/07 17:22:40  gadde
 * Move line to where it is in other programs.
 *
 * Revision 1.1  2004/03/25 19:23:37  gadde
 * Add ppm conversion, and various other updates
 *
 * Revision 1.1  2004/03/19 15:13:32  gadde
 * Major changes in datarec support, include a new 'prepare' stage
 * that initiates filtering, permutation.
 * bxh_datarec_readData has been changed to bxh_datarec_readRawData
 * to reflect this.
 * Added some fMRI QA measures.
 *
 *
 */
