static const char rcsid[] = "$Id: bxh_convert_dicom.cpp,v 1.1 2009-01-15 20:56:16 gadde Exp $";

/*
 * dicom2bxh.cpp --
 * 
 * Creates a BXH file based on given DICOM file.
 * Requires DCMTK.  Based on dcmdump.
 */


#include "bxh_utils.h"
#include "bxh_dicom.h"

#ifdef LINUX
#include <features.h>
#endif
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <math.h>
#include <ctype.h>

//#include <dcdebug.h>
#include <dctk.h>
#include <cmdlnarg.h>
#include <ofconapp.h>
#include <ofcmdln.h>
#include <dcuid.h>    /* for dcmtk version name */

#define OFFIS_CONSOLE_APPLICATION "dicom2bxh"

#ifdef __cplusplus
extern "C" {
#endif

extern char * domutil_errorbuf;

int
createBXHFromDICOM(int numinfiles,
		   const char **ifnames,
		   const char *ofname,
		   const OFBool isDataset,
		   const E_TransferSyntax xfer,
		   const OFBool showFullData,
		   const OFBool loadIntoMemory,
		   const OFBool stopOnErrors,
		   const OFBool searchForOthers,
		   const OFBool forceConcat,
		   const int sortopt,
		   const int printfields,
		   const int cornerorigin,
		   size_t *hintsize,
		   double *hintorigin,
		   double *hintspacing,
		   double *hintgap,
		   char *forceorient,
		   char *progname)
{
    int result = 0;
    int bxhretval = 0;
    BXHDocPtr docp = NULL;
    char realofname[1024];
    int multipleoutputs = 0;
    char * fullbxhpath = NULL;
    struct stat statbuf;
    void * cont = NULL;
    int juststarted = 0;
    
    fullbxhpath = bxh_datarec_getfullfilename(NULL, ofname);

    if (stat(fullbxhpath, &statbuf) == 0) {
      fprintf(stderr, "dicom2bxh: file exists ('%s')\n", fullbxhpath);
      goto FAIL;
    }

    juststarted = 1;
    while (juststarted || cont) {
	juststarted = 0;

	if ((docp = bxh_createDocument()) == NULL)
	    goto FAIL;

	if (cont == NULL) {
	    bxhretval = bxh_DICOM_readFilesStart(docp, numinfiles, ifnames, isDataset, searchForOthers, fullbxhpath, forceConcat, sortopt, printfields, cornerorigin, &cont);
	} else {
	    bxhretval = bxh_DICOM_readFilesCont(docp, fullbxhpath, &cont);
	}

	if (multipleoutputs)
	    multipleoutputs++;
	else if (cont)
	    multipleoutputs = 1;

	if (bxhretval != 0) {
	    result = -1;
	    if (cont != NULL) {
		continue;
	    }
	    goto FAIL;
	}

	if (multipleoutputs) {
	    const char * ext = strrchr(ofname, '.');
	    char numbuf[32];
	    sprintf(numbuf, "-%d", multipleoutputs);
	    memset(realofname, '\0', sizeof(realofname));
	    if (ext)
		strncpy(realofname, ofname, sizeof(char) * (ext - ofname));
	    else
		strncpy(realofname, ofname, sizeof(realofname));
	    strcat(realofname, numbuf);
	    if (ext)
		strcat(realofname, ext);
	} else {
	    strncpy(realofname, ofname, sizeof(realofname));
	}

	if (stat(realofname, &statbuf) == 0) {
	    fprintf(stderr, "dicom2bxh: file exists ('%s')\n", realofname);
	    goto FAIL;
	}

	{
	    BXHElementPtr imagedata = NULL;
	    bxhrawdatarec * datarec = NULL;
	    const char * hintnames[4] = { "x", "y", "z", "t" };
	    int dimnum;
	    if (hintsize[0] != (size_t)-1 ||
		hintsize[1] != (size_t)-1 ||
		hintsize[2] != (size_t)-1 ||
		hintsize[3] != (size_t)-1) {
		fprintf(stderr, "Warning: --hintsize* options are ignored for DICOM files\n");
	    }
	    if ((imagedata = bxh_getDatarec(docp, "image", NULL)) == NULL)
		goto FAIL;
	    if ((datarec = bxh_datarec_createFromElement(imagedata, realofname)) == NULL) {
		goto FAIL;
	    }
	    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
		bxhdimension * dimp = &datarec->dimensions[dimnum];
		int hintnum;
		for (hintnum = 0; hintnum < 4; hintnum++) {
		    if (strcmp(dimp->type, hintnames[hintnum]) == 0)
			break;
		}
		if (hintnum == 4) continue;
		if (hintorigin[hintnum] != HUGE_VAL)
		    dimp->origin = hintorigin[hintnum];
		if (hintspacing[hintnum] != HUGE_VAL)
		    dimp->spacing = hintspacing[hintnum];
		if (hintgap[hintnum] != HUGE_VAL)
		    dimp->gap = hintgap[hintnum];
		if (forceorient && hintnum < 3) {
		    if (dimp->direction == NULL)
			dimp->direction = (double *)malloc(sizeof(double)*3);
		    memset(dimp->direction, '\0', sizeof(double)*3);
		    if (tolower(forceorient[hintnum]) == 'r') {
			dimp->direction[0] = 1;
		    } else if (tolower(forceorient[hintnum]) == 'a') {
			dimp->direction[1] = 1;
		    } else if (tolower(forceorient[hintnum]) == 's') {
			dimp->direction[2] = 1;
		    } else if (tolower(forceorient[hintnum]) == 'l') {
			dimp->direction[0] = -1;
		    } else if (tolower(forceorient[hintnum]) == 'p') {
			dimp->direction[1] = -1;
		    } else if (tolower(forceorient[hintnum]) == 'i') {
			dimp->direction[2] = -1;
		    }
		}
	    }
	    if (bxh_datarec_writeToElement(imagedata, datarec) != 0)
		goto FAIL;
	    bxh_datarec_free(datarec);
	    bxh_element_unref(imagedata);
	}
    
	if (progname) {
	    if (bxh_addAutoHistoryEntry(docp, progname, (const char **)ifnames, numinfiles) != 0)
		goto FAIL;
	}
    
	if (bxh_writeFile(docp, realofname) != 0)
	    goto FAIL;

	if (docp) {
	    bxh_freeDocument(docp); docp = NULL;
	}
    }

    goto EXIT;

FAIL:
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    result = -1;

EXIT:
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }
    if (fullbxhpath) {
	free(fullbxhpath); fullbxhpath = NULL;
    }

    return result;
}

BXHDocPtr
bxh_dicom2doc(const char * filename)
{
    BXHDocPtr docp = NULL;
    void * cont = NULL;
    
    if ((docp = bxh_createDocument()) == NULL)
	goto FAIL;

    if (bxh_DICOM_readFilesStart(docp, 1, &filename, 0, 1, "dummy.bxh", 0, 0, 0, 0, &cont) != 0)
	goto FAIL;

    if (cont) {
	fprintf(stderr, "Unexpected error: multiple groups of DICOM files match.\n");
	goto FAIL;
    }

    goto EXIT;

FAIL:
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

EXIT:
    return docp;
}

#ifdef __cplusplus
};
#endif

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.46  2008/01/27 22:44:35  gadde
 * Fix memory leak.
 *
 * Revision 1.45  2006/10/27 13:56:06  gadde
 * Add --filename-sort option to dicom2bxh
 *
 * Revision 1.44  2006/03/23 17:36:51  gadde
 * Update for DCMTK 3.5.4.
 *
 * Revision 1.43  2005/09/19 16:31:58  gadde
 * Documentation and help message updates.
 *
 * Revision 1.42  2005/07/07 15:10:51  gadde
 * Memory fixes.
 *
 * Revision 1.41  2005/03/08 18:45:15  gadde
 * Some AIX fixes
 *
 * Revision 1.40  2005/03/08 18:09:57  gadde
 * Add support for dimension size/spacing/gap/orientation hints
 *
 * Revision 1.39  2004/09/16 15:03:28  gadde
 * Add missing newline to error message.
 *
 * Revision 1.38  2004/07/06 18:34:17  gadde
 * Some memory fixes.
 *
 * Revision 1.37  2004/07/06 13:56:26  gadde
 * Some memory fixes.
 *
 * Revision 1.36  2004/06/15 18:42:34  gadde
 * Update fullifnames with ifnames.
 *
 * Revision 1.35  2004/06/15 16:16:11  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.34  2004/05/17 17:54:11  gadde
 * Add force-concat option.
 *
 * Revision 1.33  2004/04/22 12:57:47  gadde
 * Remove extra extern "C".
 *
 * Revision 1.32  2004/03/12 16:29:04  gadde
 * Minor updates
 *
 * Revision 1.31  2004/02/24 18:01:07  gadde
 * Move history generation to bxh_utils.cpp, and add option to create BIRN files
 *
 * Revision 1.30  2004/02/20 18:42:48  gadde
 * Add version option and rearrange documentation
 *
 * Revision 1.29  2003/08/19 19:35:41  gadde
 * New options scheme, so that options are available using whichever form of bxhabsorb you use.
 *
 * Revision 1.28  2003/08/08 18:56:18  gadde
 * P670 updates
 *
 * Revision 1.27  2003/08/01 17:51:29  gadde
 * On the road to merging in xerces branch
 *
 * Revision 1.26  2003/06/18 16:14:14  gadde
 * win32 fixes
 *
 * Revision 1.25  2003/05/16 19:56:48  gadde
 * Require a basepath for datarec.
 *
 * Revision 1.24  2003/04/28 19:46:53  gadde
 * Move to catch-all BXH encapsulation program bxhabsorb -- dicom2bxh, pfile2bxh,
 * etc. will all be symbolic links to bxhabsorb.
 *
 * Revision 1.23  2003/03/31 17:51:36  gadde
 * Remove unparsed options.
 *
 * Revision 1.22  2003/02/12 21:07:19  gadde
 * Free some memory.
 *
 * Revision 1.21  2003/02/12 21:02:15  gadde
 * Added support for creating multiple BXH files from list of given
 * DICOM files if bxh_DICOM_readFiles doesn't use up all the files.
 * Also, added --search-for-others option which now must be specified
 * if you wish to search for other matching DICOM files in the same
 * directory.
 *
 * Revision 1.20  2003/01/17 16:16:17  gadde
 * Add version string to history
 *
 * Revision 1.19  2003/01/16 20:55:35  gadde
 * cosmetic change in history
 *
 * Revision 1.18  2003/01/06 18:41:57  gadde
 * Many changes/updates to comments.
 * Minor bug fixes.
 *
 * Revision 1.17  2002/11/25 16:21:31  gadde
 * Mega-update to merge in new library functions.
 *
 * Revision 1.16  2002/11/15 20:52:32  gadde
 * Now BXH files have the correct namespace declarations (at the expense
 * of some fragile code in dom_utils.c that breaks the GDOME abstraction
 * layer).
 *
 * Revision 1.15  2002/11/08 16:30:20  gadde
 * <imagedata> =-> <datarec type="image">
 *
 * Revision 1.14  2002/10/04 18:33:26  gadde
 * Decreased the precision for checking normals
 *
 * Revision 1.13  2002/09/18 17:31:36  gadde
 * R,A,S dims calculated differently.
 * Added bandwidth element.
 *
 * Revision 1.12  2002/09/13 19:54:02  gadde
 * Changed to non-default namespace
 *
 * Revision 1.11  2002/09/10 04:44:53  gadde
 * Comment positioning.
 *
 * Revision 1.10  2002/09/10 04:27:34  gadde
 * /mriheader -> /bxh
 *
 * Revision 1.9  2002/09/09 22:11:21  gadde
 * Added more fields that should be used in matching files.
 *
 * Revision 1.8  2002/09/09 20:02:28  gadde
 * When parent directory is ".", it is removed from filename.
 *
 * Revision 1.7  2002/09/09 19:48:10  gadde
 * Adds appropriate code for missing SpacingBetweenSlices and for
 * ignoring data not of the same type (derived vs. original) in
 * the search for associated slices.
 * Also, /mriheader -> /bxh
 *
 * Revision 1.6  2002/09/09 17:05:34  gadde
 * Now outputs orientation vectors (dimension/direction).
 *
 * Revision 1.5  2002/08/23 16:54:33  gadde
 * Compiler warning fix.
 *
 * Revision 1.4  2002/08/21 19:45:03  gadde
 * mxh -> bxh
 *
 * Revision 1.3  2002/08/20 20:11:13  gadde
 * Added const to rcsid to avoid compiler warning.
 *
 * Revision 1.2  2002/08/20 15:32:33  gadde
 * Everything moved from libxml to libgdome (DOM).  All utility functions
 * prefixed with domutil_.  XPath support still shaky in gdome, but dom_utils.c
 * accounts for this, and extra code can be removed when gdome fully supports
 * the DOM level 3 XPath specification.
 *
 */
