static const char rcsid[] = "$Id: bxh_convert_mgh.c,v 1.1 2009-01-15 20:55:18 gadde Exp $";

/*
 * mgh2bxh.cc --
 * 
 * Creates a BXH file from an .mgh/.mgz file
 */

#include "bxh_utils.h"
#include "bxh_datarec.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <zlib.h>
#include <math.h>
#include <sys/stat.h>

extern char * domutil_errorbuf;

/* the following is derived from info on the Freesurfer Wiki */
typedef struct {
    int version;
    int width;
    int height;
    int depth;
    int nframes;
    int type; /* 0=UCHAR, 1=INT, 3=FLOAT, 4=SHORT */
    int dof;
    short goodRASFlag; /* specifies whether the direction cosines (below) are populated */
    float spacingX;
    float spacingY;
    float spacingZ;
    /* direction cosines */
    float xr, xa, xs;
    float yr, ya, ys;
    float zr, za, zs;
    float cr, ca, cs;
    /* these come after the image data */
    float TR;
    float flipangle;
    float TE;
    float TI;
} MGH;

static void
grabshort(void * out_i, char * buf, size_t pos, int noswap)
{
    char * out = (char *)out_i;
    memcpy(out, &buf[pos], 2);
    if (!noswap) {
	char swapper;
	swapper = out[0];
	out[0] = out[1];
	out[1] = swapper;
    }
}

static void
grablong(void * out_i, char * buf, size_t pos, int noswap)
{
    char * out = (char *)out_i;
    memcpy(out, &buf[pos], 4);
    if (!noswap) {
	char swapper;
	swapper = out[0];
	out[0] = out[3];
	out[3] = swapper;
	swapper = out[1];
	out[1] = out[2];
	out[2] = swapper;
    }
}

BXHDocPtr
createDocFromMGH(const char **ifnames, int numinfiles, const char *ofname)
{
    BXHDocPtr docp = NULL;
    BXHElementPtr imagedata = NULL;
    BXHElementPtr acquisitiondata = NULL;
    gzFile * iffp = NULL;
    int msbfirst;
    char buf[284];
    const char * ifname = ifnames[0];
    MGH mgh;
    int elemlen;
    char * elemtype = NULL;
    bxhrawdatarec * rawdatarec = NULL;
    char numbuf[32];
    
    msbfirst = 1;
    msbfirst = (((char *)&msbfirst)[0] == 0);

    if (numinfiles > 1) {
	fprintf(stderr, "mgh2bxh: can only accept one .mgh/.mgz file at a time!\n");
	goto FAIL;
    }

    if ((iffp = gzopen(ifname, "rb")) == NULL) {
	fprintf(stderr, "mgh2bxh: error opening file '%s'\n", ifname);
	perror("fopen");
	goto FAIL;
    }

    if (gzread(iffp, &buf[0], 284) != 284) {
	fprintf(stderr, "mgh2bxh: Error reading header from '%s'\n", ifname);
	goto FAIL;
    }

    grablong(&mgh.version, &buf[0], 0, msbfirst);
    grablong(&mgh.width, &buf[0], 4, msbfirst);
    grablong(&mgh.height, &buf[0], 8, msbfirst);
    grablong(&mgh.depth, &buf[0], 12, msbfirst);
    grablong(&mgh.nframes, &buf[0], 16, msbfirst);
    grablong(&mgh.type, &buf[0], 20, msbfirst);
    grablong(&mgh.dof, &buf[0], 24, msbfirst);
    grabshort(&mgh.goodRASFlag, &buf[0], 28, msbfirst);
    grablong(&mgh.spacingX, &buf[0], 30, msbfirst);
    grablong(&mgh.spacingY, &buf[0], 34, msbfirst);
    grablong(&mgh.spacingZ, &buf[0], 38, msbfirst);
    grablong(&mgh.xr, &buf[0], 42, msbfirst);
    grablong(&mgh.xa, &buf[0], 46, msbfirst);
    grablong(&mgh.xs, &buf[0], 50, msbfirst);
    grablong(&mgh.yr, &buf[0], 54, msbfirst);
    grablong(&mgh.ya, &buf[0], 58, msbfirst);
    grablong(&mgh.ys, &buf[0], 62, msbfirst);
    grablong(&mgh.zr, &buf[0], 66, msbfirst);
    grablong(&mgh.za, &buf[0], 70, msbfirst);
    grablong(&mgh.zs, &buf[0], 74, msbfirst);
    grablong(&mgh.cr, &buf[0], 78, msbfirst);
    grablong(&mgh.ca, &buf[0], 82, msbfirst);
    grablong(&mgh.cs, &buf[0], 86, msbfirst);

    switch (mgh.type) {
    case 0:
	elemlen = 1;
	elemtype = "uint8";
	break;
    case 1:
	elemlen = 4;
	elemtype = "int32";
	break;
    case 3:
	elemlen = 4;
	elemtype = "float32";
	break;
    case 4:
	elemlen = 2;
	elemtype = "int16";
	break;
    default:
	fprintf(stderr, "mgh2bxh: unrecognized datatype %d!\n", mgh.type);
	goto FAIL;
	break;
    }

    if (gzseek(iffp, 283 + mgh.width*mgh.height*mgh.depth*mgh.nframes*elemlen, SEEK_SET) == -1) {
	fprintf(stderr, "mgh2bxh: Error seeking to end of image data in '%s'\n", ifname);
	goto FAIL;
    }

    if (gzread(iffp, &buf[0], 16) == 16) {
	grablong(&mgh.TR, &buf[0], 0, msbfirst);
	grablong(&mgh.flipangle, &buf[0], 4, msbfirst);
	grablong(&mgh.TE, &buf[0], 8, msbfirst);
	grablong(&mgh.TI, &buf[0], 12, msbfirst);
    }

    if ((docp = bxh_createDocument()) == NULL)
	goto FAIL;

    if ((imagedata = bxh_getDatarec(docp, "image", NULL)) == NULL)
	goto FAIL;
    if ((acquisitiondata = bxh_getAcquisitionData(docp)) == NULL)
	goto FAIL;
    
    rawdatarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
    memset(rawdatarec, '\0', sizeof(bxhrawdatarec));

    if (mgh.nframes > 1) {
	fprintf(stderr, "mgh2bxh: multi-frame .mgh/.mgz images not supported!\n");
	goto FAIL;
    }
    rawdatarec->numdims = 3;
    rawdatarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension)*3);
    memset(rawdatarec->dimensions, '\0', sizeof(bxhdimension)*3);
    rawdatarec->dimensions[0].type = strdup("x");
    rawdatarec->dimensions[1].type = strdup("y");
    rawdatarec->dimensions[2].type = strdup("z");
    rawdatarec->dimensions[0].units = strdup("mm");
    rawdatarec->dimensions[1].units = strdup("mm");
    rawdatarec->dimensions[2].units = strdup("mm");
    rawdatarec->dimensions[0].size = mgh.width;
    rawdatarec->dimensions[1].size = mgh.height;
    rawdatarec->dimensions[2].size = mgh.depth;
    rawdatarec->dimensions[0].spacing = mgh.spacingX;
    rawdatarec->dimensions[1].spacing = mgh.spacingY;
    rawdatarec->dimensions[2].spacing = mgh.spacingZ;
    if (mgh.goodRASFlag) {
	double ox, oy, oz;
	double or, oa, os;
	rawdatarec->dimensions[0].direction = (double *)malloc(sizeof(double)*3);
	rawdatarec->dimensions[1].direction = (double *)malloc(sizeof(double)*3);
	rawdatarec->dimensions[2].direction = (double *)malloc(sizeof(double)*3);
	rawdatarec->dimensions[0].direction[0] = mgh.xr;
	rawdatarec->dimensions[0].direction[1] = mgh.xa;
	rawdatarec->dimensions[0].direction[2] = mgh.xs;
	rawdatarec->dimensions[1].direction[0] = mgh.yr;
	rawdatarec->dimensions[1].direction[1] = mgh.ya;
	rawdatarec->dimensions[1].direction[2] = mgh.ys;
	rawdatarec->dimensions[2].direction[0] = mgh.zr;
	rawdatarec->dimensions[2].direction[1] = mgh.za;
	rawdatarec->dimensions[2].direction[2] = mgh.zs;
	/* take position of center and find origin */
	or = mgh.cr -
	    ((0.5 * mgh.width * mgh.spacingX * mgh.xr) +
	     (0.5 * mgh.height * mgh.spacingY * mgh.yr) + 
	     (0.5 * mgh.depth * mgh.spacingZ * mgh.zr));
	oa = mgh.ca -
	    ((0.5 * mgh.width * mgh.spacingX * mgh.xa) + 
	     (0.5 * mgh.height * mgh.spacingY * mgh.ya) + 
	     (0.5 * mgh.depth * mgh.spacingZ * mgh.za));
	os = mgh.cs -
	    ((0.5 * mgh.width * mgh.spacingX * mgh.xs) + 
	     (0.5 * mgh.height * mgh.spacingY * mgh.ys) + 
	     (0.5 * mgh.depth * mgh.spacingZ * mgh.zs));
	if (fabs(mgh.xr) > fabs(mgh.yr) && fabs(mgh.xr) > fabs(mgh.zr)) {
	    ox = or;
	    if (fabs(mgh.ya) > fabs(mgh.xa) && fabs(mgh.ya) > fabs(mgh.za)) {
		oy = oa; oz = os;
	    } else {
		oz = oa; oy = os;
	    }
	} else if (fabs(mgh.yr) > fabs(mgh.xr) && fabs(mgh.yr) > fabs(mgh.zr)) {
	    oy = or;
	    if (fabs(mgh.xa) > fabs(mgh.ya) && fabs(mgh.xa) > fabs(mgh.za)) {
		ox = oa; oz = os;
	    } else {
		oz = oa; ox = os;
	    }
	} else {
	    oz = or;
	    if (fabs(mgh.xa) > fabs(mgh.ya) && fabs(mgh.xa) > fabs(mgh.za)) {
		ox = oa; oy = os;
	    } else {
		oy = oa; ox = os;
	    }
	}
	rawdatarec->dimensions[0].origin = ox;
	rawdatarec->dimensions[1].origin = oy;
	rawdatarec->dimensions[2].origin = oz;
    }
    if (strlen(ifname) > 4 && strcmp(ifname + strlen(ifname) - 4, ".mgz") == 0) {
      if (bxh_datarec_setcompression(rawdatarec, "gzip") != 0) {
	goto FAIL;
      }
    }
    rawdatarec->elemtype = strdup(elemtype);
    rawdatarec->msbfirstfrags = 1;
    bxh_datarec_addfrag(rawdatarec, ifname, 284, mgh.width*mgh.height*mgh.depth*mgh.nframes*elemlen, ofname, 1);

    bxh_datarec_writeToElement(imagedata, rawdatarec);

    sprintf(&numbuf[0], "%.15g", (double)mgh.TR);
    if (bxh_appendChildElement(acquisitiondata, "tr", &numbuf[0]) != 0)
	goto FAIL;
    sprintf(&numbuf[0], "%.15g", (double)mgh.TE);
    if (bxh_appendChildElement(acquisitiondata, "te", &numbuf[0]) != 0)
	goto FAIL;
    sprintf(&numbuf[0], "%.15g", (double)mgh.TI);
    if (bxh_appendChildElement(acquisitiondata, "ti", &numbuf[0]) != 0)
	goto FAIL;
    sprintf(&numbuf[0], "%.15g", (double)mgh.flipangle);
    if (bxh_appendChildElement(acquisitiondata, "flipangle", &numbuf[0]) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

EXIT:
    if (rawdatarec)
	bxh_datarec_free(rawdatarec);
    if (acquisitiondata)
	bxh_element_unref(acquisitiondata);
    if (imagedata)
	bxh_element_unref(imagedata);

    return docp;
}

int
createBXHFromMGH(const char **ifnames, int numinfiles, const char *ofname)
{
    int result = 0;
    BXHDocPtr docp = NULL;
    struct stat statbuf;
    
    if (stat(ofname, &statbuf) == 0) {
	fprintf(stderr, "mgh2bxh: file exists ('%s')\n", ofname);
	goto FAIL;
    }

    if ((docp = createDocFromMGH(ifnames, numinfiles, ofname)) == NULL)
	goto FAIL;
    if (bxh_writeFile(docp, ofname) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    result = -1;

EXIT:
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

    return result;
}

BXHDocPtr
bxh_mgh2doc(const char *ifname)
{
    return createDocFromMGH(&ifname, 1, "dummy.bxh");
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.5  2007/02/01 18:11:39  gadde
 * Fix compression support
 *
 * Revision 1.4  2007/01/16 20:38:31  gadde
 * Fix height/width.
 *
 * Revision 1.3  2006/11/02 14:39:49  gadde
 * Add Log.
 *
 */
