/*
 * nrrd2bxh.cc --
 * 
 * Creates a BXH file based on given NRRD file
 */

#include "bxh_utils.h"
#include "bxh_datarec.h"

#include <assert.h>

#include <time.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#include <math.h>

#include <map>
#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <vector>

#ifdef WIN32
#define strcasecmp stricmp
#endif

class MyDouble {
    public:
    MyDouble() : _val(0.0/0.0) { }
    MyDouble(double val) : _val(val) { }
    operator double() const { return _val; }
    private:
    double _val;
};

class QuotedString
{
    public:
    operator std::string const& () const { return data;}

    private:
    std::string     data;
    friend std::ostream& operator<<(std::ostream& str, QuotedString const& value)
    {
        return str << value.data;
    }
    friend std::istream& operator>>(std::istream& str, QuotedString& value)
    {
        char x;
	value.data = std::string("");
        str >> x;
	if (!(str) || x != '"') {
	    str.putback(x);
	    goto FAIL;
	}
	while ((str)) {
	    str >> x;
	    if (x == '\\') {
		if (!(str)) {
		    goto FAIL;
		}
		str >> x;
		if (!(str)) {
		    goto FAIL;
		}
		if (x == '"') {
		    value.data += '"';
		} else {
		    value.data += "\\\"";
		}
	    } else if (x == '"') {
		break;
	    } else {
		value.data += x;
	    }
	}
	goto SUCCESS;

	FAIL:
	str.setstate(std::ios::failbit);

	SUCCESS:
	return str;
    }
};

struct nrrdentry {
    nrrdentry() : name(), strdata(NULL), iskeyvalue(false) { /* null */ }
    std::string name;
    std::string * strdata;
    bool iskeyvalue;
};

template <class T>
class nrrdtuple {
    public:
    nrrdtuple() : _vec(), _unknown(false) { }
    size_t size() const { return _vec.size(); }
    T & operator[] (const int index) {
	return _vec[index];
    }
    void push_back(const T & val) {
	_vec.push_back(val);
    }
    void clear() {
	_vec.clear();
    }
    bool unknown() { return _unknown; }
    void set_unknown() { _unknown = true; }
    private:
    std::vector<T> _vec;
    bool _unknown;
};

/* this does not stop at the first invalid double character so it is
 * not a full replacement for the standard "double" extraction operator.
 * It extracts the next word (delimited by non-floating-point or non-'NaN'
 * or non-'Inf') from the stream and sends it to strtod (and thereby should
 * correctly parse NaN and Inf).  This is a massively inefficient state
 * machine (i.e. hack), sorry!
 */
std::istream &
operator>> (std::istream & in, MyDouble & val) {
    std::string valstr;
    bool beforesign = true;
    bool beforeinfcheck = true;
    bool beforedot = true;
    bool beforee = true;
    bool beforeesign = true;
    bool isnegative = false;
    double tmpval;
    char * endptr = NULL;
    const char * nptr;
    while (in.good() && !in.eof()) {
	char c;
	if ((in >> c).fail()) {
	    break;
	}
	c = tolower(c);
	if (beforesign) {
	    if (isspace(c)) {
		/* ignore heading whitespace */
		continue;
	    }
	    /* sign is always in the first character, so we can set
	     * beforesign definitively here */
	    if (c == '+' || c == '-' || isdigit(c)) {
		valstr.push_back(c);
		beforesign = false;
		isnegative = (c == '-');
		continue;
	    }
	    if (c == '.') {
		valstr.push_back(c);
		beforesign = false;
		beforedot = false;
		continue;
	    }
	    if (valstr.size() == 0) {
		if (c == 'n') {
		    if ((in >> c).fail()) {
			goto FAIL;
		    }
		    c = tolower(c);
		    if (c == 'a') {
			if ((in >> c).fail()) {
			    goto FAIL;
			}
			c = tolower(c);
			if (c == 'n') {
			    valstr.append("nan");
			    if ((in >> c).fail()) {
				goto FAIL;
			    }
			    if (c != '(') {
				in.putback(c);
				break;
			    }
			    valstr.push_back(c);
			    while (!(in >> c).fail()) {
				valstr.push_back(c);
				if (c == ')') {
				    break;
				}
			    }
			    if (in.fail()) {
				goto FAIL;
			    }
			    break;
			}
		    }
		    goto FAIL;
		}
	    }
	    /* Now we have a first character that is not a sign character or
	     * a digit, and the character has not been eaten yet, so pass
	     * through */
	    beforesign = false;
	}
	if (beforeinfcheck) {
	    /* check for infinity here (after sign check) */
	    if (c == 'i') {
		if ((in >> c).fail()) {
		    goto FAIL;
		}
		c = tolower(c);
		if (c == 'n') {
		    if ((in >> c).fail()) {
			goto FAIL;
		    }
		    c = tolower(c);
		    if (c == 'f') {
			/* found infinity, done parsing */
			break;
		    }
		}
		goto FAIL;
	    }
	    beforeinfcheck = false;
	}
	if (beforedot) {
	    if (c == '.') {
		valstr.push_back(c);
		beforedot = false;
		continue;
	    }
	    if (isdigit(c)) {
		valstr.push_back(c);
		continue;
	    }
	    /* if we pass through here, we have a non-numeric character
	     * that is not a dot, and we will not expect a dot anymore
	     */
	    beforedot = false;
	}
	if (beforee) {
	    if (c == 'e') {
		valstr.push_back(c);
		beforee = false;
		continue;
	    }
	    if (isdigit(c)) {
		valstr.push_back(c);
		continue;
	    }
	    /* we didn't find an e/E or a digit here so we are at the end of the number */
	    in.putback(c);
	    break;
	}
	if (beforeesign) {
	    if (c == '-' || c == '+' || isdigit(c)) {
		valstr.push_back(c);
		beforeesign = false;
		continue;
	    }
	    /* if we pass through here, we have a non-numeric character
	     * that is not a sign, and we will not expect a sign anymore
	     */
	    beforeesign = false;
	}
	/* last thing is exponent digits */
	if (isdigit(c)) {
	    valstr.push_back(c);
	    continue;
	}
	/* done with the number */
	in.putback(c);
	break;
    }
    nptr = valstr.c_str();
    errno = 0;
    tmpval = strtod(nptr, &endptr);
    if ((errno == ERANGE && (tmpval == HUGE_VALF || tmpval == HUGE_VALL)) ||
	(errno != 0 && val == 0)) {
	goto FAIL;
    }
    if (nptr == endptr) {
	goto FAIL;
    }
    val = tmpval;
    in.clear();
    goto SUCCESS;

    FAIL:
    in.setstate(std::ios::failbit);

    SUCCESS:
    return in;
}

template <class T>
std::ostream &
operator<< (std::ostream & out, nrrdtuple<T> & tuple)
{
    if (tuple.unknown()) {
	out << "<unknown>";
	return out;
    }
    size_t numvals = tuple.size();
    out << '(';
    for (int i = 0; i < numvals; i++) {
	if (i != 0) {
	    out << ", ";
	}
	out << tuple[i];
    }
    out << ')';
}

template <class T>
std::ostream &
operator<< (std::ostream & out, std::vector<T> & vec)
{
    size_t numvals = vec.size();
    out << '[';
    for (int i = 0; i < numvals; i++) {
	if (i != 0) {
	    out << ", ";
	}
	out << vec[i];
    }
    out << ']';
}

template <class T>
std::istream &
operator>> (std::istream & in, nrrdtuple<T> & tuple)
{
    nrrdtuple<T> tmptuple;
    char c;
    size_t size = tmptuple.size();
    in >> c;
    if (c != '(') {
	if (c == 'n') {
	    in >> c;
	    if (c == 'o') {
		in >> c;
		if (c == 'n') {
		    in >> c;
		    if (c == 'e') {
			tmptuple.set_unknown();
			tuple = tmptuple;
			goto SUCCESS;
		    }
		}
	    }
	} else {
	    goto FAIL;
	}
    }
    while (1) {
	if (tmptuple.size() != 0) {
	    in >> c;
	    if (c == ')') {
		in.putback(c);
		break;
	    }
	    if (c != ',') {
		goto FAIL;
	    }
	}
	T val;
	if ((in >> val).fail()) {
	    break;
	}
	tmptuple.push_back(val);
    }
    in >> c;
    if (c != ')') {
	goto FAIL;
    }

    tuple = tmptuple;
    goto SUCCESS;

    FAIL:
    in.setstate(std::ios::failbit);

    SUCCESS:
    return in;
}

template <class T>
bool
from_string(T & t, const std::string& s, bool warnparse)
{
  std::istringstream iss(s);
  iss >> t;
  if (iss.fail()) {
      if (warnparse) {
	  fprintf(stderr, "nrrd2bxh: Error parsing value from string!\n");
      }
      return false;
  }
  return true;
}

template<>
bool
from_string<std::string>(std::string & t, const std::string& s, bool warnparse)
{
    t = s;
    return true;
}

template <class T>
bool
get_nrrd_value(std::map<std::string, nrrdentry> & map, const std::string & key, T & val, bool dowarn = false) {
    std::map<std::string, nrrdentry>::iterator iter;
    std::string lowerkey = key;
    size_t keylen = lowerkey.length();
    for (int i = 0; i < keylen; i++) {
	lowerkey[i] = std::tolower(lowerkey[i]);
    }
    if ((iter = map.find(lowerkey)) == map.end()) {
	if (dowarn) {
	    fprintf(stderr, "nrrd2bxh: Didn't find '%s' field!\n", lowerkey.c_str());
	}
	return false;
    }
    std::string * strdata = (*iter).second.strdata;
    if (strdata == NULL) {
	fprintf(stderr, "nrrd2bxh: '%s' field value is NULL (internal error?)\n", key.c_str());
	return false;
    }
    return from_string<T>(val, *strdata, dowarn);
}

template <class T>
bool
need_nrrd_value(std::map<std::string, nrrdentry> & map, const std::string & key, T & val) {
    return get_nrrd_value(map, key, val, true);
}

template <class T>
bool
try_nrrd_value(std::map<std::string, nrrdentry> & map, const std::string & key, T & val) {
    return get_nrrd_value(map, key, val, false);
}

// space-separated list of values
template <class T>
bool
get_nrrd_list(std::map<std::string, nrrdentry> & map, const std::string & key, std::vector<T> & vec, size_t expectednumvals = 0, bool dowarn = false, bool fillmissing = false, bool quoted = false) {
    std::vector<T> tmpvec;
    std::string vecstr;
    if (!get_nrrd_value<std::string>(map, key, vecstr, dowarn)) {
	return false;
    }
    std::istringstream iss(vecstr);
    while (iss.good() && !iss.eof()) {
	T val;
	if ((iss >> val).fail()) {
	    break;
	}
	tmpvec.push_back(val);
	iss >> std::ws;
    }
    if (fillmissing) {
        while (tmpvec.size() < expectednumvals) {
	    tmpvec.push_back(T());
	}
    }
    if (expectednumvals != 0 && tmpvec.size() != expectednumvals) {
        std::ostringstream oss;
	oss << tmpvec;
	fprintf(stderr, "nrrd2bxh: expected a list of %ld values in '%s' field but found %ld: %s\n", (long int)expectednumvals, key.c_str(), (long int)tmpvec.size(), oss.str().c_str());
	return false;
    }
    vec = tmpvec;
    return true;
}

template <class T>
bool
need_nrrd_list(std::map<std::string, nrrdentry> & map, const std::string & key, std::vector<T> & vec, size_t expectednumvals = 0, bool fillmissing = true) {
    return get_nrrd_list(map, key, vec, expectednumvals, true, fillmissing);
}

template <class T>
bool
try_nrrd_list(std::map<std::string, nrrdentry> & map, const std::string & key, std::vector<T> & vec, size_t expectednumvals = 0, bool fillmissing = true) {
    return get_nrrd_list(map, key, vec, expectednumvals, false, fillmissing);
}

#ifdef __cplusplus
extern "C" {
#endif

extern char * domutil_errorbuf;

BXHDocPtr
createDocFromNRRD(const char *ifname, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient)
{
    BXHDocPtr docp = NULL;
    bxhrawdatarec * datarec = NULL;
    BXHElementPtr imagedata = NULL;
    BXHElementPtr acquisitiondata = NULL;
    BXHElementPtr subject = NULL;
    struct stat statbuf;
    int msbfirst;
    std::map<std::string, nrrdentry> nrrdmap;
    FILE * iffp = NULL;
    std::ifstream inputfile(ifname);
    std::string dfname = ifname;
    bool detached = false;
    size_t datapos = 0;
    char * defaulttypes = strdup("xyzt");

    msbfirst = 1;
    msbfirst = (((char *)&msbfirst)[0] == 0);

    {
	size_t iflen = 0;
	char type[64];
	int count = 0;

	iflen = strlen(ifname);

	if (!inputfile.is_open() || !inputfile.good()) {
	    fprintf(stderr, "nrrd2bxh: Error opening file %s\n", ifname);
	    goto FAIL;
	}

	std::string magic;
	getline(inputfile, magic);
	if (magic.find("NRRD") != 0) {
	    fprintf(stderr, "nrrd2bxh: Didn't find NRRD magic in file %s\n", ifname);
	    goto FAIL;
	}

	while (inputfile.good()) {
	    nrrdentry ne;
	    std::string line;
	    getline(inputfile, line);
	    size_t linelen = line.length();
	    if (linelen > 0 && line[linelen-1] == '\r') {
		line.resize(linelen - 1);
		linelen--;
	    }
	    if (linelen > 0 && line[0] == '#') {
		/* comment line */
		continue;
	    }
	    if (linelen == 0) {
		break;
	    }
	    size_t colonpos = line.find(':');
	    if (colonpos + 1 >= linelen) {
		fprintf(stderr, "nrrd2bxh: found stray colon in line: %s\n", line.c_str());
		goto FAIL;
	    }
	    if (line[colonpos + 1] == '=') {
		ne.iskeyvalue = true;
		size_t curpos = colonpos + 2;
		size_t slashpos;
		while ((slashpos = line.find('\\', curpos)) != std::string::npos) {
		    if (slashpos + 1 < linelen) {
			if (line[slashpos + 1] == '\\') {
			    line.erase(slashpos, 1);
			} else if (line[slashpos + 1] == 'n') {
			    line.replace(slashpos, 2, "\n");
			}
		    }
		    curpos = slashpos + 1;
		}
	    } else if (line[colonpos + 1] == ' ') {
		// field
		// get rid of trailing whitespace
		size_t lastpos = line.find_last_not_of(" \t\n\r\f\v");
		if (lastpos != std::string::npos) {
		    line.resize(lastpos + 1);
		}
	    } else {
		fprintf(stderr, "nrrd2bxh: found stray colon in line: %s\n", line.c_str());
		goto FAIL;
	    }
	    ne.name = line.substr(0, colonpos);
	    size_t namelen = ne.name.length();
	    for (int i = 0; i < namelen; i++) {
		ne.name[i] = std::tolower(ne.name[i]);
	    }
	    if (ne.name.compare("datafile") == 0) {
		ne.name = "data file";
	    } else if (ne.name.compare("lineskip") == 0) {
		ne.name = "line skip";
	    } else if (ne.name.compare("byteskip") == 0) {
		ne.name = "byte skip";
	    } else if (ne.name.compare("centers") == 0) {
		ne.name = "centerings";
	    } else if (ne.name.compare("axismins") == 0) {
		ne.name = "axis mins";
	    } else if (ne.name.compare("axismaxs") == 0) {
		ne.name = "axis maxs";
	    }
	    ne.strdata = new std::string(line.substr(colonpos + 2));
	    nrrdmap[ne.name] = ne;
	}
    }

    if ((docp = bxh_createDocument()) == NULL)
	goto FAIL;

    if ((imagedata = bxh_getDatarec(docp, "image", NULL)) == NULL)
	goto FAIL;

    if ((acquisitiondata = bxh_getAcquisitionData(docp)) == NULL)
	goto FAIL;
    
    datarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
    memset(datarec, '\0', sizeof(*datarec));
    
    {
	nrrdentry ne;
	int dimnum;
	int i;
	bxhdimension * dimx = NULL;
	bxhdimension * dimy = NULL;
	bxhdimension * dimz = NULL;
	bxhdimension * dimt = NULL;

	if (!need_nrrd_value<int>(nrrdmap, "dimension", datarec->numdims)) {
	    goto FAIL;
	}

	if (try_nrrd_value<std::string>(nrrdmap, "data file", dfname)) {
	    if (dfname.find(" ") != std::string::npos) {
		fprintf(stderr, "nrrd2bxh: found a space in data file name; format-based and list-based detached data files are not supported for now.\n");
		goto FAIL;
	    }
	    datapos = 0;
	    detached = true;
	}

	std::vector<size_t> dimsizes;
	if (!need_nrrd_list<size_t>(nrrdmap, "sizes", dimsizes, datarec->numdims)) {
	    goto FAIL;
	}

	std::vector<MyDouble> spacings;
	try_nrrd_list<MyDouble>(nrrdmap, "spacings", spacings, datarec->numdims);

	std::vector<MyDouble> thicknesses;
	try_nrrd_list<MyDouble>(nrrdmap, "thicknesses", thicknesses, datarec->numdims);

	std::vector<MyDouble> axismins;
	try_nrrd_list<MyDouble>(nrrdmap, "axismins", axismins, datarec->numdims);

	std::vector<MyDouble> axismaxs;
	try_nrrd_list<MyDouble>(nrrdmap, "axismaxs", axismaxs, datarec->numdims);

	std::vector<std::string> centerings;
	try_nrrd_list<std::string>(nrrdmap, "centerings", centerings, datarec->numdims);

	std::vector<std::string> kinds;
	try_nrrd_list<std::string>(nrrdmap, "kinds", kinds, datarec->numdims);

	std::vector<QuotedString> labels;
	try_nrrd_list<QuotedString>(nrrdmap, "labels", labels, datarec->numdims);

	datarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension)*datarec->numdims);
	memset(datarec->dimensions, '\0', sizeof(bxhdimension)*datarec->numdims);
	for (int i = 0; i < datarec->numdims; i++) {
	    if (dimsizes.size() > i) {
		datarec->dimensions[i].size = dimsizes[i];
	    }
	    if (axismins.size() > i) {
		datarec->dimensions[i].origin = axismins[i];
		if (axismaxs.size() > i) {
		    bool docenter = true;
		    if (centerings.size() > i) {
			if (centerings[i].compare("node")) {
			    docenter = false;
			}
		    }
		    if (docenter) {
			datarec->dimensions[i].spacing = (axismaxs[i] - axismins[i]) / (datarec->dimensions[i].size - 1);
		    } else {
			datarec->dimensions[i].spacing = (axismaxs[i] - axismins[i]) / datarec->dimensions[i].size;
			datarec->dimensions[i].origin += (datarec->dimensions[i].spacing / 2.0);
		    }
		}
	    }
	    if (spacings.size() > i) {
		datarec->dimensions[i].spacing = spacings[i];
		if (thicknesses.size() > i) {
		    datarec->dimensions[i].gap = spacings[i] - thicknesses[i];
		}
	    }
	    if (labels.size() > i) {
		datarec->dimensions[i].type = strdup(((const std::string &) labels[i]).c_str());
	    }
	}

	std::string type;
	int elemsize = 0;
	if (!need_nrrd_value<std::string>(nrrdmap, "type", type)) {
	    goto FAIL;
	}
	if (type.compare("signed char") == 0 ||
	    type.compare("int8") == 0 ||
	    type.compare("int8_t") == 0) {
	    datarec->elemtype = strdup("int8");
	    elemsize = 1;
	} else if (type.compare("uchar") == 0 ||
		   type.compare("unsigned char") == 0 ||
		   type.compare("uint8") == 0 ||
		   type.compare("uint8_t") == 0) {
	    datarec->elemtype = strdup("uint8");
	    elemsize = 1;
	} else if (type.compare("short") == 0 ||
		   type.compare("short int") == 0 ||
		   type.compare("signed short") == 0 ||
		   type.compare("signed short int") == 0 ||
		   type.compare("int16") == 0 ||
		   type.compare("int16_t") == 0) {
	    datarec->elemtype = strdup("int16");
	    elemsize = 2;
	} else if (type.compare("ushort") == 0 ||
		   type.compare("unsigned short") == 0 ||
		   type.compare("unsigned short int") == 0 ||
		   type.compare("uint16") == 0 ||
		   type.compare("uint16_t") == 0) {
	    datarec->elemtype = strdup("uint16");
	    elemsize = 2;
	} else if (type.compare("int") == 0 ||
		   type.compare("signed int") == 0 ||
		   type.compare("int32") == 0 ||
		   type.compare("int32_t") == 0) {
	    datarec->elemtype = strdup("int32");
	    elemsize = 4;
	} else if (type.compare("uint") == 0 ||
		   type.compare("unsigned int") == 0 ||
		   type.compare("uint32") == 0 ||
		   type.compare("uint32_t") == 0) {
	    datarec->elemtype = strdup("uint32");
	    elemsize = 4;
	} else if (type.compare("longlong") == 0 ||
		   type.compare("long long") == 0 ||
		   type.compare("long long int") == 0 ||
		   type.compare("signed long long") == 0 ||
		   type.compare("signed long long int") == 0 ||
		   type.compare("int64") == 0 ||
		   type.compare("int64_t") == 0) {
	    fprintf(stderr, "nrrd2bxh: Unsupported data type '%s'!\n", type.c_str());
	    goto FAIL;
	} else if (type.compare("ulonglong") == 0 ||
		   type.compare("unsigned long long") == 0 ||
		   type.compare("unsigned long long int") == 0 ||
		   type.compare("uint64") == 0 ||
		   type.compare("uint64_t") == 0) {
	    fprintf(stderr, "nrrd2bxh: Unsupported data type '%s'!\n", type.c_str());
	    goto FAIL;
	} else if (type.compare("float") == 0) {
	    datarec->elemtype = strdup("float32");
	    elemsize = 4;
	} else if (type.compare("double") == 0) {
	    datarec->elemtype = strdup("float64");
	    elemsize = 8;
	} else {
	    fprintf(stderr, "nrrd2bxh: Unrecognized data type '%s'!\n", type.c_str());
	    goto FAIL;
	}

	std::string encoding;
	if (!need_nrrd_value<std::string>(nrrdmap, "encoding", encoding)) {
	    goto FAIL;
	}
	if (encoding.compare("raw") == 0) {
	    // nothing to do
	} else if (encoding.compare("gz") == 0 ||
		   encoding.compare("gzip") == 0) {
	    size_t dfnamelen = dfname.length();
	    if (detached) {
		if (dfnamelen > 3 && dfname.substr(dfnamelen - 3).compare(".gz") == 0) {
		    // use auto-gzip detection
		    dfname.erase(dfnamelen - 3);
		} else {
		    datarec->compression = strdup("gzip");
		}
	    } else {
		datarec->compression = strdup("gzipfrag");
	    }
	} else {
	    fprintf(stderr, "nrrd2bxh: Unsupported encoding '%s'!\n", encoding.c_str());
	    goto FAIL;
	}

	long spacedimension;
	try_nrrd_value<long>(nrrdmap, "space dimension", spacedimension);

	std::string space;
	int spaceadjust[3] = { 1, 1, 1 };
	if (try_nrrd_value<std::string>(nrrdmap, "space", space)) {
	    bool dotime = false;
	    if (space.find("scanner-xyz") == 0 ||
		space.find("scanner-xyz-time") == 0 ||
		space.find("3D-right-handed") == 0 ||
		space.find("3D-left-handed") == 0 ||
		space.find("3D-right-handed-time") == 0 ||
		space.find("3D-left-handed-time") == 0) {
		fprintf(stderr, "nrrd2bxh: Unsupported space '%s'!\n", space.c_str());
		goto FAIL;
	    } else if (space.find("right-anterior-superior") == 0 ||
		space.compare("RAS") == 0) {
		// do nothing, this is our native space
	    } else if (space.compare("left-anterior-superior") == 0 ||
		space.compare("LAS") == 0) {
		spaceadjust[0] = -1;
	    } else if (space.compare("left-posterior-superior") == 0 ||
		space.compare("LPS") == 0) {
		spaceadjust[0] = -1;
		spaceadjust[1] = -1;
	    } else if (space.find("right-anterior-superior-time") == 0 ||
		space.compare("RAST") == 0) {
		// do nothing, this is our native space
		dotime = true;
	    } else if (space.compare("left-anterior-superior-time") == 0 ||
		space.compare("LAST") == 0) {
		spaceadjust[0] = -1;
		dotime = true;
	    } else if (space.compare("left-posterior-superior-time") == 0 ||
		space.compare("LPST") == 0) {
		spaceadjust[0] = -1;
		spaceadjust[1] = -1;
		dotime = true;
	    }
	    spacedimension = 3;
	    if (dotime) {
		spacedimension = 4;
	    }
	    std::vector<nrrdtuple<MyDouble> > spacedirections;
	    if (try_nrrd_list<nrrdtuple<MyDouble> >(nrrdmap, "space directions", spacedirections, datarec->numdims)) {
		int spatialdimnum = 0;
		char * types = &defaulttypes[0];
		char * timetype = &defaulttypes[3];
		for (int i = 0; i < spacedirections.size(); i++) {
		    nrrdtuple<MyDouble> direction = spacedirections[i];
		    char * type = NULL;
		    if (datarec->dimensions[i].type == NULL) {
			/* figure out axis labels */
			if (*types != '\0' && kinds.size() > i) {
			    if (kinds[i].compare("domain") == 0) {
				type = (char *)malloc(sizeof(char) * 2);
				type[0] = *types;
				type[1] = '\0';
				types++;
			    } else if (kinds[i].compare("space") == 0) {
				if (types >= timetype) {
				    fprintf(stderr, "nrrd2bxh: found more than 3 spatial dimensions???\n");
				    goto FAIL;
				}
				type = (char *)malloc(sizeof(char) * 2);
				type[0] = *types;
				type[1] = '\0';
				types++;
			    } else if (kinds[i].compare("time") == 0) {
				if (*timetype == '\0') {
				    fprintf(stderr, "nrrd2bxh: found more than one temporal dimensions???\n");
				    goto FAIL;
				}
				type = (char *)malloc(sizeof(char) * 2);
				type[0] = *timetype;
				type[1] = '\0';
				*timetype = '\0';
			    }
			}
		    }
		    if (!direction.unknown()) {
			double spacing = sqrt((direction[0]*direction[0]) + (direction[1]*direction[1]) + (direction[2]*direction[2]));
			datarec->dimensions[i].spacing = spacing;
			datarec->dimensions[i].direction = (double *)malloc(sizeof(double) * 3);
			datarec->dimensions[i].direction[0] = (direction[0] / spacing) * spaceadjust[0];
			datarec->dimensions[i].direction[1] = (direction[1] / spacing) * spaceadjust[1];
			datarec->dimensions[i].direction[2] = (direction[2] / spacing) * spaceadjust[2];
			datarec->dimensions[i].spacing = sqrt((direction[0]*direction[0]) + (direction[1]*direction[1]) + (direction[2]*direction[2]));
			spatialdimnum++;
		    }
		    if (type == NULL) {
			char buf[16];
			buf[0] = '\0';
			snprintf(&buf[0], sizeof(buf), "dim%d", i + 1);
			datarec->dimensions[i].type = strdup(&buf[0]);
		    } else {
			datarec->dimensions[i].type = type;
		    }
		}
	    }
	}
	
	std::vector<QuotedString> spaceunits;
	if (try_nrrd_list<QuotedString>(nrrdmap, "space units", spaceunits, datarec->numdims)) {
	    for (int i = 0; i < spaceunits.size(); i++) {
		datarec->dimensions[i].units = strdup(((const std::string &)spaceunits[i]).c_str());
	    }
	}
	
	nrrdtuple<MyDouble> spaceorigin;
	if (try_nrrd_value<nrrdtuple<MyDouble> >(nrrdmap, "space origin", spaceorigin), datarec->numdims) {
	    int spacedim = 0;
	    for (int i = 0; i < datarec->numdims; i++) {
		if (kinds[i].compare("space") == 0 ||
		    kinds[i].compare("domain") == 0) {
		    datarec->dimensions[i].origin = spaceorigin[spacedim] * spaceadjust[spacedim];
		    spacedim++;
		}
	    }
	}
	
	std::vector<nrrdtuple<MyDouble> > measurementframe;
	try_nrrd_list<nrrdtuple<MyDouble> >(nrrdmap, "measurement frame", measurementframe, spacedimension);
	
	std::string endian;
	if (!need_nrrd_value<std::string>(nrrdmap, "endian", endian)) {
	    goto FAIL;
	}
	if (endian.compare("little") == 0) {
	    datarec->msbfirstfrags = 0;
	} else if (endian.compare("big") == 0) {
	    datarec->msbfirstfrags = 1;
	} else {
	    fprintf(stderr, "nrrd2bxh: unexpected endian value '%s'!\n", endian.c_str());
	    goto FAIL;
	}

	int lineskip;
	if (try_nrrd_value<int>(nrrdmap, "line skip", lineskip)) {
	    if (detached) {
		fprintf(stderr, "nrrd2bxh: 'line skip' not supported for detached NRRD data files!\n");
		goto FAIL;
	    }
	    std::string line;
	    for (int i = 0; i < lineskip; i++) {
		getline(inputfile, line);
	    }
	}

	std::string modality;
	try_nrrd_value<std::string>(nrrdmap, "modality", modality);
	if (modality.compare("DWMRI") == 0) {
	    /* do DWI-specific parsing */
	    int dwidimind = -1;
	    for (int i = 0; i < kinds.size(); i++) {
		if (kinds[i].compare("list") == 0 ||
		    kinds[i].compare("vector") == 0) {
		    dwidimind = i;
		}
	    }
	    if (dwidimind == -1) {
		fprintf(stderr, "nrrd2bxh: Can't find DWI dimension!\n");
		goto FAIL;
	    }
	    MyDouble bvalue = 0.0;
	    bvalue = (double)bvalue / (double)bvalue;
	    if (!need_nrrd_value<MyDouble>(nrrdmap, "DWMRI_b-value", bvalue)) {
		fprintf(stderr, "nrrd2bxh: DWI dataset doesn't have b-value!\n");
		goto FAIL;
	    }
	    bxhdimension * dwidim = &datarec->dimensions[dwidimind];
	    free(dwidim->type);
	    dwidim->type = strdup("diffusiondirection");
	    if (measurementframe.size() != 0) {
		dwidim->measurementframe = (double *)malloc(sizeof(dwidim->measurementframe[0]) * 9);
		dwidim->measurementframe[0] = measurementframe[0][0] * spaceadjust[0];
		dwidim->measurementframe[1] = measurementframe[0][1] * spaceadjust[1];
		dwidim->measurementframe[2] = measurementframe[0][2] * spaceadjust[2];
		dwidim->measurementframe[3] = measurementframe[1][0] * spaceadjust[0];
		dwidim->measurementframe[4] = measurementframe[1][1] * spaceadjust[1];
		dwidim->measurementframe[5] = measurementframe[1][2] * spaceadjust[2];
		dwidim->measurementframe[6] = measurementframe[2][0] * spaceadjust[0];
		dwidim->measurementframe[7] = measurementframe[2][1] * spaceadjust[1];
		dwidim->measurementframe[8] = measurementframe[2][2] * spaceadjust[2];
		free(dwidim->measurementframeversion);
		dwidim->measurementframeversion = strdup("2");
	    }

	    size_t numdiffdirs = dwidim->size;
	    char gradkey[20];
	    bxhdatapoints diffdps;
	    bxhdatapoints bvaluesdps;
	    char * bvaluesstr = NULL;
	    char bvaluebuf[64];
	    if (snprintf(&bvaluebuf[0], sizeof(bvaluebuf), "%g", (double)bvalue) >= sizeof(bvaluebuf)) {
		    fprintf(stderr, "nrrd2bxh: internal buffer for bvalue not big enough???\n");
		    goto FAIL;
	    }
	    size_t bvaluelen = strlen(&bvaluebuf[0]);
	    size_t bvaluesstrlen = 0;
	    diffdps.label = strdup("diffusiondirection");
	    diffdps.numvalues = 0;
	    diffdps.values = NULL;
	    bvaluesdps.label = strdup("bvalues");
	    bvaluesdps.numvalues = 0;
	    bvaluesdps.values = NULL;
	    for (int diffind = 0; diffind < numdiffdirs; diffind++) {
		snprintf(&gradkey[0], sizeof(gradkey), "DWMRI_gradient_%04d", diffind);
		std::string gradient;
		const char * tmpbvaluebuf = &bvaluebuf[0];
		size_t tmpbvaluelen = bvaluelen;
		if (!need_nrrd_value<std::string>(nrrdmap, &gradkey[0], gradient)) {
		    fprintf(stderr, "nrrd2bxh: Can't find gradient for volume %d!\n", (int)diffind);
		    goto FAIL;
		}
		int nex = 1;
		snprintf(&gradkey[0], sizeof(gradkey), "DWMRI_NEX_%04d", diffind);
		try_nrrd_value<int>(nrrdmap, &gradkey[0], nex);
		for (int ex = 0; ex < nex; ex++) {
		    diffdps.values = (char **)realloc(diffdps.values, sizeof(char *) * (diffdps.numvalues + 1));
		    diffdps.values[diffdps.numvalues] = strdup(gradient.c_str());
		    if (strspn(gradient.c_str(), "0. ") == gradient.length()) {
			tmpbvaluebuf = "0";
			tmpbvaluelen = 1;
		    }
		    diffdps.numvalues++;
		}
		diffind += (nex - 1);
		for (int ex = 0; ex < nex; ex++) {
		    bvaluesstr = (char *)realloc(bvaluesstr, sizeof(char) * (bvaluesstrlen + tmpbvaluelen + (diffind == 0 ? 0 : 1) + 1));
		    if (diffind != 0) {
			bvaluesstr[bvaluesstrlen] = ' ';
			bvaluesstrlen++;
		    }
		    strncpy(&bvaluesstr[bvaluesstrlen], &tmpbvaluebuf[0], sizeof(char) * tmpbvaluelen);
		    bvaluesstrlen += sizeof(char) * tmpbvaluelen;
		    bvaluesstr[bvaluesstrlen] = '\0';
		    bvaluesdps.values = (char **)realloc(bvaluesdps.values, sizeof(char *) * (bvaluesdps.numvalues + 1));
		    bvaluesdps.values[bvaluesdps.numvalues] = strdup(&tmpbvaluebuf[0]);
		    bvaluesdps.numvalues++;
		}
	    }
	    dwidim->dpstructs = (bxhdatapoints *)realloc(dwidim->dpstructs, sizeof(bxhdatapoints) * (dwidim->numdpstructs + 1));
	    dwidim->dpstructs[dwidim->numdpstructs] = diffdps;
	    dwidim->numdpstructs++;
	    dwidim->dpstructs = (bxhdatapoints *)realloc(dwidim->dpstructs, sizeof(bxhdatapoints) * (dwidim->numdpstructs + 1));
	    dwidim->dpstructs[dwidim->numdpstructs] = bvaluesdps;
	    dwidim->numdpstructs++;

	    if (bxh_appendChildElement(acquisitiondata, "bvalues", bvaluesstr) != 0) {
		free(bvaluesstr); bvaluesstr == NULL;
		goto FAIL;
	    }
	    free(bvaluesstr); bvaluesstr == NULL;
	}

	if (!detached) {
	    datapos = inputfile.tellg();
	}
	inputfile.close();
	size_t fragnumelems = 1;
	for (int i = 0; i < datarec->numdims; i++) {
	    fragnumelems *= dimsizes[i];
	}
	bxh_datarec_addfrag(datarec, dfname.c_str(), datapos, elemsize * fragnumelems, ofname, 1);
    }

    {
	const char * hintnames[4] = { "x", "y", "z", "t" };
	int dimnum;
	if (hintsize[0] != (size_t)-1 ||
	    hintsize[1] != (size_t)-1 ||
	    hintsize[2] != (size_t)-1 ||
	    hintsize[3] != (size_t)-1) {
	    fprintf(stderr, "Warning: --hintsize* options are ignored for NRRD files\n");
	}
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    bxhdimension * dimp = &datarec->dimensions[dimnum];
	    int hintnum;
	    for (hintnum = 0; hintnum < 4; hintnum++) {
		if (strcmp(dimp->type, hintnames[hintnum]) == 0)
		    break;
	    }
	    if (hintnum == 4) continue;
	    if (hintorigin[hintnum] != HUGE_VAL)
		dimp->origin = hintorigin[hintnum];
	    if (hintspacing[hintnum] != HUGE_VAL)
		dimp->spacing = hintspacing[hintnum];
	    if (hintgap[hintnum] != HUGE_VAL)
		dimp->gap = hintgap[hintnum];
	    if (forceorient && hintnum < 3) {
		if (dimp->direction == NULL)
		    dimp->direction = (double *)malloc(sizeof(double)*3);
		memset(dimp->direction, '\0', sizeof(double)*3);
		if (tolower(forceorient[hintnum]) == 'r') {
		    dimp->direction[0] = 1;
		} else if (tolower(forceorient[hintnum]) == 'a') {
		    dimp->direction[1] = 1;
		} else if (tolower(forceorient[hintnum]) == 's') {
		    dimp->direction[2] = 1;
		} else if (tolower(forceorient[hintnum]) == 'l') {
		    dimp->direction[0] = -1;
		} else if (tolower(forceorient[hintnum]) == 'p') {
		    dimp->direction[1] = -1;
		} else if (tolower(forceorient[hintnum]) == 'i') {
		    dimp->direction[2] = -1;
		}
	    }
	}
    }

    bxh_datarec_writeToElement(imagedata, datarec);

    goto EXIT;

FAIL:
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

EXIT:
    if (iffp) {
	fclose(iffp); iffp = NULL;
    }
    if (datarec)
	bxh_datarec_free(datarec);
    if (acquisitiondata)
	bxh_element_unref(acquisitiondata);
    if (imagedata)
	bxh_element_unref(imagedata);
    if (defaulttypes)
	free(defaulttypes);

    for (std::map<std::string, nrrdentry>::const_iterator it = nrrdmap.begin(); it != nrrdmap.end(); ++it) {
	nrrdentry ne = (*it).second;
	if (ne.strdata) {
	    delete ne.strdata; ne.strdata = NULL;
	}
    }

    return docp;
}

int
createBXHFromNRRD(const char *ifname, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient)
{
    int result = 0;
    BXHDocPtr docp = NULL;
    struct stat statbuf;

    if (stat(ofname, &statbuf) == 0) {
	fprintf(stderr, "nrrd2bxh: file exists ('%s')\n", ofname);
	goto FAIL;
    }

    if ((docp = createDocFromNRRD(ifname, ofname, hintsize, hintorigin, hintspacing, hintgap, forceorient)) == NULL)
	goto FAIL;
    if (bxh_writeFile(docp, ofname) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    fprintf(stderr, "nrrd2bxh: conversion failed.\n");
    result = -1;

EXIT:
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

    return result;
}

BXHDocPtr
bxh_nrrd2doc(const char *ifname)
{
    size_t hintsize[4] = { (size_t)-1, (size_t)-1, (size_t)-1, (size_t)-1 };
    double hintorigin[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    double hintspacing[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    double hintgap[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    return createDocFromNRRD(ifname, "dummy.bxh", &hintsize[0], &hintorigin[0], &hintspacing[0], &hintgap[0], NULL);
}

#ifdef __cplusplus
}
#endif
