static const char rcsid[] = "$Id: bxh_datarec.cpp,v 1.89 2009-01-15 20:55:18 gadde Exp $";

/* bxh_datarec.c --
 * 
 * Functions and structures to describe common BXH datarec fields,
 * and store BXH datarec data.
 *
 * Author: Syam Gadde (gadde@biac.duke.edu) July 2002
 */

#include <limits.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#ifndef WIN32
#include <unistd.h>
#endif
#ifdef WIN32
#include <direct.h>
#include <io.h>
#endif

#include <vector>
#include <string>

#include <zlib.h>

#include "bxh_datarec.h"
#include "bxh_utils.h"

#ifdef __cplusplus
extern "C" {
#endif
extern char * domutil_errorbuf;
#ifdef __cplusplus
}
#endif

const char * elemtypestrs[] = {
    "int8",
    "uint8",
    "int16",
    "uint16",
    "int32",
    "uint32",
    "float32",
    "float64",
    "double",
    "rgb24",
    NULL
};

off_t elemtypesizes[] = {
    1, 1, 2, 2, 4, 4, 4, 8, 8, 3
};

struct readobj {
    std::vector<int> permuteorder;
    std::vector<int> permdeletethese;
    std::vector<std::vector<size_t> > prereadselectors;
    std::vector<size_t> inputnewsizes;
    std::vector<size_t> outputnewsizes;
    std::vector<std::vector<size_t> > postreadselectors;
    std::vector<size_t> permtypelen;
};

static
size_t
path_match_vol(const char * curptr)
{
#ifdef WIN32
    if (curptr[0] != '\0' && curptr[1] != '\0') {
	if (isalpha(curptr[0]) && curptr[1] == ':') {
	    return 2;
	} else if ((curptr[0] == '\\' || curptr[0] == '/') &&
		   curptr[0] == curptr[1]) {
	    const char * hostend = NULL;
	    const char * hostend2 = NULL;
	    const char * shareend = NULL;
	    const char * shareend2 = NULL;
	    hostend = strchr(curptr+2, '\\');
	    hostend2 = strchr(curptr+2, '/');
	    if (hostend2 > hostend) hostend = hostend2;
	    if (hostend) {
		shareend = strchr(hostend+1, '\\');
		shareend2 = strchr(hostend+1, '/');
		if (shareend2 > shareend) shareend = shareend2;
		if (shareend) {
		    return (shareend - curptr);
		}
	    }
	}
    }
#endif
    return 0;
}

static
size_t
path_match_dirs(const char * curptr)
{
    const char * lastslash = NULL;
#ifdef WIN32
    {
	const char * lastslash2 = NULL;
	lastslash = strrchr(curptr, '\\');
	lastslash2 = strrchr(curptr, '/');
	if (lastslash2 > lastslash) lastslash = lastslash2;
    }
#else
    lastslash = strrchr(curptr, '/');
#endif
    if (lastslash)
	return (lastslash + 1 - curptr);
    else
	return 0;
}

#undef FUNC
#define FUNC "bxh_datarec_getfullfilename"
/**
 * Concatenate basepath portion with filename if filename is relative.
 *
 * @param basepath base pathname (must contain slash if a directory), can be a filename.  If NULL, use current working directory.
 * @param filename relative or absolute filename
 */
char *
bxh_datarec_getfullfilename(const char * basepath, const char * filename)
{
    int fulllen = 0;
    char * fullfilename = NULL;
    char * curwd = NULL;
    char * fullbasepath = NULL;
    const char * basevol = NULL;
    const char * basedirs = NULL;
    const char * basefile = NULL;
    const char * fvol = NULL;
    const char * fdirs = NULL;
    const char * ffile = NULL;
    size_t basevollen = 0;
    size_t basedirslen = 0;
    size_t basefilelen = 0;
    size_t fvollen = 0;
    size_t fdirslen = 0;
    size_t ffilelen = 0;
    size_t baselen;
    size_t filelen;
    const char * curptr = NULL;

    filelen = strlen(filename);

    curptr = filename;
    fvollen = path_match_vol(curptr);
    if (fvollen) { fvol = curptr; }
    curptr += fvollen;
    fdirslen = path_match_dirs(curptr);
    if (fdirslen) { fdirs = curptr; }
    curptr += fdirslen;
    ffile = curptr;
    ffilelen = filelen - fdirslen;

    /* Check if filename is already absolute */
#if WIN32
    if (fvollen && fdirslen &&
	(fdirs[0] == '\\' || fdirs[0] == '/'))
	return strdup(filename);
#else
    if (fdirslen && fdirs[0] == '/')
	return strdup(filename);
#endif

    if (basepath == NULL) {
	size_t curwdlen = 128;
	while (1) {
	    curwd = (char *)realloc(curwd, curwdlen);
	    basepath = getcwd(curwd, curwdlen - 1);
	    if (basepath == NULL) {
		if (errno == ERANGE) {
		    curwdlen *= 2;
		} else {
		    fprintf(stderr, "Error getting current working directory!\n");
		    free(curwd);
		    exit(-1);
		}
	    } else {
		break;
	    }
	}
#ifdef WIN32
	strcat(curwd, "\\");
#else
	strcat(curwd, "/");
#endif
    } else {
	/* make sure basepath is absolute too */
	fullbasepath = bxh_datarec_getfullfilename(NULL, basepath);
	basepath = fullbasepath;
    }

    baselen = strlen(basepath);

    curptr = basepath;
    basevollen = path_match_vol(curptr);
    if (basevollen) { basevol = curptr; }
    curptr += basevollen;
    basedirslen = path_match_dirs(curptr);
    if (basedirslen) { basedirs = curptr; }
    curptr += basedirslen;
    basefile = curptr;
    basefilelen = baselen - basedirslen;

    fulllen += strlen(basepath);
    fulllen += strlen(filename);
    fullfilename = (char *)malloc(sizeof(char) * (fulllen + 2));
    fullfilename[0] = '\0';

    if (fvollen == 0) {
	if (basevollen)
	    strncat(fullfilename, basevol, basevollen);
    } else {
	strncat(fullfilename, fvol, fvollen);
    }
#if WIN32
    if (fvollen == 0 && (fdirslen == 0 || (fdirs[0] != '\\' && fdirs[0] != '/'))) {
	if (basedirslen)
	    strncat(fullfilename, basedirs, basedirslen);
    }
#else
    if (fvollen == 0 && (fdirslen == 0 || fdirs[0] != '/')) {
	if (basedirslen)
	    strncat(fullfilename, basedirs, basedirslen);
    }
#endif
    if (fdirslen)
	strncat(fullfilename, fdirs, fdirslen);
    if (ffilelen)
	strncat(fullfilename, ffile, ffilelen);

    if (curwd) {
	free(curwd);
    }
    if (fullbasepath) {
	free(fullbasepath);
    }
    return fullfilename;
}

#undef FUNC
#define FUNC "bxh_datarec_canonicalizeFrags"
/**
 * This function takes a NULL terminated array of bxhfrag pointers,
 * sorts them by filename and fileoffset, then concatenates frags
 * that are contiguous in the same file.  On return, the array
 * may have fewer entries, and some frags may have been modified
 * or freed, so do not hold references to them across a call to
 * this function.
 *
 * @param infragps array of bxhfrag pointers
 * @param numfrags number of input frags
 * @return number of output frags
 */
static
int
bxh_datarec_canonicalizeFrags(bxhfragp * infragps, int numfrags)
{
    int fragnum = 0;
    
    for (fragnum = 0; fragnum < numfrags-1; fragnum++) {
	bxhfragp curfragp = infragps[fragnum];
	bxhfragp nextfragp = infragps[fragnum+1];
	if (strcmp(curfragp->filename, nextfragp->filename) != 0)
	    continue;
	if (curfragp->fileoffset + curfragp->fragsize !=
	    nextfragp->fileoffset)
	    continue;
	curfragp->fragsize += nextfragp->fragsize;
	free(nextfragp->filename);
	free(nextfragp);
	memmove(&infragps[fragnum+1], &infragps[fragnum+2],
		sizeof(bxhfragp) * (numfrags - (fragnum + 2)));
	fragnum--; /* restart at this one */
	numfrags--;
    }

    return numfrags;
}

#undef FUNC
#define FUNC "bxh_datarec_getFilenames"
/**
 * Constructs an array of filenames described in the document.
 * The number of files is stored in the location pointed to by
 * numfilesp.  All filenames and array should be freed by caller.
 *
 * @param datarec raw datarec struct
 * @param numfilesp number of files returned
 * @return new array of file names, or NULL on error.
 */
static
char **
bxh_datarec_getFilenames(bxhrawdatarec * datarec, BXHElementPtr elemp, int * numfilesp)
{
    BXHElementPtr * filenamenodes = NULL;
    int numfiles = 0;
    char **filenames = NULL;
    BXHElementPtr tmpnode = NULL;
    BXHElementPtr * tmpnodearray = NULL;

    /* get list of filenames */
    if ((filenamenodes = bxh_getChildElementArray(elemp, "filename")) == NULL)
	goto EXIT;
    for (numfiles = 0; filenamenodes[numfiles]; numfiles++) { /* null */ }
    filenames = NULL;
    if ((tmpnodearray = bxh_getChildElementArray(elemp, "filenameprintfdimensions")) != NULL) {
	char * tmplt = NULL;
	int * pdims = NULL;
	int * sortedpdims = NULL;
	int * porigins = NULL;
	int * newdimsizes = NULL;
	int dimnum;
	int numpdims;
	int numporigins;
	int * counters = NULL;
	char * tmpstr = NULL;
	int numrepls;
	int numfpds;
	BXHElementPtr fpd = NULL;
	int failed = 0;

	for (numfpds = 0; tmpnodearray[numfpds]; numfpds++) { /* null */ }
	if (numfpds > 1) {
	    fprintf(stderr, "Only one instance of 'filenameprintfdimensions' allowed!\n");
	    goto FPDFAIL;
	}
	fpd = tmpnodearray[0];
	free(tmpnodearray); tmpnodearray = NULL;
	if (numfiles > 1) {
	    fprintf(stderr, "Only one instance of 'filename' allowed when using 'filenameprintfdimensions'!\n");
	    goto FPDFAIL;
	}
	if (bxh_getElementStringValue(filenamenodes[0], &tmplt) == -1)
	    goto FPDFAIL;
	
	if (datarec->numdims > 10) {
	    fprintf(stderr, "More than 10 dimensions not supported with filenameprintfdimensions!\n");
	    goto FPDFAIL;
	}
	if ((numpdims = bxh_getElementIntListValue(fpd, &pdims)) == -1)
	    goto FPDFAIL;
	if (numpdims > 10) {
	    fprintf(stderr, "More than 10 dimensions not supported with filenameprintfdimensions!\n");
	    goto FPDFAIL;
	}
	pdims = (int *)realloc(pdims, sizeof(int) * 10);
	memset(pdims + numpdims, '\0', sizeof(int) * (10 - numpdims));
	tmpnode = bxh_getChildElement(elemp, "filenameprintforigins");
	numporigins = bxh_getElementIntListValue(tmpnode, &porigins);
	bxh_element_unref(tmpnode); tmpnode = NULL;
	if (numporigins != 0 && numporigins != numpdims) {
	    fprintf(stderr, "list in filenameprintforigins must have same number of elements\nas filenameprintfdimensions!\n");
	    goto FPDFAIL;
	}
	/* correct for indexing of printfdimensions (start at 0) */
	{
	    int pdnum;
	    for (pdnum = 0; pdnum < numpdims; pdnum++) {
		pdims[pdnum]--;
	    }
	}
	numrepls = 0;
	for (tmpstr = tmplt; *tmpstr; tmpstr++) {
	    if (*tmpstr != '%')
		continue;
	    tmpstr++;
	    tmpstr += strspn(tmpstr, "#0- +"); /* flags */
	    if (*tmpstr == '0')
		continue; /* error and exit? */
	    tmpstr += strspn(tmpstr, "0123456789"); /* field width */
	    if (*tmpstr == '.') {
		tmpstr++;
		tmpstr += strspn(tmpstr, "0123456789");
	    }
	    if (*tmpstr != 'd') {
		fprintf(stderr, "Only simple %%d conversions are supported in filename!\n");
		goto FPDFAIL;
	    }
	    numrepls++;
	}
	if (numrepls != numpdims) {
	    fprintf(stderr, "Number of %%d conversions in filename does not equal number of\nfilenameprintfdimensions!\n");
	    goto FPDFAIL;
	}
	newdimsizes = (int *)malloc(sizeof(int)*datarec->numdims);
	memset(newdimsizes, '\0', sizeof(int)*datarec->numdims);
	/* number of filenames */
	numfiles = 1;
	{
	    int i, j;
	    int slowestpdim = -1;
	    for (i = 0; i < numpdims; i++) {
		if (pdims[i] > slowestpdim)
		    slowestpdim = pdims[i];
	    }
	    /* these are the dimensions in filenameprintfdimensions */
	    for (i = 0; i < numpdims; i++) {
		newdimsizes[pdims[i]] = datarec->dimensions[pdims[i]].size;
		numfiles *= newdimsizes[pdims[i]];
	    }
	    /* these are the slower-moving dimensions that will also
	     * take up filenames */
	    for (j = 0; j < datarec->numdims; j++) {
		for (i = 0; i < numpdims; i++) {
		    if (pdims[i] >= j)
			break;
		}
		if (i == numpdims) {
		    numfiles *= datarec->dimensions[j].size;
		    newdimsizes[slowestpdim] *= datarec->dimensions[j].size;
		}
	    }
	}
	filenames = (char **)malloc(sizeof(char *) * (numfiles+1));
	memset(filenames, '\0', sizeof(char *) * (numfiles+1));
	{
	    static char buf[4096];
	    int pdnum;
	    int filenum;

	    /* bubble-sort printf dimensions */
	    sortedpdims = (int *)malloc(sizeof(int)*numpdims);
	    memcpy(sortedpdims, pdims, sizeof(int)*numpdims);
	    {
		int i, j;
		for (i = 0; i < numpdims; i++) {
		    for (j = i + 1; j < numpdims; j++) {
			if (sortedpdims[j] < sortedpdims[i]) {
			    int swap = sortedpdims[j];
			    sortedpdims[j] = sortedpdims[i];
			    sortedpdims[i] = swap;
			}
		    }
		}
	    }

	    /* initialize counters */
	    counters = (int *)malloc(sizeof(int) * datarec->numdims);
	    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
		counters[dimnum] = 0;
	    }
	    for (pdnum = 0; pdnum < numpdims; pdnum++) {
		counters[pdims[pdnum]] = porigins[pdnum];
	    }
	    if (strlen(tmplt) > sizeof(buf) / 2) {
		fprintf(stderr, "Filename '%s' too big!\n", tmplt);
		goto FPDFAIL;
	    }
	    
	    for (filenum = 0; filenum < numfiles; filenum++) {
		int pdnum;
		sprintf(buf, tmplt, counters[pdims[0]], counters[pdims[1]], counters[pdims[2]], counters[pdims[3]], counters[pdims[4]], counters[pdims[5]], counters[pdims[6]], counters[pdims[7]], counters[pdims[8]], counters[pdims[9]]);
		filenames[filenum] = strdup(&buf[0]);
		/* increment fastest dimension first */
		counters[sortedpdims[0]]++;
		/* bump remaining slower dimensions */
		pdnum = 0;
		while (pdnum < (numpdims - 1) &&
		       counters[sortedpdims[pdnum]] >= newdimsizes[sortedpdims[pdnum]] + porigins[pdnum]) {
		    counters[sortedpdims[pdnum]] = porigins[pdnum];
		    pdnum++;
		    counters[sortedpdims[pdnum]]++;
		}
	    }
	}
	goto FPDEXIT;

    FPDFAIL:
	if (domutil_errorbuf[0])
	    fprintf(stderr, "%s", domutil_errorbuf);
	failed = 1;

    FPDEXIT:
	if (tmpnodearray) {
	    int i;
	    for (i = 0; tmpnodearray[i]; i++) {
		bxh_element_unref(tmpnodearray[i]);
	    }
	    free(tmpnodearray);
	}
	if (fpd) bxh_element_unref(fpd);
	if (newdimsizes) free(newdimsizes);
	if (counters) free(counters);
	if (sortedpdims) free(sortedpdims);
	if (pdims) free(pdims);
	if (porigins) free(porigins);
	if (tmplt) free(tmplt);
	if (failed)
	    goto FAIL;
    } else {
	int i;
	filenames = (char **)malloc(sizeof(char *) * (numfiles + 1));
	for (i = 0; i < numfiles; i++) {
	    if ((bxh_getElementStringValue(filenamenodes[i], &filenames[i])) == -1)
		goto FAIL;
	}
    }
    filenames[numfiles] = NULL;
    if (numfilesp)
	*numfilesp = numfiles;
    goto EXIT;

FAIL:
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", &domutil_errorbuf[0]);
    }
    if (filenames) {
	int filenum;
	for (filenum = 0; filenum < numfiles; filenum++) {
	    if (filenames[filenum])
		free(filenames[filenum]);
	}
	free(filenames);
	filenames = NULL;
    }

EXIT:
    if (filenamenodes) {
	int i;
	for (i = 0; filenamenodes[i]; i++) {
	    bxh_element_unref(filenamenodes[i]);
	}
	free(filenamenodes);
    }
    if (tmpnode)
	bxh_element_unref(tmpnode);
    return filenames;
}

#undef FUNC
#define FUNC "bxh_datarec_fillFrags"
/**
 * Constructs an array of bxhfrag pointers, as described by the given
 * datarec element, and stores it in the raw datarec structure.
 * All frags and array should be freed by caller.
 *
 * @param datarec raw datarec struct
 * @param elemp BXH datarec element pointer
 * @return 0 on success, non-zero on error.
 */
static
int
bxh_datarec_fillFrags(bxhrawdatarec * datarec, BXHElementPtr elemp)
{
    int dimnum;
    int numfiles;
    char ** filenames = NULL;
    BXHElementPtr * dataoffsetnodes = NULL;
    BXHElementPtr * datasizenodes = NULL;
    int numdataoffsets = -1;
    int numdatasizes = 0;
    int retval = 0;

    bxhfragp * frags = NULL;
    int numfrags = 0;
    off_t totaldatasize = 0;

    filenames = bxh_datarec_getFilenames(datarec, elemp, &numfiles);

    /* calculate total size of data */
    totaldatasize = 1;
    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	totaldatasize *= datarec->dimensions[dimnum].size;
    }
    totaldatasize *= datarec->elemsize;

    if (filenames == NULL)
	goto FINISHEDFILES;

    dataoffsetnodes = bxh_getChildElementArray(elemp, "fileoffset");
    if (dataoffsetnodes) {
	for (numdataoffsets = 0; dataoffsetnodes[numdataoffsets]; numdataoffsets++) { /* null */ }
    }
    datasizenodes = bxh_getChildElementArray(elemp, "filerecordsize");
    if (datasizenodes) {
	for (numdatasizes = 0; datasizenodes[numdatasizes]; numdatasizes++) { /* null */ }
    }
    
    if (dataoffsetnodes != NULL && numdataoffsets != 1 && numdataoffsets != numfiles) {
	fprintf(stderr, "Number of fileoffset elements is wacky!\n");
	goto FAIL;
    }
    if (datasizenodes != NULL && numdatasizes != 1 && numdatasizes != numfiles) {
	fprintf(stderr, "Number of recordsize elements is wacky!\n");
	goto FAIL;
    }
    
    /* now construct frag list */
    {
	int filenum;
	int fragnum;
	frags = NULL;
	for (filenum = 0; filenum < numfiles; filenum++) {
	    long long int * offsets = NULL;
	    long long int * sizes = NULL;
	    int numoffsets;
	    int numsizes;
	    int offsetnum;
	    if (numdataoffsets == -1) {
		numoffsets = 1;
		offsets = (long long int *)malloc(sizeof(long long int) * 1);
		offsets[0] = 0;
	    } else {
		BXHElementPtr ctxt = NULL;
		if (numdataoffsets == 1)
		    ctxt = dataoffsetnodes[0];
		else
		    ctxt = dataoffsetnodes[filenum];
		numoffsets = bxh_getElementLongLongIntListValue(ctxt, &offsets);
		if (numoffsets == -1) {
		    offsets = (long long int *)malloc(sizeof(long long int)*1);
		    offsets[0] = 0;
		    numoffsets = 1;
		}
	    }
	    numsizes = -1;
	    if (datasizenodes != NULL) {
		BXHElementPtr ctxt = NULL;
		if (numdatasizes == 1)
		    ctxt = datasizenodes[0];
		else
		    ctxt = datasizenodes[filenum];
		if ((numsizes = bxh_getElementLongLongIntListValue(ctxt, &sizes)) == -1) {
		    fprintf(stderr, "%s", domutil_errorbuf);
		    free(offsets);
		    goto FAIL;
		}
	    }
	    if (numsizes != -1 && numsizes != 1 && numsizes != numoffsets) {
		fprintf(stderr, "datasize and dataoffset fields don't have matching number of elements!\n");
		free(offsets);
		if (sizes) free(sizes);
		goto FAIL;
	    }
	    frags = (bxhfragp *)realloc(frags, sizeof(bxhfragp) * (numfrags + numoffsets));
	    for (offsetnum = 0; offsetnum < numoffsets; offsetnum++) {
		fragnum = numfrags;
		frags[fragnum] = (bxhfragp)malloc(sizeof(bxhfrag));
		frags[fragnum]->filename = strdup(filenames[filenum]);
		frags[fragnum]->fileoffset = offsets[offsetnum];
		numfrags++;
		if (numsizes != -1) {
		    if (numsizes == 1)
			frags[fragnum]->fragsize = sizes[0];
		    else
			frags[fragnum]->fragsize = sizes[offsetnum];
		} else {
		    struct stat statbuf;
		    char * fullfilename = NULL;
		    if ((fullfilename = bxh_datarec_getfullfilename(datarec->bxhpath, filenames[filenum])) == NULL ||
			stat(fullfilename, &statbuf) == -1) {
			char * fullfilenamegz = NULL;
			int fd = -1;
			unsigned char tmpbuf[4];
			fullfilenamegz = (char *)malloc(sizeof(char)*strlen(fullfilename) + 4);
			strcpy(fullfilenamegz, fullfilename);
			strcat(fullfilenamegz, ".gz");
			if (stat(fullfilenamegz, &statbuf) == -1 ||
			    (fd = open(fullfilenamegz, O_RDONLY)) == -1 ||
			    lseek(fd, -4, SEEK_END) == -1 ||
			    read(fd, (char *)&tmpbuf[0], 4) != 4) {
			    fprintf(stderr, "Error accessing file '%s' to find its size\n", filenames[filenum]);
			    perror("stat");
			    if (fullfilename)
				free(fullfilename);
			    free(offsets);
			    free(sizes);
			    if (fd != -1) {
				close(fd);
			    }
			    goto FAIL;
			}
			statbuf.st_size =
			    ((tmpbuf[3] << 24) |
			     (tmpbuf[2] << 16) |
			     (tmpbuf[1] << 8) |
			     tmpbuf[0]);
			free(fullfilenamegz); fullfilenamegz = NULL;
		    }
		    free(fullfilename); fullfilename = NULL;
		    frags[fragnum]->fragsize = statbuf.st_size - offsets[offsetnum];
		}
	    }
	    free(offsets);
	    free(sizes);
	}
    }

FINISHEDFILES:
    /* validate size */
    {
	int fragnum;
	off_t calcsize = 0;
	for (fragnum = 0; fragnum < numfrags; fragnum++) {
	    calcsize += frags[fragnum]->fragsize;
	}
	if (calcsize != totaldatasize) {
	    fprintf(stderr, "Sizes of frags (%lu) and dimensions (%lu) don't match!\n", (unsigned long)calcsize, (unsigned long) totaldatasize);
	    if (filenames != NULL) {
	      goto FAIL;
	    }
	}
	datarec->datasize = totaldatasize;
    }

    datarec->numfrags = numfrags;
    datarec->frags = frags;
    goto EXIT;

FAIL:
    if (frags) {
	int i;
	for (i = 0; i < numfrags; i++) {
	    if (frags[i]) {
		free(frags[i]->filename);
		free(frags[i]);
	    }
	}
	free(frags);
    }
    frags = NULL;
    numfrags = 0;
    retval = -1;

EXIT:
    if (filenames) {
	int filenum;
	for (filenum = 0; filenum < numfiles; filenum++)
	    free(filenames[filenum]);
	free(filenames);
    }
    if (dataoffsetnodes) {
	int i;
	for (i = 0; dataoffsetnodes[i]; i++)
	    bxh_element_unref(dataoffsetnodes[i]);
	free(dataoffsetnodes);
    }
    if (datasizenodes) {
	int i;
	for (i = 0; datasizenodes[i]; i++)
	    bxh_element_unref(datasizenodes[i]);
	free(datasizenodes);
    }
    return retval;
}

#undef FUNC
#define FUNC "bxh_datarec_fillParams"
/**
 * Populate raw datarec struct with info from the document, such as
 * dimensions, byteorder, etc.
 *
 * @param datarec raw datarec struct
 * @param elemp pointer to BXH element
 */
static
int
bxh_datarec_fillParams(bxhrawdatarec * datarec, BXHElementPtr elemp)
{
    BXHElementPtr * dimps = NULL;
    BXHElementPtr rasorigin = NULL;
    int numdims = 0;
    int i;
    int retval = 0;

    if ((dimps = bxh_getChildElementArray(elemp, "dimension")) == NULL)
	goto FAIL;
    for (numdims = 0; dimps[numdims]; numdims++) { /* null */ }
    datarec->numdims = numdims;
    datarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension) * (numdims+1));
    for (i = 0; i < numdims; i++) {
	BXHElementPtr dimnode = dimps[i];
	BXHElementPtr tmpnode = NULL;
	bxhdimension * dim;
	BXHElementPtr * dpnodes = NULL;
	int dpnodeind = 0;
	char * tmpstr = NULL;
	dim = &datarec->dimensions[i];
	memset(dim, '\0', sizeof(bxhdimension));
	if ((dim->type = bxh_getAttributeStringValue(dimnode, "type")) == NULL)
	    goto FAIL;
	if ((tmpnode = bxh_getChildElement(dimnode, "units")) != NULL) {
	    bxh_element_unref(tmpnode); tmpnode = NULL;
	    dim->units = bxh_getChildElementStringValue(dimnode, "units");
	} else {
	    /* fill in default units if known */
	    if ((dim->type[0] == 'x' || dim->type[0] == 'y' || dim->type[0] == 'z') && (dim->type[1] == '\0' || strncmp(&dim->type[1], "-split", 6))) {
		dim->units = strdup("mm");
	    } else if (dim->type[0] == 't' && (dim->type[1] == '\0' || strncmp(&dim->type[1], "-split", 6))) {
		dim->units = strdup("ms");
	    }
	}
	if ((tmpnode = bxh_getChildElement(dimnode, "direction")) != NULL) {
	    double * tmpdirection = NULL;
	    if (bxh_getElementDoubleListValue(tmpnode, &tmpdirection) != 3) {
		bxh_element_unref(tmpnode);
		if (tmpdirection)
		    free(tmpdirection);
		fprintf(stderr, FUNC ": direction field does not have three (R,A,S) coordinates\n");
		goto FAIL;
	    }
	    dim->direction = (double *)malloc(sizeof(double) * 3);
	    dim->direction[0] = tmpdirection[0];
	    dim->direction[1] = tmpdirection[1];
	    dim->direction[2] = tmpdirection[2];
	    free(tmpdirection);
	    bxh_element_unref(tmpnode); tmpnode = NULL;
	}
	if ((tmpnode = bxh_getChildElement(dimnode, "measurementframe")) != NULL) {
	    BXHElementPtr * vectorelems = NULL;
	    char * mfversion = NULL;
	    int numvectors = 0;
	    double * vector1 = NULL;
	    double * vector2 = NULL;
	    double * vector3 = NULL;
	    if ((vectorelems = bxh_getChildElementArray(tmpnode, "vector")) == NULL)
		goto FAIL;
	    numvectors = 0;
	    while (vectorelems[numvectors] != NULL) {
		numvectors++;
	    }
	    if (numvectors != 3) {
		int vnum;
		fprintf(stderr, FUNC ": ERROR: wrong number of vectors in measurementframe element\n");
		for (vnum = 0; vnum < numvectors; vnum++) {
		    if (vectorelems[vnum]) {
			bxh_element_unref(vectorelems[vnum]);
			vectorelems[vnum] = NULL;
		    }
		}
		free(vectorelems);
		goto FAIL;
	    }
	    dim->measurementframe = (double *)malloc(sizeof(double)*9);
	    if (bxh_getElementDoubleListValue(vectorelems[0], &vector1) != 3 ||
		bxh_getElementDoubleListValue(vectorelems[1], &vector2) != 3 ||
		bxh_getElementDoubleListValue(vectorelems[2], &vector3) != 3) {
		bxh_element_unref(vectorelems[0]);
		bxh_element_unref(vectorelems[1]);
		bxh_element_unref(vectorelems[2]);
		free(vectorelems);
		free(tmpnode);
		if (vector1) free(vector1);
		if (vector2) free(vector2);
		if (vector3) free(vector3);
		fprintf(stderr, FUNC ": measurement frame vectors do not have three values each\n");
		goto FAIL;
	    }
	    dim->measurementframe[0] = vector1[0];
	    dim->measurementframe[1] = vector1[1];
	    dim->measurementframe[2] = vector1[2];
	    dim->measurementframe[3] = vector2[0];
	    dim->measurementframe[4] = vector2[1];
	    dim->measurementframe[5] = vector2[2];
	    dim->measurementframe[6] = vector3[0];
	    dim->measurementframe[7] = vector3[1];
	    dim->measurementframe[8] = vector3[2];
	    free(vector1);
	    free(vector2);
	    free(vector3);
	    bxh_element_unref(vectorelems[0]); vectorelems[0] = NULL;
	    bxh_element_unref(vectorelems[1]); vectorelems[1] = NULL;
	    bxh_element_unref(vectorelems[2]); vectorelems[2] = NULL;
	    free(vectorelems);

	    mfversion = bxh_getAttributeStringValue(tmpnode, "version");
	    if (mfversion != NULL) {
		dim->measurementframeversion = mfversion;
	    }

	    bxh_element_unref(tmpnode); tmpnode = NULL;
	}
	if ((tmpstr = bxh_getChildElementStringValue(dimnode, "size")) == NULL)
	    goto FAIL;
	dim->size = strtoul(tmpstr, NULL, 10);
	free(tmpstr); tmpstr = NULL;
	if ((tmpstr = bxh_getAttributeStringValue(dimnode, "outputselect")) != NULL && tmpstr[0] != '\0') {
	    unsigned long sel;
	    char * curptr = NULL;
	    char * endptr = NULL;
	    curptr = tmpstr;
	    dim->outputselect = NULL;
	    dim->numoutputselect = 0;
	    while (!((sel = strtoul(curptr, &endptr, 10)) == ULONG_MAX &&
		     errno == ERANGE) &&
		   endptr != curptr) {
		dim->numoutputselect++;
		dim->outputselect = (int *)realloc(dim->outputselect, sizeof(int) * dim->numoutputselect);
		dim->outputselect[dim->numoutputselect-1] = sel;
		curptr = endptr;
	    }
	}
	free(tmpstr); tmpstr = NULL;
	dpnodes = bxh_getChildElementArray(dimnode, "datapoints");
	
	for (dpnodeind = 0;
	     dpnodes != NULL && dpnodes[dpnodeind] != NULL;
	     dpnodeind++) {
	    tmpnode = dpnodes[dpnodeind];
	    unsigned int numdatapoints;
	    char * label = NULL;
	    BXHElementPtr * valuenodes = NULL;
	    label = bxh_getAttributeStringValue(tmpnode, "label"); /* NULL is OK */
	    if ((valuenodes = bxh_getChildElementArray(tmpnode, "value")) != NULL) {
		unsigned int dpnum;
		for (numdatapoints = 0; valuenodes[numdatapoints]; numdatapoints++) { /* null */ }
		dim->dpstructs = (bxhdatapoints *)realloc(dim->dpstructs, sizeof(bxhdatapoints) * (dim->numdpstructs + 1));
		bxhdatapoints * datapoints = &dim->dpstructs[dim->numdpstructs];
		dim->numdpstructs++;
		datapoints->label = label;
		datapoints->values = (char **)malloc(sizeof(char *) * numdatapoints);
		datapoints->numvalues = numdatapoints;
		for (dpnum = 0; dpnum < numdatapoints; dpnum++) {
		    if (bxh_getElementStringValue(valuenodes[dpnum], &datapoints->values[dpnum]) != 0) {
			goto FAIL;
		    }
		}
		for (dpnum = 0; dpnum < numdatapoints; dpnum++) {
		    bxh_element_unref(valuenodes[dpnum]);
		    valuenodes[dpnum] = NULL;
		}
		free(valuenodes);
	    } else {
		char * value = NULL;
		char * valpos = NULL;
		if (bxh_getElementStringValue(tmpnode, &value) != 0) {
		    goto FAIL;
		}
		dim->dpstructs = (bxhdatapoints *)realloc(dim->dpstructs, sizeof(bxhdatapoints) * (dim->numdpstructs + 1));
		bxhdatapoints * datapoints = &dim->dpstructs[dim->numdpstructs];
		memset(&dim->dpstructs[dim->numdpstructs], 0, sizeof(*datapoints));
		dim->numdpstructs++;
		datapoints->label = label;
		numdatapoints = 0;
		valpos = value;
		do {
		    char * laststart = valpos;
		    while (*valpos && isspace(*valpos)) { valpos++; }
		    laststart = valpos;
		    while (*valpos && !isspace(*valpos)) { valpos++; }
		    if (laststart != valpos) {
			if (*valpos) {
			    *valpos = '\0';
			    valpos++;
			}
			datapoints->values = (char **)realloc(datapoints->values, sizeof(char *) * (numdatapoints + 1));
			datapoints->values[numdatapoints] = strdup(laststart);
			numdatapoints++;
		    }
		} while (*valpos);
		free(value);
		datapoints->numvalues = numdatapoints;
		if (dim->size == 0) {
		    dim->size = numdatapoints;
		}
	    }
	    bxh_element_unref(tmpnode); tmpnode = NULL;
	    {
		unsigned long dimsize = dim->size;
		if (dim->outputselect != NULL) {
		    dimsize = dim->numoutputselect;
		}
		if (numdatapoints != dimsize) {
		    fprintf(stderr, FUNC ": WARNING: ignoring datapoints element '%s' because it has %u elements (should have %u)\n", label, (unsigned int)numdatapoints, (unsigned int)dim->size);
		    dim->numdpstructs--;
		    bxh_datarec_datapoints_free(&dim->dpstructs[dim->numdpstructs]);
		}
	    }
	}
	if (dpnodes != NULL) {
	  free(dpnodes); dpnodes = NULL;
	}
	/* the following are optional values (i.e. they have defaults) */
	dim->origin = 0;
	if ((tmpstr = bxh_getChildElementStringValue(dimnode, "origin")) != NULL) {
	    dim->origin = strtod(tmpstr, NULL);
	    free(tmpstr); tmpstr = NULL;
	}
	dim->gap = 0;
	if ((tmpstr = bxh_getChildElementStringValue(dimnode, "gap")) != NULL) {
	    dim->gap = strtod(tmpstr, NULL);
	    free(tmpstr); tmpstr = NULL;
	}
	dim->spacing = 1;
	if ((tmpstr = bxh_getChildElementStringValue(dimnode, "spacing")) != NULL) {
	    dim->spacing = strtod(tmpstr, NULL);
	    free(tmpstr); tmpstr = NULL;
	}
    }
    for (i = 0; dimps[i]; i++) { bxh_element_unref(dimps[i]); }
    free(dimps);

    if ((rasorigin = bxh_getChildElement(elemp, "rasorigin")) != NULL) {
	double * rasoriginvals = NULL;
	if (bxh_getElementDoubleListValue(rasorigin, &rasoriginvals) == 3) {
	    bxhdimension * dimx = NULL;
	    bxhdimension * dimy = NULL;
	    bxhdimension * dimz = NULL;
	    bxhdimension * dimr = NULL;
	    bxhdimension * dima = NULL;
	    bxhdimension * dims = NULL;
	    int dimnum;
	    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
		bxhdimension * dimp = &datarec->dimensions[dimnum];
		if (dimp->direction && dimp->type) {
		    if (strcmp(dimp->type, "x") == 0) {
			dimx = dimp;
		    } else if (strcmp(dimp->type, "y") == 0) {
			dimy = dimp;
		    } else if (strcmp(dimp->type, "z") == 0) {
			dimz = dimp;
		    }
		}
	    }
	    if (dimx && dimy && dimz) {
		double Xr = dimx->direction[0];
		double Xa = dimx->direction[1];
		/*double Xs = dimx->direction[2];*/
		double Yr = dimy->direction[0];
		double Ya = dimy->direction[1];
		/*double Ys = dimy->direction[2];*/
		double Zr = dimz->direction[0];
		double Za = dimz->direction[1];
		/*double Zs = dimz->direction[2];*/
		if (fabs(Xr) > fabs(Yr) && fabs(Xr) > fabs(Zr)) {
		    dimr = dimx;
		    if (fabs(Ya) > fabs(Xa) && fabs(Ya) > fabs(Za)) {
			dima = dimy; dims = dimz;
		    } else {
			dima = dimz; dims = dimy;
		    }
		} else if (fabs(Yr) > fabs(Xr) && fabs(Yr) > fabs(Zr)) {
		    dimr = dimy;
		    if (fabs(Xa) > fabs(Ya) && fabs(Xa) > fabs(Za)) {
			dima = dimx; dims = dimz;
		    } else {
			dima = dimz; dims = dimx;
		    }
		} else {
		    dimr = dimz;
		    if (fabs(Xa) > fabs(Ya) && fabs(Xa) > fabs(Za)) {
			dima = dimx; dims = dimy;
		    } else {
			dima = dimy; dims = dimx;
		    }
		}
	    }
	    if (dimr && dima && dims) {
		dimr->origin = rasoriginvals[0];
		dima->origin = rasoriginvals[1];
		dims->origin = rasoriginvals[2];
	    }
	    free(rasoriginvals);
	}
	bxh_element_unref(rasorigin);
    }

    {
	char * elementtypestr = NULL;
	const char * typestr = NULL;
	int typenum;

	if ((elementtypestr = bxh_getChildElementStringValue(elemp, "elementtype")) == NULL)
	    goto FAIL;
	for (typenum = 0, typestr = elemtypestrs[0];
	     typestr;
	     typenum++, typestr = elemtypestrs[typenum]) {
	    if (strcmp(elementtypestr, typestr) == 0) {
		datarec->elemsize = elemtypesizes[typenum];
		break;
	    }
	}
	if (typestr == NULL) {
	    fprintf(stderr, "elementtype '%s' not supported!\n", elementtypestr);
	    free(elementtypestr);
	    goto FAIL;
	}
	datarec->elemtype = strdup(typestr);
	free(elementtypestr);
    }

    {
	char * byteorderstr = NULL;
	if ((byteorderstr = bxh_getChildElementStringValue(elemp, "byteorder")) == NULL)
	    goto FAIL;
	if (strcmp(byteorderstr, "msbfirst") == 0)
	    datarec->msbfirstfrags = 1;
	else if (strcmp(byteorderstr, "lsbfirst") == 0)
	    datarec->msbfirstfrags = 0;
	else {
	    fprintf(stderr, "byteorder '%s' not supported!\n", byteorderstr);
	    free(byteorderstr);
	    goto FAIL;
	}
	free(byteorderstr);
    }

    {
	char * compression = NULL;
	if ((compression = bxh_getChildElementStringValue(elemp, "compression")) == NULL)
	    goto FAIL;
	if (strcmp(compression, "") == 0)
	    datarec->compression = NULL;
	else if (strcmp(compression, "gzip") == 0 || strcmp(compression, "gzipfrag") == 0)
	    datarec->compression = strdup(compression);
	else {
	    fprintf(stderr, "compression type '%s' not supported!\n", compression);
	    free(compression); compression = NULL;
	    goto FAIL;
	}
	free(compression); compression = NULL;
    }

    goto EXIT;

FAIL:
    if (domutil_errorbuf[0])
	fprintf(stderr, "%s", domutil_errorbuf);
    if (datarec->dimensions) {
	int i;
	for (i = 0; i < datarec->numdims; i++) {
	    bxh_datarec_dimdata_free(&datarec->dimensions[i]);
	}
	free(datarec->dimensions);
    }
    datarec->dimensions = NULL;
    datarec->numdims = -1;
    datarec->elemsize = -1;
    retval = -1;

EXIT:
    return retval;
}

#undef FUNC
#define FUNC "do_dim_selection"
/**
 * Modify #dimp to only refer to selected points, given by #dimselect.
 *
 */
static
void
do_dim_selection(bxhdimension * dimp, std::vector<size_t> & dimselect)
{
    size_t selsize = dimselect.size();
    if (selsize == 0)
	return; /* selector is empty for this dimension */
    for (int dpstructind = 0; dpstructind < dimp->numdpstructs; dpstructind++) {
	if (dimp->dpstructs == NULL || dimp->dpstructs[dpstructind].values == NULL) {
	    continue;
	}
	size_t oldindex = 0;
	size_t newindex = 0;
	char ** datapoints = dimp->dpstructs[dpstructind].values;
	char ** newdatapoints = (char **)malloc(sizeof(char *) * selsize);
	for (newindex = 0; newindex < selsize; newindex++) {
	    newdatapoints[newindex] = datapoints[dimselect[newindex]];
	    datapoints[dimselect[newindex]] = NULL;
	}
	for (oldindex = 0; oldindex < dimp->dpstructs[dpstructind].numvalues; oldindex++) {
	    if (datapoints[oldindex]) {
		free(datapoints[oldindex]);
	    }
	}
	dimp->dpstructs[dpstructind].values = newdatapoints;
	free(datapoints);
	dimp->dpstructs[dpstructind].numvalues = selsize;
    }
    dimp->size = selsize;
    if (dimp->origin != 0 || dimp->spacing != 0 || dimp->gap != 0) {
	/* no datapoints, so adjust origin/spacing/gap */
	int evenlyspaced = 1;
	size_t selspacing = 0;
	if (selsize >= 2) {
	    size_t selind;
	    selspacing = dimselect[1] - dimselect[0];
	    for (selind = 2; selind < selsize; selind++) {
		if (dimselect[selind] - dimselect[selind-1] != selspacing) {
		    evenlyspaced = 0;
		    break;
		}
	    }
	}
	if (evenlyspaced) {
	    dimp->origin += dimp->spacing * dimselect[0];
	    if (selsize >= 2) {
		dimp->gap += dimp->spacing * (selspacing - 1);
		dimp->spacing *= selspacing;
	    } else {
		dimp->gap = 0;
		dimp->spacing = 0;
	    }
	} else {
	    dimp->origin = 0;
	    dimp->gap = 0;
	    dimp->spacing = 0;
	}
    }
}

#undef FUNC
#define FUNC "bxh_datarec_postprocess"
/**
 * Create a new datarec that would be returned if all
 * permutations/filters/selectors are applied.
 * New datarec will not have any frags (since it is no longer
 * describing any on-disk data).
 *
 * @param datarec_in raw datarec struct
 * @return new datarec on success, NULL on error
 */
bxhrawdatarec *
bxh_datarec_postprocess(const bxhrawdatarec * datarec_in)
{
    int dimnum;
    readobj * robj = NULL;
    const bxhrawdatarec * datarec = datarec_in;
    bxhrawdatarec * datareccopy = NULL;
    bxhrawdatarec * newdatarec;
    if (datarec_in->readobj == NULL) {
	datareccopy = bxh_datarec_copy(datarec_in);
	datarec = datareccopy;
	bxh_datarec_prepareForRead(datareccopy);
    }
    robj = (readobj *)datarec->readobj;

    /* copy over old datarec to new datarec */
    newdatarec = bxh_datarec_copy(datarec);
    bxh_datarec_frags_free(newdatarec);

    /* do pre-read selection */
    {
	std::vector<std::vector<size_t> > & prereadselectors =
	    robj->prereadselectors;
	if (!prereadselectors.empty()) {
	    for (dimnum = 0; dimnum < newdatarec->numdims; dimnum++) {
		bxhdimension * dimp = &newdatarec->dimensions[dimnum];
		std::vector<size_t> & dimselect = prereadselectors[dimnum];
		do_dim_selection(dimp, dimselect);
	    }
	}
    }

    /* permute dimensions */
    if (!robj->permuteorder.empty()) {
	int i;
	bxhdimension * newdims = NULL;
	std::vector<int> & permuteorder = robj->permuteorder;
	newdims = (bxhdimension *)malloc(sizeof(bxhdimension)*newdatarec->numdims);
	for (i = 0; i < newdatarec->numdims; i++) {
	    memcpy(&newdims[i], &newdatarec->dimensions[permuteorder[i]], sizeof(bxhdimension));
	}
	free(newdatarec->dimensions);
	newdatarec->dimensions = newdims;
    }

    /* Fix dimension names if needed */
    {
	std::vector<size_t> & permtypelen = robj->permtypelen;
	for (dimnum = 0; dimnum < newdatarec->numdims; dimnum++) {
	    if (permtypelen[dimnum] != 0) {
		newdatarec->dimensions[dimnum].type[permtypelen[dimnum]] = '\0';
	    }
	}
    }

    /* delete the redundant dimensions by copying to new dimension array */
    {
	bxhdimension * newdims = NULL;
	int newnumdims = 0;
	std::vector<int> & permdeletethese = robj->permdeletethese;
	newdims = (bxhdimension *)malloc(sizeof(bxhdimension) * newdatarec->numdims);
	for (dimnum = 0; dimnum < newdatarec->numdims; dimnum++) {
	    bxhdimension * dimp = &newdatarec->dimensions[dimnum];
	    if (permdeletethese[dimnum]) {
		bxh_datarec_dimdata_free(dimp);
	    } else {
		memcpy(&newdims[newnumdims], &newdatarec->dimensions[dimnum], sizeof(bxhdimension));
		newnumdims++;
	    }
	}
	free(newdatarec->dimensions);
	newdatarec->dimensions = newdims;
	newdatarec->numdims = newnumdims;
    }

    /* do post-read selection */
    {
	std::vector<std::vector<size_t> > & postreadselectors =
	    robj->postreadselectors;
	if (!postreadselectors.empty()) {
	    for (dimnum = 0; dimnum < newdatarec->numdims; dimnum++) {
		bxhdimension * dimp = &newdatarec->dimensions[dimnum];
		std::vector<size_t> & dimselect = postreadselectors[dimnum];
		do_dim_selection(dimp, dimselect);
	    }
	}
    }

    /* copy the new sizes */
    for (dimnum = 0; dimnum < newdatarec->numdims; dimnum++) {
	newdatarec->dimensions[dimnum].size = robj->outputnewsizes[dimnum];
    }

    newdatarec->datasize = newdatarec->elemsize;
    for (dimnum = 0; dimnum < newdatarec->numdims; dimnum++) {
	newdatarec->datasize *= newdatarec->dimensions[dimnum].size;
    }

    /* remove output selects */
    for (dimnum = 0; dimnum < newdatarec->numdims; dimnum++) {
	if (newdatarec->dimensions[dimnum].outputselect) {
	    free(newdatarec->dimensions[dimnum].outputselect);
	    newdatarec->dimensions[dimnum].outputselect = NULL;
	    newdatarec->dimensions[dimnum].numoutputselect = 0;
	}
    }

    if (datareccopy)
	bxh_datarec_free(datareccopy);
    
    return newdatarec;
}

#undef FUNC
#define FUNC "bxh_datarec_prepareForRead"
/**
 * Prepare rawdatarec for reading.
 * Modifies input datarec to store some info (used to
 * actually read data in a later pass).
 * Also prepares for mergeing/reordering "split" dimensions (i.e. those
 * named specialdimname-split1, specialdimname-split2, etc.).
 *
 * @param datarec raw datarec struct
 * @return 0 on success, non-zero on error
 */
int
bxh_datarec_prepareForRead(bxhrawdatarec * datarec)
{
    return bxh_datarec_prepareForReadPermuteSelect(datarec, NULL, 0, NULL);
}

#undef FUNC
#define FUNC "bxh_datarec_prepareForReadPermute"
/**
 * Prepare rawdatarec to have dimensions permuted to be in order
 * given by #ordereddimnames.
 * Modifies input datarec to store permutation info (used to
 * actually read data in a later pass).
 * Any existing dimensions not specified in ordereddimnames list are
 * moved to end of dimension list.
 * Also prepares for mergeing/reordering "split" dimensions (i.e. those
 * named specialdimname-split1, specialdimname-split2, etc.).
 *
 * @param datarec raw datarec struct
 * @param orddimnames list of dimension names, in the order desired for new data.  If NULL, no permutation is done.
 * @param numorddims number of ordereddims sent in ordereddimnames.  Ignored if #ordereddimnames is NULL.
 * @return 0 on success, non-zero on error
 */
int
bxh_datarec_prepareForReadPermute(bxhrawdatarec * datarec, const char ** orddimnames, int numorddims)
{
    return bxh_datarec_prepareForReadPermuteSelect(datarec, orddimnames, numorddims, NULL);
}

#undef FUNC
#define FUNC "bxh_datarec_prepareForReadPermuteSelect"
/**
 * Prepare rawdatarec to have dimensions permuted to be in order
 * given by #ordereddimnames, and selected by indices given in
 * #ordselectors.
 * Modifies input datarec to store permutation info (used to
 * actually read data in a later pass).
 * Any existing dimensions not specified in ordereddimnames list are
 * moved to end of dimension list.
 * Also prepares for mergeing/reordering "split" dimensions (i.e. those
 * named specialdimname-split1, specialdimname-split2, etc.).
 *
 * @param datarec raw datarec struct
 * @param orddimnames list of dimension names, in the order desired for new data.  If NULL, no permutation is done.
 * @param numorddims number of ordereddims sent in ordereddimnames.  Ignored if #ordereddimnames is NULL.
 * @param ordselectors pointer to array of #numorddims elements.  Each element is a pointer to a selector array, containing a list of indices in the given dimension that should be read.  Data corresponding to indices not in these selectors will not be returned on read.  Selector array indices should be monotonically increasing, and be terminated with a -1.  The array of selectors are ordered as given by orddimnames (#selectors[n] corresponds to the dimension named #orddimnames[n]).  If NULL, no selection is done (i.e. all data is returned).
 * @return 0 on success, non-zero on error
 */
int
bxh_datarec_prepareForReadPermuteSelect(bxhrawdatarec * datarec, const char ** orddimnames, int numorddims, int ** ordselectors)
{
    int retval = 0;
    readobj * robj = NULL;

    int dimnum;
    int orderednum;

    if (datarec->readobj) {
	delete ((readobj *)datarec->readobj);
    }
    robj = new readobj;
    datarec->readobj = (void *)robj;

    /* these arrays map a dimension index to a value (index->value) */

    /* new index -> old index (permutation) */
    std::vector<int> & permuteorder = robj->permuteorder;
    /* old1 index -> old2 index: merge old1 into old2 */
    std::vector<int> mergeinto;
    /* old index -> bool: if 1, delete after merge */
    std::vector<int> deletethese;
    /* new1 index -> new2 index: mergeinto permuted */
    std::vector<int> permmergeinto;
    /* new index -> bool: deletethese permuted */
    std::vector<int> & permdeletethese = robj->permdeletethese;
    /* new index -> typelen: length of actual dimension name  */
    std::vector<size_t> & permtypelen = robj->permtypelen;

    std::vector<size_t> & inputnewsizes = robj->inputnewsizes;
    std::vector<size_t> & outputnewsizes = robj->outputnewsizes;
    std::vector<std::vector<size_t> > & prereadselectors = robj->prereadselectors;
    std::vector<std::vector<size_t> > & postreadselectors = robj->postreadselectors;

    /* these are populated per dimension */
    std::vector<int> spinds; /* dimensions that are splits of current dimension */
    std::vector<int> spnums; /* this stores the split number for sorting */

    std::vector<std::string> ordereddimnames;

    mergeinto = std::vector<int>(datarec->numdims, 0);
    deletethese = std::vector<int>(datarec->numdims, 0);
    permmergeinto = std::vector<int>(datarec->numdims, 0);
    permdeletethese = std::vector<int>(datarec->numdims, 0);
    permtypelen = std::vector<size_t>(datarec->numdims, 0);

    /* init mergeinto array (merging a dimension into itself is a no-op) */
    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	mergeinto[dimnum] = dimnum;
    }

    /* first add the dimensions given explicitly by caller */
    if (orddimnames) {
	for (orderednum = 0; orderednum < numorddims; orderednum++) {
	    ordereddimnames.push_back(std::string(orddimnames[orderednum]));
	}
    }
    /* now add dimensions that weren't specified in arguments */
    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	int tmpnum;
	char * splitloc = NULL;
	bxhdimension * dimp = &datarec->dimensions[dimnum];
	std::string dimname(dimp->type);
	/* see if it's a split dimension */
	if ((splitloc = strstr(dimp->type, "-split")) != NULL) {
	    long spnum;
	    char *endptr = NULL;
	    char *nptr = splitloc + 6;
	    spnum = strtol(nptr, &endptr, 10);
	    if (*nptr != '\0' && *endptr == '\0') {
		/* -split is followed by an integer */
		dimname = dimname.substr(0, splitloc - dimp->type);
	    }
	}
	/* check if it's in the list already */
	for (tmpnum = 0; tmpnum < (int)ordereddimnames.size(); tmpnum++) {
	    if (ordereddimnames[tmpnum] == dimname)
		break;
	}
	if (tmpnum == (int)ordereddimnames.size()) {
	    /* not found, add it */
	    ordereddimnames.push_back(dimname);
	}
    }

    for (orderednum = 0; orderednum < (int)ordereddimnames.size(); orderednum++) {
	const char * specname = ordereddimnames[orderednum].c_str();
	/* search for unadulterated dimension */
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    bxhdimension * dimp = NULL;
	    dimp = &datarec->dimensions[dimnum];
	    if (strcmp(dimp->type, specname) == 0) {
		permuteorder.push_back(dimnum);
		permtypelen[permuteorder.size()-1] = strlen(specname);
		break;
	    }
	}

	if (dimnum != datarec->numdims)
	    continue;

	/* pure dim not found, check for split dims */
	spnums.clear();
	spinds.clear();
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    bxhdimension * dimp = NULL;
	    int speclen = strlen(specname);
	    dimp = &datarec->dimensions[dimnum];
	    /* split dim will have type NAME-splitNUM */
	    if (strncmp(dimp->type, specname, speclen) == 0 &&
		strncmp(dimp->type + speclen, "-split", 6) == 0) {
		long spnum;
		char *endptr = NULL;
		char *nptr = dimp->type + speclen + 6;
		spnum = strtol(nptr, &endptr, 10);
		if (*nptr == '\0' || *endptr != '\0') {
		    /* -split isn't followed by an integer */
		    fprintf(stderr, FUNC ": Bad split dimension '%s'\n", dimp->type);
		    goto FAIL;
		}
		spnums.push_back(spnum);
		spinds.push_back(dimnum);
	    }
	}
	if (spnums.empty()) /* no splits for this special dimension */
	    continue;

	/* pre-condition: spinds contains the indexes of dimensions
	 * that are splits of the current dimension.  spnums contains
	 * the "split numbers" (number after "-split") */

	{
	    /* bubble sort split indexes using spnums */
	    int i;
	    for (i = 0; i < (int)spnums.size(); i++) {
		int j;
		for (j = i + 1; j < (int)spnums.size(); j++) {
		    int swap;
		    if (spnums[i] > spnums[j]) {
			swap = spnums[i];
			spnums[i] = spnums[j];
			spnums[j] = swap;
			swap = spinds[i];
			spinds[i] = spinds[j];
			spinds[j] = swap;
		    }
		}
	    }
	}
	{
	    /* add these dimensions to the ordered dimension list,
	     * and indicate that we need to merge all lower-numbered
	     * splits into highest-numbered split */
	    int splitnum;
	    for (splitnum = 0; splitnum < (int)spnums.size(); splitnum++) {
		permuteorder.push_back(spinds[splitnum]);
		permtypelen[permuteorder.size()-1] = strlen(specname);
		if (splitnum < (int)spnums.size() - 1)
		    deletethese[spinds[splitnum]] = 1;
		mergeinto[spinds[splitnum]] = spinds[spnums.size()-1];
	    }
	}
    }

    /* permute the mergeinto and deletethese arrays, so we can apply
     * them after the data and datarec are permuted */
    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	int newmergedim;
	for (newmergedim = 0; newmergedim < datarec->numdims; newmergedim++) {
	    if (permuteorder[newmergedim] == mergeinto[permuteorder[dimnum]])
		break;
	}
	permmergeinto[dimnum] = newmergedim;
	permdeletethese[dimnum] = deletethese[permuteorder[dimnum]];
    }

    /*** init selectors ***/
    {
	int keeppresel = 0;
	int keeppostsel = 0;
	prereadselectors = std::vector<std::vector<size_t> >(datarec->numdims);
	postreadselectors.clear();
	/* Create initial outputselect array from "outputselect" dimensions.
	 * Create it in input order (i.e. before permutation) for now. */
	{
	    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
		int osind;
		std::vector<size_t> os;
		bxhdimension * dimp = NULL;
		dimp = &datarec->dimensions[dimnum];
		if (dimp->numoutputselect)
		    keeppostsel = 1;
		for (osind = 0; osind < (int)dimp->numoutputselect; osind++) {
		    os.push_back(dimp->outputselect[osind]);
		}
		postreadselectors.push_back(os);
	    }
	}

	if (ordselectors) {
	    /* ordselectors is in output order.
	     * make sure to convert to input order */
	    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
		int issplit = 0;
		int ismonotonous = 0;
		const int * curinsel = NULL;
		std::vector<size_t> newsel;
		int pdimnum = -1;
		int selnum = -1;
		for (pdimnum = 0; pdimnum < datarec->numdims; pdimnum++) {
		    if (permuteorder[pdimnum] == dimnum)
			break;
		}
		if (pdimnum == datarec->numdims) {
		    fprintf(stderr, FUNC ": Internal error!\n");
		    goto FAIL;
		}
		selnum = pdimnum;
		if (permdeletethese[pdimnum]) {
		    /* this will be a deleted dimension */
		    continue;
		}
		int tmpdimnum;
		for (tmpdimnum = pdimnum - 1; tmpdimnum >= 0; tmpdimnum--) {
		    if (permdeletethese[tmpdimnum])
			selnum--;
		}
		if (selnum >= numorddims) {
		    /* no selector for this dimension */
		    continue;
		}
		/* selnum now holds the output order dimension number
		 * corresponding to dimnum */
		curinsel = ordselectors[selnum];
		while (*curinsel != -1) {
		    newsel.push_back((size_t)*curinsel);
		    curinsel++;
		}
		if (newsel.empty())
		    continue;
		/* We have a selector corresponding to input dimension dimnum.
		 * Find out if we can apply it on input (we can't if this is a
		 * split dimension or if it is non-monotonously increasing). */
		for (tmpdimnum = 0; tmpdimnum < datarec->numdims; tmpdimnum++) {
		    if (permmergeinto[tmpdimnum] == pdimnum &&
			tmpdimnum != pdimnum)
			break;
		}
		if (tmpdimnum < datarec->numdims)
		    issplit = 1;
		{
		    ismonotonous = 1;
		    for (int i = 1; i < (int)newsel.size(); i++) {
			if (newsel[i-1] >= newsel[i]) {
			    ismonotonous = 0;
			    break;
			}
		    }
		}
		if (issplit or !ismonotonous) {
		    if (postreadselectors[dimnum].empty()) {
			postreadselectors[dimnum] = newsel;
		    } else {
			/* select from the existing selector */
			std::vector<size_t> newoutsel;
			int i;
			for (i = 0; i < (int)newsel.size(); i++) {
			    newoutsel.push_back(postreadselectors[dimnum][newsel[i]]);
			}
			postreadselectors[dimnum] = newoutsel;
		    }
		    keeppostsel = 1;
		} else {
		    prereadselectors[dimnum] = newsel;
		    keeppresel = 1;
		}
	    }
	}
    
	if (!keeppresel)
	    prereadselectors.clear();
	if (!keeppostsel)
	    postreadselectors.clear();
    }

    /* rearrange outputselector to be in output order */
    if (!postreadselectors.empty() && !permuteorder.empty()) {
	std::vector<std::vector<size_t> > tmpselectors(datarec->numdims);
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    tmpselectors[dimnum] = postreadselectors[permuteorder[dimnum]];
	}
	postreadselectors.clear();
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    if (!permdeletethese[dimnum])
		postreadselectors.push_back(tmpselectors[dimnum]);
	}
    }

    /* calculate input/output sizes */
    {
	std::vector<size_t> tmpnewsizes;
	tmpnewsizes.clear();
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    if (!prereadselectors.empty() &&
		!prereadselectors[dimnum].empty()) {
		tmpnewsizes.push_back(prereadselectors[dimnum].size());
	    } else {
		tmpnewsizes.push_back(datarec->dimensions[dimnum].size);
	    }
	}
	/* permute order */
	inputnewsizes.clear();
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    inputnewsizes.push_back(tmpnewsizes[permuteorder[dimnum]]);
	}
	/* merge the split dimensions (multiply their size) */
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    if (permmergeinto[dimnum] != dimnum) {
		inputnewsizes[permmergeinto[dimnum]] *= inputnewsizes[dimnum];
	    }
	}
	/* delete merged dimensions */
	tmpnewsizes.clear();
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    if (!permdeletethese[dimnum]) {
		tmpnewsizes.push_back(inputnewsizes[dimnum]);
	    }
	}
	inputnewsizes = tmpnewsizes;

	outputnewsizes = inputnewsizes;
	for (dimnum = 0; dimnum < (int)postreadselectors.size(); dimnum++) {
	    if (!postreadselectors[dimnum].empty()) {
		outputnewsizes[dimnum] = postreadselectors[dimnum].size();
	    }
	}
    }

    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	if (permuteorder[dimnum] != dimnum)
	    break;
    }
    if (dimnum == datarec->numdims) {
	/* no actual permutation done */
	permuteorder.clear();
    }

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    return retval;
}

#undef FUNC
#define FUNC "bxh_datarec_readRawData"
/**
 * Fills buf with size bytes starting at offset bytes into the
 * raw data described by the datarec.
 * Any permutations, filtering, or selectors that were specified
 * in previous calls to bxh_datarec_prepareForRead or
 * bxh_datarec_prepareForReadPermute are ignored.
 * Data is byte-swapped if desired endianness (given by msbfirst)
 * does not match that of frags (stored in the msbfirstfrags field
 * of datarec).
 *
 * @param datarec raw datarec struct
 * @param buf pointer to pre-allocated data area
 * @param offset offset into datarec dataset
 * @param size number of bytes to read
 * @param msbfirst desired byte order, 0 if little-endian, 1 if big-endian
 * @return 0 on success, non-zero on error
 */
int
bxh_datarec_readRawData(bxhrawdatarec * datarec, void * buf, off_t offset, size_t size, int msbfirst)
{
    int fragnum;
    off_t curdatapos = 0;
    off_t curbufpos = 0;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if (offset >= datarec->datasize) {
	fprintf(stderr, FUNC ": offset >= datasize!");
	goto FAIL;
    }

    curdatapos = 0;
    for (fragnum = 0; curbufpos < (off_t)size && fragnum < datarec->numfrags; fragnum++) {
	bxhfragp curfragp = datarec->frags[fragnum];
	off_t fragoffset = 0;
	off_t readsize = 0;
	if (offset >= curdatapos + curfragp->fragsize) {
	    curdatapos += curfragp->fragsize;
	    continue;
	}
	fragoffset = 0;
	if (offset > curdatapos)
	    fragoffset = offset - curdatapos;
	/* optimization to reuse existing File *, if possible.
	 * Make sure to re-open if we are doing any per-frag gzip compression
	 * or if we need to go backwards in a compressed dataset */
	/* datarec->lastfpisgz == 0 : uncompressed
	 * datarec->lastfpisgz == 1 : gzip compressed, entire file
	 * datarec->lastfpisgz == 2 : gzip compressed, per-frag
	 */
	if (datarec->lastfp == NULL ||
	    (strcmp(datarec->lastfilename, curfragp->filename) != 0 &&
	     (datarec->lastfpisgz == 0 ||
	      datarec->lastfpisgz == 2 ||
	      (datarec->compression != NULL && strcmp(datarec->compression, "gzipfrag") == 0) || 
	      curfragp->fileoffset + fragoffset < gztell((gzFile)datarec->lastfp)))) {
	    char * fullfilename = NULL;
	    if (datarec->lastfp) {
		if (datarec->lastfpisgz != 0) {
		    gzclose((gzFile)datarec->lastfp);
		} else {
		    fclose((FILE *)datarec->lastfp);
		}
		datarec->lastfp = NULL;
	    }
	    if (datarec->lastfilename) {
		free(datarec->lastfilename);
		datarec->lastfilename = NULL;
	    }
	    if (datarec->bxhpath) {
		if ((fullfilename = bxh_datarec_getfullfilename(datarec->bxhpath, curfragp->filename)) == NULL) {
		    fprintf(stderr, FUNC ": error getting filename\n");
		    goto FAIL;
		}
	    } else {
		fullfilename = strdup(curfragp->filename);
	    }
	    if (datarec->compression != NULL) {
		if (strcmp(datarec->compression, "gzip") == 0) {
		    datarec->lastfp = (void *)gzopen(fullfilename, "rb");
		    datarec->lastfpisgz = 1;
		} else if (strcmp(datarec->compression, "gzipfrag") == 0) {
		    FILE * tmpfp = fopen(fullfilename, "rb");
		    if (tmpfp == NULL) {
			fprintf(stderr, FUNC ": error opening file %s (%s)\n",
				fullfilename, strerror(errno));
			goto FAIL;
		    }
		    if (fseek(tmpfp, curfragp->fileoffset, SEEK_SET) == -1) {
			fprintf(stderr, FUNC ": error seeking to file %s, byte position %lu (%s)\n",
				bxh_datarec_getfullfilename(datarec->bxhpath, curfragp->filename), (unsigned long)curfragp->fileoffset, strerror(errno));
			goto FAIL;
		    }
		    datarec->lastfp = (void *)gzdopen(fileno(tmpfp), "rb");
		    if (gzseek((gzFile)datarec->lastfp, fragoffset, SEEK_SET) == -1) {
			fprintf(stderr, FUNC ": error seeking to file %s, byte position %lu within compressed frag at position %lu (%s)\n",
				bxh_datarec_getfullfilename(datarec->bxhpath, curfragp->filename), (unsigned long)fragoffset, (unsigned long)curfragp->fileoffset, strerror(errno));
			goto FAIL;
		    }
		    datarec->lastfpisgz = 2;
		}
	    } else {
		datarec->lastfp = (void *)fopen(fullfilename, "rb");
		datarec->lastfpisgz = 0;
	    }
	    if (datarec->compression == NULL && datarec->lastfp == NULL) {
		/* file open failed */
		char * fullfilenamegz = NULL;
		fullfilenamegz = (char *)malloc(sizeof(char)*strlen(fullfilename) + 4);
		strcpy(fullfilenamegz, fullfilename);
		strcat(fullfilenamegz, ".gz");
		if ((datarec->lastfp = (void *)gzopen(fullfilenamegz, "rb")) == NULL) {
		    fprintf(stderr, FUNC ": error opening file %s (%s)\n",
			    curfragp->filename, strerror(errno));
		    free(fullfilename);
		    goto FAIL;
		}
		datarec->lastfpisgz = 1;
		free(fullfilenamegz);
	    }
	    free(fullfilename);
	    datarec->lastfilename = strdup(curfragp->filename);
	}
	readsize = curfragp->fragsize - fragoffset;
	if (curbufpos + readsize > size)
	    readsize = size - curbufpos;
	/* need to loop because gzread supports at most 4GB reads */
	off_t bytesRead = 0;
	while (bytesRead < readsize) {
	    off_t bytesToRead = readsize - bytesRead;
	    unsigned int chunksize = UINT_MAX; /* (2^32)-1 */
	    if ((unsigned long long int)chunksize > readsize) {
		chunksize = (unsigned int)readsize;
	    }
	    if ((datarec->lastfpisgz == 1 &&
		 (gzseek((gzFile)datarec->lastfp, curfragp->fileoffset + fragoffset + bytesRead, SEEK_SET) == -1 ||
		  gzread((gzFile)datarec->lastfp, (char *)buf + curbufpos, chunksize) != chunksize)) ||
		(datarec->lastfpisgz == 2 && /* already did seek */
		 gzread((gzFile)datarec->lastfp, (char *)buf + curbufpos, chunksize) != chunksize) ||
		(datarec->lastfpisgz == 0 &&
		 (fseek((FILE *)datarec->lastfp, curfragp->fileoffset + fragoffset, SEEK_SET) == -1 ||
		  fread((char *)buf + curbufpos, 1, chunksize, (FILE *)datarec->lastfp) != chunksize))) {
		fprintf(stderr, FUNC ": error reading file %s, byte position %lu (%s)\n",
			bxh_datarec_getfullfilename(datarec->bxhpath, curfragp->filename), (unsigned long)(curfragp->fileoffset + fragoffset + bytesRead), strerror(errno));
		goto FAIL;
	    }
	    bytesRead += chunksize;
	}
	if (msbfirst != datarec->msbfirstfrags) {
	    char * curp;
	    char * endp = (char *)buf + curbufpos + readsize;
	    for (curp = (char *)buf + curbufpos;
		 curp + datarec->elemsize <= endp;
		 curp += datarec->elemsize) {
		if (datarec->elemsize == 2) {
		    char swap = curp[0];
		    curp[0] = curp[1];
		    curp[1] = swap;
		} else if (datarec->elemsize == 4) {
		    char swap = curp[0];
		    curp[0] = curp[3];
		    curp[3] = swap;
		    swap = curp[1];
		    curp[1] = curp[2];
		    curp[2] = swap;
		} else if (datarec->elemsize == 8) {
		    char swap = curp[0];
		    curp[0] = curp[7];
		    curp[7] = swap;
		    swap = curp[1];
		    curp[1] = curp[6];
		    curp[6] = swap;
		    swap = curp[2];
		    curp[2] = curp[5];
		    curp[5] = swap;
		    swap = curp[3];
		    curp[3] = curp[4];
		    curp[4] = swap;
		}
	    }
	}
	curbufpos += readsize;
	curdatapos += curfragp->fragsize;
    }
    goto EXIT;

  FAIL:
    if (datarec->lastfilename) {
	free(datarec->lastfilename);
	datarec->lastfilename = NULL;
    }
    if (datarec->lastfp) {
	if (datarec->lastfpisgz != 0) {
	    gzclose((gzFile)datarec->lastfp);
	} else {
	    fclose((FILE *)datarec->lastfp);
	}
	datarec->lastfp = NULL;
    }
    retval = -1;

  EXIT:
    return retval;
}

static
size_t
coords2page(size_t * coords,
	    size_t * sizes,
	    const std::vector<std::vector<size_t> > & selector,
	    int numdims,
	    int basepagedim)
{
    size_t retval = 0;
    size_t ind;
    int dimnum;
    for (dimnum = numdims - 1; dimnum >= basepagedim; dimnum--) {
	if (selector[dimnum].empty())
	    ind = coords[dimnum];
	else
	    ind = selector[dimnum][coords[dimnum]];
	retval = retval + ind;
	if (dimnum > basepagedim)
	    retval *= sizes[dimnum-1];
    }
    return retval;
}

static
void
coordsinc(size_t * coords,
	  size_t * nextcoords,
	  size_t * sizes,
	  const std::vector<std::vector<size_t> > & selector,
	  int numdims,
	  int basepagedim)
{
    int dimtoinc = basepagedim;
    memcpy(nextcoords, coords, sizeof(size_t)*numdims);
    while (dimtoinc < numdims) {
	const std::vector<size_t> & cursel = selector[dimtoinc];
	size_t curcoord = nextcoords[dimtoinc];
	size_t cursize = sizes[dimtoinc];
	if ((cursel.empty() && curcoord == cursize - 1) ||
	    (!cursel.empty() && curcoord == cursel.size() - 1)) {
	    nextcoords[dimtoinc] = 0;
	    dimtoinc = dimtoinc + 1;
	} else {
	    nextcoords[dimtoinc] = curcoord + 1;
	    break;
	}
    }
    if (dimtoinc >= numdims)
	nextcoords[0] = (size_t)-1;
    return;
}

#undef FUNC
#define FUNC "bxh_datarec_readData"
/**
 * Reads data described by datarec, automatically performing permutation,
 * filtering, and selection if there has been a previous call to
 * bxh_datarec_prepareForRead, bxh_datarec_prepareForReadPermute
 * or bxh_datarec_prepareForReadPermuteSelect.
 * Data is byte-swapped if desired endianness (given by msbfirst)
 * does not match that of frags (stored in the msbfirstfrags field
 * of datarec).
 *
 * @param datarec raw datarec struct
 * @param bufptr pointer to pointer to buf (modified on return)
 * @param msbfirst desired byte order, 0 if little-endian, 1 if big-endian
 * @return new datarec on success, NULL on error
 */
bxhrawdatarec *
bxh_datarec_readData(bxhrawdatarec * datarec, void ** bufptr, int msbfirst)
{
    bxhrawdatarec * newdatarec = NULL; /* return value */
    
    readobj * robj = NULL;

    int basepagedim = 0;
    int dochunk = 0;
    size_t totalinselsize = 0;
    const size_t minunreadchunksize = 1 << 10;
    const size_t maxreadchunksize = 1 << 20;
    size_t elemsinpage = 0;
    size_t numpages = 0;
    size_t pagenum = 0;

    size_t bufsize = 0;
    void * savebuf = NULL;
    void * newbuf = NULL;
    void * readbuf = NULL;
    void * newreadbuf = NULL;
    void * permutebuf = NULL;

    size_t * coords = NULL;
    std::vector<std::vector<size_t> > postselect;
    std::vector<std::vector<size_t> > rangejumps(datarec->numdims);
    size_t * cumfragsizes = NULL;
    size_t * sizes = NULL;
    size_t * cumsizes = NULL;
    size_t * inselsizes = NULL;
    size_t * cuminselsizes = NULL;
    size_t * unreadchunksperdim = NULL;
    size_t * readchunksperdim = NULL;
    std::vector<std::vector<size_t> > prereadselectors;

    int dimnum = 0;
    int fragnum = 0;

    coords = (size_t *)malloc(sizeof(size_t)*datarec->numdims);
    cumfragsizes = (size_t *)malloc(sizeof(size_t)*datarec->numfrags);
    sizes = (size_t *)malloc(sizeof(size_t)*datarec->numdims);
    cumsizes = (size_t *)malloc(sizeof(size_t)*datarec->numdims);
    inselsizes = (size_t *)malloc(sizeof(size_t)*datarec->numdims);
    cuminselsizes = (size_t *)malloc(sizeof(size_t)*datarec->numdims);
    unreadchunksperdim = (size_t *)malloc(sizeof(size_t)*(datarec->numdims+1));
    readchunksperdim = (size_t *)malloc(sizeof(size_t)*(datarec->numdims+1));

    memset(coords, '\0', sizeof(size_t)*datarec->numdims);
    memset(cumfragsizes, '\0', sizeof(size_t)*datarec->numfrags);
    memset(sizes, '\0', sizeof(size_t)*datarec->numdims);
    memset(cumsizes, '\0', sizeof(size_t)*datarec->numdims);
    memset(inselsizes, '\0', sizeof(size_t)*datarec->numdims);
    memset(cuminselsizes, '\0', sizeof(size_t)*datarec->numdims);
    memset(unreadchunksperdim, '\0', sizeof(size_t)*(datarec->numdims+1));
    memset(readchunksperdim, '\0', sizeof(size_t)*(datarec->numdims+1));

    if (bufptr == NULL) {
	fprintf(stderr, FUNC ": input argument bufptr is NULL!\n");
	goto FAIL;
    }

    if (datarec->readobj == NULL)
	bxh_datarec_prepareForRead(datarec);
    robj = (readobj *)datarec->readobj;
    newdatarec = bxh_datarec_postprocess(datarec);
    newdatarec->msbfirstfrags = msbfirst;

    prereadselectors = robj->prereadselectors;
    /* fill in prereadselectors if it doesn't have enough members */
    while (prereadselectors.size() < (unsigned int)datarec->numdims) {
	prereadselectors.push_back(std::vector<size_t>());
    }
    prereadselectors.push_back(std::vector<size_t>(1,1)); /* sentry */

    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	sizes[dimnum] = datarec->dimensions[dimnum].size;
	if (dimnum == 0)
	    cumsizes[dimnum] = sizes[dimnum];
	else
	    cumsizes[dimnum] = cumsizes[dimnum - 1] * sizes[dimnum];
	if (prereadselectors[dimnum].empty())
	    inselsizes[dimnum] = sizes[dimnum];
	else
	    inselsizes[dimnum] = prereadselectors[dimnum].size();
	if (dimnum == 0)
	    cuminselsizes[dimnum] = inselsizes[dimnum];
	else
	    cuminselsizes[dimnum] = cuminselsizes[dimnum - 1] * inselsizes[dimnum];
    }
    for (fragnum = 0; fragnum < datarec->numfrags; fragnum++) {
	if (fragnum > 0)
	    cumfragsizes[fragnum] = cumfragsizes[fragnum - 1];
	cumfragsizes[fragnum] += datarec->frags[fragnum]->fragsize;
    }

    savebuf = *bufptr;
    if (*bufptr == NULL) {
	/* caller did not provide a buffer, so we need to allocate it */
	/* make it big enough to hold unfiltered data */
	if ((*bufptr = malloc(sizeof(char) * cuminselsizes[datarec->numdims-1] * (size_t)datarec->elemsize)) == NULL) {
	    fprintf(stderr, FUNC ": Couldn't allocate enough memory!\n");
	    goto FAIL;
	}
	newbuf = *bufptr; /* needs to be freed on failure */
    }
    /* if we permute, we need two buffers.
     * Otherwise read directly into *bufptr */
    if (robj->permuteorder.empty()) {
	permutebuf = NULL;
	readbuf = *bufptr;
    } else {
	permutebuf = *bufptr;
	readbuf = malloc(sizeof(char)*cuminselsizes[datarec->numdims-1]*(size_t)datarec->elemsize);
	newreadbuf = readbuf; /* needs to be freed on exit */
    }
    

    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	//if (inselsizes[dimnum] == sizes[dimnum])
	//    continue;
	int unreadchunks = 0;
	int readchunks = 0;
	std::vector<size_t> & curselect = prereadselectors[dimnum];
	int i;
	if (curselect.empty()) {
	    unreadchunks = 0;
	    readchunks = 1;
	} else {
	    readchunks++;
	    if (curselect[0] > 0) {
		/* selection is not flush left */
		unreadchunks++;
	    }
	    for (i = 1; i < (int)curselect.size(); i++) {
		if (curselect[i] - curselect[i-1] != 1) {
		    unreadchunks++;
		    readchunks++;
		}
	    }
	    if (sizes[dimnum] - *(curselect.end()-1) > 1) {
		/* selection is not flush right */
		unreadchunks++;
	    }
	}
	unreadchunksperdim[dimnum] = unreadchunks;
	readchunksperdim[dimnum] = readchunks;
    }
    unreadchunksperdim[datarec->numdims] = 1;
    readchunksperdim[datarec->numdims] = 1;
    
    /* figure out optimal chunking size */
    {
	size_t numunreadchunks = 0;
	double avgunreadchunksize = 0;
	size_t numreadchunks = 0;
	double avgreadchunksize = 0;
	basepagedim = 0;
	/* find first dim that is non-contiguous */
	while (basepagedim < datarec->numdims &&
	       inselsizes[basepagedim] == sizes[basepagedim] &&
	       readchunksperdim[basepagedim] <= 1) {
	    basepagedim++;
	}
	while (1) {
	    if (basepagedim >= datarec->numdims)
		break; /* reached last dim */
	    numunreadchunks = unreadchunksperdim[basepagedim];
	    if (numunreadchunks == 0)
		break; /* already at an optimal size */
	    totalinselsize = datarec->elemsize;
	    numreadchunks = readchunksperdim[basepagedim];
	    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
		if (dimnum < basepagedim)
		    totalinselsize *= sizes[dimnum];
		else
		    totalinselsize *= inselsizes[dimnum];
		if (dimnum > basepagedim) {
		    numunreadchunks *= sizes[dimnum];
		}
	    }
	    avgunreadchunksize = totalinselsize / numunreadchunks;
	    avgreadchunksize = totalinselsize / numreadchunks;
	    if (avgunreadchunksize >= minunreadchunksize ||
		avgreadchunksize >= maxreadchunksize)
		break; /* reached target chunk size */

	    /* want to chunk larger than current page size -- go up one
	     * more dimension */
	    dochunk = 1;
	    basepagedim++;
	}
    }

    if (dochunk) {
	/* move selection of fastest dimensions to after disk read */
	for (dimnum = 0; dimnum < basepagedim && dimnum < datarec->numdims; dimnum++) {
	    postselect.push_back(prereadselectors[dimnum]);
	    prereadselectors[dimnum].clear();
	}
    }

    elemsinpage = 1;
    for (dimnum = 0; dimnum < basepagedim && dimnum < datarec->numdims; dimnum++) {
	elemsinpage *= sizes[dimnum];
    }
    numpages = 1;
    for (dimnum = basepagedim; dimnum < datarec->numdims; dimnum++) {
	numpages *= sizes[dimnum];
    }
	
    bufsize = 0;
    /* rangejumps is structured like prereadselectors, but indicates the
     * index corresponding to the end of the contiguous range of
     * selected pages within that dimension */
    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	if (prereadselectors[dimnum].empty())
	    continue;
	std::vector<size_t> & curselect = prereadselectors[dimnum];
	int jumpto = curselect.size() - 1;
	int i;
	rangejumps[dimnum] = curselect;
	for (i = curselect.size() - 1; i >= 0; i--) {
	    if (i < (int)curselect.size() - 1 &&
		curselect[i + 1] - curselect[i] > 1)
		jumpto = i;
	    rangejumps[dimnum][i] = jumpto;
	}
    }

    pagenum = coords2page(coords, sizes, prereadselectors, datarec->numdims, basepagedim);
    while (pagenum < numpages) {
	/* Precondition: "pagenum" is the number of the first
	 *   unprocessed page containing valid data.  We are
	 *   currently looking for the end of a range of valid
	 *   data.  The last page in this range we have seen so far
	 *   is "lastvalidpage".  Page numbers start at 1.
	 *   "startfragnum" is for efficiency, so we don't parse
	 *   frags we know we don't need anymore. */
	size_t firstvalidpage = pagenum;
	size_t lastvalidpage = 0;

	/* update coords/pagenum here for next page */
	if (basepagedim >= datarec->numdims) {
	    /* shortcut */
	    firstvalidpage = 0;
	    lastvalidpage = numpages - 1;
	    pagenum = numpages;
	} else {
	    std::vector<size_t> & baseselect = prereadselectors[basepagedim];
	    /* jump to end of region in basepagedim */
	    if (!baseselect.empty()) {
		coords[basepagedim] = rangejumps[basepagedim][coords[basepagedim]];
	    }
	    lastvalidpage = coords2page(coords, sizes, prereadselectors, datarec->numdims, basepagedim);
	    size_t * nextcoords = (size_t *)malloc(sizeof(size_t) * datarec->numdims);
	    while (1) {
		coordsinc(coords, nextcoords, sizes, prereadselectors, datarec->numdims, basepagedim);
		if (nextcoords[0] == (size_t)-1)
		    break;
		size_t nextpage = coords2page(nextcoords, sizes, prereadselectors, datarec->numdims, basepagedim);
		memcpy(coords, nextcoords, sizeof(size_t) * datarec->numdims);
		if (nextpage != lastvalidpage + 1)
		    break;
		lastvalidpage = nextpage;
	    }
	    /* calculate next page number */
	    if (nextcoords[0] == (size_t)-1) {
		/* made it all the way up, so must be end of data */
		pagenum = numpages + 1;
	    } else {
		pagenum = coords2page(coords, sizes, prereadselectors, datarec->numdims, basepagedim);
	    }
	    free(nextcoords);
	}

	/* Precondition:  We have a range of valid data, from
	 *   "firstvalidpage" to "lastvalidpage". */
	size_t validbytesstart = (firstvalidpage * elemsinpage) * datarec->elemsize;
	size_t validbytesstop = ((lastvalidpage + 1) * elemsinpage) * datarec->elemsize;
	void * pages = NULL;
	size_t pagepos = 0;
	size_t numpagestoread = lastvalidpage + 1 - firstvalidpage;

// 	fprintf(stderr, "(first, last) valid page (%d,%d)/%d\n", (int)firstvalidpage, (int)lastvalidpage, (int)numpages);
// 	fprintf(stderr, "valid bytes (start, stop) (%d,%d)/%d\n", (int)validbytesstart, (int)validbytesstop, (int)datarec->datasize);
	
	if (!postselect.empty()) {
	    pages = malloc(sizeof(char)*(size_t)datarec->elemsize*cumsizes[basepagedim-1] * numpagestoread);
	}

	int startfragnum = 0;
	int endfragnum = -1;
	if (datarec->numfrags > 0) {
	    for (startfragnum = 0; startfragnum < datarec->numfrags; startfragnum++) {
		if (cumfragsizes[startfragnum] > validbytesstart)
		    break;
	    }
	    for (endfragnum = 0; endfragnum < datarec->numfrags; endfragnum++) {
		if (cumfragsizes[endfragnum] >= validbytesstop)
		    break;
	    }
	}
      
// 	fprintf(stderr, "(start, end) frag num (%u,%u)/%u\n", (unsigned int)startfragnum, (unsigned int)endfragnum, (unsigned int)datarec->numfrags);
	for (fragnum = startfragnum; fragnum <= endfragnum; fragnum++) {
	    bxhfragp fragp = datarec->frags[fragnum];
	    bxhfrag readfrag;
	    size_t fragstart = 0;
	    size_t fragend = 0;
	    int stopnow = 0;
	    readfrag.filename = NULL;
	    if (fragnum > 0)
		fragstart = cumfragsizes[fragnum - 1];
	    fragend = cumfragsizes[fragnum];
// 	    fprintf(stderr, "frag (num, start, end)  = (%u/%u, %u, %u)\n", (unsigned int)fragnum, (unsigned int)datarec->numfrags, (unsigned int)fragstart, (unsigned int)fragend);
	    if (fragend <= validbytesstart) {
		/* this frag is before valid range, do nothing */
	    } else if (fragstart >= validbytesstop) {
		/* this frag is after valid range, so stop here */
		stopnow = 1;
	    } else if (validbytesstart <= fragstart &&
		       fragend <= validbytesstop) {
		/* frag is completely valid, so read it unchanged */
		readfrag.filename = fragp->filename;
		readfrag.fileoffset = fragp->fileoffset;
		readfrag.fragsize = fragp->fragsize;
	    } else {
		/* valid range starts and/or ends in this frag so make
		 * new frag to replace this one */
		off_t offsetdiff = 0; /* will be added */
		off_t sizediff = 0;   /* will be subtracted */
		if (fragstart <= validbytesstart) {
		    /* need to remove a bit from the front */
		    offsetdiff = offsetdiff + validbytesstart - fragstart;
		    sizediff = sizediff + validbytesstart - fragstart;
		}
		if (validbytesstop <= fragend) {
		    /* need to truncate the end */
		    sizediff = sizediff + fragend - validbytesstop;
		    /* we're at the end of the range so stop after reading */
		    stopnow = 1;
		}
		readfrag.filename = fragp->filename;
		readfrag.fileoffset = fragp->fileoffset + offsetdiff;
		readfrag.fragsize = fragp->fragsize - sizediff;
	    }

	    if (readfrag.filename != NULL) {
		/* read the frag */
		void * curreadptr = NULL;

// 		fprintf(stderr, "frag (num, filename, fileoffset, fragsize) = (%d/%d, %s, %u, %u)\n", (unsigned int)fragnum, (unsigned int)datarec->numfrags, readfrag.filename, (unsigned int)readfrag.fileoffset, (unsigned int)readfrag.fragsize);
		/* curfpisgz == 0 : uncompressed
		 * curfpisgz == 1 : gzip compressed, entire file
		 * curfpisgz == 2 : gzip compressed, per-frag
		 */
		int curfpisgz = 0;
		if (datarec->compression != NULL) {
		    if (strcmp(datarec->compression, "gzip") == 0) {
		        curfpisgz = 1;
		    } else if (strcmp(datarec->compression, "gzipfrag") == 0) {
		        curfpisgz = 2;
		    }
		}
		/* first open the file (check if open already) */
		/* optimization to reuse existing File *, if possible.
		 * Make sure to re-open if we are doing any per-frag gzip compression
		 * or if we need to go backwards in a compressed dataset */
		if (datarec->lastfp == NULL ||
		    strcmp(datarec->lastfilename, readfrag.filename) != 0 ||
		    datarec->lastfpisgz != curfpisgz ||
		    /* filename and compression scheme are equal;
		     * check if they are gzipfrag */
		    curfpisgz == 2 ||
		    /* they are both uncompressed or pure gzip compression;
		     * check if it's a compressed backwards seek */
		    (curfpisgz == 1 && readfrag.fileoffset < gztell((gzFile)datarec->lastfp))) {
		    char * fullfilename = NULL;
		    if (datarec->lastfp) {
			if (datarec->lastfpisgz) {
			    gzclose((gzFile)datarec->lastfp);
			} else {
			    fclose((FILE *)datarec->lastfp);
			}
			datarec->lastfp = NULL;
		    }
		    if (datarec->lastfilename) {
			free(datarec->lastfilename);
			datarec->lastfilename = NULL;
		    }
		    if (datarec->bxhpath) {
			if ((fullfilename = bxh_datarec_getfullfilename(datarec->bxhpath, readfrag.filename)) == NULL) {
			    fprintf(stderr, FUNC ": error getting filename\n");
			    goto FAIL;
			}
		    } else {
			fullfilename = strdup(readfrag.filename);
		    }
		    if (curfpisgz != 0) {
			if (curfpisgz == 1) {
			    datarec->lastfp = (void *)gzopen(fullfilename, "rb");
			    gzbuffer((gzFile)datarec->lastfp, 256*1024); // default is 8K
			    datarec->lastfpisgz = 1;
			} else if (curfpisgz == 2) {
			    /* make sure to fseek into beginning of original
			     * frag (fragp) and then gzseek within the frag
			     * to the offset required */
			    int tmpfd = open(fullfilename, O_RDONLY);
			    if (tmpfd < 0) {
				fprintf(stderr, FUNC ": error opening file %s (%s)\n",
					fullfilename, strerror(errno));
				goto FAIL;
			    }
			    if (lseek(tmpfd, fragp->fileoffset, SEEK_SET) == -1) {
				fprintf(stderr, FUNC ": error seeking to file %s, byte position %lu (%s)\n",
					bxh_datarec_getfullfilename(datarec->bxhpath, readfrag.filename), (unsigned long)readfrag.fileoffset, strerror(errno));
				goto FAIL;
			    }
			    datarec->lastfp = (void *)gzdopen(tmpfd, "rb");
			    if (gzseek((gzFile)datarec->lastfp, readfrag.fileoffset - fragp->fileoffset, SEEK_SET) == -1) {
				fprintf(stderr, FUNC ": error seeking to file %s, byte position %lu within compressed frag at position %lu (%s)\n",
					bxh_datarec_getfullfilename(datarec->bxhpath, readfrag.filename), (unsigned long)(readfrag.fileoffset - fragp->fileoffset), (unsigned long)readfrag.fileoffset, strerror(errno));
				goto FAIL;
			    }
			    datarec->lastfpisgz = 2;
			}
		    } else {
			datarec->lastfp = (void *)fopen(fullfilename, "rb");
			datarec->lastfpisgz = 0;
		    }
		    if (datarec->lastfp == NULL) {
			char * fullfilenamegz = NULL;
			fullfilenamegz = (char *)malloc(sizeof(char)*strlen(fullfilename) + 4);
			strcpy(fullfilenamegz, fullfilename);
			strcat(fullfilenamegz, ".gz");
			if ((datarec->lastfp = gzopen(fullfilenamegz, "rb")) == NULL) {
			    fprintf(stderr, FUNC ": error opening file %s (%s)\n",
				    readfrag.filename, strerror(errno));
			    free(fullfilename);
			    goto FAIL;
			}
			gzbuffer((gzFile)datarec->lastfp, 256*1024); // default is 8K
			datarec->lastfpisgz = 1;
			free(fullfilenamegz);
		    }
		    free(fullfilename);
		    datarec->lastfilename = strdup(readfrag.filename);
		}
		if ((datarec->lastfpisgz == 1 &&
		     gzseek((gzFile)datarec->lastfp, readfrag.fileoffset, SEEK_SET) == -1) ||
		    (datarec->lastfpisgz == 0 &&
		     fseek((FILE *)datarec->lastfp, readfrag.fileoffset, SEEK_SET) == -1)) {
		    fprintf(stderr, FUNC ": error seeking in file %s (%s)\n",
			    bxh_datarec_getfullfilename(datarec->bxhpath, readfrag.filename), strerror(errno));
		    goto FAIL;
		}
		/* need to loop because gzread supports at most 4GB reads */
		off_t bytesRead = 0;
		while (bytesRead < readfrag.fragsize) {
		    if (postselect.empty())
			curreadptr = (char *)readbuf + bufsize;
		    else
			curreadptr = (char *)pages + pagepos;
		    off_t bytesToRead = readfrag.fragsize - bytesRead;
		    unsigned int chunksize = UINT_MAX; /* (2^32)-1 */
		    if ((unsigned long long int)chunksize > bytesToRead) {
			chunksize = (unsigned int)bytesToRead;
		    }
		    if ((datarec->lastfpisgz != 0 &&
			 gzread((gzFile)datarec->lastfp, curreadptr, chunksize) != chunksize) ||
			(datarec->lastfpisgz == 0 &&
			 fread(curreadptr, 1, chunksize, (FILE *)datarec->lastfp) != chunksize)) {
			if ((datarec->lastfpisgz != 0 && gzeof((gzFile)datarec->lastfp)) ||
			    (datarec->lastfpisgz == 0 && feof((FILE *)datarec->lastfp))) {
			    fprintf(stderr, FUNC ": reached end-of-file %s (%s)\n",
				    bxh_datarec_getfullfilename(datarec->bxhpath, readfrag.filename), strerror(errno));
			    goto FAIL;
			} else {
			    fprintf(stderr, FUNC ": error reading file %s, byte position %lu-%lu\n",
				    bxh_datarec_getfullfilename(datarec->bxhpath, readfrag.filename), (unsigned long)(readfrag.fileoffset + bytesRead), (unsigned long)(readfrag.fileoffset + bytesRead + chunksize - 1));
			    goto FAIL;
			}
		    }
		    bytesRead += chunksize;
		    if (postselect.empty())
			bufsize = bufsize + chunksize;
		    else
			pagepos = pagepos + chunksize;
		}
	    }
	    if (stopnow)
		break;
	}
	if (!postselect.empty()) {
	    /* do input selection and copy data (pagepos is size of data) */
	    std::vector<int> coords(postselect.size(), 0);
	    int elemsize = datarec->elemsize;
	    for (dimnum = basepagedim - 1; dimnum >= 0; dimnum--) {
		size_t oldpos = 0;
		size_t newpos = 0;
		std::vector<size_t> & curselect = postselect[dimnum];
		size_t curpagesize = 0;
		
		if (curselect.empty())
		    continue;
		if (dimnum > 0)
		    curpagesize = cumsizes[dimnum - 1] * elemsize;
		else
		    curpagesize = elemsize;

		while (oldpos < pagepos) {
		    int i;
		    size_t curselsize = curselect.size();
		    for (i = 0; i < (int)curselsize; i++) {
			memmove((char *)pages + newpos, (char *)pages + oldpos + (curselect[i] * curpagesize), curpagesize);
			newpos += curpagesize;
		    }
		    oldpos += sizes[dimnum] * curpagesize;
		}
		pagepos = newpos;
	    }
	    memcpy((char *)readbuf + bufsize, (char *)pages, pagepos);
	    bufsize = bufsize + pagepos;
	}
	firstvalidpage = 0;
	lastvalidpage = 0;
	free(pages);
    }
    
    /* Check that we read enough data */
    if (bufsize != cuminselsizes[datarec->numdims - 1] * datarec->elemsize) {
	fprintf(stderr, "Not enough data in files (%u) to fill output array (%u)!\n", (unsigned int)bufsize, (unsigned int)(cuminselsizes[datarec->numdims - 1] * datarec->elemsize));
	goto FAIL;
    }

    /* fix byte order */
    if (msbfirst != datarec->msbfirstfrags) {
	int elemsize = datarec->elemsize;
	char * curp;
	char * endp = (char *)readbuf + bufsize;
	for (curp = (char *)readbuf;
		     curp + elemsize <= endp;
		     curp += elemsize) {
	    char swap;
	    switch (elemsize) {
	    case 2:
		swap = curp[0];
		curp[0] = curp[1];
		curp[1] = swap;
		break;
	    case 4:
		swap = curp[0];
		curp[0] = curp[3];
		curp[3] = swap;
		swap = curp[1];
		curp[1] = curp[2];
		curp[2] = swap;
		break;
	    case 8:
		swap = curp[0];
		curp[0] = curp[7];
		curp[7] = swap;
		swap = curp[1];
		curp[1] = curp[6];
		curp[6] = swap;
		swap = curp[2];
		curp[2] = curp[5];
		curp[5] = swap;
		swap = curp[3];
		curp[3] = curp[4];
		curp[4] = swap;
		break;
	    }
	}
    }

    /* permute */
    if (!robj->permuteorder.empty()) {
	std::vector<int> & permuteorder = ((readobj *)datarec->readobj)->permuteorder;
	char * tmpbuf = NULL;
	size_t tmpbufsize = 0;
	size_t curpos = 0;
	int i = 0;
	unsigned int * curcoords = 0;

	curcoords = (unsigned int *)malloc(sizeof(unsigned int) * datarec->numdims);
	memset(curcoords, '\0', sizeof(unsigned int) * datarec->numdims);
    
	/* calculate tmpbufsize as largest contiguous unpermuted n-d section */
	tmpbufsize = datarec->elemsize;
	for (i = 0; i < datarec->numdims && permuteorder[i] == i; i++) {
	    tmpbufsize *= inselsizes[permuteorder[i]];
	}

	tmpbuf = (char *)malloc(sizeof(char *) * tmpbufsize);

	/* read data in chunks of tmpbufsize */
	curpos = 0;
	while (curpos < bufsize) {
	    off_t bufind; /* index within buf in elems, not bytes */
	    off_t bufelems;
	    int elemsize;
	    int numdims;
	    size_t readsize = tmpbufsize;
	    if (curpos + tmpbufsize > bufsize) {
		readsize = bufsize - curpos;
	    }
	    memcpy(tmpbuf, (char *)readbuf + curpos, readsize);
	    elemsize = datarec->elemsize;
	    numdims = datarec->numdims;
	    bufelems = (off_t)tmpbufsize/elemsize;
	    for (bufind = 0; bufind < bufelems; bufind++, curpos += elemsize) {
		/* indexes within total data, in elems, not bytes */
		off_t newind = 0;
		int i;
		/* now permute to get the new coords */
		for (i = numdims-1; i >= 0; i--) {
		    newind += curcoords[permuteorder[i]];
		    if (i != 0)
			newind *= inselsizes[permuteorder[i-1]];
		}
		/* place data item in the correct place in new array */
		/* memcpy(&((char *)buf)[newind*datarec->elemsize], &tmpbuf[bufind*datarec->elemsize], datarec->elemsize); */
		for (i = 0; i < elemsize; i++) {
		    ((char *)permutebuf)[newind*elemsize + i] = tmpbuf[bufind*elemsize + i];
		}
		/* increment curcoords */
		for (i = 0; i < numdims; i++) {
		    curcoords[i]++;
		    if (curcoords[i] < inselsizes[i])
			break; /* our job is done */
		    curcoords[i] = 0;
		    /* loop around to increment next dimension */
		}
	    }
	}
	free(tmpbuf);
	free(curcoords);
    }
    
    /* postreadselectors */
    if (!robj->postreadselectors.empty()) {
	std::vector<size_t> filteredsizes = robj->inputnewsizes;
	size_t filtereddatasize = cuminselsizes[datarec->numdims-1] * datarec->elemsize; /* old (input filtered) size */
	for (dimnum = newdatarec->numdims - 1; dimnum >= 0; dimnum--) {
	    /* go from slowest moving dimension (i.e. largest page size)
	     * to fastest moving, to minimize number of calls to memmove,
	     * though we risk moving more data */
	    size_t oldpos = 0;
	    size_t newpos = 0;
	    size_t pagesize = 0;
	    int tmpnum;
	    std::vector<size_t> & osel = robj->postreadselectors[dimnum];
	    if (osel.empty()) { continue; }

	    pagesize = datarec->elemsize;
	    for (tmpnum = 0; tmpnum < dimnum; tmpnum++) {
		pagesize *= filteredsizes[tmpnum];
	    }
	    while (oldpos < filtereddatasize) {
		int sind = 0;
		for (sind = 0; sind < (int)osel.size(); sind++) {
		    memmove(&((char *)*bufptr)[newpos], &((char *)*bufptr)[oldpos + (osel[sind] * pagesize)], pagesize);
		    newpos += pagesize;
		}
		oldpos += filteredsizes[dimnum] * pagesize;
	    }
	    filtereddatasize = newpos;
	    filteredsizes[dimnum] = osel.size();
	}
	/* if we potentially allocated more memory than we needed,
	 * reduce memory size if we can */
	if (newbuf) {
	    void * tmpptr = realloc(newbuf, newdatarec->datasize);
	    if (tmpptr) {
		newbuf = *bufptr = tmpptr;
	    }
	}
    }

    goto EXIT;

  FAIL:
    if (newbuf) {
	free(newbuf);
	*bufptr = savebuf;
    }
    if (newdatarec) {
	bxh_datarec_free(newdatarec);
	newdatarec = NULL;
    }

  EXIT:
    if (datarec->lastfilename) {
	free(datarec->lastfilename);
	datarec->lastfilename = NULL;
    }
    if (datarec->lastfp) {
	if (datarec->lastfpisgz) {
	    gzclose((gzFile)datarec->lastfp);
	} else {
	    fclose((FILE *)datarec->lastfp);
	}
	datarec->lastfp = NULL;
    }
    if (newreadbuf) {
	free(newreadbuf);
    }
    if (coords) { free(coords); }
    if (cumfragsizes) { free(cumfragsizes); }
    if (sizes) { free(sizes); }
    if (cumsizes) { free(cumsizes); }
    if (inselsizes) { free(inselsizes); }
    if (cuminselsizes) { free(cuminselsizes); }
    if (unreadchunksperdim) { free(unreadchunksperdim); }
    if (readchunksperdim) { free(readchunksperdim); }

    return newdatarec;
}

#undef FUNC
#define FUNC "bxh_datarec_readCanonicalData"
/**
 * Returns copy of rawdatarec with dimensions permuted to be in order
 * given by ordereddimnames, and fills buf with permuted data.
 * Any existing dimensions not specified in ordereddimnames list are
 * moved to end of dimension list.
 * Also merges/reorders "split" dimensions (i.e. those named
 * specialdimname-split1, specialdimname-split2, etc.).  This only
 * works for dimensions explicitly specified in ordereddimnames.
 * New datarec will not have any frags (since it is no longer
 * describing any on-disk data).
 * Data is byte-swapped if desired endianness (given by msbfirst)
 * does not match that of frags (stored in the msbfirstfrags field
 * of datarec).
 *
 * @param datarec raw datarec struct
 * @param buf pointer to pre-allocated data area
 * @param ordereddimnames list of dimension names, in the order desired for new data
 * @param numordereddims number of ordereddims sent in ordereddimnames
 * @param msbfirst desired byte order, 0 if little-endian, 1 if big-endian
 * @return new permuted bxhrawdatarec, NULL if error
 */
bxhrawdatarec *
bxh_datarec_readCanonicalData(bxhrawdatarec * datarec, void * buf, const char ** ordereddimnames, int numordereddims, int msbfirst)
{
    bxhrawdatarec * newdatarec = NULL;
    if (bxh_datarec_prepareForReadPermute(datarec, ordereddimnames, numordereddims) != 0) {
	fprintf(stderr, FUNC ": prepareForReadPermute failed!\n");
	goto FAIL;
    }
    if ((newdatarec = bxh_datarec_readData(datarec, &buf, msbfirst)) == NULL) {
	fprintf(stderr, FUNC ": readData failed!\n");
	goto FAIL;
    }
    goto EXIT;

  FAIL:
    if (newdatarec)
	bxh_datarec_free(newdatarec);
    newdatarec = NULL;

  EXIT:
    return newdatarec;
}

#undef FUNC
#define FUNC "bxh_datarec_readXYZTData"
/**
 * This is the common form of bxh_datarec_readCanonicalData,
 * sending "x", "y", "z", "t" as the dimension order.
 * See that function for more info.
 *
 * @param datarec raw datarec struct
 * @param buf pointer to pre-allocated data area
 * @param msbfirst desired byte order, 0 if little-endian, 1 if big-endian
 * @return new permuted bxhrawdatarec, NULL if error
 */
bxhrawdatarec *
bxh_datarec_readXYZTData(bxhrawdatarec * datarec, void * buf, int msbfirst)
{
    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    int numordereddims = 4;
    return bxh_datarec_readCanonicalData(datarec, buf, &ordereddimnames[0], numordereddims, msbfirst);
}

#undef FUNC
#define FUNC "bxh_datarec_createFromElement"
/**
 * Given a BXH datarec, this function reads in selected metadata
 * described to by the element, and returns a pointer to a
 * newly allocated bxhrawdatarec holding this data.  The caller
 * is responsible for freeing this, perhaps
 * by calling bxh_datarec_free().
 *
 * @param elemp BXH datarec element pointer
 * @param bxhpath if non-NULL, specifies the base path for relative filenames in the BXH file (typically the full path of the BXH file).  If path is a directory name, it must end with a path separator (e.g. slash '/')
 * @return pointer to bxhrawdatarec
 */
bxhrawdatarec *
bxh_datarec_createFromElement(const BXHElementPtr elemp, const char * bxhpath)
{
    bxhrawdatarec * datarec = NULL;

    datarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
    memset(datarec, '\0', sizeof(bxhrawdatarec));
    if (bxhpath)
	datarec->bxhpath = strdup(bxhpath);
    
    if (bxh_datarec_fillParams(datarec, elemp) != 0)
	goto FAIL;
    if (bxh_datarec_fillFrags(datarec, elemp) != 0)
	goto FAIL;
    if (datarec->frags)
	datarec->numfrags =
	    bxh_datarec_canonicalizeFrags(datarec->frags, datarec->numfrags);

    goto EXIT;
	
FAIL:
    if (datarec)
	bxh_datarec_free(datarec);
    datarec = NULL;

EXIT:
    return datarec;
}

#undef FUNC
#define FUNC "bxh_datarec_writeToElement"
/**
 * Replace the appropriate fields in the BXH datarec element
 * with those in the raw datarec.
 *
 * @param elemp BXH element pointer for the datarec
 * @param datarec pointer to raw datarec struct
 * @return 0 on success, non-zero on error.
 */
int
bxh_datarec_writeToElement(const BXHElementPtr elemp, bxhrawdatarec * datarec)
{
    int retval = 0;
    BXHElementPtr * nodearray = NULL;
    BXHElementPtr dimelem = NULL;
    int numnodes = 0;
    int nodenum;
    int fragnum;
    int fieldnum;
    int dimnum;
    char * offbuf = NULL;
    size_t offbuflen = 0;
    char * recsizebuf = NULL;
    size_t recsizebuflen = 0;
    const char * fieldlist[] = {
	"rasorigin",
	"dimension",
	"compression",
	"byteorder",
	"elementtype",
	"filename",
	"fileoffset",
	"filerecordsize",
	"filenameprintfdimensions",
	"filenameprintforigins"
    };

    /* get rid of old stuff */
    for (fieldnum = 0; fieldnum < (int)sizeof(fieldlist)/(int)sizeof(char *); fieldnum++) {
	if ((nodearray = bxh_getChildElementArray(elemp, fieldlist[fieldnum])) == NULL) {
	    numnodes = 0;
	} else {
	    for (numnodes = 0; nodearray[numnodes]; numnodes++) { /* null */ } 
	}
	for (nodenum = 0; nodenum < numnodes; nodenum++) {
	    BXHElementPtr noderef = NULL;
	    noderef = bxh_removeChildElement(elemp, nodearray[nodenum]);
	    bxh_element_unref(noderef); noderef = NULL;
	    bxh_element_unref(nodearray[nodenum]); nodearray[nodenum] = NULL;
	}
	free(nodearray); nodearray = NULL;
    }

    bxh_removeAutogenComments(elemp);
    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	bxhdimension * dimp = &datarec->dimensions[dimnum];
	if (strcmp(dimp->type, "z") == 0 ||
	    strncmp(dimp->type, "z-split", 7) == 0) {
	    double * dir = dimp->direction;
	    if (dir) {
		BXHCommentPtr tmpcomment = NULL;
		static char orientation[128] = "";
		double d0, d1, d2;
		orientation[0] = '\0';
		strcat(&orientation[0], "AUTOGEN: Orientation is ");
		d0 = fabs(dir[0]);
		d1 = fabs(dir[1]);
		d2 = fabs(dir[2]);
		if (d1 != 1 && d2 != 1 && d2 != 1)
		    strcat(orientation, "oblique ");
		if (d0 > d1 && d0 > d2)
		    strcat(orientation, "sagittal ");
		else if (d1 > d0 && d1 > d2)
		    strcat(orientation, "coronal ");
		else
		    strcat(orientation, "axial ");
		tmpcomment = bxh_appendComment(elemp, orientation, 1);
		bxh_comment_unref(tmpcomment);
	    }
	}
    }
    if (datarec->comments != NULL) {
	for (char ** commentp = datarec->comments; *commentp != NULL; commentp++) {
	    BXHCommentPtr tmpcomment = bxh_appendComment(elemp, *commentp, 1);
	    bxh_comment_unref(tmpcomment);
	}
    }

    {
	double originr = 0, origina = 0, origins = 0;
	int foundr = 0, founda = 0, founds = 0;
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    bxhdimension * dimp = &datarec->dimensions[dimnum];
	    if ((strncmp(dimp->type, "x", 1) == 0 ||
		 strncmp(dimp->type, "y", 1) == 0 ||
		 strncmp(dimp->type, "z", 1) == 0) &&
		(dimp->type[1] == '\0' ||
		 strncmp(&dimp->type[1], "-split", 6) == 0)) {
		double *direction = dimp->direction;
		double absdirR, absdirA, absdirS;
		if (direction == NULL) continue;
		absdirR = fabs(direction[0]);
		absdirA = fabs(direction[1]);
		absdirS = fabs(direction[2]);
		if (absdirR > absdirA && absdirR > absdirS) {
		    foundr = 1;
		    originr = dimp->origin;
		} else if (absdirA > absdirR && absdirA > absdirS) {
		    founda = 1;
		    origina = dimp->origin;
		} else if (absdirS > absdirR && absdirS > absdirA) {
		    founds = 1;
		    origins = dimp->origin;
		}
	    }
	}
	if (foundr && founda && founds) {
	    static char buf[1024];
	    sprintf(&buf[0], "%.15g %.15g %.15g", originr, origina, origins);
	    if (bxh_appendChildElement(elemp, "rasorigin", &buf[0]) == -1)
		goto FAIL;
	}
    }
    
    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	bxhdimension * dimp = &datarec->dimensions[dimnum];
	static char buf[1024];
	if ((dimelem = bxh_appendAndReturnChildElement(elemp, "dimension", NULL)) == NULL)
	    goto FAIL;
	if (bxh_setAttribute(dimelem, "type", dimp->type) == -1)
	    goto FAIL;
	if (dimp->units &&
	    bxh_appendChildElement(dimelem, "units", dimp->units) == -1)
	    goto FAIL;
	sprintf(&buf[0], "%d", (int)dimp->size);
	if (bxh_appendChildElement(dimelem, "size", &buf[0]) == -1)
	    goto FAIL;
	if (dimp->outputselect) {
	    char * curbuf = NULL;
	    char * selbuf = NULL;
	    unsigned int sind;
	    selbuf = (char *)malloc(sizeof(char) * ((int)ceil(log10((double)dimp->size)) + 2) * dimp->numoutputselect + 1);
	    curbuf = selbuf;
	    for (sind = 0; sind < dimp->numoutputselect; sind++) {
		curbuf += sprintf(curbuf, "%d", dimp->outputselect[sind]);
		if (sind < dimp->numoutputselect - 1) {
		    curbuf[0] = ' ';
		    curbuf++;
		}
	    }
	    curbuf[0] = '\0';
	    if (bxh_setAttribute(dimelem, "outputselect", selbuf) == -1)
		goto FAIL;
	    free(selbuf);
	}
	if (dimp->numdpstructs == 0 ||
	    (dimp->origin != 0 || dimp->gap != 0 || dimp->spacing != 0)) {
	    sprintf(&buf[0], "%.15g", dimp->origin);
	    if (bxh_appendChildElement(dimelem, "origin", &buf[0]) == -1)
		goto FAIL;
	    sprintf(&buf[0], "%.15g", dimp->gap);
	    if (bxh_appendChildElement(dimelem, "gap", &buf[0]) == -1)
		goto FAIL;
	    sprintf(&buf[0], "%.15g", dimp->spacing);
	    if (bxh_appendChildElement(dimelem, "spacing", &buf[0]) == -1)
		goto FAIL;
	}
	for (int dpstructind = 0; dpstructind < dimp->numdpstructs; dpstructind++) {
	    bxhdatapoints * dpstructp = &dimp->dpstructs[dpstructind];
	    int protect = 0;
	    unsigned int dpnum;
	    for (dpnum = 0; dpnum < dpstructp->numvalues; dpnum++) {
		if (strpbrk(dpstructp->values[dpnum], " \t\v\r\n")) {
		    /* found whitespace, need to protect */
		    protect = 1;
		    break;
		}
	    }
	    if (protect) {
		unsigned int dpnum = 0;
		BXHElementPtr dpnode = NULL;
		if ((dpnode = bxh_appendAndReturnChildElement(dimelem, "datapoints", NULL)) == NULL)
		    goto FAIL;
		if (dpstructp->label != NULL &&
		    bxh_setAttribute(dpnode, "label", dpstructp->label) != 0) {
		    bxh_element_unref(dpnode); dpnode = NULL;
		    goto FAIL;
		}
		for (dpnum = 0; dpnum < dpstructp->numvalues; dpnum++) {
		    if (bxh_appendChildElement(dpnode, "value", dpstructp->values[dpnum]) == -1) {
			bxh_element_unref(dpnode); dpnode = NULL;
			goto FAIL;
		    }
		}
		bxh_element_unref(dpnode); dpnode = NULL;
	    } else {
		BXHElementPtr dpnode = NULL;
		char * dpstr = NULL;
		unsigned int dpnum = 0;
		int dplen = 0;
		dpstr = (char *)malloc(sizeof(char) * 1);
		dpstr[0] = '\0';
		for (dpnum = 0; dpnum < dpstructp->numvalues; dpnum++) {
		    int len;
		    len = strlen(dpstructp->values[dpnum]);
		    dpstr = (char *)realloc(dpstr, sizeof(char) * (len + dplen + 1 + 1));
		    if (dpnum != 0) {
			strcpy(dpstr + dplen, " ");
			dplen++;
		    }
		    strncpy(dpstr + dplen, dpstructp->values[dpnum], len);
		    dplen += len;
		}
		dpstr[dplen] = '\0';
		if ((dpnode = bxh_appendAndReturnChildElement(dimelem, "datapoints", dpstr)) == NULL) {
		    bxh_element_unref(dpnode); dpnode = NULL;
		    free(dpstr);
		    goto FAIL;
		}
		free(dpstr);
		if (dpstructp->label != NULL &&
		    bxh_setAttribute(dpnode, "label", dpstructp->label) != 0) {
		    bxh_element_unref(dpnode); dpnode = NULL;
		    goto FAIL;
		}
		bxh_element_unref(dpnode); dpnode = NULL;
	    }
	}

	if (dimp->direction) {
	    int i;
	    for (i = 0; i < 3; i++) {
		if (dimp->direction[i] == -0) {
		    dimp->direction[i] = 0;
		}
	    }
	    sprintf(&buf[0], "%.7g %.7g %.7g", dimp->direction[0], dimp->direction[1], dimp->direction[2]);
	    if (bxh_appendChildElement(dimelem, "direction", &buf[0]) == -1)
		goto FAIL;
	}
	if (dimp->measurementframe) {
	    BXHElementPtr dmfelem = NULL;
	    int i;
	    for (i = 0; i < 9; i++) {
		if (dimp->measurementframe[i] == -0) {
		    dimp->measurementframe[i] = 0;
		}
	    }
	    if ((dmfelem = bxh_appendAndReturnChildElement(dimelem, "measurementframe", NULL)) == NULL)
		goto FAIL;
	    if (dimp->measurementframeversion != NULL) {
		if (bxh_setAttribute(dmfelem, "version", dimp->measurementframeversion) != 0) {
		    bxh_element_unref(dmfelem);
		    goto FAIL;
		}
	    }
	    sprintf(&buf[0], "%.7g %.7g %.7g",
		    dimp->measurementframe[0],
		    dimp->measurementframe[1],
		    dimp->measurementframe[2]);
	    if (bxh_appendChildElement(dmfelem, "vector", &buf[0]) != 0) {
		bxh_element_unref(dmfelem);
		goto FAIL;
	    }
	    sprintf(&buf[0], "%.7g %.7g %.7g",
		    dimp->measurementframe[3],
		    dimp->measurementframe[4],
		    dimp->measurementframe[5]);
	    if (bxh_appendChildElement(dmfelem, "vector", &buf[0]) != 0) {
		bxh_element_unref(dmfelem);
		goto FAIL;
	    }
	    sprintf(&buf[0], "%.7g %.7g %.7g",
		    dimp->measurementframe[6],
		    dimp->measurementframe[7],
		    dimp->measurementframe[8]);
	    if (bxh_appendChildElement(dmfelem, "vector", &buf[0]) != 0) {
		bxh_element_unref(dmfelem);
		goto FAIL;
	    }
	    bxh_element_unref(dmfelem); dmfelem = NULL;
	}

	bxh_element_unref(dimelem); dimelem = NULL;
    }

    if (datarec->compression &&
	bxh_appendChildElement(elemp, "compression", datarec->compression) == -1)
	    goto FAIL;

    if (datarec->msbfirstfrags &&
	bxh_appendChildElement(elemp, "byteorder", "msbfirst") == -1)
	    goto FAIL;
    else if (!datarec->msbfirstfrags &&
	     bxh_appendChildElement(elemp, "byteorder", "lsbfirst") == -1)
	goto FAIL;

    if (datarec->elemtype &&
	bxh_appendChildElement(elemp, "elementtype", datarec->elemtype) == -1)
	goto FAIL;
	
    if (datarec->frags)
	datarec->numfrags =
	    bxh_datarec_canonicalizeFrags(datarec->frags, datarec->numfrags);
    for (fragnum = 0; fragnum < datarec->numfrags; fragnum++) {
	static char buf[32];
	sprintf(&buf[0], "%lu", (unsigned long)datarec->frags[fragnum]->fileoffset);
	if (offbuf == NULL) {
	    offbuf = (char *)malloc(sizeof(char) * (strlen(buf) + 1));
	    offbuflen = 0;
	} else {
	    offbuf = (char *)realloc(offbuf, sizeof(char) * (offbuflen + strlen(buf) + 2));
	    offbuf[offbuflen++] = ' ';
	    offbuf[offbuflen] = '\0';
	}
	strcpy(&offbuf[offbuflen], &buf[0]);
	offbuflen += strlen(buf);
	sprintf(&buf[0], "%lu", (unsigned long)datarec->frags[fragnum]->fragsize);
	if (recsizebuf == NULL) {
	    recsizebuf = (char *)malloc(sizeof(char) * (strlen(buf) + 1));
	    recsizebuflen = 0;
	} else {
	    recsizebuf = (char *)realloc(recsizebuf, sizeof(char) * (recsizebuflen + strlen(buf) + 2));
	    recsizebuf[recsizebuflen++] = ' ';
	    recsizebuf[recsizebuflen] = '\0';
	}
	strcpy(&recsizebuf[recsizebuflen], &buf[0]);
	recsizebuflen += strlen(buf);
	if (fragnum == datarec->numfrags - 1 ||
	    strcmp(datarec->frags[fragnum]->filename, datarec->frags[fragnum+1]->filename) != 0) {
	    if (bxh_appendChildElement(elemp, "filename", datarec->frags[fragnum]->filename) == -1)
		goto FAIL;
	    if (bxh_appendChildElement(elemp, "fileoffset", &offbuf[0]) == -1)
		goto FAIL;
	    if (bxh_appendChildElement(elemp, "filerecordsize", &recsizebuf[0]) == -1)
		goto FAIL;
	    free(offbuf); offbuf = NULL; offbuflen = 0;
	    free(recsizebuf); recsizebuf = NULL; recsizebuflen = 0;
	}
    }

    goto EXIT;

FAIL:
    if (domutil_errorbuf[0]) {
	fprintf(stderr, FUNC ": %s\n", domutil_errorbuf);
    }
    retval = -1;

EXIT:
    if (nodearray) {
	int i;
	for (i = 0; nodearray[i]; i++) { bxh_element_unref(nodearray[i]); }
	free(nodearray);
    }
    if (dimelem) bxh_element_unref(dimelem);
    return retval;
}


#undef FUNC
#define FUNC "bxh_datarec_copy"
/**
 * This function returns a copy of datarec.  All dynamically allocated
 * storage referred to by datarec will be re-allocated for returned datarec.
 *
 * @param datarec pointer to bxhrawdatarec
 */
bxhrawdatarec *
bxh_datarec_copy(const bxhrawdatarec * datarec)
{
    bxhrawdatarec * newdatarec = NULL;
    int dimnum = 0;

    newdatarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
    memset(newdatarec, '\0', sizeof(bxhrawdatarec));
    newdatarec->numdims = datarec->numdims;
    newdatarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension)*datarec->numdims);
    memset(newdatarec->dimensions, '\0', sizeof(bxhdimension)*datarec->numdims);
    for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	bxhdimension * olddim = &datarec->dimensions[dimnum];
	bxhdimension * newdim = &newdatarec->dimensions[dimnum];
	newdim->type = strdup(olddim->type);
	if (olddim->units)
	    newdim->units = strdup(olddim->units);
	newdim->size = olddim->size;
	if (olddim->outputselect) {
	    newdim->outputselect = (int *)malloc(sizeof(int)*olddim->numoutputselect);
	    memcpy(newdim->outputselect, olddim->outputselect, sizeof(int)*olddim->numoutputselect);
	    newdim->numoutputselect = olddim->numoutputselect;
	}
	newdim->origin = olddim->origin;
	newdim->gap = olddim->gap;
	newdim->spacing = olddim->spacing;
	if (olddim->direction) {
	    newdim->direction = (double *)malloc(sizeof(double)*3);
	    memcpy(newdim->direction, olddim->direction, sizeof(double)*3);
	}
	if (olddim->measurementframe) {
	    newdim->measurementframe = (double *)malloc(sizeof(double)*9);
	    memcpy(newdim->measurementframe, olddim->measurementframe, sizeof(double)*9);
	}
	if (olddim->measurementframeversion) {
	    newdim->measurementframeversion = strdup(olddim->measurementframeversion);
	}
	if (olddim->numdpstructs > 0) {
	    newdim->numdpstructs = olddim->numdpstructs;
	    newdim->dpstructs = (bxhdatapoints *)malloc(sizeof(bxhdatapoints) * olddim->numdpstructs);
	}
	for (int dpstructind = 0; dpstructind < olddim->numdpstructs; dpstructind++) {
	    bxhdatapoints * olddpstructp = &olddim->dpstructs[dpstructind];
	    bxhdatapoints * newdpstructp = &newdim->dpstructs[dpstructind];
	    char ** olddatapoints = olddpstructp->values;
	    unsigned int dpnum;
	    memset(newdpstructp, '\0', sizeof(bxhdatapoints));
	    if (olddpstructp->label != NULL) {
		newdpstructp->label = strdup(olddpstructp->label);
	    }
	    char ** newdatapoints = (char **)malloc(sizeof(char *) * olddpstructp->numvalues);
	    newdpstructp->values = newdatapoints;
	    newdpstructp->numvalues = olddpstructp->numvalues;
	    for (dpnum = 0; dpnum < olddpstructp->numvalues; dpnum++) {
		newdatapoints[dpnum] = strdup(olddatapoints[dpnum]);
	    }
	}
    }
    newdatarec->elemtype = strdup(datarec->elemtype);
    newdatarec->msbfirstfrags = datarec->msbfirstfrags;
    newdatarec->numfrags = datarec->numfrags;
    if (datarec->frags) {
	int fragnum;
	newdatarec->frags = (bxhfragp *)malloc(sizeof(bxhfragp)*datarec->numfrags);
	for (fragnum = 0; fragnum < datarec->numfrags; fragnum++) {
	    bxhfragp oldfrag = NULL;
	    bxhfragp newfrag = NULL;
	    newdatarec->frags[fragnum] = (bxhfragp)malloc(sizeof(bxhfrag));
	    oldfrag = datarec->frags[fragnum];
	    newfrag = newdatarec->frags[fragnum];
	    newfrag->filename = strdup(oldfrag->filename);
	    newfrag->fileoffset = oldfrag->fileoffset;
	    newfrag->fragsize = oldfrag->fragsize;
	}
    }
    newdatarec->elemsize = datarec->elemsize;
    newdatarec->datasize = datarec->datasize;
    newdatarec->bxhpath = strdup(datarec->bxhpath);
    newdatarec->lastfp = NULL;
    newdatarec->lastfilename = NULL;
    return newdatarec;
}

#undef FUNC
#define FUNC "bxh_datarec_addcomment"
/**
 * This function adds a new comment to the datarec
 *
 * @param datarec pointer to bxhrawdatarec
 * @param comment the comment string
 * @return 0 on success, non-zero on failure
 */
int
bxh_datarec_addcomment(bxhrawdatarec * datarec,
		       const char * comment)
{
    int numcomments = 0;
    char ** commentp = NULL;
    char ** comments = datarec->comments;
    if (comments != NULL) {
	for (commentp = comments; *commentp != NULL; commentp++) {
	    numcomments++;
	}
    }
    comments = (char **)realloc(comments, sizeof(char *) * (numcomments + 2));
    if (comments == NULL) {
	return -1;
    }
    datarec->comments = comments;
    comments[numcomments] = strdup(comment);
    comments[numcomments + 1] = NULL;
    return 0;
}


#undef FUNC
#define FUNC "bxh_datarec_addfrag"
/**
 * This function adds a new frag to the datarec
 *
 * @param datarec pointer to bxhrawdatarec
 * @param filename pointer to filename (absolute or relative)
 * @param expandfilename if 1, assume #filename is relative to the current directory, and expand appropriately.  If #filename is already an absolute pathname, this has no effect.
 * @param fileoffset frag offset within file
 * @param fragsize size of frag
 * @param bxhpath if non-NULL, specifies the base path for relative filenames in the frag (typically the full path of the output BXH file).  If #bxhpath is a directory name, it must end with a path separator (e.g. slash '/').  If #filename (after expansion, see #expandfilename) is absolute and if #filename is found to be contained within (or below) the same directory as #bxhpath, then filename will be stored as a relative path.  #bxhpath is assumed to be relative to the current directory
 * @return 0 on success, non-zero on failure
 */
int
bxh_datarec_addfrag(bxhrawdatarec * datarec,
		    const char * filename,
		    size_t fileoffset,
		    size_t fragsize,
		    const char * bxhpath,
		    int expandfilename)
{
    int retval = 0;
    bxhfragp newfragp = NULL;
    size_t filelen;
    char * newbxhpath = NULL;
    char * newfilename = NULL;

    filelen = strlen(filename);
    if (filename == NULL || strlen(filename) == 0) {
	fprintf(stderr, FUNC ": filename is NULL or empty!\n");
	goto FAIL;
    }

    if (expandfilename) {
	newfilename = bxh_datarec_getfullfilename(NULL, filename);
	filename = newfilename;
    }
    newbxhpath = bxh_datarec_getfullfilename(NULL, bxhpath);
    bxhpath = newbxhpath;

    newfragp = (bxhfrag *)malloc(sizeof(bxhfrag));
    datarec->frags = (bxhfragp *)realloc(datarec->frags, sizeof(bxhfragp) * (datarec->numfrags + 1));
    datarec->frags[datarec->numfrags] = newfragp;
    datarec->numfrags++;
    newfragp->fileoffset = fileoffset;
    newfragp->fragsize = fragsize;
    if (bxhpath == NULL) {
	newfragp->filename = strdup(filename);
    } else {
	const char * basevol = NULL;
	const char * basedirs = NULL;
	const char * basefile = NULL;
	const char * fvol = NULL;
	const char * fdirs = NULL;
	const char * ffile = NULL;
	size_t basevollen = 0;
	size_t basedirslen = 0;
	size_t basefilelen = 0;
	size_t fvollen = 0;
	size_t fdirslen = 0;
	size_t ffilelen = 0;
	size_t baselen;
	const char * curptr = NULL;
	baselen = strlen(bxhpath);
	filelen = strlen(filename);

	curptr = bxhpath;
	basevollen = path_match_vol(curptr);
	if (basevollen) { basevol = curptr; }
	curptr += basevollen;
	basedirslen = path_match_dirs(curptr);
	if (basedirslen) { basedirs = curptr; }
	curptr += basedirslen;
	basefile = curptr;
	basefilelen = baselen - basedirslen;

	curptr = filename;
	fvollen = path_match_vol(curptr);
	if (fvollen) { fvol = curptr; }
	curptr += fvollen;
	fdirslen = path_match_dirs(curptr);
	if (fdirslen) { fdirs = curptr; }
	curptr += fdirslen;
	ffile = curptr;
	ffilelen = filelen - fdirslen;

	if (basevollen == fvollen &&
	    (basevollen == 0 || strncmp(basevol, fvol, basevollen) == 0) &&
	    basedirslen <= fdirslen &&
	    (basedirslen == 0 || strncmp(basedirs, fdirs, basedirslen) == 0)) {
	    newfragp->filename = (char *)malloc(sizeof(char)*((fdirslen-basedirslen)+ffilelen+1));
	    strcpy(newfragp->filename, &filename[basevollen+basedirslen]);
	} else {
	    newfragp->filename = strdup(filename);
	}
    }
    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    if (newfilename)
	free(newfilename);
    if (newbxhpath)
	free(newbxhpath);
    return retval;
}

#undef FUNC
#define FUNC "bxh_datarec_setcompression"
/**
 * This function sets (or unsets) the compression field.
 * Currently the only valid values are NULL, and "gzip".
 *
 * @param datarec pointer to bxhrawdatarec
 * @param compression the type of compression used by frags
 * @return 0 on success, non-zero on failure
 */
int
bxh_datarec_setcompression(bxhrawdatarec * datarec,
			   const char * compression)
{
    int retval = 0;
    if (compression == NULL) {
      datarec->compression = NULL;
    } else if (strcmp(compression, "gzip") == 0) {
      datarec->compression = strdup(compression);
    } else if (strcmp(compression, "gzipfrag") == 0) {
      datarec->compression = strdup(compression);
    } else {
      fprintf(stderr, FUNC ": Bad compression type %s\n", compression);
      goto FAIL;
    }
    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    return retval;
}

#undef FUNC
#define FUNC "bxh_datarec_datapoints_free"
void
bxh_datarec_datapoints_free(bxhdatapoints * dpstructp)
{
    if (dpstructp->label != NULL) {
	free(dpstructp->label);
	dpstructp->label = NULL;
    }
    if (dpstructp->values != NULL) {
	unsigned int dpnum;
	for (dpnum = 0; dpnum < dpstructp->numvalues; dpnum++) {
	    free(dpstructp->values[dpnum]);
	    dpstructp->values[dpnum] = NULL;
	}
    }
    free(dpstructp->values);
    dpstructp->values = NULL;
}

#undef FUNC
#define FUNC "bxh_datarec_dimdata_free"
/**
 * This function frees the data associated with a dimension
 * (but not the dimension itself).
 *
 * @param dimp pointer to bxhdimension
 */
void
bxh_datarec_dimdata_free(bxhdimension * dimp)
{
    free(dimp->type);
    dimp->type = NULL;
    free(dimp->units);
    dimp->units = NULL;
    if (dimp->outputselect) {
	free(dimp->outputselect);
	dimp->outputselect = NULL;
    }
    if (dimp->numdpstructs > 0) {
	for (int dpstructind = 0; dpstructind < dimp->numdpstructs; dpstructind++) {
	    bxh_datarec_datapoints_free(&dimp->dpstructs[dpstructind]);
	}
	free(dimp->dpstructs);
	dimp->dpstructs = NULL;
	dimp->numdpstructs = 0;
    }
    if (dimp->direction) {
	free(dimp->direction);
	dimp->direction = NULL;
    }
    if (dimp->measurementframe) {
	free(dimp->measurementframe);
	dimp->measurementframe = NULL;
    }
    if (dimp->measurementframeversion) {
	free(dimp->measurementframeversion);
	dimp->measurementframeversion = NULL;
    }
}

#undef FUNC
#define FUNC "bxh_datarec_dimensions_free"
/**
 * This function frees the data associated with all dimensions
 * in a datarec (and sets the numdims to 0)
 *
 * @param datarec pointer to bxhrawdatarec
 */
void
bxh_datarec_dimensions_free(bxhrawdatarec * datarec)
{
    int dimnum;
    if (datarec->dimensions) {
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    bxhdimension * dimp = &datarec->dimensions[dimnum];
	    bxh_datarec_dimdata_free(dimp);
	}
	free(datarec->dimensions);
	datarec->dimensions = NULL;
    }
    datarec->numdims = 0;
}

#undef FUNC
#define FUNC "bxh_datarec_fragdata_free"
/**
 * This function frees the data associated with a frag
 * (but not the frag itself).
 *
 * @param fragp pointer to bxhfrag
 */
void
bxh_datarec_fragdata_free(bxhfrag * fragp)
{
    if (fragp->filename) {
	free(fragp->filename);
	fragp->filename = NULL;
    }
}

#undef FUNC
#define FUNC "bxh_datarec_frags_free"
/**
 * This function frees the data associated with all frags
 * in a datarec (and sets the numfrags to 0)
 *
 * @param datarec pointer to bxhrawdatarec
 */
void
bxh_datarec_frags_free(bxhrawdatarec * datarec)
{
    int fragnum;
    if (datarec->frags) {
	for (fragnum = 0; fragnum < datarec->numfrags; fragnum++) {
	    bxhfragp curfragp = datarec->frags[fragnum];
	    bxh_datarec_fragdata_free(curfragp);
	    free(curfragp);
	}
	free(datarec->frags);
	datarec->frags = NULL;
    }
    datarec->numfrags = 0;
}

#undef FUNC
#define FUNC "bxh_datarec_free"
/**
 * This function frees the data associated with datarec, which typically
 * is generated by bxh_datarec_createFromElement().
 *
 * @param datarec pointer to bxhrawdatarec
 */
void
bxh_datarec_free(bxhrawdatarec * datarec)
{
    if (datarec == NULL)
	return;
    if (datarec->elemtype) {
	free(datarec->elemtype);
	datarec->elemtype = NULL;
    }
    if (datarec->dimensions) {
	int dimnum;
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    bxh_datarec_dimdata_free(&datarec->dimensions[dimnum]);
	}
	free(datarec->dimensions);
	datarec->dimensions = NULL;
    }
    bxh_datarec_frags_free(datarec);
    if (datarec->bxhpath) {
	free(datarec->bxhpath);
	datarec->bxhpath = NULL;
    }
    if (datarec->compression) {
	free(datarec->compression);
	datarec->compression = NULL;
    }
    if (datarec->lastfilename) {
	free(datarec->lastfilename);
	datarec->lastfilename = NULL;
    }
    if (datarec->lastfp) {
	if (datarec->lastfpisgz) {
	    gzclose((gzFile)datarec->lastfp);
	} else {
	    fclose((FILE *)datarec->lastfp);
	}
	datarec->lastfp = NULL;
    }
    if (datarec->readobj) {
	delete (readobj *)datarec->readobj;
	datarec->readobj = NULL;
    }
    if (datarec->comments) {
	for (char **commentp = datarec->comments; *commentp != NULL; commentp++) {
	    free(*commentp);
	    *commentp = NULL;
	}
	free(datarec->comments);
	datarec->comments = NULL;
    }
    free(datarec);
}

#undef FUNC
#define FUNC "bxh_datarec_constructSelector"
/**
 * This function constructs a (-1)-terminated selector array
 * from the given string, which should be a comma-separated list of
 * index ranges, which can be any of the following forms:
 *   <index>                         (scalar)
 *   <startindex>:<endindex>         (contiguous range)
 *   <startindex>:<step>:<endindex>  (non-contiguous range)
 * startindex and endindex are optional.
 * If startindex is empty, range starts at #min.
 * If endindex is empty, range ends at #max.
 * If string is empty, or NULL, it is equivalent to ":".
 * This can be used to construct selectors for
 * bxh_datarec_prepareForReadPermuteSelect().
 *
 * @param instr input string
 * @param min minimum index for bounds checking
 * @param max maximum index for bounds checking
 * @param retsizep number of indices in output list will be stored in the size_t pointed to by this argument.
 */
int *
bxh_datarec_constructSelector(const char * instr, size_t min, size_t max, size_t * retsizep)
{
    int * retval = NULL;
    size_t retsize = 0;
    char * selstr = NULL;
    char * curptr = NULL;
    char * commaptr = NULL;
    if (instr == NULL || instr[0] == '\0') {
	instr = ":";
    }
    selstr = strdup(instr);
    curptr = selstr;
    while (*curptr) {
	unsigned long start;
	unsigned long end;
	unsigned long step = 1;
	unsigned long ind;
	char * colonptr = NULL;
	if ((commaptr = (char *)strchr(curptr, ',')) != NULL) {
	    *commaptr = '\0';
	}
	start = strtoul(curptr, NULL, 10);
	end = start;
	if ((colonptr = (char *)strchr(curptr, ':')) != NULL) {
	    char * colon2ptr = NULL;
	    *colonptr = '\0';
	    if (colonptr == curptr)
		start = min; /* nothing before colon */
	    if ((colon2ptr = (char *)strchr(colonptr+1, ':')) != NULL) {
		/* get step */
		*colon2ptr = '\0';
		step = strtoul(colonptr+1, NULL, 10);
		colonptr = colon2ptr;
	    }
	    /* get end of range.
	     * if nothing after colon, use actual size of dimension */
	    if (colonptr[1])
		end = atol(colonptr+1);
	    else
		end = max;
	}
	if (start > max || end > max || start < min || end < min) {
	    fprintf(stderr, "Point range %u:%u out of range (must be between %u and %u).\n", (unsigned int)start, (unsigned int)end, (unsigned int)min, (unsigned int)max);
	    goto FAIL;
	}
	for (ind = start; ind <= end; ind += step) {
	    retval = (int *)realloc(retval, sizeof(int)*(retsize+1));
	    retval[retsize] = ind;
	    retsize++;
	}
	if (commaptr) {
	    curptr = commaptr + 1;
	} else {
	    curptr += strlen(curptr);
	}
    }
    retval = (int *)realloc(retval, sizeof(int)*(retsize+1));
    retval[retsize] = -1;
    if (retsizep) {
	*retsizep = retsize;
    }
    goto EXIT;

  FAIL:
    if (retval)
	free(retval);
    retval = NULL;

  EXIT:
    if (selstr)
	free(selstr);
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.88  2008/07/28 18:35:44  gadde
 * Fix relative path resolver bug
 *
 * Revision 1.87  2008/05/22 14:20:54  gadde
 * Reduce output precision for direction and measurement frame values.
 *
 * Revision 1.86  2008/05/22 13:46:50  gadde
 * Change "negative zero" to "regular zero" for direction and measurementframe
 * fields.
 *
 * Revision 1.85  2008/03/07 23:05:07  gadde
 * Stop using off_t for signed data
 *
 * Revision 1.84  2008/03/07 18:18:59  gadde
 * Forgot to clear some vectors before using them.
 *
 * Revision 1.83  2008/02/12 19:53:17  gadde
 * Only use zlib functions (gzopen, gzread, etc.) when we know we are reading
 * gzipped data.  Otherwise binary data that looks like gzipped data (but isn't)
 * will get read incorrectly.
 *
 * Revision 1.82  2007/12/07 17:41:54  gadde
 * Fix a memory leak.
 *
 * Revision 1.81  2007/02/05 18:01:01  gadde
 * win32 fix
 *
 * Revision 1.80  2007/02/01 18:11:39  gadde
 * Fix compression support
 *
 * Revision 1.79  2007/02/01 15:42:24  gadde
 * Fix misspelling.
 *
 * Revision 1.78  2007/02/01 15:39:22  gadde
 * Add compression field
 *
 * Revision 1.77  2007/01/16 19:34:14  gadde
 * Memory fixes.
 *
 * Revision 1.76  2007/01/09 18:04:24  gadde
 * Deal with gzipped files without a corresponding <filerecordsize>.
 *
 * Revision 1.75  2006/11/16 17:33:15  gadde
 * If rasorigin exists in input, use it!
 *
 * Revision 1.74  2006/05/31 14:32:15  gadde
 * Remove unnecessary consts
 *
 * Revision 1.73  2006/05/25 18:23:30  gadde
 * Add support for reading gzipped files.
 *
 * Revision 1.72  2006/04/07 19:08:04  gadde
 * Add implicit BIAC tensors and measurementframe
 *
 * Revision 1.71  2006/03/23 22:16:18  gadde
 * Use errno.h instead of sys/errno.h.
 *
 * Revision 1.70  2006/02/09 21:39:55  gadde
 * Write out rasorigin field.
 *
 * Revision 1.69  2005/07/07 15:10:51  gadde
 * Memory fixes.
 *
 * Revision 1.68  2005/06/17 14:03:49  gadde
 * win32 updates
 *
 * Revision 1.67  2005/06/15 16:28:50  gadde
 * Don't set units to the empty string if it doesn't exist.
 *
 * Revision 1.66  2005/05/10 19:46:57  gadde
 * Update AUTOGEN orientation comment.
 *
 * Revision 1.65  2005/05/09 20:45:45  gadde
 * Add orientation AUTOGEN comment.
 * Also fix a basepath calculation problem.
 *
 * Revision 1.64  2005/04/25 13:52:05  gadde
 * Not sure why this change helps but it does -- XXX TODO: look at
 * bxh_element_unref's in bxh_datarec_writeToElement.
 *
 * Revision 1.63  2005/03/15 15:06:46  gadde
 * When selecting, also do origin/spacing/gap even if datapoints exists.
 *
 * Revision 1.62  2005/03/14 21:10:39  gadde
 * Fix some permutation/selection bugs.
 * Also output origin/spacing/etc if non-zero, even if datapoints
 * field is populated.
 *
 * Revision 1.61  2004/12/20 18:52:53  gadde
 * Allow selector string to be NULL or empty.
 *
 * Revision 1.60  2004/12/14 19:39:03  gadde
 * Fix calculation of permuted dimension number.
 *
 * Revision 1.59  2004/11/12 15:23:00  gadde
 * Change arg to const.
 *
 * Revision 1.58  2004/09/14 15:14:24  gadde
 * Allow interchangable / and \ on WIN32.
 *
 * Revision 1.57  2004/07/07 17:01:50  gadde
 * Memory fixes.
 *
 * Revision 1.56  2004/07/06 14:04:31  gadde
 * Some memory fixes.
 *
 * Revision 1.55  2004/06/22 17:50:45  gadde
 * Clean up a little more when freeing memory.
 *
 * Revision 1.54  2004/06/21 20:44:27  gadde
 * free pages array.
 *
 * Revision 1.53  2004/06/21 17:37:07  gadde
 * Use proper array length for outputnewsizes.
 *
 * Revision 1.52  2004/06/18 19:21:36  gadde
 * Fix memory free() bug when selecting dimension datapoints.
 *
 * Revision 1.51  2004/06/17 22:53:08  gadde
 * Fix up data selection code.
 *
 * Revision 1.50  2004/06/15 20:43:17  gadde
 * Remove unused variable.
 *
 * Revision 1.49  2004/06/15 20:39:08  gadde
 * Remove incorrect absolute file name check.
 *
 * Revision 1.48  2004/06/15 16:16:11  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.47  2004/06/11 15:04:00  gadde
 * Remove unnecessary use of std::vector.
 *
 * Revision 1.46  2004/06/10 19:56:24  gadde
 * update docs and html
 *
 * Revision 1.45  2004/06/01 20:31:37  gadde
 * Some more memory fixes.
 *
 * Revision 1.44  2004/06/01 20:23:25  gadde
 * Fix strtol error check
 *
 * Revision 1.43  2004/05/31 16:38:07  gadde
 * Some memory fixes
 *
 * Revision 1.42  2004/05/28 21:16:55  gadde
 * Fix bug introduced in coordsinc.
 *
 * Revision 1.41  2004/05/28 20:26:26  gadde
 * Get rid of some STL usage (may bring it back).
 *
 * Revision 1.40  2004/05/27 22:00:55  gadde
 * Fixes for selector code (not completely fixed yet).
 *
 * Revision 1.39  2004/04/07 17:20:24  gadde
 * Minor bug fix and add generic selector parsing routine.
 *
 * Revision 1.38  2004/03/19 15:13:31  gadde
 * Major changes in datarec support, include a new 'prepare' stage
 * that initiates filtering, permutation.
 * bxh_datarec_readData has been changed to bxh_datarec_readRawData
 * to reflect this.
 * Added some fMRI QA measures.
 *
 * Revision 1.37  2004/03/12 16:29:04  gadde
 * Minor updates
 *
 * Revision 1.36  2004/01/22 14:33:05  gadde
 * Add double as a synonym for float64.
 *
 * Revision 1.35  2004/01/02 18:41:43  gadde
 * -Wall fixes
 *
 * Revision 1.34  2003/11/14 20:13:08  gadde
 * Performance enhancements.
 *
 * Revision 1.33  2003/11/13 21:10:41  gadde
 * Make getfullfilename a public function called bxh_datarec_getfullfilename.
 *
 * Revision 1.32  2003/10/22 17:15:42  gadde
 * Fix some bugs uncovered on AIX
 *
 * Revision 1.31  2003/10/22 16:08:48  gadde
 * Fix some compiler warnings
 *
 * Revision 1.30  2003/10/22 15:04:50  gadde
 * Minor bug fixes
 *
 * Revision 1.29  2003/10/21 19:31:26  gadde
 * Change from splitignorestart to more generic outputselect.
 *
 * Revision 1.28  2003/10/14 16:47:49  gadde
 * Hack to get rid of extra slices in Siemens Mosaic (maybe it can be applied generally to other formats with split dimensions)
 *
 * Revision 1.27  2003/08/21 18:39:24  gadde
 * Don't use memcpy for small copies.
 *
 * Revision 1.26  2003/08/21 16:01:47  gadde
 * Fix byte order.
 *
 * Revision 1.25  2003/08/19 19:37:09  gadde
 * Fix some -Wall warnings.
 *
 * Revision 1.24  2003/08/14 16:10:46  gadde
 * Update docs.
 *
 * Revision 1.23  2003/08/14 15:38:58  gadde
 * Update docs.
 *
 * Revision 1.22  2003/08/12 20:42:08  gadde
 * Add functions to deal with split dimensions and arbitrary
 * permutations of data.
 *
 * Revision 1.21  2003/07/29 15:19:12  gadde
 * Some -Wall fixes
 *
 * Revision 1.20  2003/07/25 20:43:52  gadde
 * Windows fixes for the C++ conversion of some files.
 * Also reordered some headers so "interface" is not
 * defined before including gdome headers.
 *
 * Revision 1.19  2003/07/21 16:46:49  gadde
 * Code-prettiness updates, for the most part, to further protect BXH
 * library users from particulars of DOM implementation (esp. C vs. C++).
 *
 * Revision 1.18  2003/06/18 16:06:13  gadde
 * fopen in binary mode
 *
 * Revision 1.17  2003/06/18 15:52:28  gadde
 * More filename fixes for Windows.
 *
 * Revision 1.16  2003/06/18 15:41:51  gadde
 * Debug.
 *
 * Revision 1.15  2003/06/17 21:06:59  gadde
 * Windows pathname update.
 *
 * Revision 1.14  2003/06/04 13:54:27  gadde
 * When writing to element, collapse multiple frags that have same
 * filename.
 *
 * Revision 1.13  2003/05/30 19:51:02  gadde
 * Allow simple filenames as basepath.
 *
 * Revision 1.12  2003/05/30 14:29:55  gadde
 * Oops -- initialize numdataoffsets to -1 as it was intended.
 *
 * Revision 1.11  2003/05/19 15:40:24  gadde
 * bxhpath never updated! oops.
 *
 * Revision 1.10  2003/05/16 19:56:44  gadde
 * Require a basepath for datarec.
 *
 * Revision 1.9  2003/05/14 21:44:12  gadde
 * Increase precision of output doubles.
 *
 * Revision 1.8  2003/05/13 14:12:29  gadde
 * Make more robust to empty (NULL) datarec fields.
 *
 * Revision 1.7  2003/05/10 18:19:34  gadde
 * Don't null-terminate frags.
 *
 * Revision 1.6  2003/04/17 19:43:02  gadde
 * Fix random problems in allocation, zero-terminating strings, and
 * missing increment operator.  Also use new bxh_appendAndReturnChildElement.
 *
 * Revision 1.5  2003/03/24 16:43:59  gadde
 * Add support for vector datapoints
 *
 * Revision 1.4  2003/02/12 19:29:29  gadde
 * Updated doxygen documentation.
 *
 * Revision 1.3  2003/01/28 17:20:34  gadde
 * Updated datapoints usage.
 *
 * Revision 1.2  2003/01/06 18:41:57  gadde
 * Many changes/updates to comments.
 * Minor bug fixes.
 *
 * Revision 1.1  2002/12/04 17:21:50  gadde
 * Adding new module files
 *
 */
