static const char rcsid[] = "$Id: bxh_dom_utils.cpp,v 1.41 2009-02-17 18:32:54 gadde Exp $";

/* dom_utils.c --
 *
 * Some useful XPath-based utitility functions for use in conjunction
 * with libgdome.
 *
 * Author: Syam Gadde (gadde@biac.duke.edu), July 2002.
 */

#include <ctype.h>
#include <limits.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <gdome.h>
#include <gdome-xpath.h>

#include "bxh_dom_utils.h"

/*** you didn't see this */
#ifdef __cplusplus
extern "C" {
#endif

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xmlmemory.h>
#include <libxml/hash.h>
typedef struct {
    GdomeNode super;
    const void *vtab;
    int refcnt;
    xmlNode *n;
    GdomeAccessType accessType;
    void *ll;
} Gdome_xml_Node;
extern void gdome_xmlSetOldNs (xmlDoc *doc, xmlNs *ns);
extern int gdome_xmlLinkNsDecl(xmlNode *elem, xmlChar *prefix, xmlChar *href);

#ifdef __cplusplus
}
#endif
/*** you didn't see that */

static char domutil_errorarray[2048];

/*
 * For functions that return an error, this string may contain
 * a textual description of the error.  All functions initially
 * set this to the null string.  Critical errors are reported
 * straight to stderr.
 */
#ifdef __cplusplus
extern "C" {
#endif
char * domutil_errorbuf = &domutil_errorarray[0];
#ifdef __cplusplus
}
#endif

/**
 * This function initializes the DOM implementation
 */
int
domutil_initialize()
{
    /* nothing to do for gdome */
    return 0;
}

/**
 * This function creates a comment with the given text and appends it
 * to the given parent element.
 *
 * @param doc the document needed to create new nodes
 * @param parent the parent of the new comment
 * @param valuestr text that will appear inside the comment
 *            If NULL, then the element will be empty.
 * @param prependnewline if non-zero, will prepend a newline before comment
 * @return The new child element, or NULL on error
 */
GdomeComment *
domutil_appendComment(GdomeDocument * doc,
		      GdomeElement * parent,
		      const char * valuestr,
		      int prependnewline)
{
    GdomeException exc = 0;
    GdomeDOMString * value = NULL;
    GdomeComment * comment = NULL;
    GdomeNode * tmpnode = NULL;
    GdomeNode * tmpnode2 = NULL;

    if (prependnewline) {
	value = gdome_str_mkref("\n");
	if ((tmpnode = (GdomeNode *)gdome_doc_createTextNode(doc, value, &exc)) == NULL) {
	    fprintf(stderr, "Document.createTextNode: failed\n\tException #%d\n", exc);
	    goto FAIL;
	}
	gdome_str_unref(value); value = NULL;
	if ((tmpnode2 = gdome_el_appendChild(parent, tmpnode, &exc)) == NULL) {
	    fprintf(stderr, "Element.appendChild: failed\n\tException #%d\n", exc);
	    goto FAIL;
	}
	if (exc) {
	    sprintf(domutil_errorbuf, "Appending text to imagedata failed\n\tException #%d\n", exc);
	    gdome_t_unref((GdomeText *)tmpnode, &exc); tmpnode = NULL;
	    goto FAIL;
	}
	gdome_t_unref((GdomeText *)tmpnode, &exc); tmpnode = NULL;
	gdome_n_unref(tmpnode2, &exc); tmpnode2 = NULL;
    }

    value = gdome_str_mkref_dup(valuestr);
    comment = gdome_doc_createComment(doc, value, &exc);
    if (exc) {
	fprintf(stderr, "Document.createComment: failed\n\tException #%d\n", exc);
	goto FAIL;
    }
    gdome_str_unref(value); value = NULL;

    if ((tmpnode = gdome_el_appendChild(parent, (GdomeNode *)comment, &exc)) == NULL) {
	fprintf(stderr, "Element.appendChild: failed\n\tException #%d\n", exc);
	goto FAIL;
    }
    gdome_n_unref(tmpnode, &exc); tmpnode = NULL;
    if (exc) {
	sprintf(domutil_errorbuf, "Appending comment to imagedata failed\n\tException #%d\n", exc);
	goto FAIL;
    }

    goto EXIT;

FAIL:
    comment = NULL;

EXIT:
    if (tmpnode) {
	gdome_n_unref(tmpnode, &exc);
	tmpnode = NULL;
    }
    if (tmpnode2) {
	gdome_n_unref(tmpnode2, &exc);
	tmpnode2 = NULL;
    }
    return comment;
}

/**
 * This function is used to create an element which encloses a simple
 * text value, and add this new element as a child to a given parent
 * element.  Caller should dereference the returned element using
 * gdome_el_unref() if it does not need to keep it around.
 *
 * @param doc the document needed to create new nodes
 * @param parent the parent of the new child
 * @param namestr tag name of the new child element
 * @param valuestr text that will appear inside the new child element.
 *            If NULL, then the element will be empty.
 * @param namespaceURI namespace for prefix, if given, or NULL
 * @return The new child element, or NULL on error
 */
GdomeElement *
domutil_appendNewChildWithTextValue(GdomeDocument * doc,
				    GdomeElement * parent,
				    const char * namestr,
				    const char * valuestr,
				    const char * namespaceURI)
{
    GdomeException exc = 0;
    char * qnamestr = NULL;
    GdomeDOMString * name = NULL;
    GdomeDOMString * value = NULL;
    GdomeDOMString * ns = NULL;
    GdomeElement * retelem = NULL;
    GdomeNode * tmpnode = NULL;
    GdomeText * tmptxt = NULL;
    const char * tempnamespaceprefix = "dummyns:";

    domutil_errorbuf[0] = '\0';

    if (namespaceURI && !strchr(namestr, ':')) {
	qnamestr = (char *)malloc(sizeof(char *) * (strlen(namestr) + strlen(tempnamespaceprefix) + 1));
	strcpy(qnamestr, tempnamespaceprefix);
	strcat(qnamestr, namestr);
    } else {
	qnamestr = strdup(namestr);
    }

    if (namespaceURI && (ns = gdome_str_mkref_dup(namespaceURI)) == NULL) {
	sprintf(domutil_errorbuf, "domutil_appendNewChildWithValue: create string failed\n");
	goto FAIL;
    }
    if ((name = gdome_str_mkref_dup(qnamestr)) == NULL) {
	sprintf(domutil_errorbuf, "domutil_appendNewChildWithValue: create string failed\n");
	goto FAIL;
    }
    if ((namespaceURI && (retelem = gdome_doc_createElementNS(doc, ns, name, &exc)) == NULL) ||
	(!namespaceURI && (retelem = gdome_doc_createElement(doc, name, &exc)) == NULL)) {
	sprintf(domutil_errorbuf, "Create '%s' element failed\n\tException #%d\n", namestr, exc);
	goto FAIL;
    }
    gdome_str_unref(name); name = NULL;
    if ((tmpnode = gdome_el_appendChild(parent, (GdomeNode *)retelem, &exc)) != (GdomeNode *)retelem) {
	GdomeException newexc = 0;
	GdomeDOMString * tmpstr = NULL;
	tmpstr = gdome_el_nodeName(parent, &newexc);
	sprintf(domutil_errorbuf, "Appending '%s' to parent '%s' failed\n\tException #%d\n", namestr, tmpstr->str, exc);
	gdome_str_unref(tmpstr); tmpstr = NULL;
	gdome_n_unref(tmpnode, &exc); tmpnode = NULL;
	goto FAIL;
    }
    gdome_n_unref(tmpnode, &exc); tmpnode = NULL;
    if (valuestr) {
	value = gdome_str_mkref_dup(valuestr);
	if ((tmptxt = gdome_doc_createTextNode(doc, value, &exc)) == NULL) {
	    sprintf(domutil_errorbuf, "Document.createTextNode failed\n\tException #%d\n", exc);
	    goto FAIL;
	}
	gdome_str_unref(value); value = NULL;
	if ((tmpnode = gdome_el_appendChild(retelem, (GdomeNode *)tmptxt, &exc)) != (GdomeNode *)tmptxt) {
	    sprintf(domutil_errorbuf, "Appending text to '%s' failed\n\tException #%d\n", namestr, exc);
	    goto FAIL;
	}
	gdome_n_unref(tmpnode, &exc); tmpnode = NULL;
	gdome_t_unref(tmptxt, &exc); tmptxt = NULL;
    }

    goto EXIT;

FAIL:
    if (retelem)
	gdome_el_unref(retelem, &exc);
    retelem = NULL;

EXIT:
    if (qnamestr)
	free(qnamestr);
    if (ns)
	gdome_str_unref(ns);
    if (name)
	gdome_str_unref(name);
    if (value)
	gdome_str_unref(value);
    if (tmpnode)
	gdome_n_unref(tmpnode, &exc);
    if (tmptxt)
	gdome_t_unref(tmptxt, &exc);
    return retelem;
}

/**
 * This function is used to create an attribute which encloses a simple
 * text value, and add this new attribute as a child to a given parent
 * element.  Caller should dereference the returned attribute using
 * gdome_a_unref() if it does not need to keep it around.
 *
 * @param doc the document needed to create new nodes
 * @param parent the parent of the new attribute
 * @param namestr tag name of the new attribute element
 * @param valuestr text that will appear inside the new attribute.
 *            If NULL, then the attribute will be empty.
 * @param namespaceURI namespace for prefix, if given, or NULL
 * @return 0 on success, -1 on error
 */
int
domutil_setAttributeWithTextValue(GdomeDocument * doc,
				  GdomeElement * parent,
				  const char * namestr,
				  const char * valuestr,
				  const char * namespaceURI)
{
    GdomeException exc = 0;
    char * qnamestr = NULL;
    GdomeDOMString * name = NULL;
    GdomeDOMString * value = NULL;
    GdomeDOMString * ns = NULL;
    GdomeDOMString * tmpstr = NULL;
    const char * tempnamespaceprefix = "dummyns:";
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if (namespaceURI && !strchr(namestr, ':') && strcmp(namestr, "xmlns") != 0) {
	qnamestr = (char *)malloc(sizeof(char *) * (strlen(namestr) + strlen(tempnamespaceprefix) + 1));
	strcpy(qnamestr, tempnamespaceprefix);
	strcat(qnamestr, namestr);
    } else {
	qnamestr = strdup(namestr);
    }

    if (namespaceURI && (ns = gdome_str_mkref_dup(namespaceURI)) == NULL) {
	sprintf(domutil_errorbuf, "domutil_appendNewChildWithValue: create string failed\n");
	goto FAIL;
    }
    if ((name = gdome_str_mkref_dup(qnamestr)) == NULL) {
	sprintf(domutil_errorbuf, "domutil_appendNewChildWithValue: create string failed (namestr may be NULL?)\n");
	goto FAIL;
    }
    if ((value = gdome_str_mkref_dup(valuestr)) == NULL) {
	sprintf(domutil_errorbuf, "domutil_appendNewChildWithValue: create string failed (valuestr may be NULL?)\n");
	goto FAIL;
    }
    if (namespaceURI) {
	if (strcmp(namestr, "xmlns") == 0) {
	    /* a bug in gdome2 forces us to do this -- cover your eyes! */
	    gdome_xmlLinkNsDecl (((Gdome_xml_Node *)parent)->n, (xmlChar *)NULL, (xmlChar *)valuestr);
	} else {
	    gdome_el_setAttributeNS(parent, ns, name, value, &exc);
	}
    } else {
	gdome_el_setAttribute(parent, name, value, &exc);
    }
    if (exc != 0) {
	GdomeException newexc = 0;
	tmpstr = gdome_el_nodeName(parent, &newexc);
	sprintf(domutil_errorbuf, "Appending '%s' to parent '%s' failed\n\tException #%d\n", namestr, tmpstr->str, exc);
	gdome_str_unref(tmpstr); tmpstr = NULL;
	goto FAIL;
    }
    gdome_str_unref(name); name = NULL;
    gdome_str_unref(value); value = NULL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (qnamestr)
	free(qnamestr);
    if (ns)
	gdome_str_unref(ns);
    if (name)
	gdome_str_unref(name);
    if (value)
	gdome_str_unref(value);
    return retval;
}

/**
 * This function evaluates an XPath expression in a given context node.
 *
 * @param ctxt the context node in which to evaluate the xpath expression
 * @param exprstr an XPath expression
 * @param nodetype a GDOME type to cast the result
 * @return the result of evaluation, or NULL on error
 */
GdomeXPathResult *
domutil_getxpresult(GdomeNode * ctxt, const char * exprstr, unsigned short int nodetype)
{
    GdomeException exc = 0;
    GdomeXPathEvaluator * eval = NULL;
    GdomeDOMString * expr = NULL;
    GdomeXPathResult * result = NULL;
    GdomeXPathNSResolver * nsresolv = NULL;
    char * newexprstr = NULL;

    domutil_errorbuf[0] = '\0';

    newexprstr = (char *)malloc(sizeof(char)*(strlen(exprstr)+16));
    switch (nodetype) {
    case GDOME_STRING_TYPE:
	sprintf(newexprstr, "string(%s)", exprstr);
	break;
    case GDOME_NUMBER_TYPE:
	sprintf(newexprstr, "number(%s)", exprstr);
	break;
    default:
	sprintf(newexprstr, "%s", exprstr);
	break;
    }

    eval = gdome_xpeval_mkref();

    if ((nsresolv = gdome_xpeval_createNSResolver(eval, ctxt, &exc)) == NULL) {
	sprintf(domutil_errorbuf, "Creating NSResolver failed.\n\tException #%d\n", exc);
	goto FAIL;
    }
    expr = gdome_str_mkref_dup(newexprstr);
    if ((result = gdome_xpeval_evaluate(eval, expr, ctxt, nsresolv, nodetype, NULL, &exc)) == NULL) {
	sprintf(domutil_errorbuf, "Attempt to find '%s' failed.\n\tException #%d\n", exprstr, exc);
	goto FAIL;
    }
    gdome_str_unref(expr); expr = NULL;
    
    goto EXIT;

FAIL:
    if (result)
	gdome_xpresult_unref(result, &exc);
    result = NULL;

EXIT:
    if (newexprstr)
	free(newexprstr);
    if (nsresolv)
	gdome_xpnsresolv_unref(nsresolv, &exc);
    if (expr)
	gdome_str_unref(expr);
    if (eval)
	gdome_xpeval_unref(eval, &exc);
    return result;
}

/**
 * Given an XPath expression and a context node, this function returns
 * a newly malloc()ed array of pointers to all matching nodes.  The
 * pointer to the new array will be stored in the location pointed to by
 * retp.  The returned array must eventually be free()d by the caller,
 * and all nodes within the array should also be dereferenced by the
 * caller with gdome_n_unref().  Both can be done using
 * domutil_freeNodeArray(), if desired.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param nodetype type of returned node list (should be a node list type)
 * @param retp pointer to result
 * @return number of nodes in array, or -1 on error.  On error, *retp
 *          is unchanged.
 */
int
domutil_getNodeArray(GdomeNode * ctxt, const char * path, unsigned short int nodetype, GdomeNode *** retp)
{
    GdomeException exc = 0;
    GdomeXPathResult * result = NULL;
    GdomeNode * tmpnode = NULL;
    GdomeNode ** nodearray = NULL;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if ((result = domutil_getxpresult(ctxt, path, nodetype)) == NULL) {
	sprintf(domutil_errorbuf, "getNodeArray: Error getting path '%s'\n", path);
	goto FAIL;
    }
    retval = 0;
    while ((tmpnode = gdome_xpresult_iterateNext(result, &exc)) != NULL) {
	nodearray = (GdomeNode **)realloc(nodearray, sizeof(GdomeNode *) * (retval + 2));
	if (nodearray == NULL)
	    goto FAIL;
	nodearray[retval] = tmpnode;
	retval++;
    }
    if (retval == 0) {
	sprintf(domutil_errorbuf, "getNodeArray: Error getting path '%s'\n", path);
	goto FAIL;
    }
    nodearray[retval] = NULL;
    *retp = nodearray;
    goto EXIT;

FAIL:
    if (nodearray) {
	int i;
	for (i = 0; i < retval; i++) {
	    gdome_n_unref(nodearray[i], &exc); nodearray[i] = NULL;
	}
	free(nodearray);
    }
    retval = -1;

EXIT:
    if (result)
	gdome_xpresult_unref(result, &exc);
    return retval;
}

/**
 * Given a name and a parent node, this function returns a newly malloc()ed
 * array of pointers to all children of the parent whose name matches the
 * given name.  The pointer to the new array will be stored in the location
 * pointed to by retp.  The returned array must eventually be free()d by the
 * caller, and all nodes within the array should also be dereferenced by the
 * caller with gdome_n_unref().  Both can be done using
 * domutil_freeNodeArray(), if desired.
 *
 * @param parent parent node
 * @param name child node name
 * @param ns child node namespace (can be NULL for empty namespace)
 * @param retp pointer to result
 * @return number of nodes in array, or -1 on error.  On error, *retp
 *          is unchanged.
 */
int
domutil_getChildArray(GdomeElement * parent, const char * name, const char * ns, GdomeNode *** retp)
{
    GdomeException exc = 0;
    GdomeNode ** nodearray = NULL;
    int retval = 0;
    GdomeNode * curchild = NULL;
    GdomeNode * lastchild = NULL;

    domutil_errorbuf[0] = '\0';

    retval = 0;
    for (curchild = gdome_el_firstChild(parent, &exc); curchild; curchild = gdome_n_nextSibling(lastchild, &exc)) {
	GdomeDOMString * childname = NULL;
	GdomeDOMString * childns = NULL;
	if (lastchild) {
	    gdome_n_unref(lastchild, &exc);
	    lastchild = NULL;
	}
	if ((childname = gdome_n_localName(curchild, &exc)) != NULL &&
	    strcmp(childname->str, name) == 0) {
	    childns = gdome_n_namespaceURI(curchild, &exc);
	    if ((childns == NULL && ns == NULL) ||
		(childns != NULL && ns != NULL &&
		 strcmp(childns->str, ns) == 0)) {
		GdomeNode ** newnodearray = NULL;
		newnodearray = (GdomeNode **)realloc(nodearray, sizeof(GdomeNode *) * (retval + 2));
		if (newnodearray == NULL)
		    goto FAIL;
		nodearray = newnodearray;
		nodearray[retval] = curchild;
		gdome_n_ref(curchild, &exc);
		retval++;
	    }
	}
	if (childname) {
	    gdome_str_unref(childname); childname = NULL;
	}
	if (childns) {
	    gdome_str_unref(childns); childns = NULL;
	}
	lastchild = curchild;
    }
    if (lastchild) {
	gdome_n_unref(lastchild, &exc);
	lastchild = NULL;
    }
    if (retval == 0)
	goto FAIL;
    nodearray[retval] = NULL;
    *retp = nodearray;
    goto EXIT;

FAIL:
    if (nodearray) {
	int i;
	for (i = 0; i < retval; i++) {
	    gdome_n_unref(nodearray[i], &exc); nodearray[i] = NULL;
	}
	free(nodearray);
    }
    retval = -1;

EXIT:
    return retval;
}

/**
 * A convenient way to free a node array of the kind returned by
 * domutil_getNodeArray().  Send a *pointer to* the pointer
 * to the node pointer array -- i.e. the same argument you sent
 * to domutil_getNodeArray();
 *
 * @param nap pointer to the location holding a pointer to a node pointer array
 */
void
domutil_freeNodeArray(GdomeNode *** nap)
{
    GdomeException exc = 0;
    GdomeNode ** cap;

    domutil_errorbuf[0] = '\0';

    for (cap = *nap; *cap != NULL; cap++) {
	gdome_n_unref(*cap, &exc);
    }
    free(*nap);
/*     *nap = NULL; */
}

/**
 * Given an XPath expression and a context node, this function returns
 * a pointer to the single node that matches the expression.  The pointer
 * to the node will be stored in the location pointed to by retp.
 * The returned node eventually should be dereferenced by the caller with
 * gdome_n_unref().  It is an error if the XPath expression matches more
 * than one node.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return 0 on success, or -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getSingleNode(GdomeNode * ctxt, const char * path, GdomeNode ** retp)
{
    GdomeException exc = 0;
    GdomeNode ** nodearray = NULL;
    int numnodes;
    int retval = 0;
    
    domutil_errorbuf[0] = '\0';

    if ((numnodes = domutil_getNodeArray(ctxt, path, GDOME_UNORDERED_NODE_ITERATOR_TYPE, &nodearray)) == -1)
	goto FAIL;
    if (numnodes != 1) {
	sprintf(domutil_errorbuf, "getSingleNode: path '%s' has %d elements (should be 1)\n", path, numnodes);
	goto FAIL;
    }
    *retp = nodearray[0];
    goto EXIT;

FAIL:
    retval = -1;
    if (numnodes != -1) {
	int i;
	for (i = 0; i < numnodes; i++) {
	    gdome_n_unref(nodearray[i], &exc);
	}
    }

EXIT:
    if (nodearray)
	free(nodearray);
    return retval;
}

/**
 * Given an XPath expression and a context node, this function returns
 * a the double-precision floating point number stored in the node that
 * matches the expression.  The double will be stored in the location
 * pointed to by retp.  It is an error if the XPath expression matches
 * more than one node, or if the text in the node is not numeric.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return 0 on success, or -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getDoubleValue(GdomeNode * ctxt, const char * path, double * retp)
{
    GdomeException exc = 0;
    GdomeXPathResult * result = NULL;
    double value;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if ((result = domutil_getxpresult(ctxt, path, GDOME_NUMBER_TYPE)) == NULL)
	goto FAIL;
    
    value = gdome_xpresult_numberValue(result, &exc);
    if (exc != 0)
	goto FAIL;
    *retp = value;
    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (result)
	gdome_xpresult_unref(result, &exc);
    return retval;
}

/**
 * Given an XPath expression and a context node, this function returns
 * an array of double-precision floating point numbers generated from
 * the space-separated list of numbers stored in the node that
 * matches the expression.  A pointer to the new double array is stored
 * in the location pointed to by retp.  It is an error if the XPath
 * expression matches more than one node, or if the text in the node is
 * not a space-separated list of numbers.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return number of items in list, or -1 on error.  On error, *retp
 *          is unchanged.
 */
int
domutil_getDoubleListValue(GdomeNode * ctxt, const char * path, double ** retp)
{
    GdomeException exc = 0;
    GdomeXPathResult * result = NULL;
    GdomeDOMString * valdomstr = NULL;
    char * content = NULL;
    char * tmpcontent = NULL;
    char * endptr = NULL;
    double value;
    int numvalues;
    double * dlist = NULL;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if (path[0] == '.' && path[1] == '\0') {
	if (domutil_getElementStringValue(ctxt, &content) != 0) {
	    goto FAIL;
	}
    } else {
	if ((result = domutil_getxpresult(ctxt, path, GDOME_STRING_TYPE)) == NULL)
	    goto FAIL;
    
	valdomstr = gdome_xpresult_stringValue(result, &exc);
	if (exc != 0)
	    goto FAIL;
	content = (char *)valdomstr->str;
    }

    endptr = NULL;
    numvalues = 0;
    tmpcontent = content;
    while (!((value = strtod(tmpcontent, &endptr)) == 0 &&
	     (endptr == tmpcontent))) {
	numvalues++;
	tmpcontent = endptr;
    }
    if (numvalues == 0 ||
	strspn(tmpcontent, " \t\r\n\f\v") != strlen(tmpcontent)) {
	sprintf(domutil_errorbuf, "getDoubleListValue: content of '%s' ('%s') is not a list of doubles\n", path, content);
	goto FAIL;
    }
    if ((dlist = (double *)malloc(sizeof(double)*numvalues)) == NULL)
	goto FAIL;
    tmpcontent = content;
    numvalues = 0;
    while (!((value = strtod(tmpcontent, &endptr)) == 0 &&
	     (endptr == tmpcontent))) {
	dlist[numvalues] = value;
	numvalues++;
	tmpcontent = endptr;
    }
    if (strspn(endptr, " \t\r\n\f\v") != strlen(endptr)) {
	sprintf(domutil_errorbuf, "domutil_getDoubleListValue: content of '%s' ('%s') is not a valid list of numbers\n", path, content);
	goto FAIL;
    }
    if (valdomstr) {
	gdome_str_unref(valdomstr); valdomstr = NULL;
    } else if (content) {
	free(content); content = NULL;
    }
    retval = numvalues;
    *retp = dlist;
    goto EXIT;

FAIL:
    if (dlist)
	free(dlist);
    retval = -1;

EXIT:
    if (valdomstr)
	gdome_str_unref(valdomstr);
    else if (content)
	free(content);
    if (result)
	gdome_xpresult_unref(result, &exc);
    return retval;
}

/**
 * Given an XPath expression and a context node, this function returns
 * a the (signed) integer stored in the node that matches the expression.
 * The integer will be stored in the location pointed to by retp.  It is
 * an error if the XPath expression matches more than one node, or if the
 * text in the node is not numeric, or if the numeric value cannot be
 * represented as a 32-bit signed integer.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return 0 on success, or -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getIntValue(GdomeNode * ctxt, const char * path, int * retp)
{
    double dval;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if (domutil_getDoubleValue(ctxt, path, &dval) == -1)
	goto FAIL;
    if (dval != floor(dval)) {
	sprintf(domutil_errorbuf, "domutil_getIntValue: content of '%s' is not an integer (%g)\n", path, dval);
	goto FAIL;
    }
    if (dval > (double)INT_MAX || dval < (double)INT_MIN) {
	sprintf(domutil_errorbuf, "domutil_getIntValue: content of '%s' results in integer over/underflow (%g)\n", path, dval);
	goto FAIL;
    }
    *retp = (int)dval;
    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    return retval;
}

/**
 * Given an XPath expression and a context node, this function creates
 * an array of (signed) integers generated from the space-separated list
 * of numbers stored in the node that matches the expression.  A pointer
 * to the integer array will be stored in the location pointed to by retp.
 * It is an error if the XPath expression matches more than one node, or
 * if the text in the node is not a space-separated list of numbers, or
 * if the numbers cannot be represented as 32-bit signed integers.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return number of items in list, or -1 on error.  On error, *retp
 *          is unchanged.
 */
int
domutil_getIntListValue(GdomeNode * ctxt, const char * path, int ** retp)
{
    double * dlist = NULL;
    double * curdlist = NULL;
    int * ilist = NULL;
    int numvals;
    int valnum;

    domutil_errorbuf[0] = '\0';

    if ((numvals = domutil_getDoubleListValue(ctxt, path, &dlist)) == -1)
	goto FAIL;
    curdlist = dlist;
    if ((ilist = (int *)malloc(sizeof(int) * numvals)) == NULL)
	goto FAIL;
    *retp = ilist;
    valnum = 0;
    while (valnum++ < numvals) {
	double dval = *curdlist;
	if (dval != floor(dval)) {
	    sprintf(domutil_errorbuf, "domutil_getIntListValue: content of '%s' is not an integer (%g)\n", path, dval);
	    goto FAIL;
	}
	if (dval > (double)INT_MAX || dval < (double)INT_MIN) {
	    sprintf(domutil_errorbuf, "domutil_getIntListValue: content of '%s' results in integer over/underflow (%g)\n", path, dval);
	    goto FAIL;
	}
	*ilist = (int)dval;
	ilist++;
	curdlist++;
    }
    goto EXIT;

FAIL:
    if (ilist)
	free(ilist);
    numvals = -1;

EXIT:
    if (dlist)
	free(dlist);
    return numvals;
}

/**
 * Given an XPath expression and a context node, this function returns
 * a the unsigned integer stored in the node that matches the expression.
 * The integer will be stored in the location pointed to by retp.  It is
 * an error if the XPath expression matches more than one node, or if the
 * text in the node is not numeric, or if the numeric value cannot be
 * represented as a 32-bit unsigned integer.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return 0 on success, or -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getUnsignedIntValue(GdomeNode * ctxt, const char * path, unsigned int * retp)
{
    double dval;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if (domutil_getDoubleValue(ctxt, path, &dval) == -1)
	goto FAIL;
    if (dval != floor(dval)) {
	sprintf(domutil_errorbuf, "domutil_getUnsignedIntValue: content of '%s' is not an integer (%g)\n", path, dval);
	goto FAIL;
    }
    if (dval < 0 || dval > (double)UINT_MAX) {
	sprintf(domutil_errorbuf, "domutil_getUnsignedIntValue: content of '%s' results in unsigned integer over/underflow (%g)\n", path, dval);
	goto FAIL;
    }
    *retp = (unsigned int)dval;
    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    return retval;
}

/**
 * Given an XPath expression and a context node, this function creates
 * an array of unsigned integers generated from the space-separated list
 * of numbers stored in the node that matches the expression.  A pointer
 * to the unsigned integer array will be stored in the location pointed
 * to by retp.  It is an error if the XPath expression matches more than
 * one node, or if the text in the node is not a space-separated list of
 * numbers, or if the numbers cannot be represented as 32-bit unsigned
 * integers.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return number of items in list, or -1 on error.  On error, *retp
 *          is unchanged.
 */
int
domutil_getUnsignedIntListValue(GdomeNode * ctxt, const char * path, unsigned int ** retp)
{
    double * dlist = NULL;
    unsigned int * ilist = NULL;
    int numvals;
    int valnum;

    domutil_errorbuf[0] = '\0';

    if ((numvals = domutil_getDoubleListValue(ctxt, path, &dlist)) == -1)
	goto FAIL;
    if ((ilist = (unsigned int *)malloc(sizeof(int) * numvals)) == NULL)
	goto FAIL;
    *retp = ilist;
    valnum = 0;
    while (valnum++ < numvals) {
	double dval = *dlist;
	if (dval != floor(dval)) {
	    sprintf(domutil_errorbuf, "domutil_getUnsignedIntListValue: content of '%s' is not an integer (%g)\n", path, dval);
	    goto FAIL;
	}
	if (dval < 0 || dval > (double)UINT_MAX) {
	    sprintf(domutil_errorbuf, "domutil_getUnsignedIntListValue: content of '%s' results in unsigned integer over/underflow (%g)\n", path, dval);
	    goto FAIL;
	}
	*ilist = (unsigned int)dval;
	ilist++;
	dlist++;
    }
    goto EXIT;

FAIL:
    if (dlist)
	free(dlist);
    if (ilist)
	free(ilist);
    numvals = -1;

EXIT:
    return numvals;
}

/**
 * Given an XPath expression and a context node, this function creates
 * an array of (signed) long long integers generated from the space-separated
 * list of numbers stored in the node that matches the expression.  A pointer
 * to the long long int array will be stored in the location pointed to by retp.
 * It is an error if the XPath expression matches more than one node, or
 * if the text in the node is not a space-separated list of numbers, or
 * if the numbers cannot be represented as 32-bit signed integers.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return number of items in list, or -1 on error.  On error, *retp
 *          is unchanged.
 */
int
domutil_getLongLongIntListValue(GdomeNode * ctxt, const char * path, long long int ** retp)
{
    char * ptr = NULL;
    char * curptr = NULL;
    long long int * ilist = NULL;
    int numvals;
    long long int curval = 0;
    long long int threshmin = -1 * (LLONG_MIN / 10);
    long long int threshmax = LLONG_MAX / 10;

    domutil_errorbuf[0] = '\0';

    if (domutil_getStringValue(ctxt, path, &ptr) == -1)
	goto FAIL;
    numvals = 0;
    curptr = ptr;
    while (*curptr != '\0') {
	int isnegative = 0;
	while (isspace(*curptr)) {
	    curptr++;
	}
	if (*curptr == '+') {
	    /* no-op */
	    curptr++;
	}
	if (*curptr == '-') {
	    isnegative = 1;
	    curptr++;
	}
	if (*curptr == '\0') {
	    break;
	}
	if (!isdigit(*curptr)) {
	    sprintf(domutil_errorbuf, "domutil_getLongLongIntListValue: content of '%s' has invalid character starting at: %s\n", path, curptr);
	    goto FAIL;
	}
	curval = 0;
	while (isdigit(*curptr)) {
	    if (isnegative) {
		if (curval > threshmin) {
		    sprintf(domutil_errorbuf, "domutil_getLongLongIntListValue: content of '%s' has a value that will underflow starting at character: %s\n", path, curptr);
		    goto FAIL;
		}
	    } else {
		if (curval > threshmax) {
		    sprintf(domutil_errorbuf, "domutil_getLongLongIntListValue: content of '%s' has a value that will overflow starting at character: %s\n", path, curptr);
		    goto FAIL;
		}
	    }
	    curval *= 10;
	    curval += *curptr - '0';
	    curptr++;
	}
	if (isnegative) {
	  curval *= -1;
	}
	ilist = (long long int *)realloc(ilist, sizeof(long long int) * (numvals + 1));
	if (ilist == NULL) {
	    sprintf(domutil_errorbuf, "domutil_getLongLongIntListValue: memory allocation error!\n");
	    goto FAIL;
	}
	ilist[numvals] = curval;
	numvals++;
    }
    *retp = ilist;
    goto EXIT;

FAIL:
    if (ilist)
	free(ilist);
    numvals = -1;

EXIT:
    if (ptr)
	free(ptr);
    return numvals;
}

/**
 * Given an XPath expression and a context node, this function returns
 * the text stored in the node that matches the expression.  The
 * character array will be stored in the location pointed to by retp.
 * It is an error if the XPath expression matches more than one node.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return 0 on success, or -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getStringValue(GdomeNode * ctxt, const char * path, char ** retp)
{
    GdomeException exc = 0;
    GdomeXPathResult * result = NULL;
    GdomeDOMString * value = NULL;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if (path[0] == '.' && path[1] == '\0') {
	if (domutil_getElementStringValue(ctxt, retp) != 0) {
	    goto FAIL;
	}
    } else {
	if ((result = domutil_getxpresult(ctxt, path, GDOME_STRING_TYPE)) == NULL)
	    goto FAIL;
	value = gdome_xpresult_stringValue(result, &exc);
	if (exc != 0)
	    goto FAIL;
	*retp = strdup(value->str);
    }
    
    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (value)
	gdome_str_unref(value);
    if (result)
	gdome_xpresult_unref(result, &exc);
    return retval;
}


/**
 * Given an element node, this function returns a newly malloc()ed string
 * representing the concatenation of its text node descendants.
 *
 * @param elem element
 * @param retp pointer to result
 * @return 0 on success, -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getElementStringValue(GdomeNode * elem, char ** retp)
{
    GdomeException exc = 0;
    GdomeElement ** nodestack = NULL;
    unsigned int stacksize = 0;
    int retval = 0;
    GdomeNode * curchild = NULL;
    GdomeNode * lastchild = NULL;
    char * newstr = NULL;
    unsigned int newstrlen = 0;
    
    if (gdome_n_nodeType(elem, &exc) != GDOME_ELEMENT_NODE) {
	goto FAIL;
    }

    domutil_errorbuf[0] = '\0';

    retval = 0;

    newstr = (char *)malloc(sizeof(char)*1);
    newstr[0] = '\0';
    nodestack = (GdomeElement **)realloc(nodestack, sizeof(GdomeElement *)*(stacksize+1));
    gdome_el_ref((GdomeElement *)elem, &exc);
    nodestack[stacksize] = (GdomeElement *)elem;
    stacksize++;
    while (stacksize > 0) {
	GdomeElement * parent = NULL;
	unsigned int firstchildstackpos = stacksize - 1;
	unsigned int lastchildstackpos = stacksize - 1;
	parent = nodestack[stacksize-1];
	stacksize--;
	for (curchild = gdome_el_firstChild(parent, &exc); curchild; curchild = gdome_n_nextSibling(lastchild, &exc)) {
	    unsigned int nodetype = 0;
	    if (lastchild) {
		gdome_n_unref(lastchild, &exc);
		lastchild = NULL;
	    }
	    nodetype = gdome_n_nodeType(curchild, &exc);
	    if (nodetype == GDOME_ELEMENT_NODE) {
		nodestack = (GdomeElement **)realloc(nodestack, sizeof(GdomeElement *)*(stacksize+1));
		gdome_el_ref((GdomeElement *)curchild, &exc);
		nodestack[stacksize] = (GdomeElement *)curchild;
		lastchildstackpos = stacksize;
		stacksize++;
	    } else if (nodetype == GDOME_TEXT_NODE) {
		GdomeDOMString * content = NULL;
		unsigned int contentlen = 0;
		content = gdome_t_nodeValue((GdomeText *)curchild, &exc);
		contentlen = strlen(content->str);
		newstr = (char *)realloc(newstr, sizeof(char)*(newstrlen+contentlen+1));
		strncpy(newstr+newstrlen, content->str, contentlen+1);
		newstrlen += contentlen;
		gdome_str_unref(content);
		content = NULL;
	    }
	    lastchild = curchild;
	}
	if (lastchild) {
	    gdome_n_unref(lastchild, &exc);
	    lastchild = NULL;
	}
	gdome_el_unref(parent, &exc);
	parent = NULL;
	/* child elements were added to stack in document order, but we need
	 * to reverse that if we want to pop them off the stack in the same
	 * order
	 */
	while (firstchildstackpos < lastchildstackpos) {
	    GdomeElement * swap = nodestack[firstchildstackpos];
	    nodestack[firstchildstackpos] = nodestack[lastchildstackpos];
	    nodestack[lastchildstackpos] = swap;
	    firstchildstackpos++;
	    lastchildstackpos--;
	}
    }
    *retp = newstr;
    goto EXIT;

FAIL:
    if (newstr) {
	free(newstr);
	newstr = NULL;
    }
    retval = -1;

EXIT:
    if (nodestack) {
	free(nodestack);
	nodestack = NULL;
    }
    return retval;
}

struct nsdef;
static void domutil_normalizeNamespaces_aux(GdomeElement * cur, struct nsdef * oldnsdefs);

/**
 * This function implements, more or less, the namespace normalization
 * algorithm defined in the DOM-Level-3-Core Appendix B.
 * One modification is to replace prefixes with the default namespace
 * if at all possible.
 * There is no return value, this will bomb if something goes wrong.
 *
 * @param start root of the subtree to normalize
 */
void
domutil_normalizeNamespaces(GdomeElement * start)
{
    domutil_normalizeNamespaces_aux(start, NULL);
}

typedef struct nsdef {
    int valid;
    char * prefix;
    char * ns;
} nsdef;

static void
domutil_normalizeNamespaces_aux(GdomeElement * cur, nsdef * oldnsdefs)
{
    GdomeException exc = 0;
    nsdef * nsdefs = NULL;
    int numdefs;

    numdefs = 0;
    nsdefs = (nsdef *)malloc(sizeof(nsdef) * 1);
    nsdefs[numdefs].valid = 0;
    nsdefs[numdefs].prefix = NULL;
    nsdefs[numdefs].ns = NULL;
    
    /* copy old defs */
    {
	int numolddefs = 0;
	int olddefnum;
	for (numolddefs = 0; oldnsdefs && oldnsdefs[numolddefs].valid; numolddefs++) { /* null */ }
	nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef) * (numdefs + numolddefs + 1));
	/* put at end of list, for these are old, so just move terminator */
	memmove(nsdefs + numdefs + numolddefs, nsdefs + numdefs, sizeof(nsdef) * 1);
	for (olddefnum = 0; olddefnum < numolddefs; olddefnum++) {
	    nsdefs[numdefs + olddefnum].valid = 1;
	    if (oldnsdefs[olddefnum].prefix)
		nsdefs[numdefs + olddefnum].prefix = strdup(oldnsdefs[olddefnum].prefix);
	    else
		nsdefs[numdefs + olddefnum].prefix = NULL;
	    if (oldnsdefs[olddefnum].ns)
		nsdefs[numdefs + olddefnum].ns = strdup(oldnsdefs[olddefnum].ns);
	    else
		nsdefs[numdefs + olddefnum].ns = NULL;
	}
	numdefs += numolddefs;
    }

    /* find namespace attributes from current element */
    {
	GdomeNamedNodeMap * nnm = NULL;
	int numattrs;
	int attrnum;
	nnm = gdome_el_attributes(cur, &exc);
	numattrs = gdome_nnm_length(nnm, &exc);
	for (attrnum = 0; attrnum < numattrs; attrnum++) {
	    GdomeAttr * attr = NULL;
	    GdomeDOMString * namespaceURI = NULL;
	    GdomeDOMString * prefix = NULL;
	    GdomeDOMString * name = NULL;
	    GdomeDOMString * value = NULL;
	    attr = (GdomeAttr *)gdome_nnm_item(nnm, attrnum, &exc);
	    namespaceURI = gdome_a_namespaceURI(attr, &exc);
	    prefix = gdome_a_prefix(attr, &exc);
	    name = gdome_a_nodeName(attr, &exc);
	    value = gdome_a_nodeValue(attr, &exc);
	    if ((prefix && strcmp(prefix->str, "xmlns") == 0) ||
		(!prefix && name && strcmp(name->str, "xmlns") == 0)) {
		nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef) * (numdefs + 1 + 1));
		memmove(nsdefs + 1, nsdefs, sizeof(nsdef) * (numdefs + 1));
		if (prefix && strcmp(prefix->str, "xmlns") == 0) {
		    nsdefs[0].valid = 1;
		    nsdefs[0].prefix = strdup(name->str+6);
		    nsdefs[0].ns = strdup(value->str);
		} else {
		    nsdefs[0].valid = 1;
		    nsdefs[0].prefix = NULL;
		    if (value)
			nsdefs[0].ns = strdup(value->str);
		    else
			nsdefs[0].ns = NULL;
		}
		numdefs++;
	    }
	    if (name)
		gdome_str_unref(name);
	    if (value)
		gdome_str_unref(value);
	    if (prefix)
		gdome_str_unref(prefix);
	    if (namespaceURI)
		gdome_str_unref(namespaceURI);
	    if (attr)
		gdome_a_unref(attr, &exc);
	}
	gdome_nnm_unref(nnm, &exc);
    }

    /* Now using algorithm in DOM-Level-3-Core Appendix B */

    /* do normalization for this node */
    {
	GdomeDOMString * namespaceURI = NULL;
	GdomeDOMString * prefix = NULL;
	nsdef * curdefaultns = NULL;
	for (curdefaultns = nsdefs; curdefaultns->valid && curdefaultns->prefix != NULL; curdefaultns++) { /* null */ }
	if (!curdefaultns->valid)
	    curdefaultns = NULL;
	namespaceURI = gdome_el_namespaceURI(cur, &exc);
	prefix = gdome_el_prefix(cur, &exc);
	if (namespaceURI != NULL) {
	    if (curdefaultns && curdefaultns->ns &&
		strcmp(curdefaultns->ns, namespaceURI->str) == 0) {
		if (prefix != NULL) {
		    /* unsetting the prefix!  We do something DANGEROUS here
		     * because gdome doesn't allow for sending NULL to
		     * set_prefix() */
		    /* gdome_el_set_prefix(cur, NULL, &exc); */
		    xmlNode * curn = NULL;
		    const xmlChar * nsuri = NULL;
		    xmlNs * ns = NULL;
		    curn = ((Gdome_xml_Node *)cur)->n;
		    if (curn && curn->ns)
			nsuri = curn->ns->href;
		    ns = xmlNewNs (NULL, nsuri, NULL);
		    if (curn->doc != NULL) {
			if (curn->doc->oldNs == NULL) {
			    gdome_xmlSetOldNs(curn->doc, ns);
			} else {
			    ns->next = curn->doc->oldNs->next;
			    curn->doc->oldNs->next = ns;
			}
		    }
		    xmlSetNs(curn, ns);
		}
	    } else {
		nsdef * match;
		for (match = nsdefs; match->valid; match++) {
		    if ((prefix && match->prefix && strcmp(match->prefix, prefix->str) == 0) ||
			(prefix == NULL && match->prefix == NULL)) {
			break;
		    }
		}
		if (match->valid) {
		    if (strcmp(match->ns, namespaceURI->str) != 0) {
			free(match->ns);
			match->ns = strdup(namespaceURI->str);
		    }
		} else {
		    nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef) * (numdefs + 1 + 1));
		    memmove(nsdefs + 1, nsdefs, sizeof(nsdef) * (numdefs + 1));
		    nsdefs[0].valid = 1;
		    if (prefix == NULL)
			nsdefs[0].prefix = NULL;
		    else
			nsdefs[0].prefix = strdup(prefix->str);
		    nsdefs[0].ns = strdup(namespaceURI->str);
		    numdefs++;
		}
	    }
	} else {
	    if (prefix) {
		fprintf(stderr, "domutil_normalizeNamespaces: Element with prefix but no namespace!\n");
	    } else {
		/* element has no prefix and no namespace */
		if (curdefaultns && curdefaultns->ns == NULL) {
		    /* default namespace is empty, so do nothing */
		} else {
		    if (curdefaultns) {
			/* default namespace is not empty, so change it */
			free(curdefaultns->ns);
			curdefaultns->ns = NULL;
		    } else {
			/* create an "empty" default namespace */
			GdomeDOMString * xmlnsstr = NULL;
			GdomeDOMString * emptystr = NULL;
			xmlnsstr = gdome_str_mkref("xmlns");
			emptystr = gdome_str_mkref("");
			gdome_el_setAttribute(cur, xmlnsstr, emptystr, &exc);
			gdome_str_unref(emptystr);
			gdome_str_unref(xmlnsstr);
			/* add to nsdefs */
			nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef)*(numdefs+1+1));
			memmove(nsdefs + 1, nsdefs, sizeof(nsdef) * (numdefs + 1));
			nsdefs[0].valid = 1;
			nsdefs[0].prefix = NULL;
			nsdefs[0].ns = NULL;
			numdefs++;
		    }
		}
	    }
	}
	if (namespaceURI)
	    gdome_str_unref(namespaceURI);
	if (prefix)
	    gdome_str_unref(prefix);
    }

    /* do the attributes */
    {
	GdomeNamedNodeMap * nnm = NULL;
	int numattrs;
	int attrnum;
	nsdef * curdefaultns = NULL;

	for (curdefaultns = nsdefs; curdefaultns->valid && curdefaultns->prefix != NULL; curdefaultns++) { /* null */ }
	if (!curdefaultns->valid)
	    curdefaultns = NULL;

	nnm = gdome_el_attributes(cur, &exc);
	numattrs = gdome_nnm_length(nnm, &exc);
	for (attrnum = 0; attrnum < numattrs; attrnum++) {
	    /* for all attrs of element */
	    GdomeAttr * attr = NULL;
	    GdomeDOMString * namespaceURI = NULL;
	    GdomeDOMString * prefix = NULL;
	    attr = (GdomeAttr *)gdome_nnm_item(nnm, attrnum, &exc);
	    namespaceURI = gdome_a_namespaceURI(attr, &exc);
	    prefix = gdome_a_prefix(attr, &exc);
	    if (namespaceURI) {
#if 0
		if (curdefaultns && curdefaultns->ns &&
		    strcmp(curdefaultns->ns, namespaceURI->str) == 0) {
		    /* unsetting the prefix!  We do something DANGEROUS here
		     * because gdome doesn't allow for sending NULL to
		     * set_prefix() */
		    /* gdome_el_set_prefix(cur, NULL, &exc); */
		    xmlNode * curn = NULL;
		    const xmlChar * nsuri = NULL;
		    xmlNs * ns = NULL;
		    curn = ((Gdome_xml_Node *)attr)->n;
		    if (curn && curn->ns)
			nsuri = curn->ns->href;
		    ns = xmlNewNs (NULL, nsuri, NULL);
		    if (curn->doc != NULL)
			gdome_xmlSetOldNs(curn->doc, ns);
		    xmlSetNs(curn, ns);
		} else {
#endif
		    /* if attr has no prefix, or has a prefix that conflicts
		     * with a binding already in scope */
		    nsdef * match;
		    for (match = nsdefs; match->valid; match++) {
			if (prefix && match->prefix && match->ns &&
			    strcmp(match->prefix, prefix->str) == 0 &&
			    strcmp(match->ns, namespaceURI->str) != 0)
			    break;
		    }
		    if (prefix == NULL || (match->valid && match->prefix != NULL)) {
			/* if Element is in the scope of a non-default binding
			 * for this namespace */
			nsdef * match = NULL;
			for (match = oldnsdefs; match && match->valid; match++) {
			    if (match->ns &&
				strcmp(match->ns, namespaceURI->str) == 0)
				break;
			}
			if (oldnsdefs && match->valid && match->prefix != NULL) {
			    /* if one or more prefix bindings are available */
			    nsdef * match;
			    for (match = nsdefs; match->valid; match++) {
				if (match->prefix && match->ns &&
				    strcmp(match->ns, namespaceURI->str) == 0)
				    break;
			    }
			    if (match->valid) {
				/* pick any one but prefer local bindings */
				GdomeDOMString * newprefix = NULL;
				newprefix = gdome_str_mkref_dup(match->prefix);
				gdome_a_set_prefix(attr, newprefix, &exc);
				gdome_str_unref(newprefix);
			    } else {
				/* create a local namespace declaration attr
				 * for this namespace */
				GdomeDOMString * xmlnsstr = NULL;
				GdomeDOMString * nsstr = NULL;
				GdomeDOMString * prefixstr = NULL;
				static char newprefix[16];
				static char nsattrname[32];
				int nsnum = 1;
				while (1) {
				    nsdef * match;
				    sprintf(&newprefix[0], "xmlns:NS%d", nsnum);
				    for (match = nsdefs; match->valid; match++) {
					if (match->prefix &&
					    strcmp(match->prefix, &newprefix[0]) == 0)
					    break;
				    }
				    if (!match->valid)
					break;
				}
				strcpy(nsattrname, "xmlns:");
				strcat(nsattrname, &newprefix[0]);
				xmlnsstr = gdome_str_mkref_dup(&nsattrname[0]);
				nsstr = gdome_str_mkref_dup(namespaceURI->str);
				gdome_el_setAttribute(cur, xmlnsstr, nsstr, &exc);
				gdome_str_unref(nsstr);
				gdome_str_unref(xmlnsstr);
				/* add to nsdefs */
				nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef)*(numdefs+1+1));
				memmove(nsdefs + 1, nsdefs, sizeof(nsdef) * (numdefs + 1));
				nsdefs[0].valid = 1;
				nsdefs[0].prefix = strdup(&newprefix[0]);
				nsdefs[0].ns = strdup(namespaceURI->str);
				numdefs++;
				/* Change the Attr to use this prefix */
				prefixstr = gdome_str_mkref_dup(&newprefix[0]);
				gdome_a_set_prefix(attr, prefixstr, &exc);
				gdome_str_unref(prefixstr);
			    }
			}
		    } else {
			/* prefix does match but... */
			GdomeDOMString * attrname = NULL;
			attrname = gdome_a_nodeName(attr, &exc);
			if (strcmp(namespaceURI->str, "http://www.w3.org/2000/xmlns/") == 0 &&
			    !(prefix && strcmp(prefix->str, "xmlns") == 0) &&
			    strcmp(attrname->str, "xmlns") != 0) {
			    /* if there is a non-default binding for this
			     * namespace in scope with a prefix other than
			     * "xmlns" */
			    nsdef * match;
			    for (match = nsdefs; match->valid; match++) {
				if (match->prefix && match->ns &&
				    strcmp(match->ns, namespaceURI->str) == 0 &&
				    strcmp(match->prefix, "xmlns") != 0)
				    break;
			    }
			    if (match->valid) {
				/* pick any one but prefer local bindings */
				GdomeDOMString * newprefix = NULL;
				newprefix = gdome_str_mkref_dup(match->prefix);
				gdome_a_set_prefix(attr, newprefix, &exc);
				gdome_str_unref(newprefix);
			    } else {
				/* create a local namespace declaration attr
				 * for this namespace */
				GdomeDOMString * xmlnsstr = NULL;
				GdomeDOMString * nsstr = NULL;
				GdomeDOMString * prefixstr = NULL;
				static char newprefix[16];
				static char nsattrname[32];
				int nsnum = 1;
				while (1) {
				    nsdef * match;
				    sprintf(&newprefix[0], "xmlns:NS%d", nsnum);
				    for (match = nsdefs; match->valid; match++) {
					if (match->prefix &&
					    strcmp(match->prefix, &newprefix[0]) == 0)
					    break;
				    }
				    if (!match->valid)
					break;
				}
				strcpy(nsattrname, "xmlns:");
				strcat(nsattrname, &newprefix[0]);
				xmlnsstr = gdome_str_mkref_dup(&nsattrname[0]);
				nsstr = gdome_str_mkref_dup(namespaceURI->str);
				gdome_el_setAttribute(cur, xmlnsstr, nsstr, &exc);
				gdome_str_unref(nsstr);
				gdome_str_unref(xmlnsstr);
				/* add to nsdefs */
				nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef)*(numdefs+1+1));
				memmove(nsdefs + 1, nsdefs, sizeof(nsdef) * (numdefs + 1));
				nsdefs[0].valid = 1;
				nsdefs[0].prefix = strdup(&newprefix[0]);
				nsdefs[0].ns = strdup(namespaceURI->str);
				numdefs++;
				/* Change the Attr to use this prefix */
				prefixstr = gdome_str_mkref_dup(&newprefix[0]);
				gdome_a_set_prefix(attr, prefixstr, &exc);
				gdome_str_unref(prefixstr);
			    }
			}
			gdome_str_unref(attrname);
		    }
#if 0
		}
#endif
	    } else {
		if (prefix) {
		    fprintf(stderr, "domutil_normalizeNamespaces: Attribute with prefix but no namespace!\n");
		} else {
		    /* attr has no namespaceURI and no prefix */
		}
	    }
	    if (namespaceURI)
		gdome_str_unref(namespaceURI);
	    if (prefix)
		gdome_str_unref(prefix);
	    gdome_a_unref(attr, &exc);
	}
	gdome_nnm_unref(nnm, &exc);

	/* recurse */
	{
	    GdomeNode * curchild = NULL;
	    GdomeNode * lastchild = NULL;
	    for (curchild = gdome_el_firstChild(cur, &exc); curchild; curchild = gdome_n_nextSibling(lastchild, &exc)) {
		if (lastchild) {
		    gdome_n_unref(lastchild, &exc);
		    lastchild = NULL;
		}
		if (gdome_n_nodeType(curchild, &exc) == GDOME_ELEMENT_NODE) {
		    domutil_normalizeNamespaces_aux((GdomeElement *)curchild, nsdefs);
		}
		lastchild = curchild;
	    }
	    if (lastchild) {
		gdome_n_unref(lastchild, &exc);
		lastchild = NULL;
	    }
	}
    }
    {
	int defnum;
	for (defnum = 0; defnum < numdefs; defnum++) {
	    free(nsdefs[defnum].prefix);
	    free(nsdefs[defnum].ns);
	}
	free(nsdefs);
    }
}

/**
 * Make the document pretty by removing extraneous text elements
 * and adding appropriate spacing for indentation.
 * There is no return value, this will bomb if something goes wrong.
 *
 * @param start root of the subtree to prettify
 * @param startindent number of spaces to indent start element
 * @param indentstep how many spaces to indent each level
 */
void
domutil_prettify(GdomeNode * start, int startindent, int indentstep)
{
    GdomeException exc = 0;
    GdomeNode * parent = NULL;
    GdomeDocument * doc = NULL;
    GdomeNode * tmpnode = NULL;
    int nodetype;

    if (start == NULL)
	return;

    if (gdome_n_nodeType(start, &exc) == GDOME_DOCUMENT_NODE) {
	doc = (GdomeDocument *)start;
	gdome_n_ref((GdomeNode *)doc, &exc);
    } else {
	doc = gdome_n_ownerDocument(start, &exc);
    }
    if (doc == NULL)
	return;

    nodetype = gdome_n_nodeType(start, &exc);

    parent = gdome_n_parentNode(start, &exc);

    /* fix up space before the element */
    if (parent != NULL) {
	GdomeNode * cur = NULL;
	GdomeNode * prev = NULL;
	char * indentstr = NULL;
	GdomeText * indentnode = NULL;
	GdomeDOMString * value;
	indentstr = (char *)malloc(sizeof(char *) * (startindent + 2));
	indentstr[0] = '\n';
	memset(indentstr+1, ' ', startindent);
	indentstr[startindent+1] = '\0';
	cur = start;
	gdome_n_ref(cur, &exc);
	while ((prev = gdome_n_previousSibling(cur, &exc)) != NULL) {
	    int nodeType;
	    int spacey = 1;
	    nodeType = gdome_n_nodeType(prev, &exc);
	    if (nodeType == GDOME_ATTRIBUTE_NODE) {
		gdome_n_unref(cur, &exc); cur = NULL;
		cur = prev;
		continue;
	    }
	    if (nodeType != GDOME_TEXT_NODE) {
		gdome_n_unref(prev, &exc); prev = NULL;
		break;
	    }
	    value = gdome_n_nodeValue(prev, &exc);
	    if (strspn(value->str, " \n\t\v\f") != strlen(value->str))
		spacey = 0;
	    gdome_str_unref(value); value = NULL;
	    if (spacey) {
		GdomeNode * removed = NULL;
		removed = gdome_n_removeChild(parent, prev, &exc);
		gdome_n_unref(removed, &exc); removed = NULL;
	    }
	    gdome_n_unref(prev, &exc); prev = NULL;
	    if (!spacey)
		break;
	}
	gdome_n_unref(cur, &exc);
	value = gdome_str_mkref_dup(indentstr);
	indentnode = gdome_doc_createTextNode(doc, value, &exc);
	gdome_str_unref(value);
	free(indentstr); indentstr = NULL;
	tmpnode = gdome_n_insertBefore(parent, (GdomeNode *)indentnode, start, &exc);
	gdome_n_unref(tmpnode, &exc); tmpnode = NULL;
	gdome_t_unref(indentnode, &exc); indentnode = NULL;
    }

    if (parent == NULL) {
	startindent -= indentstep;
    }

    /* recurse */
    if (nodetype == GDOME_DOCUMENT_NODE ||
	nodetype == GDOME_ELEMENT_NODE) {
	GdomeNode * curnode = NULL;
	curnode = gdome_n_firstChild(start, &exc);
	while (curnode) {
	    GdomeNode * nextnode = NULL;
	    int curnodetype;
	    curnodetype = gdome_n_nodeType(curnode, &exc);
	    if (curnodetype == GDOME_ELEMENT_NODE ||
		curnodetype == GDOME_COMMENT_NODE) {
		domutil_prettify(curnode, startindent+indentstep, indentstep);
	    }
	    nextnode = gdome_n_nextSibling(curnode, &exc);
	    gdome_n_unref(curnode, &exc); curnode = NULL;
	    curnode = nextnode;
	}
    }

    /* fix up space after the element */
    if (parent != NULL) {
	GdomeNode * cur = NULL;
	GdomeNode * next = NULL;
	char * indentstr = NULL;
	GdomeText * indentnode = NULL;
	GdomeDOMString * value;
	indentstr = (char *)malloc(sizeof(char *) * (startindent + 2));
	indentstr[0] = '\n';
	if (startindent > indentstep) {
	    memset(indentstr+1, ' ', startindent - indentstep);
	    indentstr[startindent - indentstep + 1] = '\0';
	} else {
	    indentstr[1] = '\0';
	}
	cur = start;
	gdome_n_ref(cur, &exc);
	while ((next = gdome_n_nextSibling(cur, &exc)) != NULL) {
	    int nodeType;
	    int spacey = 1;
	    nodeType = gdome_n_nodeType(next, &exc);
	    if (nodeType == GDOME_ATTRIBUTE_NODE) {
		gdome_n_unref(cur, &exc); cur = NULL;
		cur = next;
		continue;
	    }
	    if (nodeType != GDOME_TEXT_NODE) {
		gdome_n_unref(next, &exc); next = NULL;
		break;
	    }
	    value = gdome_n_nodeValue(next, &exc);
	    if (strspn(value->str, " \n\t\v\f") != strlen(value->str))
		spacey = 0;
	    gdome_str_unref(value); value = NULL;
	    if (spacey) {
		GdomeNode * removed = NULL;
		removed = gdome_n_removeChild(parent, next, &exc);
		gdome_n_unref(removed, &exc); removed = NULL;
	    }
	    gdome_n_unref(next, &exc); next = NULL;
	    if (!spacey)
		break;
	}
	value = gdome_str_mkref_dup(indentstr);
	indentnode = gdome_doc_createTextNode(doc, value, &exc);
	gdome_str_unref(value);
	free(indentstr); indentstr = NULL;
	next = gdome_n_nextSibling(cur, &exc); /* null is OK */
	tmpnode = gdome_n_insertBefore(parent, (GdomeNode *)indentnode, next, &exc);
	if (next)
	    gdome_n_unref(next, &exc);
	gdome_n_unref(tmpnode, &exc); tmpnode = NULL;
	gdome_t_unref(indentnode, &exc); indentnode = NULL;
	gdome_n_unref(cur, &exc);
    }
    
    if (parent) {
	gdome_n_unref(parent, &exc); parent = NULL;
    }

    gdome_doc_unref(doc, &exc); doc = NULL;
}

/**
 * Remove auto-generated comments (tagged by the AUTOGEN: prefix).
 *
 * @param start root of the subtree
 */
void
domutil_removeAutogenComments(GdomeNode * start)
{
    GdomeException exc = 0;
    int nodetype;

    if (start == NULL)
	return;

    nodetype = gdome_n_nodeType(start, &exc);

    /* recurse */
    if (nodetype == GDOME_DOCUMENT_NODE ||
	nodetype == GDOME_ELEMENT_NODE) {
	GdomeNode * curnode = NULL;
	curnode = gdome_n_firstChild(start, &exc);
	while (curnode) {
	    GdomeNode * nextnode = NULL;
	    int curnodetype;
	    curnodetype = gdome_n_nodeType(curnode, &exc);
	    if (curnodetype == GDOME_ELEMENT_NODE) {
		domutil_removeAutogenComments(curnode);
	    }
	    nextnode = gdome_n_nextSibling(curnode, &exc);
	    if (curnodetype == GDOME_COMMENT_NODE) {
		GdomeDOMString * domstr = gdome_c_data((GdomeComment *)curnode, &exc);
		if (strncmp(domstr->str, "AUTOGEN:", 8) == 0) {
		    GdomeNode * removed = gdome_n_removeChild(start, curnode, &exc);
		    gdome_n_unref(removed, &exc);
		}
		gdome_str_unref(domstr);
	    }
	    gdome_n_unref(curnode, &exc); curnode = NULL;
	    curnode = nextnode;
	}
    }
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.40  2009/01/15 20:52:42  gadde
 * Performance enhancements
 *
 * Revision 1.39  2007/02/23 23:25:56  gadde
 * Fix a nasty memory error due to our onconventional use of gdome2 and libxslt, as well as some other minor memory issues.
 *
 * Revision 1.38  2006/07/07 18:36:34  gadde
 * A couple memory management fixes
 *
 * Revision 1.37  2006/05/31 21:07:36  gadde
 * Unref return value from gdome_n_removeChild.
 *
 * Revision 1.36  2005/05/09 20:33:25  gadde
 * Add function to remove AUTOGEN comments.
 *
 * Revision 1.35  2004/09/07 21:16:08  gadde
 * Attributes don't have a default namespace, so don't assume one.
 *
 * Revision 1.34  2004/07/06 18:35:17  gadde
 * Some memory fixes.
 *
 * Revision 1.33  2004/07/06 13:52:14  gadde
 * Some memory fixes.
 *
 * Revision 1.32  2004/06/18 19:31:58  gadde
 * Code clean up.
 *
 * Revision 1.31  2004/06/09 14:23:27  gadde
 * libgdome/ now  part of -I paths
 *
 * Revision 1.30  2004/05/31 16:38:07  gadde
 * Some memory fixes
 *
 * Revision 1.29  2004/05/28 20:25:29  gadde
 * Initialize some values (to quiet warnings by memory checkers).
 *
 * Revision 1.28  2004/04/02 16:23:53  gadde
 * Move dom_utils to bxh_dom_utils, and add some missing dependencies
 *
 * Revision 1.27  2003/08/01 20:40:28  gadde
 * Some updates to help ease the migration of code between DOM implementations
 *
 * Revision 1.26  2003/07/25 20:43:53  gadde
 * Windows fixes for the C++ conversion of some files.
 * Also reordered some headers so "interface" is not
 * defined before including gdome headers.
 *
 * Revision 1.25  2003/07/25 17:42:14  gadde
 * Random fixes in the move from C to C++
 *
 * Revision 1.24  2003/07/23 14:44:52  gadde
 * More code-prettiness and cleanup (namespace is a reserved word in C++)
 *
 * Revision 1.23  2003/07/21 16:46:49  gadde
 * Code-prettiness updates, for the most part, to further protect BXH
 * library users from particulars of DOM implementation (esp. C vs. C++).
 *
 * Revision 1.22  2003/06/27 15:04:52  gadde
 * Fix early unref()
 *
 * Revision 1.21  2003/06/06 19:43:07  gadde
 * Make child iteration more efficient.
 *
 * Revision 1.20  2003/06/06 02:29:56  gadde
 * Some performance enhancements (to make up for some questionable
 * code in libxml).
 *
 * Revision 1.19  2003/03/31 17:50:48  gadde
 * Prettify indentation of comments too.
 *
 * Revision 1.18  2003/01/06 18:41:58  gadde
 * Many changes/updates to comments.
 * Minor bug fixes.
 *
 * Revision 1.17  2002/12/04 17:21:50  gadde
 * Adding new module files
 *
 * Revision 1.16  2002/12/03 21:08:35  gadde
 * Fixed some docs
 *
 * Revision 1.15  2002/12/03 20:41:04  gadde
 * Big update --
 *  add new datarec module, convert more programs from domutil_ to bxh_,
 *  add new bxh_getElement* functions, and domutil_prettify.
 *
 * Revision 1.14  2002/11/25 16:21:31  gadde
 * Mega-update to merge in new library functions.
 *
 * Revision 1.13  2002/11/15 21:38:19  gadde
 * Added documentation for domutil_normalizeNamespaces
 *
 * Revision 1.12  2002/11/15 21:16:51  gadde
 * More uncommitted namespace changes
 *
 * Revision 1.11  2002/11/15 20:52:33  gadde
 * Now BXH files have the correct namespace declarations (at the expense
 * of some fragile code in dom_utils.c that breaks the GDOME abstraction
 * layer).
 *
 * Revision 1.10  2002/10/25 20:29:51  gadde
 * Moved comment to README
 *
 * Revision 1.9  2002/09/09 19:50:37  gadde
 * Added domutil_appendNewAttributeWithTextValue().
 *
 * Revision 1.8  2002/08/30 14:34:49  gadde
 * Documentation update.
 *
 * Revision 1.7  2002/08/23 20:21:46  gadde
 * *** empty log message ***
 *
 * Revision 1.6  2002/08/23 16:54:18  gadde
 * Whoops.  Fixed some logic errors in number checking.
 * Added more documentation.
 *
 * Revision 1.5  2002/08/22 15:45:57  gadde
 * Fixed some overflow logic.
 *
 * Revision 1.4  2002/08/22 15:39:39  gadde
 * Forgot to #include <limits.h>
 *
 * Revision 1.3  2002/08/22 15:37:58  gadde
 * Added documentation, and more error checking.
 *
 * Revision 1.2  2002/08/20 20:11:29  gadde
 * Added const to rcsid to avoid compiler warning.
 *
 * Revision 1.1  2002/08/20 15:32:33  gadde
 * Everything moved from libxml to libgdome (DOM).  All utility functions
 * prefixed with domutil_.  XPath support still shaky in gdome, but dom_utils.c
 * accounts for this, and extra code can be removed when gdome fully supports
 * the DOM level 3 XPath specification.
 *
 * Revision 1.1  2002/08/16 19:41:15  gadde
 * Initial import
 *
 */
