static const char rcsid[] = "$Id: bxh_niftilib.c,v 1.5 2009-04-03 00:52:33 gadde Exp $";

/*
 * bxh_niftilib.c --
 * 
 * Functions and structures for writing analyze and NIfTI-1 headers
 * from BXH/XCEDE structs.
 *
 * Author: Syam Gadde (gadde@biac.duke.edu)
 */

#include <time.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>
#if defined(WIN32) && !defined(__MINGW32__)
#include <pstdint.h>
#else
#include <stdint.h>
#endif

#include "nifti1_io.h"

#include "bxh_utils.h"
#include "bxh_datarec.h"

#include "bxh_niftilib.h"

static int getOrientation(bxhrawdatarec *datarec, char *orientVal);
static int MatFileWrite(const char *fname, float *data, int rows, int cols, char *name);

/* FROM http://nifti.nimh.nih.gov/dfwg/src/nifti1.h */
/* this is included via nifti1_io.h, so it's just here for reference */
#if 0
/* [SG] offsets added */
struct nifti_1_header { /* NIFTI-1 usage         */  /* ANALYZE 7.5 field(s) */
                        /*************************/  /************************/

                                           /*--- was header_key substruct ---*/
 int   sizeof_hdr;    /*!< MUST be 348         0 */  /* int sizeof_hdr;      */
 char  data_type[10]; /*!< ++UNUSED++          4 */  /* char data_type[10];  */
 char  db_name[18];   /*!< ++UNUSED++         14 */  /* char db_name[18];    */
 int   extents;       /*!< ++UNUSED++         32 */  /* int extents;         */
 short session_error; /*!< ++UNUSED++         36 */  /* short session_error; */
 char  regular;       /*!< ++UNUSED++         38 */  /* char regular;        */
 char  dim_info;      /*!< MRI slice ordering.39 */  /* char hkey_un0;       */

                                      /*--- was image_dimension substruct ---*/
 short dim[8];        /*!< Data array dimensions. 40*/  /* short dim[8];     */
 float intent_p1 ;    /*!< 1st intent parameter.  56*/  /* short unused8;    */
                                                        /* short unused9;    */
 float intent_p2 ;    /*!< 2nd intent parameter.  60*/  /* short unused10;   */
                                                        /* short unused11;   */
 float intent_p3 ;    /*!< 3rd intent parameter.  64*/  /* short unused12;   */
                                                        /* short unused13;   */
 short intent_code ;  /*!< NIFTI_INTENT_* code.   68*/  /* short unused14;   */
 short datatype;      /*!< Defines data type!     70*/  /* short datatype;   */
 short bitpix;        /*!< Number bits/voxel.     72*/  /* short bitpix;     */
 short slice_start;   /*!< First slice index.     74*/  /* short dim_un0;    */
 float pixdim[8];     /*!< Grid spacings.         76*/  /* float pixdim[8];  */
 float vox_offset;    /*!< Offset into .nii file 108*/  /* float vox_offset; */
 float scl_slope ;    /*!< Data scaling: slope.  112*/  /* float funused1;   */
 float scl_inter ;    /*!< Data scaling: offset. 116*/  /* float funused2;   */
 short slice_end;     /*!< Last slice index.     120*/  /* float funused3;   */
 char  slice_code ;   /*!< Slice timing order.   122*/
 char  xyzt_units ;   /*!< Units of pixdim[1..4] 123*/
 float cal_max;       /*!< Max display intensity 124*/  /* float cal_max;    */
 float cal_min;       /*!< Min display intensity 128*/  /* float cal_min;    */
 float slice_duration;/*!< Time for 1 slice.     132*/  /* float compressed; */
 float toffset;       /*!< Time axis shift.      136*/  /* float verified;   */
 int   glmax;         /*!< ++UNUSED++            140*/  /* int glmax;        */
 int   glmin;         /*!< ++UNUSED++            144*/  /* int glmin;        */

                                         /*--- was data_history substruct ---*/
 char  descrip[80];   /*!< any text you like.  148*/ /* char descrip[80];    */
 char  aux_file[24];  /*!< auxiliary filename. 228*/ /* char aux_file[24];   */

 short qform_code ;   /*!< NIFTI_XFORM_* code. 252*/ /*-- all ANALYZE 7.5 ---*/
 short sform_code ;   /*!< NIFTI_XFORM_* code. 254*/ /*   fields below here  */
                                                     /*   are replaced       */
 float quatern_b ;    /*!< Quaternion b param. 256*/
 float quatern_c ;    /*!< Quaternion c param. 260*/
 float quatern_d ;    /*!< Quaternion d param. 264*/
 float qoffset_x ;    /*!< Quaternion x shift. 268*/
 float qoffset_y ;    /*!< Quaternion y shift. 272*/
 float qoffset_z ;    /*!< Quaternion z shift. 276*/

 float srow_x[4] ;    /*!< 1st row affine transform.   280 */
 float srow_y[4] ;    /*!< 2nd row affine transform.   296 */
 float srow_z[4] ;    /*!< 3rd row affine transform.   312 */

 char intent_name[16];/*!< 'name' or meaning of data.  328 */

 char magic[4] ;      /*!< MUST be "ni1\0" or "n+1\0". 344*/

} ;                   /**** 348 bytes total ****/

/* NIFTI-1 data types */
#define NIFTI_TYPE_UINT8           2  /*! unsigned char. */
#define NIFTI_TYPE_INT16           4  /*! signed short. */
#define NIFTI_TYPE_INT32           8  /*! signed int. */
#define NIFTI_TYPE_FLOAT32        16  /*! 32 bit float. */
#define NIFTI_TYPE_COMPLEX64      32  /*! 64 bit complex = 2 32 bit floats. */
#define NIFTI_TYPE_FLOAT64        64  /*! 64 bit float = double. */
#define NIFTI_TYPE_RGB24         128  /*! 3 8 bit bytes. */
#define NIFTI_TYPE_INT8          256  /*! signed char. */
#define NIFTI_TYPE_UINT16        512  /*! unsigned short. */
#define NIFTI_TYPE_UINT32        768  /*! unsigned int. */
#define NIFTI_TYPE_INT64        1024  /*! signed long long. */
#define NIFTI_TYPE_UINT64       1280  /*! unsigned long long. */
#define NIFTI_TYPE_FLOAT128     1536  /*! 128 bit float = long double. */
#define NIFTI_TYPE_COMPLEX128   1792  /*! 128 bit complex = 2 64 bit floats. */
#define NIFTI_TYPE_COMPLEX256   2048  /*! 256 bit complex = 2 128 bit floats */

#define NIFTI_UNITS_UNKNOWN 0  /*! NIFTI code for unspecified units. */
/** Space codes are multiples of 1. **/
#define NIFTI_UNITS_METER   1  /*! NIFTI code for meters. */
#define NIFTI_UNITS_MM      2  /*! NIFTI code for millimeters. */
#define NIFTI_UNITS_MICRON  3  /*! NIFTI code for micrometers. */
/** Time codes are multiples of 8. **/
#define NIFTI_UNITS_SEC     8  /*! NIFTI code for seconds. */
#define NIFTI_UNITS_MSEC   16  /*! NIFTI code for milliseconds. */
#define NIFTI_UNITS_USEC   24  /*! NIFTI code for microseconds. */
/*** These units are for spectral data: ***/
#define NIFTI_UNITS_HZ     32  /*! NIFTI code for Hertz. */
#define NIFTI_UNITS_PPM    40  /*! NIFTI code for ppm. */

   /* [qs]form_code value:  */      /* x,y,z coordinate system refers to:    */
   /*-----------------------*/      /*---------------------------------------*/
#define NIFTI_XFORM_UNKNOWN      0  /*! Arbitrary coordinates (Method 1). */
#define NIFTI_XFORM_SCANNER_ANAT 1  /*! Scanner-based anatomical coordinates */
#define NIFTI_XFORM_ALIGNED_ANAT 2  /*! Coordinates aligned to another file's, 
				        or to anatomical "truth".            */
#define NIFTI_XFORM_TALAIRACH    3  /*! Coordinates aligned to Talairach-
				        Tournoux Atlas; (0,0,0)=AC, etc. */
#define NIFTI_XFORM_MNI_152      4  /*! MNI 152 normalized coordinates. */     
#endif

struct analyze_header_key {
    int sizeof_hdr;
    char data_type[10];
    char db_name[18];
    int extents;
    short int session_error;
    char regular;
    char hkey_un0;
};

struct analyze_image_dimension {
    short int dim[8];
    char vox_units[4];
    char cal_units[4];   /* some show cal_units[8] */
    short int unused1;
    short int unused2;
    short int unused3;
    short int datatype;
    short int bitpix;
    short int dim_un0;
    float pixdim[8];
    float vox_offset;
    float funused1;
    float funused2;
    float funused3;
    float cal_max;
    float cal_min;
    float compressed;
    float verified;
    int glmax,glmin;
};

struct analyze_data_history {
    char descrip[80];
    char aux_file[24];
    char orient;
    char originator[10];
    char generated[10];
    char scannum[10];
    char patient_id[10];
    char exp_date[10];
    char exp_time[10];
    char hist_un0[3];
    int views;
    int vols_added;
    int start_field;
    int field_skip;
    int omax, omin;
    int smax, smin;
};

struct analyze_dsr {
    struct analyze_header_key hk;
    struct analyze_image_dimension dime;
    struct analyze_data_history hist;
};

typedef struct {
    float real_v;
    float imag_v;
} COMPLEX;

#define DT_NONE			0
#define DT_UNKNOWN		0
#define DT_BINARY		1	
#define DT_UNSIGNED_CHAR	2
#define DT_SIGNED_SHORT		4
#define DT_SIGNED_INT		8
#define DT_FLOAT		16
#define DT_COMPLEX		32
#define DT_DOUBLE		64
#define DT_SIGNED_CHAR		130 /* SPM */
#define DT_UNSIGNED_SHORT	132 /* SPM */
#define DT_UNSIGNED_INT		136 /* SPM */
#define DT_RGB			128
#define DT_ALL			255                                    

static int
check_opts(struct bxh2analyze_opts * opts)
{
    if (opts->version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (opts->nii || opts->niigz) {
	opts->niftihdr = 1;
	opts->dontSplitVols = 1;
	opts->dontWriteImg = 0;
	opts->dontWriteHdr = 0;
	opts->dontWriteMat = 1;
    }
    if ((!!opts->niftihdr + !!opts->spmhdr + !!opts->analyzehdr) > 1) {
	fprintf(stderr, "Only one of --niftihdr, --spmhdr, --analyzehdr can be specified.\n");
	return -1;
    }
    if (opts->preferanalyzetypes && !opts->spmhdr && !opts->niftihdr) {
	fprintf(stderr, "--preferanalyzetypes can only be used with --niftihdr or --spmhdr.\n");
	return -1;
    }
    if ((opts->bxh ? 1 : 0) + (opts->xcede ? 1 : 0) + (opts->xcede2 ? 1 : 0) > 1) {
	fprintf(stderr, "only one of --bxh, --xcede, or --xcede2 can be specified!\n");
	return -1;
    }
    if (opts->bxh) {
#ifdef HAVE_LIBXSLT
	bxh_doXCEDE = 0;
#ifdef HAVE_LIBEXSLT
	bxh_doXCEDE2 = 0;
#endif
#endif
    }
    if (opts->xcede) {
#ifdef HAVE_LIBXSLT
	bxh_doXCEDE = 1;
#else
	fprintf(stderr, "Sorry, this program was not compiled with XSLT support, so XCEDE support is not available!\n");
	return -1;
#endif
    }
    if (opts->xcede2) {
#ifdef HAVE_LIBEXSLT
	bxh_doXCEDE2 = 1;
#else
	fprintf(stderr, "Sorry, this program was not compiled with EXSLT/XSLT support, so XCEDE2 support is not available!\n");
	return -1;
#endif
    }
    return 0;
}

/* inputdata must match bdr;
 * bxhxcede == 0: do whatever bxh_doXCEDE or bxh_doXCEDE2 says (or write BXH if no XSLT)
 * bxhxcede == 1: write XCEDE-1
 * bxhxcede == 2: write XCEDE-2
 */
int writeBXHAndNIIGZ(const char *outputbase, struct bxhdataread * bdr, void * inputdata, int xcede)
{
    struct bxh2analyze_opts b2a_opts;
    memset(&b2a_opts, '\0', sizeof(b2a_opts));
    b2a_opts.dontWriteMat = 1;
    b2a_opts.dontSplitVols = 1;
    b2a_opts.niigz = 1;
    b2a_opts.preferanalyzetypes = 1;
    b2a_opts.overwrite = 1;
    switch (xcede) {
    case 0:
#ifdef HAVE_LIBXSLT
	b2a_opts.xcede = bxh_doXCEDE;
#ifdef HAVE_LIBEXSLT
	if (bxh_doXCEDE == 0) {
	    b2a_opts.xcede = bxh_doXCEDE2;
	}
#endif
#endif
	break;
    case 1:
	b2a_opts.xcede = 1;
	break;
    case 2:
	b2a_opts.xcede2 = 1;
	break;
    default:
	break;
    }
    return convertBXHToAnalyze(NULL, outputbase, bdr, inputdata, &b2a_opts);
}

/* inputdata, if non-NULL, must match bdr */
int
convertBXHToAnalyze(const char *ifname, const char *ofname, struct bxhdataread * bdrp, void * inputdata, struct bxh2analyze_opts * opts)
{
    struct stat         statbuf;
    char                *outHeader = NULL;
    char                *outImage = NULL;
    char                *outMat = NULL;
    char                *outBXHHeader = NULL;
    void                *outBuffer = NULL;
    FILE                *outHeaderFile = NULL;
    FILE                *outImageFile = NULL;
    gzFile              *outniigzFile = NULL;
    BXHDocPtr           docp = NULL;
    BXHElementPtr       imagedatap = NULL;
    BXHElementPtr       acquisitiondatap = NULL; 
    BXHElementPtr       tempPtr = NULL; 
    char                *tempVal;
    bxhrawdatarec       *datarec = NULL;
    char                *databuf = NULL;
    int                 needpermute = 0;
    int                 overflow = 0;
    int                 overflowtype = 0;
    char *              overflowtypestr = 0;
    int                 overflowbitpix = 0;
    int                 fixelemtype = 0;
    char *		fileext = NULL;
    const char *	ordereddimnames[] = { "x", "y", "z", "t" };
    struct bxhdataread	newbdr;

    struct analyze_dsr  hdr; 
    struct nifti_1_header *nifp = (struct nifti_1_header *)&hdr;
    int                 result, i, msbfirst, numImgFiles;
    size_t		curpos, curposoverflow;
    size_t		bytesToRead, bytesRead, bytesToWrite;

    {
	int retval = check_opts(opts);
	if (retval != 0) {
	    return retval;
	}
    }

    if (inputdata) {
	databuf = (char *)inputdata;
    }

#ifdef HAVE_LIBXSLT
    if (bxh_doXCEDE)
	fileext = ".xml";
#ifdef HAVE_LIBEXSLT
    else if (bxh_doXCEDE2)
	fileext = ".xcede2";
#endif
    else
#endif
	fileext = ".bxh";
    
    result = 0;
    curpos = 0;
    curposoverflow = 0;
    numImgFiles = 1;
    msbfirst = 1;
    msbfirst = (((char *)&msbfirst)[0] == 0);

    if (bdrp == NULL) {
	if (ifname == NULL) {
	    fprintf(stderr, "No document or input file provided!\n");
	    goto FAIL;
	}
	if (stat(ifname, &statbuf) != 0) {
	    fprintf(stderr, "convertBXHToAnalyze(): Input BXH file('%s') does not exist \n", ifname); 
	    goto FAIL;
	}
	memset(&newbdr, '\0', sizeof(newbdr));
	if (bxh_dataReadFileStart(ifname, "image", NULL, 4, ordereddimnames, NULL, &newbdr) != 0 || 
	    bxh_dataReadFinish(&newbdr, NULL) != 0) {
	    fprintf(stderr, "Reading input file '%s' failed.\n", ifname); 
	    goto FAIL;
	}
	databuf = (char *)newbdr.dataptr;
	bdrp = &newbdr;
    }
    docp = bdrp->docp;
    imagedatap = bdrp->imagedatap;
    acquisitiondatap = bdrp->acqdatap;
    datarec = bdrp->datarec;
    if (datarec->numdims > 7) {
      fprintf(stderr, "Data with more than 7 dimensions not supported.\n");
      goto FAIL; 
    }

    if (datarec->msbfirstfrags != msbfirst &&
	!opts->dontWriteHdr &&
	opts->dontWriteBxh &&
	opts->dontWriteImg) {
      fprintf(stderr, "Warning: this machine's byte order does not match the byte order of the data\ndescribed in the XML file.  If you are writing .hdr files to match already\nexisting data, be aware that the byte order may be incorrect.\n");
    }

    if (databuf == NULL && bdrp != &newbdr) {
	/* if data is properly ordered, don't do anything, otherwise permute */
	for (i = 0; i < datarec->numdims; i++) {
	    bxhdimension * dimp = &datarec->dimensions[i];
	    if (i == 0 && strcmp(dimp->type, "x") != 0)
		needpermute = 1;
	    else if (i == 1 && strcmp(dimp->type, "y") != 0)
		needpermute = 1;
	    else if (i == 2 && strcmp(dimp->type, "z") != 0)
		needpermute = 1;
	    else if (i == 3 && strcmp(dimp->type, "t") != 0)
		needpermute = 1;
	}
	if (needpermute) {
	    bxhrawdatarec * newdatarec = NULL;
	    fprintf(stderr, "Permuting data...\n");
	    if ((databuf = (char *)malloc(sizeof(char)*datarec->datasize)) == NULL) {
		fprintf(stderr, "  Couldn't allocate enough memory!\n");
		goto FAIL;
	    }
	    if ((newdatarec = bxh_datarec_readXYZTData(datarec, databuf, msbfirst)) == NULL) {
		fprintf(stderr, "convertBXHToAnalyze(): Error in permuting data to XYZT order!\n");
		goto FAIL;
	    }
	    bxh_datarec_free(datarec);
	    datarec = newdatarec;
	}
    }

    /* Set the values in the header */
    /* Values are set basically in the order they appear in the header */

    memset(&hdr, '\0', sizeof(hdr));

    hdr.hk.sizeof_hdr = 348;            /*required value*/
    if (!opts->niftihdr) {
	hdr.hk.extents = 16384;             /*required value*/
	hdr.hk.session_error = 0;
	hdr.hk.regular = 'r';               /*required value*/
	hdr.hk.hkey_un0 = '\0';
    }

    if (opts->niftihdr) {
	BXHElementPtr * fdelemp = NULL;
	unsigned int freqdir = 0;
	if ((fdelemp = bxh_getChildElement(acquisitiondatap, "frequencydirection")) != NULL &&
	    bxh_getElementUnsignedIntValue(fdelemp, &freqdir) == 0) {
	    int freq_dim = 0;
	    int phase_dim = 0;
	    int slice_dim = 0;
	    if (freqdir == 1) {
		freq_dim = 1;
		phase_dim = 2;
		slice_dim = 3; /* XXX is this always true? */
	    } else if (freqdir == 2) {
		phase_dim = 2;
		freq_dim = 1;
		slice_dim = 3; /* XXX is this always true? */
	    }
	    nifp->dim_info =
		((((char)freq_dim) & 0x03)) |
		((((char)phase_dim) & 0x03) << 2) |
		((((char)slice_dim) & 0x03) << 4);
	    bxh_element_unref(fdelemp); fdelemp = NULL;
	}
    }

    hdr.dime.dim[0] = datarec->numdims;
    hdr.dime.pixdim[0] = 0;
    for(i = 0; hdr.dime.dim[0] - i > 0; i++) {
      int isspatial = 0;
      int istemporal = 0;
      if (strcmp(datarec->dimensions[i].type, "x") == 0 ||
	  strcmp(datarec->dimensions[i].type, "y") == 0 ||
	  strcmp(datarec->dimensions[i].type, "z") == 0) {
	  isspatial = 1;
      } else if (strcmp(datarec->dimensions[i].type, "t") == 0) {
	  istemporal = 1;
      }
      if (i == 0) {
	hdr.dime.dim[1] = datarec->dimensions[i].size;         /*number of pixels per...*/
	hdr.dime.pixdim[1] = (float)datarec->dimensions[i].spacing;   /*pixel spacing in mm*/
	if (isspatial && opts->spatialunits)
	  strcpy(hdr.dime.vox_units, opts->spatialunits);
	else if (isspatial && datarec->dimensions[i].units)
	  strcpy(hdr.dime.vox_units, datarec->dimensions[i].units);
      }
      else if (i == 1) {
	hdr.dime.dim[2] = datarec->dimensions[i].size;
	hdr.dime.pixdim[2] = (float)datarec->dimensions[i].spacing;
	if (isspatial && opts->spatialunits)
	  strcpy(hdr.dime.vox_units, opts->spatialunits);
	else if (isspatial && datarec->dimensions[i].units)
	  strcpy(hdr.dime.vox_units, datarec->dimensions[i].units);
      }
      else if (i == 2) {
        hdr.dime.dim[3] = datarec->dimensions[i].size;
	hdr.dime.pixdim[3] = (float)datarec->dimensions[i].spacing;   
	if (isspatial && opts->spatialunits)
	  strcpy(hdr.dime.vox_units, opts->spatialunits);
	else if (isspatial && datarec->dimensions[i].units)
	  strcpy(hdr.dime.vox_units, datarec->dimensions[i].units);
      }
      else if (i == 3) {
	if (!opts->dontSplitVols) {
	  hdr.dime.dim[4] = 1; 
	  hdr.dime.pixdim[4] = (float)datarec->dimensions[i].spacing;
	  numImgFiles = datarec->dimensions[i].size;
	}
	else {
	  hdr.dime.dim[4] = datarec->dimensions[i].size;
	  hdr.dime.pixdim[4] = (float)datarec->dimensions[i].spacing;
	  numImgFiles = 1; 
	}
	if (isspatial && opts->spatialunits)
	  strcpy(hdr.dime.vox_units, opts->spatialunits);
	else if (isspatial && datarec->dimensions[i].units)
	  strcpy(hdr.dime.vox_units, datarec->dimensions[i].units);
      }
      else if (i >= 3 && i <= 7) {
	if (!opts->dontSplitVols) {
	  numImgFiles *= datarec->dimensions[i].size;
	}
	if (i == 3) {
	  hdr.dime.dim[4] = datarec->dimensions[i].size;
	} else {
	  hdr.dime.dim[4] *= datarec->dimensions[i].size;
	}
      }
      else {
	fprintf(stderr, "Too many dimensions (%d)!\n", (int)hdr.dime.dim[0]);
	goto FAIL;
      }
    }

    /* if t dimension exists, change t size depending on whether we split
     * volumes or not */
    if (datarec->numdims >= 3) {
	if (!opts->dontSplitVols) {
	  hdr.dime.dim[0] = 3;
	} else if (datarec->numdims >= 4) {
	  hdr.dime.dim[0] = 4;
	}
    }

    for(i = hdr.dime.dim[0] + 1; i < 8; i++) { 
      hdr.dime.dim[i] = 1;
    }

    if (opts->niftihdr) {
      nifp->intent_code = 0;
      nifp->intent_name[0] = '\0';
    } else {
      memset(hdr.dime.cal_units, '\0', 4);
      hdr.dime.unused1 = 0;
      hdr.dime.unused2 = 0;
      hdr.dime.unused3 = 0;
    }

    if (strcmp(datarec->elemtype, "uint8") == 0) {
      hdr.dime.datatype = DT_UNSIGNED_CHAR;
      hdr.dime.bitpix = 8;
    }
    else if (strcmp(datarec->elemtype, "int16") == 0) {
      hdr.dime.datatype = DT_SIGNED_SHORT;
      hdr.dime.bitpix = 16;
    }
    else if (strcmp(datarec->elemtype, "int32") == 0) {
      hdr.dime.datatype = DT_SIGNED_INT;
      hdr.dime.bitpix = 32;
    }
    else if (strcmp(datarec->elemtype, "float32") == 0) {
      hdr.dime.datatype = DT_FLOAT;
      hdr.dime.bitpix = 32;
    }
    else if (strcmp(datarec->elemtype, "float64") == 0 ||
	     strcmp(datarec->elemtype, "double") == 0) {
      hdr.dime.datatype = DT_DOUBLE;
      hdr.dime.bitpix = 64;
    }
    /* the following are not natively supported by the Analyze header,
       so the data needs to be type-converted later */
    else if (strcmp(datarec->elemtype, "int8") == 0) {
      if (opts->spmhdr && !opts->analyzetypes && !opts->preferanalyzetypes) {
	hdr.dime.datatype = DT_SIGNED_CHAR;
      } else if (opts->niftihdr && !opts->analyzetypes && !opts->preferanalyzetypes) {
	nifp->datatype = NIFTI_TYPE_INT8;
      } else {
	hdr.dime.datatype = DT_UNSIGNED_CHAR;
	fixelemtype = 1;
	if (opts->preferanalyzetypes) {
	  fprintf(stderr, "Trying uint8 data type (instead of input int8 datatype).\n");
	  /* if data overflows, we use these non-preferred types */
	  overflowtypestr = "int8";
	  overflowbitpix = 8;
	  if (opts->spmhdr) {
	      overflowtype = DT_SIGNED_CHAR;
	  } else if (opts->niftihdr) {
	      overflowtype = NIFTI_TYPE_INT8;
	  }
	} else {
	  fprintf(stderr, "Warning: int8 is not supported by Analyze header, will attempt to write as uint8 instead.\nIf overflow (underflow) is detected, will write as int16.\n");
	  overflowtypestr = "int16";
	  overflowbitpix = 16;
	  overflowtype = DT_SIGNED_SHORT;
	}
      }
      hdr.dime.bitpix = 8;
    }
    else if (strcmp(datarec->elemtype, "uint16") == 0) {
      if (opts->spmhdr && !opts->analyzetypes && !opts->preferanalyzetypes) {
	hdr.dime.datatype = DT_UNSIGNED_SHORT;
      } else if (opts->niftihdr && !opts->analyzetypes && !opts->preferanalyzetypes) {
	nifp->datatype = NIFTI_TYPE_UINT16;
      } else {
	hdr.dime.datatype = DT_SIGNED_SHORT;
	fixelemtype = 1;
	if (opts->preferanalyzetypes) {
	  fprintf(stderr, "Trying int16 data type (instead of input uint16 datatype).\n");
	  /* if data overflows, we use these non-preferred types */
	  overflowtypestr = "uint16";
	  overflowbitpix = 16;
	  if (opts->spmhdr) {
	      overflowtype = DT_UNSIGNED_SHORT;
	  } else if (opts->niftihdr) {
	      overflowtype = NIFTI_TYPE_UINT16;
	  }
	} else {
	  fprintf(stderr, "Warning: uint16 is not supported by Analyze header, will attempt to write as int16 instead.\nIf overflow is detected, will write as int32.\n");
	  overflowtypestr = "int32";
	  overflowbitpix = 32;
	  overflowtype = DT_SIGNED_INT;
	}
      }
      hdr.dime.bitpix = 16;
    }
    else if (strcmp(datarec->elemtype, "uint32") == 0) {
      if (opts->spmhdr && !opts->analyzetypes && !opts->preferanalyzetypes) {
	hdr.dime.datatype = DT_UNSIGNED_INT;
      } else if (opts->niftihdr && !opts->analyzetypes && !opts->preferanalyzetypes) {
	nifp->datatype = NIFTI_TYPE_UINT32;
      } else {
	hdr.dime.datatype = DT_SIGNED_INT;
	fixelemtype = 1;
	if (opts->preferanalyzetypes) {
	  fprintf(stderr, "Trying int32 data type (instead of input uint32 datatype).\n");
	  /* if data overflows, we use these non-preferred types */
	  overflowtypestr = "uint32";
	  overflowbitpix = 32;
	  if (opts->spmhdr) {
	      overflowtype = DT_UNSIGNED_INT;
	  } else if (opts->niftihdr) {
	      overflowtype = NIFTI_TYPE_UINT32;
	  }
	} else {
	  fprintf(stderr, "Warning: uint32 is not supported by Analyze header, will attempt to write as int32 instead.\nIf overflow is detected, will write as float32.\n");
	  overflowtypestr = "float32";
	  overflowbitpix = 32;
	  overflowtype = DT_FLOAT;
	}
      }
      hdr.dime.bitpix = 32;
    }
    else if (!opts->analyzetypes && opts->niftihdr && strcmp(datarec->elemtype, "int64") == 0) {
      nifp->datatype = NIFTI_TYPE_INT64;
      hdr.dime.bitpix = 64;
    }
    else if (!opts->analyzetypes && opts->niftihdr && strcmp(datarec->elemtype, "uint64") == 0) {
      nifp->datatype = NIFTI_TYPE_UINT64;
      hdr.dime.bitpix = 64;
    }
    else if (strcmp(datarec->elemtype, "rgb24") == 0) {
      hdr.dime.datatype = DT_RGB;
      hdr.dime.bitpix = 24;
    }
    else {
      fprintf(stderr, "convertBXHToAnalyze(): The '%s' format is not currently supported!\n", datarec->elemtype);
      goto FAIL; 
    }

    if (opts->niftihdr) {
      int spatialunits = 0;
      int temporalunits = 0;

      /* XXX need to populate these once slice timing is in BXH header */
      nifp->slice_code = 0;
      nifp->slice_start = 0;
      nifp->slice_end = 0;

      if (opts->spatialunits) {
	  if (strcmp(opts->spatialunits, "m") == 0) {
	      spatialunits = NIFTI_UNITS_METER;
	  } else if (strcmp(opts->spatialunits, "mm") == 0) {
	      spatialunits = NIFTI_UNITS_MM;
	  } else if (strcmp(opts->spatialunits, "um") == 0) {
	      spatialunits = NIFTI_UNITS_MICRON;
	  } else {
	      fprintf(stderr, "Unsupported spatial unit type '%s'\n",
		      opts->spatialunits);
	  }
      } else if (datarec->dimensions[0].units != NULL &&
	  (datarec->numdims < 2 ||
	   ((datarec->dimensions[1].units != NULL &&
	     strcmp(datarec->dimensions[0].units,
		    datarec->dimensions[1].units) == 0) &&
	    (datarec->numdims < 3 ||
	     (datarec->dimensions[2].units != NULL &&
	      strcmp(datarec->dimensions[1].units,
		     datarec->dimensions[2].units) == 0))))) {
	/* all spatial dimensions are the same units */
	if (strcmp(datarec->dimensions[0].units, "m") == 0) {
	  spatialunits = NIFTI_UNITS_METER;
	} else if (strcmp(datarec->dimensions[0].units, "mm") == 0) {
	  spatialunits = NIFTI_UNITS_MM;
	} else if (strcmp(datarec->dimensions[0].units, "um") == 0) {
	  spatialunits = NIFTI_UNITS_MICRON;
	}
      } else {
	spatialunits = NIFTI_UNITS_UNKNOWN;
      }
      if (opts->temporalunits) {
	  if (strcmp(opts->temporalunits, "s") == 0) {
	      temporalunits = NIFTI_UNITS_SEC;
	  } else if (strcmp(opts->temporalunits, "ms") == 0) {
	      temporalunits = NIFTI_UNITS_MSEC;
	  } else if (strcmp(opts->temporalunits, "us") == 0) {
	      temporalunits = NIFTI_UNITS_USEC;
	  } else {
	      fprintf(stderr, "Unsupported temporal unit type '%s'\n",
		      opts->temporalunits);
	  }
      } else if (datarec->numdims >= 4 &&
	  strcmp(datarec->dimensions[3].type, "t") == 0) {
	if (datarec->dimensions[3].units == NULL ||
	    strcmp(datarec->dimensions[3].units, "ms") == 0) {
	    temporalunits = NIFTI_UNITS_MSEC;
	} else if (strcmp(datarec->dimensions[3].units, "s") == 0) {
	    temporalunits = NIFTI_UNITS_SEC;
	} else if (strcmp(datarec->dimensions[3].units, "us") == 0) {
	    temporalunits = NIFTI_UNITS_USEC;
	} else {
	    fprintf(stderr, "Unsupported temporal unit type '%s'\n",
		    datarec->dimensions[3].units);
	}
	nifp->slice_duration = (float)(datarec->dimensions[3].spacing - datarec->dimensions[3].gap) / datarec->dimensions[2].size;
	nifp->toffset = (float)datarec->dimensions[3].origin;
      } else {
	temporalunits = NIFTI_UNITS_UNKNOWN;
      }
      nifp->xyzt_units = spatialunits | temporalunits;
      nifp->cal_max = 0;
      nifp->cal_min = 0;
    } else {
      hdr.dime.dim_un0 = 0;
      hdr.dime.vox_offset = 0;
      hdr.dime.funused1 = 0;
      hdr.dime.funused2 = 0;
      hdr.dime.funused3 = 0;
      hdr.dime.cal_max = 0;
      hdr.dime.cal_min = 0;
      hdr.dime.compressed = 0; 
      hdr.dime.verified = 0; 
      hdr.dime.glmax = 0;
      hdr.dime.glmin = 0;
      hdr.dime.pixdim[4] /= (float)1000.0; /* convert it to seconds, which is standard (?) */
    }

    memset(hdr.hist.descrip, '\0', 80);
    memset(hdr.hist.aux_file, '\0', 24);

    if (opts->niftihdr) {
      /* calculations from nifti1_io.c */
      if (datarec->numdims >= 3 &&
	  datarec->dimensions[0].direction != NULL &&
	  datarec->dimensions[1].direction != NULL &&
	  datarec->dimensions[2].direction != NULL) {
	bxhdimension * dimx = &datarec->dimensions[0];
	bxhdimension * dimy = &datarec->dimensions[1];
	bxhdimension * dimz = &datarec->dimensions[2];
	struct bxhdimension *dimr, *dima, *dims;
	double Rr, Ra, Rs; /* row */
	double Cr, Ca, Cs; /* column */
	double Nr, Na, Ns; /* slice */
	mat44 R;

	Rr = dimx->direction[0];
	Ra = dimx->direction[1];
	Rs = dimx->direction[2];
	Cr = dimy->direction[0];
	Ca = dimy->direction[1];
	Cs = dimy->direction[2];
	Nr = dimz->direction[0];
	Na = dimz->direction[1];
	Ns = dimz->direction[2];

	if (fabs(Rr) > fabs(Cr) && fabs(Rr) > fabs(Nr)) {
	    dimr = dimx;
	    if (fabs(Ca) > fabs(Ra) && fabs(Ca) > fabs(Na)) {
		dima = dimy; dims = dimz;
	    } else {
		dima = dimz; dims = dimy;
	    }
	} else if (fabs(Cr) > fabs(Rr) && fabs(Cr) > fabs(Nr)) {
	    dimr = dimy;
	    if (fabs(Ra) > fabs(Ca) && fabs(Ra) > fabs(Na)) {
		dima = dimx; dims = dimz;
	    } else {
		dima = dimz; dims = dimx;
	    }
	} else {
	    dimr = dimz;
	    if (fabs(Ra) > fabs(Ca) && fabs(Ra) > fabs(Na)) {
		dima = dimx; dims = dimy;
	    } else {
		dima = dimy; dims = dimx;
	    }
	}

	/* compute quaternion */

	R.m[0][0] = (float)Rr;
	R.m[1][0] = (float)Ra;
	R.m[2][0] = (float)Rs;
	R.m[0][1] = (float)Cr;
	R.m[1][1] = (float)Ca;
	R.m[2][1] = (float)Cs;
	R.m[0][2] = (float)Nr;
	R.m[1][2] = (float)Na;
	R.m[2][2] = (float)Ns;
	if (opts->zeroorigin) {
	    R.m[0][3] = 0;
	    R.m[1][3] = 0;
	    R.m[2][3] = 0;
	} else {
	    R.m[0][3] = (float)dimr->origin;
	    R.m[1][3] = (float)dima->origin;
	    R.m[2][3] = (float)dims->origin;
	}
	R.m[3][0] = 0;
	R.m[3][1] = 0;
	R.m[3][2] = 0;
	R.m[3][3] = 1;

	nifti_mat44_to_quatern(
	    R,
	    &nifp->quatern_b, &nifp->quatern_c, &nifp->quatern_d,
	    &nifp->qoffset_x, &nifp->qoffset_y, &nifp->qoffset_z,
	    NULL, NULL, NULL,
	    &nifp->pixdim[0]);
	/* NULLs above replace
	 *   &nifp->pixdim[1], &nifp->pixdim[2], &nifp->pixdim[3]
	 * because direction vectors are already normalized,
	 * and we don't need to recalculate spacing.
	 */
	nifp->qform_code = NIFTI_XFORM_SCANNER_ANAT;

	if (!opts->nosform) {
	    /* Also put orientation/positioning information into sform for
	     * those tools that require it */
	    nifp->srow_x[0] = R.m[0][0] * (float)dimx->spacing;
	    nifp->srow_x[1] = R.m[0][1] * (float)dimy->spacing;
	    nifp->srow_x[2] = R.m[0][2] * (float)dimz->spacing;
	    nifp->srow_y[0] = R.m[1][0] * (float)dimx->spacing;
	    nifp->srow_y[1] = R.m[1][1] * (float)dimy->spacing;
	    nifp->srow_y[2] = R.m[1][2] * (float)dimz->spacing;
	    nifp->srow_z[0] = R.m[2][0] * (float)dimx->spacing;
	    nifp->srow_z[1] = R.m[2][1] * (float)dimy->spacing;
	    nifp->srow_z[2] = R.m[2][2] * (float)dimz->spacing;
	    nifp->srow_x[3] = R.m[0][3];
	    nifp->srow_y[3] = R.m[1][3];
	    nifp->srow_z[3] = R.m[2][3];
	    nifp->sform_code = NIFTI_XFORM_SCANNER_ANAT;
	}

#if 0
	/* check to see if the quaternion calculation is somewhat invertible */
	{
	    mat44 S;
	    int i, j;
	    int equal = 1;
	    S = nifti_quatern_to_mat44(
		nifp->quatern_b, nifp->quatern_c, nifp->quatern_d,
		nifp->qoffset_x, nifp->qoffset_y, nifp->qoffset_z,
		1, 1, 1,
		nifp->pixdim[0]);
	    for (i = 0; i < 3; i++) {
		for (j = 0; j < 3; j++) {
		    if (R.m[i][j] != S.m[i][j])
			equal = 0;
		}
	    }
	    if (!equal) {
		fprintf(stderr, "Transformation matrices not equal:\n");
		fprintf(stderr, "Before:                             | After:\n");
		for (i = 0; i < 3; i++) {
		    fprintf(stderr, "%8g %8g %8g %8g | %8g %8g %8g %8g\n",
			    R.m[i][0], R.m[i][1], R.m[i][2], R.m[i][3], 
			    S.m[i][0], S.m[i][1], S.m[i][2], S.m[i][3]);
		}
	    }
	}
#endif

	/* check if slice order info is available */
	{
	  const int TRY_SEQ_INC =  0x01;
	  const int TRY_SEQ_DEC =  0x02;
	  const int TRY_ALT_INC =  0x04;
	  const int TRY_ALT_DEC =  0x08;
	  const int TRY_ALT_INC2 = 0x10;
	  const int TRY_ALT_DEC2 = 0x20;
	  int trythese = TRY_SEQ_INC | TRY_SEQ_DEC | TRY_ALT_INC | TRY_ALT_DEC | TRY_ALT_INC2 | TRY_ALT_DEC2;
	  int dpsind;
	  for (dpsind = 0; dpsind < dimz->numdpstructs; dpsind++) {
	    int valueind;
	    struct bxhdatapoints * dpstructp = &dimz->dpstructs[dpsind];
	    int numvalues = dpstructp->numvalues;
	    int numvalueshalf = (dpstructp->numvalues + 1) / 2;
	    if (strcmp(dpstructp->label, "acquisitiontimeindex") != 0) {
	      continue;
	    }
	    for (valueind = 0; trythese != 0 && valueind < numvalues; valueind++) {
	      static char * endptr;
	      char * value = dpstructp->values[valueind];
	      long int acqtimeind = strtol(value, &endptr, 10);
	      if (value[0] == '\0' || endptr[0] != '\0') {
		trythese = 0;
		break;
	      }
	      if (trythese & TRY_SEQ_INC && acqtimeind != valueind + 1) {
		trythese &= ~TRY_SEQ_INC;
	      }
	      if (trythese & TRY_SEQ_DEC && acqtimeind != numvalues - valueind) {
		trythese &= ~TRY_SEQ_DEC;
	      }
	      if (trythese & TRY_ALT_INC && acqtimeind != (valueind / 2) + ((valueind % 2) == 0 ? 0 : numvalueshalf) + 1) {
		trythese &= ~TRY_ALT_INC;
	      }
	      if (trythese & TRY_ALT_DEC && acqtimeind != ((numvalues - valueind - 1) / 2) + (((numvalues - valueind - 1) % 2) == 0 ? 0 : numvalueshalf) + 1) {
		trythese &= ~TRY_ALT_DEC;
	      }
	      if (trythese & TRY_ALT_INC2 && acqtimeind != (valueind / 2) + ((valueind % 2) == 1 ? 0 : numvalueshalf - 1) + 1) {
		trythese &= ~TRY_ALT_INC2;
	      }
	      if (trythese & TRY_ALT_DEC2 && acqtimeind != ((numvalues - valueind - 1) / 2) + (((numvalues - valueind - 1) % 2) == 1 ? 0 : numvalueshalf - 1) + 1) {
		trythese &= ~TRY_ALT_DEC2;
	      }
	    }
	    if (trythese & TRY_SEQ_INC) {
	      nifp->slice_code = NIFTI_SLICE_SEQ_INC;
	    } else if (trythese & TRY_SEQ_DEC) {
	      nifp->slice_code = NIFTI_SLICE_SEQ_DEC;
	    } else if (trythese & TRY_ALT_INC) {
	      nifp->slice_code = NIFTI_SLICE_ALT_INC;
	    } else if (trythese & TRY_ALT_DEC) {
	      nifp->slice_code = NIFTI_SLICE_ALT_DEC;
	    } else if (trythese & TRY_ALT_INC2) {
	      nifp->slice_code = NIFTI_SLICE_ALT_INC2;
	    } else if (trythese & TRY_ALT_DEC2) {
	      nifp->slice_code = NIFTI_SLICE_ALT_DEC2;
	    }
	    nifp->slice_start = 0;
	    nifp->slice_end = dpstructp->numvalues - 1;
	  }
	}
      } else {
	nifp->qform_code = 0;
	nifp->sform_code = 0;
      }

      /* add NIFTI-1 magic number */
      if (opts->nii || opts->niigz) {
	  strcpy(&nifp->magic[0], "n+1");
      } else {
	  strcpy(&nifp->magic[0], "ni1");
      }
    } else {
      if (getOrientation(datarec, &hdr.hist.orient) < 0) {
	fprintf(stderr, "WARNING: not a supported Analyze orientation --\n labeling as transverse unflipped (0) anyway...\n");
      }

      memset(hdr.hist.originator, '\0', 10);
      memset(hdr.hist.generated, '\0', 10);

      memset(hdr.hist.scannum, '\0', 10);  
      tempPtr = bxh_getChildElement(acquisitiondatap, "examnumber");
      if (tempPtr) {
	i = bxh_getElementStringValue(tempPtr, &tempVal);
	strcpy(hdr.hist.scannum, tempVal);
	free(tempVal);
	bxh_element_unref(tempPtr);
      }

      memset(hdr.hist.patient_id, '\0', 10);
      memset(hdr.hist.exp_date, '\0', 10);
      tempPtr = bxh_getChildElement(acquisitiondatap, "scandate");
      if (tempPtr) {
	i = bxh_getElementStringValue(tempPtr, &tempVal);
	strcpy(hdr.hist.exp_date, tempVal); 
	free(tempVal);
	bxh_element_unref(tempPtr);
      }

      memset(hdr.hist.exp_time, '\0', 10);
      tempPtr = bxh_getChildElement(acquisitiondatap, "scantime");
      if (tempPtr) {
	i = bxh_getElementStringValue(tempPtr, &tempVal);
	strcpy(hdr.hist.exp_time, tempVal); 
	free(tempVal);
	bxh_element_unref(tempPtr);
      }

      memset(hdr.hist.hist_un0, '\0', 3);
      hdr.hist.views = 0;
      hdr.hist.vols_added = 0;
      hdr.hist.start_field = 0;
      hdr.hist.field_skip = 0;
      hdr.hist.omax = 0;
      hdr.hist.omin = 0;
      hdr.hist.smax = 0;
      hdr.hist.smin = 0;
    }
    /* Done setting values for the header */

    bytesToRead = hdr.dime.bitpix / 8;
    for(i = 0; i < hdr.dime.dim[0]; i++) {
      bytesToRead *= hdr.dime.dim[i+1];
    }
    bytesToWrite = bytesToRead;

    outHeader = (char *)malloc(sizeof(char)*(strlen(ofname) + 9));
    outImage = (char *)malloc(sizeof(char)*(strlen(ofname) + 9));
    outMat = (char *)malloc(sizeof(char)*(strlen(ofname) + 9));
    outBXHHeader = (char *)malloc(sizeof(char)*(strlen(ofname) + 9));

    if (!opts->overwrite) {
	for(i = 0; i < numImgFiles; i++) {
	    if (opts->niigz) {
		sprintf(outHeader, "%s%s", ofname, ".nii.gz");
		sprintf(outImage, "%s%s", ofname, ".nii.gz");
		sprintf(outMat, "%s%s", ofname, ".mat");
	    } else if (opts->nii) {
		sprintf(outHeader, "%s%s", ofname, ".nii");
		sprintf(outImage, "%s%s", ofname, ".nii");
		sprintf(outMat, "%s%s", ofname, ".mat");
	    } else if (opts->dontSplitVols) {
		sprintf(outHeader, "%s%s", ofname, ".hdr");
		sprintf(outImage, "%s%s", ofname, ".img");
		sprintf(outMat, "%s%s", ofname, ".mat");
	    } else {
		sprintf(outHeader, "%s%04d%s", ofname, i + 1, ".hdr");
		sprintf(outImage, "%s%04d%s", ofname, i + 1, ".img");
		sprintf(outMat, "%s%04d%s", ofname, i + 1, ".mat");
	    }

	    if (!opts->dontWriteHdr && stat(outHeader, &statbuf) == 0) {
		fprintf(stderr, "convertBXHToAnalyze(): Analyze Header file already exists ('%s')\n", outHeader);
		goto FAIL;
	    }
	    if (!opts->dontWriteImg && stat(outImage, &statbuf) == 0) {
		fprintf(stderr, "convertBXHToAnalyze(): Analyze Image file already exists ('%s')\n", outImage);
		goto FAIL;
	    }
	    if (!opts->dontWriteMat && stat(outMat, &statbuf) == 0) {
		fprintf(stderr, "convertBXHToAnalyze(): SPM MAT file already exists ('%s')\n", outMat);
		goto FAIL;
	    }
	    if (!opts->dontWriteBxh && opts->splitBxh) {
		sprintf(outBXHHeader, "%s%04d%s", ofname, i + 1, fileext);
		if (stat(outBXHHeader, &statbuf) == 0) {
		    fprintf(stderr, "convertBXHToAnalyze(): output file already exists ('%s')\n", outBXHHeader);
		    goto FAIL;
		}
	    }
	}
	if (!opts->dontWriteBxh && !opts->splitBxh) {
	    sprintf(outBXHHeader, "%s%s", ofname, fileext);
	    if (stat(outBXHHeader, &statbuf) == 0) {
		fprintf(stderr, "convertBXHToAnalyze(): output file already exists ('%s')\n", outBXHHeader);
		goto FAIL;
	    }
	}
    }

    if (opts->nii || opts->niigz) {
	nifp->vox_offset = sizeof(struct analyze_dsr) + 4;
    }

    if ((!opts->spmhdr && !opts->niftihdr) || opts->preferanalyzetypes || opts->analyzetypes) {
	/* the following are not natively supported by the Analyze header,
	   so we will check for overflow now */
	if (strcmp(datarec->elemtype, "int8") == 0 || 
	    strcmp(datarec->elemtype, "uint16") == 0 ||
	    strcmp(datarec->elemtype, "uint32") == 0) {
	    fprintf(stderr, "Checking for overflow...\n");
	    bytesRead = 0;
	    outBuffer = malloc(1024 * 1024);
	    while (bytesRead < bytesToRead) {  
		size_t readsize = bytesToRead - bytesRead; 
		long bufind;
		if (readsize > 1024 * 1024)
		    readsize = 1024 * 1024;
		if (databuf == NULL) {
		    if (bxh_datarec_readRawData(datarec, outBuffer, curposoverflow, readsize, msbfirst) == -1) {
			fprintf(stderr, "Error reading %lu bytes of data at position %lu", (unsigned long)readsize, (unsigned long)curposoverflow);
			return -1;
		    }
		} else {
		    memcpy(outBuffer, &databuf[curposoverflow], readsize);
		}
		if (strcmp(datarec->elemtype, "int8") == 0) {
		    for (bufind = 0; bufind < (int)(readsize/sizeof(int8_t)); bufind++) {
			if (((int8_t *)outBuffer)[bufind] < 0) {
			    overflow = 1;
			    break;
			}
		    }
		}
		else if (strcmp(datarec->elemtype, "uint16") == 0) {
		    for (bufind = 0; bufind < (int)(readsize/sizeof(uint16_t)); bufind++) {
			if (((uint16_t *)outBuffer)[bufind] > (1UL<<15)-1) {
			    overflow = 1;
			    break;
			}
		    }
		}
		else if (strcmp(datarec->elemtype, "uint32") == 0) {
		    for (bufind = 0; bufind < (int)(readsize/sizeof(unsigned int)); bufind++) {
			if (((uint32_t *)outBuffer)[bufind] > (1UL<<31)-1) {
			    overflow = 1;
			    break;
			}
		    }
		}
		curposoverflow += readsize;
		bytesRead += readsize;
	    }
	    if (overflow) {
		if (opts->preferanalyzetypes) {
		    /* this was only a suggestion, so go back to the more appropriate data type */
		    fixelemtype = 0;
		    overflow = 0;
		    fprintf(stderr, "Detected overflow -- going back to original data type %s.\n", datarec->elemtype); 
		    if (opts->spmhdr) {
			hdr.dime.datatype = overflowtype;
			hdr.dime.bitpix = overflowbitpix;
		    } else if (opts->niftihdr) {
			nifp->datatype = overflowtype;
			nifp->bitpix = overflowbitpix;
		    }
		} else {
		    fprintf(stderr, "Overflow detected -- will convert data to datatype %s\n", overflowtypestr);
		    if (opts->niftihdr) {
			nifp->datatype = overflowtype;
			nifp->bitpix = overflowbitpix;
		    } else {
			hdr.dime.datatype = overflowtype;
			hdr.dime.bitpix = overflowbitpix;
		    }
		}
	    } else {
		fixelemtype = 0;
	    }
	    free(outBuffer); outBuffer = NULL;
	}
    }

    for(i = 0; i < numImgFiles; i++) {
      if (opts->niigz) {
	  sprintf(outHeader, "%s%s", ofname, ".nii.gz");
	  sprintf(outImage, "%s%s", ofname, ".nii.gz");
	  sprintf(outMat, "%s%s", ofname, ".mat");
      } else if (opts->nii) {
	  sprintf(outHeader, "%s%s", ofname, ".nii");
	  sprintf(outImage, "%s%s", ofname, ".nii");
	  sprintf(outMat, "%s%s", ofname, ".mat");
      } else if (opts->dontSplitVols) {
	  sprintf(outHeader, "%s%s", ofname, ".hdr");
	  sprintf(outImage, "%s%s", ofname, ".img");
	  sprintf(outMat, "%s%s", ofname, ".mat");
      } else {
	  sprintf(outHeader, "%s%04d%s", ofname, i + 1, ".hdr");
	  sprintf(outImage, "%s%04d%s", ofname, i + 1, ".img");
	  sprintf(outMat, "%s%04d%s", ofname, i + 1, ".mat");
      }

      if (!opts->dontWriteHdr) {
	  if (opts->niigz) {
	      if ((outniigzFile = gzopen(outHeader, "wb")) == NULL) {
		  fprintf(stderr, "convertBXHToAnalyze(): Error opening file '%s'\n", outHeader);
		  if (errno == 0) {
		      fprintf(stderr, "gzopen: Z_MEM_ERROR\n");
		  } else {
		      perror("gzopen");
		  }
		  goto FAIL; 
	      }  
	      if (gzwrite(outniigzFile, &hdr, sizeof(struct analyze_dsr)) != sizeof(struct analyze_dsr)) {
		  fprintf(stderr, "convertBXHToAnalyze(): Error writing header to file '%s'\n", outHeader);
		  perror("gzwrite");
		  goto FAIL;
	      }
	  } else {
	      if ((outHeaderFile = fopen(outHeader, "wb")) == NULL) {
		  fprintf(stderr, "convertBXHToAnalyze(): Error opening file '%s'\n", outHeader);
		  perror("fopen"); 
		  goto FAIL; 
	      }  
	      if (fwrite(&hdr, sizeof(struct analyze_dsr), 1, outHeaderFile) != 1) {
		  fprintf(stderr, "convertBXHToAnalyze(): Error writing header file '%s'\n", outHeader);
		  perror("fwrite");
		  goto FAIL;
	      }
	  }
	  if (opts->niigz) {
	      char extension[4] = "\0\0\0\0";
	      if (gzwrite(outniigzFile, &extension[0], sizeof(extension)) != sizeof(extension)) {
		  fprintf(stderr, "convertBXHToAnalyze(): Error writing to header file '%s'\n", outHeader);
		  perror("gzwrite");
		  goto FAIL;
	      }
	  } else if (opts->nii) {
	      char extension[4] = "\0\0\0\0";
	      if (fwrite(&extension[0], sizeof(extension), 1, outHeaderFile) != 1) {
		  fprintf(stderr, "convertBXHToAnalyze(): Error writing to header file '%s'\n", outHeader);
		  perror("fwrite");
		  goto FAIL;
	      }
	  }
	  
	  if (!opts->niigz) {
	      fclose(outHeaderFile);
	  }
      }

      if (!opts->dontWriteImg) {
	  /* Write out the raw image data file */
	  if (opts->niigz) {
	      /* append to existing (already opened) file */
	  } else if (opts->nii) {
	      /* append to existing file */
	      outImageFile = fopen(outImage, "ab");
	  } else {
	      outImageFile = fopen(outImage, "wb");
	  }
	  if (overflow &&
	      (strcmp(datarec->elemtype, "int8") == 0 ||
	       strcmp(datarec->elemtype, "uint16") == 0)) {
	      outBuffer = malloc(1024 * 1024 * 2);
	  } else {
	      outBuffer = malloc(1024 * 1024);
	  }
	  bytesRead = 0;
	  while (bytesRead < bytesToRead) {  
	      size_t readsize;
	      size_t writesize;
	      readsize = bytesToRead - bytesRead;
	      if (readsize > 1024 * 1024)
		  readsize = 1024 * 1024;
	      writesize = readsize;
	      if (databuf == NULL) {
		  if (bxh_datarec_readRawData(datarec, outBuffer, curpos, readsize, msbfirst) == -1) {
		      fprintf(stderr, "Error reading %lu bytes of data at position %lu", (unsigned long)readsize, (unsigned long)curpos);
		      return -1;
		  }
	      } else {
		  memcpy(outBuffer, &databuf[curpos], readsize);
	      }

	      if (overflow) {
		  /* overflow detected: convert to new type */
		  long bufind;
		  if (strcmp(datarec->elemtype, "int8") == 0) {
		      /* datatype sizes are different; go backwards so we don't overwrite data we haven't converted yet */
		      for (bufind = (long)((readsize/sizeof(int8_t)) - 1); bufind >= 0; bufind--) {
			  ((int16_t *)outBuffer)[bufind] = (int16_t)(((int8_t *)outBuffer)[bufind]);
		      }
		      writesize = readsize * 2;
		      bytesToWrite = bytesToRead * 2;
		  } else if (strcmp(datarec->elemtype, "uint16") == 0) {
		      /* datatype sizes are different; go backwards so we don't overwrite data we haven't converted yet */
		      for (bufind = (long)((readsize/sizeof(uint16_t)) - 1); bufind >= 0; bufind--) {
			  ((int32_t *)outBuffer)[bufind] = (int32_t)((uint16_t *)outBuffer)[bufind];
		      }
		      writesize = readsize * 2;
		      bytesToWrite = bytesToRead * 2;
		  } else if (strcmp(datarec->elemtype, "uint32") == 0) {
		      /* datatype sizes are the same; go backwards anyway, because the others do */
		      for (bufind = (long)((readsize/sizeof(uint32_t)) - 1); bufind >= 0; bufind--) {
			  ((float *)outBuffer)[bufind] = (float)((uint32_t *)outBuffer)[bufind];
		      }
		  }
	      }

	      if (opts->niigz) {
		  if (gzwrite(outniigzFile, outBuffer, writesize) != writesize) {
		      fprintf(stderr, "convertBXHToAnalyze(): Error writing image to '%s'\n", outImage);
		      perror("gzwrite");
		      goto FAIL;
		  }
	      } else {
		  if (fwrite(outBuffer, writesize, 1, outImageFile) != 1) {
		      perror("fwrite");
		      goto FAIL;
		  }
	      }
	      curpos += readsize;
	      bytesRead += readsize;
	  } 
	  free(outBuffer); outBuffer = NULL;
	  if (opts->niigz) {
	      gzclose(outniigzFile);
	  } else {
	      fclose(outImageFile);
	  }
      }

      /* Write out the SPM .mat file */
      if (!opts->dontWriteMat &&
	  datarec->dimensions[0].direction != NULL &&
	  datarec->dimensions[1].direction != NULL &&
	  datarec->dimensions[2].direction != NULL) {
	float M[16];
	float origin1[4];
	double Rr, Ra, Rs; /* row */
	double Cr, Ca, Cs; /* column */
	double Nr, Na, Ns; /* slice */
	struct bxhdimension *dimx, *dimy, *dimz;
	struct bxhdimension *dimr, *dima, *dims;
	double xspacing, yspacing, zspacing;

	dimx = &datarec->dimensions[0];
	dimy = &datarec->dimensions[1];
	dimz = &datarec->dimensions[2];
	Rr = dimx->direction[0];
	Ra = dimx->direction[1];
	Rs = dimx->direction[2];
	Cr = dimy->direction[0];
	Ca = dimy->direction[1];
	Cs = dimy->direction[2];
	Nr = dimz->direction[0];
	Na = dimz->direction[1];
	Ns = dimz->direction[2];

	if (fabs(Rr) > fabs(Cr) && fabs(Rr) > fabs(Nr)) {
	    dimr = dimx;
	    if (fabs(Ca) > fabs(Ra) && fabs(Ca) > fabs(Na)) {
		dima = dimy; dims = dimz;
	    } else {
		dima = dimz; dims = dimy;
	    }
	} else if (fabs(Cr) > fabs(Rr) && fabs(Cr) > fabs(Nr)) {
	    dimr = dimy;
	    if (fabs(Ra) > fabs(Ca) && fabs(Ra) > fabs(Na)) {
		dima = dimx; dims = dimz;
	    } else {
		dima = dimz; dims = dimx;
	    }
	} else {
	    dimr = dimz;
	    if (fabs(Ra) > fabs(Ca) && fabs(Ra) > fabs(Na)) {
		dima = dimx; dims = dimy;
	    } else {
		dima = dimy; dims = dimx;
	    }
	}

	xspacing = dimx->spacing;
	yspacing = dimy->spacing;
	zspacing = dimz->spacing;

	/* here is the matrix given 0,0,0 as first voxel */
	/* first row: 0-3, second row: 4-7, etc. */
	M[ 0] = (float)(dimx->direction[0] * xspacing);
	M[ 1] = (float)(dimy->direction[0] * yspacing);
	M[ 2] = (float)(dimz->direction[0] * zspacing);
	M[ 4] = (float)(dimx->direction[1] * xspacing);
	M[ 5] = (float)(dimy->direction[1] * yspacing);
	M[ 6] = (float)(dimz->direction[1] * zspacing);
	M[ 8] = (float)(dimx->direction[2] * xspacing);
	M[ 9] = (float)(dimy->direction[2] * yspacing);
	M[10] = (float)(dimz->direction[2] * zspacing);
	if (opts->zeroorigin) {
	    M[ 3] = 0;
	    M[ 7] = 0;
	    M[11] = 0;
	} else {
	    M[ 3] = (float)dimr->origin;
	    M[ 7] = (float)dima->origin;
	    M[11] = (float)dims->origin;
	}
	M[12] = 0;
	M[13] = 0;
	M[14] = 0;
	M[15] = 1;

	/* correct so that volume coordinate 1,1,1 can point to the origin */
	if (!opts->zeroorigin) {
	    origin1[0] = M[0] + M[1] + M[2] + M[3];
	    origin1[1] = M[4] + M[5] + M[6] + M[7];
	    origin1[2] = M[8] + M[9] + M[10] + M[11];
	    /* P0 = 2*P0 - P1 */
	    M[3] = M[3] * 2 - origin1[0];
	    M[7] = M[7] * 2 - origin1[1];
	    M[11] = M[11] * 2 - origin1[2];
	}
	MatFileWrite(outMat,M,4,4,"M");
      }
    }

    /* Write out the BXH Header file */ 
    if (!opts->dontWriteBxh) {
      /* correct elemtype if unsupported Analyze type */
      if (fixelemtype) {
	free(datarec->elemtype);
	datarec->elemtype = strdup(overflowtypestr);
      }

      datarec->msbfirstfrags = msbfirst;
 
      {
	  const char * inputs[2] = {ifname, ofname};
	  if (ifname == NULL) {
	      inputs[0] = "(no input file specified)";
	  }
	  if (bxh_addAutoHistoryEntry(docp, "bxh2analyze", &inputs[0], 2) != 0)
	      goto FAIL;
      }

      if (opts->splitBxh) {
	datarec->numdims = hdr.dime.dim[0];
	for (i = 0; i < hdr.dime.dim[0]; i++) {
	    datarec->dimensions[i].size = hdr.dime.dim[i+1];
	}
	for(i = 0; i < numImgFiles; i++) {
	  if (opts->dontSplitVols) {
	      sprintf(outBXHHeader, "%s%s", ofname, fileext);
	      sprintf(outImage, "%s%s", ofname, ".img");
	  } else {
	      sprintf(outBXHHeader, "%s%04d%s", ofname, i + 1, fileext);
	      sprintf(outImage, "%s%04d%s", ofname, i + 1, ".img");
	  }

	  bxh_datarec_frags_free(datarec);
	  bxh_datarec_addfrag(datarec, outImage, 0, bytesToWrite, outBXHHeader, 1);
	  if (bxh_datarec_writeToElement(imagedatap, datarec) == -1) {
	    fprintf(stderr, "Error writing new frags to datarec element\n");
	    goto FAIL;
	  } 
	
	  if (bxh_writeFile(docp, outBXHHeader) != 0) {
	    fprintf(stderr, "Error writing output file %s\n", outBXHHeader);
	    goto FAIL;
	  } 
	}
      } else {
	bxh_datarec_frags_free(datarec);
	sprintf(outBXHHeader, "%s%s", ofname, fileext);
	for(i = 0; i < numImgFiles; i++) {
	  if (opts->nii || opts->niigz) {
	      /* even .nii.gz will point to just .nii */
	      sprintf(outImage, "%s%s", ofname, ".nii");
	  } else if (opts->dontSplitVols) {
	      sprintf(outImage, "%s%s", ofname, ".img");
	  } else {
	      sprintf(outImage, "%s%04d%s", ofname, i + 1, ".img");
	  }
	  if (opts->nii || opts->niigz) {
	      bxh_datarec_addfrag(datarec, outImage, sizeof(struct analyze_dsr) + 4, bytesToWrite, outBXHHeader, 1);
	  } else {
	      bxh_datarec_addfrag(datarec, outImage, 0, bytesToWrite, outBXHHeader, 1);
	  }
	}
	if (bxh_datarec_writeToElement(imagedatap, datarec) == -1) {
	  fprintf(stderr, "Error writing new frags to datarec element\n");
	  goto FAIL;
	} 
	
	if (bxh_writeFile(docp, outBXHHeader) != 0) {
	  fprintf(stderr, "Error writing output file %s\n", outBXHHeader);
	  goto FAIL;
	} 
      }
    }

    goto EXIT;

FAIL:
    fprintf(stderr, "convertBXHToAnalyze(): conversion failed.\n");
    result = -1;

EXIT:
    if (outBXHHeader)
      free(outBXHHeader);
    if (outHeader)
      free(outHeader);
    if (outImage)
      free(outImage);
    if (outMat)
      free(outMat);

    if (databuf && databuf != inputdata && databuf != (char *)newbdr.dataptr)
      free(databuf);

    if (bdrp == &newbdr) {
	bxh_datareaddata_free(&newbdr);
    }

    return result;
}

static int
maxPosition(double * nums)
{
  if (fabs(nums[0]) > fabs(nums[1])) {
    if (fabs(nums[0]) > fabs(nums[2]))
      return 1;
    else
      return 3;
  }
  else {    
    if (fabs(nums[1]) > fabs(nums[2])) 
      return 2;
    else
      return 3;
  }  
}

static int
getOrientation(bxhrawdatarec *datarec, char *orientVal)
{
  int maxX , maxY, maxZ, i;
  maxX = maxY = maxZ = 0;

  if (datarec->numdims < 3)
    return -1;

  if (datarec->dimensions[0].direction == NULL) {
    orientVal = 0;
    return -1;
  }

  for(i = 0; i < 3; i++) {
    if (strcmp(datarec->dimensions[i].type, "x") == 0)
      maxX = maxPosition(datarec->dimensions[i].direction);
    else if (strcmp(datarec->dimensions[i].type, "y") == 0)
      maxY = maxPosition(datarec->dimensions[i].direction);
    else if (strcmp(datarec->dimensions[i].type, "z") == 0)
      maxZ = maxPosition(datarec->dimensions[i].direction);
  }

  /* transverse unflipped (LAS) */
  if (maxX == 1 && datarec->dimensions[0].direction[0] < 0 && /* X -> L */
      maxY == 2 && datarec->dimensions[1].direction[1] > 0 && /* Y -> A */
      maxZ == 3 && datarec->dimensions[2].direction[2] > 0) { /* Z -> S */
    *orientVal = 0;
    return 1;
  }
  /* coronal unflipped (LSA) */
  else if (maxX == 1 && datarec->dimensions[0].direction[0] < 0 && /* X -> L */
	   maxY == 3 && datarec->dimensions[1].direction[2] > 0 && /* Y -> S */
	   maxZ == 2 && datarec->dimensions[2].direction[1] > 0) { /* Z -> A */
    *orientVal = 1;
    return 1;
  }
  /* sagittal unflipped (ASL) */
  else if (maxX == 2 && datarec->dimensions[0].direction[1] > 0 && /* X -> A */
	   maxY == 3 && datarec->dimensions[1].direction[2] > 0 && /* Y -> S */
	   maxZ == 1 && datarec->dimensions[2].direction[0] < 0) { /* Z -> L */
    *orientVal = 2;
    return 1;
  }
  /* transverse flipped (LPS) */
  else if (maxX == 1 && datarec->dimensions[0].direction[0] < 0 && /* X -> L */
	   maxY == 2 && datarec->dimensions[1].direction[1] < 0 && /* Y -> P */
	   maxZ == 3 && datarec->dimensions[2].direction[2] > 0) { /* Z -> S */
    *orientVal = 3;
    return 1;
  }
  /* coronal flipped (LIA) */
  else if (maxX == 1 && datarec->dimensions[0].direction[0] < 0 && /* X -> L */
	   maxY == 3 && datarec->dimensions[1].direction[2] < 0 && /* Y -> I */
	   maxZ == 2 && datarec->dimensions[2].direction[1] > 0) { /* Z -> A */
    *orientVal = 4;
    return 1;
  }
  /* sagittal flipped (AIL) */
  else if (maxX == 2 && datarec->dimensions[0].direction[1] > 0 && /* X -> A */
	   maxY == 3 && datarec->dimensions[1].direction[2] < 0 && /* Y -> I */
	   maxZ == 1 && datarec->dimensions[2].direction[0] < 0) { /* Z -> L */
    *orientVal = 5;
    return 1;
  }
  else {
    *orientVal = 0;
    return -1;
  }
}


/*** following code courtesy of Bruce Fischl via Doug Greve ***/

#define MATFILE_PC      0000
#define MATFILE_SPARC   1000
#define MATFILE_DOUBLE  00
#define MATFILE_FULL    0

typedef struct
{
    int32_t  type ;
    int32_t  mrows ;
    int32_t  ncols ;
    int32_t  imagf ;
    int32_t  namlen ;
} MATHD ;

typedef struct
{
    int32_t  type ;
    int32_t  mrows ;
    int32_t  ncols ;
    int32_t  imagf ;
    int32_t  namlen ;
    char  *data ;
    char  *idata ;
} MATFILE ;

static int
MatFileWrite(const char *fname, float *data, int rows, int cols, char *name)
{
    int     row, col, nitems, mtype ;
    float   *fptr ;   
    FILE    *fp ;
    double  dval ;
    MATFILE mf ;
    int     msbfirst;

    msbfirst = 1;
    msbfirst = (((char *)&msbfirst)[0] == 0);

    mf.namlen = (int32_t)strlen(name)+1 ;
    mf.mrows = (int32_t)rows ;
    mf.ncols = (int32_t)cols ;
    mf.imagf = 0L ;
    if (msbfirst) {
	mtype = MATFILE_SPARC ;
    } else {
	mtype = MATFILE_PC ;
    }
    mf.type = mtype + MATFILE_DOUBLE + MATFILE_FULL ;

    fp = fopen(fname, "wb") ;
    if (!fp)return(-1) ;

    nitems = fwrite(&mf, 1, sizeof(MATHD), fp) ;
    if (nitems != sizeof(MATHD))
	{
	    fclose(fp) ;
	    return(-2) ;
	}

    nitems = fwrite(name, sizeof(char), (int)mf.namlen, fp) ;
    if (nitems != (int)mf.namlen)
	{
	    fclose(fp) ;
	    return(-3) ;
	}

    /* write out in column-major format */
    for (col = 0 ; col < cols ; col++)
	{
	    for (row = 0 ; row < rows ; row++)
		{
		    fptr = data + row * cols + col ;
		    dval = (double)(*fptr) ;
		    nitems = fwrite(&dval, 1, sizeof(double), fp) ;
		    if (nitems != sizeof(double))
			{
			    fclose(fp) ;
			    return(-2) ;
			}
		}
	}
  
    fclose(fp) ;
    return(0) ;
}

