static const char rcsid[] = "$Id: bxh_signa.c,v 1.49 2008-05-21 18:07:03 gadde Exp $";

/* bxh_signa.cpp --
 *
 * Extract SIGNA data into BXH files using these functions.
 *
 * Author: Syam Gadde (gadde@biac.duke.edu), Nov. 2002.
 */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "bxh_utils.h"
#include "bxh_datarec.h"
#include "bxh_signa.h"
#include "bxh_signamaps.h"

#ifndef TOPLEVELELEM
#define TOPLEVELELEM "bxh"
#endif

static struct signamaps * signamap3list[4] = { &signamaps3align0, &signamaps3align2, &signamaps3align4, NULL };
static struct signamaps * signamap11list[4] = { &signamaps11align0, &signamaps11align2, &signamaps11align4, NULL };
static struct signamaps * signamap20list[4] = { &signamaps20align0, &signamaps20align2, &signamaps20align4, NULL };
static struct signamaps * signamap24list[4] = { &signamaps24align0, &signamaps24align2, &signamaps24align4, NULL };
static struct signamaps * signamap27list[4] = { &signamaps27align0, &signamaps27align2, &signamaps27align4, NULL };

static int grabAndSwapVal(const void * source, off_t off, size_t size, void * data, int valismsbfirst);

#undef FUNC
#define FUNC "bxh_SIGNA_readPFileFromFile"
/**
 * Extract metadata from a file containing a PFile header and adds
 * metadata to the given BXH document.
 *
 * @param docp BXH document pointer
 * @param filename name of input PFile
 * @param numslices number of slices (z) in this dataset
 * @param numtimepoints number of timepoints (t) in this dataset
 * @param opt_dimorder string of form "x,y,z,t", "x,y,t,z", etc.
 * @param datamsbfirst 1 if data is big-endian, 0 if little-endian
 * @param forceversion if non-negative, use this as version number of pfile
 * @return 0 on success, or 1 on error.
 */
int
bxh_SIGNA_readPFileFromFile(BXHDocPtr docp, const char *filename, unsigned int numslices, unsigned int numtimepoints, const char * opt_dimorder, int datamsbfirst, float forceversion)
{
    int retval = 0;
    FILE * fp = NULL;
    void * examdata = NULL;
    void * seriesdata = NULL;
    void * mrimagedata = NULL;
    char hdrverbuf[4];
    float hdrver = 0;
    int hdrbigend = 1;

    size_t HSIZE_RDB_HEADER_REC = 2048;
    size_t HSIZE_PER_PASS = 4096;
    size_t HSIZE_UNLOCK = 4096;
    size_t HSIZE_DATA_ACQ = 20480; /* or 40960 for >= 8.0 revs */
    size_t HSIZE_NEX = 2052;
    size_t HSIZE_NEX_ABORT = 2052;
    size_t HSIZE_TOOLSDATA = 2048;
    size_t HSIZE_PRESCAN = 0;
    size_t HSIZE_EXAMDATA = 1040;
    size_t HSIZE_SERIESDATA = 1028; /* or 1536 for >= 9.0 revs, 2048 for >= 11.0 */
    size_t HSIZE_MRIMAGEDATA = 1044; /* or 1536 for >= 9.0 revs */

    off_t HOFFSET_RDB_HEADER_REC = 0;
    off_t HOFFSET_PER_PASS = 0;
    off_t HOFFSET_UNLOCK = 0;
    off_t HOFFSET_DATA_ACQ = 0;
    off_t HOFFSET_NEX = 0;
    off_t HOFFSET_NEX_ABORT = 0;
    off_t HOFFSET_TOOLSDATA = 0;
    off_t HOFFSET_PRESCAN = 0;
    off_t HOFFSET_EXAMDATA = 0;
    off_t HOFFSET_SERIESDATA = 0;
    off_t HOFFSET_MRIMAGEDATA = 0;
    off_t HOFFSET_USERDATA = 0;

    if ((fp = fopen(filename, "rb")) == NULL) {
	fprintf(stderr, "Error opening pfile '%s'\n", filename);
	perror("open");
	goto FAIL;
    }

    if (fseek(fp, 0, SEEK_SET) == -1) {
	perror("fseek");
	goto FAIL;
    }
    if (fread(&hdrverbuf[0], 4, 1, fp) != 1) {
	fprintf(stderr, "fread: error (magic)\n");
	goto FAIL;
    }
    grabAndSwapVal(&hdrverbuf[0], 0, 4, &hdrver, 0);
    if (hdrver != 9.0 && hdrver != 11.0 && hdrver >= 20.0) {
	grabAndSwapVal(&hdrverbuf[0], 0, 4, &hdrver, 1);
    }
    if (forceversion >= 0) {
	hdrver = forceversion;
    }
    if (hdrver >= 8.0) {
	HSIZE_DATA_ACQ = 40960;
    }
    if (hdrver >= 9.0) {
	HSIZE_SERIESDATA = 1536;
	HSIZE_MRIMAGEDATA = 1536;
	hdrbigend = 0;
    }
    if (hdrver >= 11.0) {
	HSIZE_DATA_ACQ = 45056;
	HSIZE_SERIESDATA = 2048;
    }
    if (hdrver >= 20.0) {
	HSIZE_RDB_HEADER_REC = 4096;
	HSIZE_PER_PASS = 16384;
	HSIZE_UNLOCK = 16384;
	HSIZE_DATA_ACQ = 98304;
	HSIZE_NEX = 2052;
	HSIZE_NEX_ABORT = 2052;
	HSIZE_TOOLSDATA = 3548;
	HSIZE_EXAMDATA = 1960;
	HSIZE_SERIESDATA = 2560;
	HSIZE_MRIMAGEDATA = 2448;
    }
    if (hdrver >= 27.0) {
	HSIZE_DATA_ACQ = 147456;
	HSIZE_TOOLSDATA = 2048;
	HSIZE_PRESCAN = 3188;
    }

    /* now that we have the correct sizes of each section, calculate offsets */
    HOFFSET_RDB_HEADER_REC = 0;
    HOFFSET_PER_PASS = ((HOFFSET_RDB_HEADER_REC) + (HSIZE_RDB_HEADER_REC));
    HOFFSET_UNLOCK = ((HOFFSET_PER_PASS) + (HSIZE_PER_PASS));
    HOFFSET_DATA_ACQ = ((HOFFSET_UNLOCK) + (HSIZE_UNLOCK));
    HOFFSET_NEX = ((HOFFSET_DATA_ACQ) + (HSIZE_DATA_ACQ));
    HOFFSET_NEX_ABORT = ((HOFFSET_NEX) + (HSIZE_NEX));
    HOFFSET_TOOLSDATA = ((HOFFSET_NEX_ABORT) + (HSIZE_NEX_ABORT));
    HOFFSET_PRESCAN = ((HOFFSET_TOOLSDATA) + (HSIZE_TOOLSDATA));
    HOFFSET_EXAMDATA = ((HOFFSET_PRESCAN) + (HSIZE_PRESCAN));
    HOFFSET_SERIESDATA = ((HOFFSET_EXAMDATA) + (HSIZE_EXAMDATA));
    HOFFSET_MRIMAGEDATA = ((HOFFSET_SERIESDATA) + (HSIZE_SERIESDATA));
    HOFFSET_USERDATA = ((HOFFSET_MRIMAGEDATA) + (HSIZE_MRIMAGEDATA));

    if (fseek(fp, HOFFSET_EXAMDATA, SEEK_SET) == -1) {
	perror("fseek");
	goto FAIL;
    }
    examdata = (void *)malloc(HSIZE_EXAMDATA);
    if (fread(examdata, HSIZE_EXAMDATA, 1, fp) != 1) {
	fprintf(stderr, "fread: error (exam)\n");
	goto FAIL;
    }
    
    if (fseek(fp, HOFFSET_SERIESDATA, SEEK_SET) == -1) {
	perror("fseek");
	goto FAIL;
    }
    seriesdata = (void *)malloc(HSIZE_SERIESDATA);
    if (fread(seriesdata, HSIZE_SERIESDATA, 1, fp) != 1) {
	fprintf(stderr, "fread: error (series)\n");
	goto FAIL;
    }
    
    if (fseek(fp, HOFFSET_MRIMAGEDATA, SEEK_SET) == -1) {
	perror("fseek");
	goto FAIL;
    }
    mrimagedata = (void *)malloc(HSIZE_MRIMAGEDATA);
    if (fread(mrimagedata, HSIZE_MRIMAGEDATA, 1, fp) != 1) {
	fprintf(stderr, "fread: error (mrimage)\n");
	goto FAIL;
    }
    
    retval = bxh_SIGNA_readPFileFromHeaders(docp, examdata, seriesdata, mrimagedata, numslices, numtimepoints, opt_dimorder, datamsbfirst, hdrbigend, hdrver);

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (fp)
	fclose(fp);
    if (examdata)
	free(examdata);
    if (seriesdata)
	free(seriesdata);
    if (mrimagedata)
	free(mrimagedata);
    return retval;
}

#undef FUNC
#define FUNC "bxh_SIGNA_readPFileFromHeaders"
/**
 * Extract metadata from SIGNA PFile headers and adds to BXH document.
 *
 * @param docp BXH document pointer
 * @param examdata "exam" portion of PFile header
 * @param seriesdata "series" portion of PFile header
 * @param mrimagedata "mr" portion of PFile header
 * @param numslices number of slices (z) in this dataset
 * @param numtimepoints number of timepoints (t) in this dataset
 * @param opt_dimorder string of form "x,y,z,t", "x,y,t,z", etc.
 * @param datamsbfirst 1 if data is big-endian, 0 if little-endian
 * @return 0 on success, or 1 on error.
 */
int
bxh_SIGNA_readPFileFromHeaders(BXHDocPtr docp, const void * examdata, const void * seriesdata, const void * mrimagedata, unsigned int numslices, unsigned int numtimepoints, const char * opt_dimorder, int datamsbfirst, int hdrbigend, float hdrver)
{
    return bxh_SIGNA_addHeaders(docp, 4, examdata, seriesdata, mrimagedata, numslices, numtimepoints, opt_dimorder, datamsbfirst, hdrbigend, hdrver);
}

#undef FUNC
#define FUNC "bxh_SIGNA_readIMGFromFile"
/**
 * Extract metadata from SIGNA IMG file and adds it to BXH document.
 *
 * @param docp BXH document pointer
 * @param filename name of input PFile
 * @param numslices number of slices in this dataset
 * @param numtimepoints number of timepoints (t) in this dataset
 * @param opt_dimorder string of form "x,y,z,t", "x,y,t,z", etc.
 * @return 0 on success, or 1 on error.
 */
int
bxh_SIGNA_readIMGFromFile(BXHDocPtr docp, const char *filename, unsigned int numslices, unsigned int numtimepoints, const char * opt_dimorder)
{
    int retval = 0;
    FILE * fp = NULL;
    void * examdata = NULL;
    void * seriesdata = NULL;
    void * mrimagedata = NULL;

    size_t ISIZE_FILEDATA = 2304;
    size_t ISIZE_SUITEDATA = 114;
    size_t ISIZE_EXAMDATA = 1024;
    size_t ISIZE_SERIESDATA = 1020;
    size_t ISIZE_MRIMAGEDATA = 1022;
/*    size_t ISIZE_USERDATA = 2420;*/

    off_t IOFFSET_FILEDATA = 0;
    off_t IOFFSET_SUITEDATA = ((IOFFSET_FILEDATA) + (ISIZE_FILEDATA));
    off_t IOFFSET_EXAMDATA = ((IOFFSET_SUITEDATA) + (ISIZE_SUITEDATA));
    off_t IOFFSET_SERIESDATA = ((IOFFSET_EXAMDATA) + (ISIZE_EXAMDATA));
    off_t IOFFSET_MRIMAGEDATA = ((IOFFSET_SERIESDATA) + (ISIZE_SERIESDATA));
/*    off_t IOFFSET_USERDATA = ((IOFFSET_MRIMAGEDATA) + (ISIZE_MRIMAGEDATA));*/

    if ((fp = fopen(filename, "rb")) == NULL) {
	fprintf(stderr, "Error opening signa image '%s'\n", filename);
	perror("open");
	goto FAIL;
    }

    if (fseek(fp, IOFFSET_EXAMDATA, SEEK_SET) == -1) {
	perror("fseek");
	goto FAIL;
    }
    examdata = (void *)malloc(ISIZE_EXAMDATA);
    if (fread(examdata, ISIZE_EXAMDATA, 1, fp) != 1) {
	fprintf(stderr, "fread: error (exam)\n");
	goto FAIL;
    }
    
    if (fseek(fp, IOFFSET_SERIESDATA, SEEK_SET) == -1) {
	perror("fseek");
	goto FAIL;
    }
    seriesdata = (void *)malloc(ISIZE_SERIESDATA);
    if (fread(seriesdata, ISIZE_SERIESDATA, 1, fp) != 1) {
	fprintf(stderr, "fread: error (series)\n");
	goto FAIL;
    }
    
    if (fseek(fp, IOFFSET_MRIMAGEDATA, SEEK_SET) == -1) {
	perror("fseek");
	goto FAIL;
    }
    mrimagedata = (void *)malloc(ISIZE_MRIMAGEDATA);
    if (fread(mrimagedata, ISIZE_MRIMAGEDATA, 1, fp) != 1) {
	fprintf(stderr, "fread: error (mrimage)\n");
	goto FAIL;
    }
    
    retval = bxh_SIGNA_readIMGFromHeaders(docp, examdata, seriesdata, mrimagedata, numslices, numtimepoints, opt_dimorder, 1);

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (fp)
	fclose(fp);
    if (examdata)
	free(examdata);
    if (seriesdata)
	free(seriesdata);
    if (mrimagedata)
	free(mrimagedata);
    return retval;
}

#undef FUNC
#define FUNC "bxh_SIGNA_readIMGFromHeaders"
/**
 * Extract metadata from SIGNA headers and adds to BXH document.
 *
 * @param docp BXH document pointer
 * @param examdata "exam" portion of IMG header
 * @param seriesdata "series" portion of IMG header
 * @param mrimagedata "mr" portion of IMG header
 * @param numslices number of slices in this dataset
 * @param numtimepoints number of timepoints (t) in this dataset
 * @param opt_dimorder string of form "x,y,z,t", "x,y,t,z", etc.
 * @return 0 on success, or 1 on error.
 */
int
bxh_SIGNA_readIMGFromHeaders(BXHDocPtr docp, const void * examdata, const void * seriesdata, const void * mrimagedata, unsigned int numslices, unsigned int numtimepoints, const char * opt_dimorder, int hdrbigend)
{
    return bxh_SIGNA_addHeaders(docp, 2, examdata, seriesdata, mrimagedata, numslices, numtimepoints, opt_dimorder, 1, hdrbigend, -1);
}

#undef FUNC
#define FUNC "bxh_SIGNA_readXIMGFromFile"
/**
 * Extract metadata from SIGNA XIMG file and adds it to BXH document.
 *
 * @param docp BXH document pointer
 * @param filename name of input PFile
 * @param numslices number of slices in this dataset
 * @param numtimepoints number of timepoints (t) in this dataset
 * @param opt_dimorder string of form "x,y,z,t", "x,y,t,z", etc.
 * @return 0 on success, or 1 on error.
 */
int
bxh_SIGNA_readXIMGFromFile(BXHDocPtr docp, const char *filename, unsigned int numslices, unsigned int numtimepoints, const char * opt_dimorder)
{
    int retval = 0;
    FILE * fp = NULL;
    void * examdata = NULL;
    void * seriesdata = NULL;
    void * mrimagedata = NULL;

    size_t XSIZE_SUITEDATA = 116;
    size_t XSIZE_EXAMDATA = 1040;
    size_t XSIZE_SERIESDATA = 1028;
    size_t XSIZE_MRIMAGEDATA = 1044;
/*    size_t XSIZE_FILEDATA = 2304;*/

    off_t XOFFSET_SUITEDATA = 0;
    off_t XOFFSET_EXAMDATA = ((XOFFSET_SUITEDATA) + (XSIZE_SUITEDATA));
    off_t XOFFSET_SERIESDATA = ((XOFFSET_EXAMDATA) + (XSIZE_EXAMDATA));
    off_t XOFFSET_MRIMAGEDATA = ((XOFFSET_SERIESDATA) + (XSIZE_SERIESDATA));
/*    off_t XOFFSET_FILEDATA = ((XOFFSET_MRIMAGEDATA) + (XSIZE_MRIMAGEDATA));*/

    if ((fp = fopen(filename, "rb")) == NULL) {
	fprintf(stderr, "Error opening ximg '%s'\n", filename);
	perror("open");
	goto FAIL;
    }

    if (fseek(fp, XOFFSET_EXAMDATA, SEEK_SET) == -1) {
	perror("fseek");
	goto FAIL;
    }
    examdata = (void *)malloc(XSIZE_EXAMDATA);
    if (fread(examdata, XSIZE_EXAMDATA, 1, fp) != 1) {
	fprintf(stderr, "fread: error (exam)\n");
	goto FAIL;
    }
    
    if (fseek(fp, XOFFSET_SERIESDATA, SEEK_SET) == -1) {
	perror("fseek");
	goto FAIL;
    }
    seriesdata = (void *)malloc(XSIZE_SERIESDATA);
    if (fread(seriesdata, XSIZE_SERIESDATA, 1, fp) != 1) {
	fprintf(stderr, "fread: error (series)\n");
	goto FAIL;
    }
    
    if (fseek(fp, XOFFSET_MRIMAGEDATA, SEEK_SET) == -1) {
	perror("fseek");
	goto FAIL;
    }
    mrimagedata = (void *)malloc(XSIZE_MRIMAGEDATA);
    if (fread(mrimagedata, XSIZE_MRIMAGEDATA, 1, fp) != 1) {
	fprintf(stderr, "fread: error (mrimage)\n");
	goto FAIL;
    }
    
    retval = bxh_SIGNA_readXIMGFromHeaders(docp, examdata, seriesdata, mrimagedata, numslices, numtimepoints, opt_dimorder, 1);

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (fp)
	fclose(fp);
    if (examdata)
	free(examdata);
    if (seriesdata)
	free(seriesdata);
    if (mrimagedata)
	free(mrimagedata);
    return retval;
}

#undef FUNC
#define FUNC "bxh_SIGNA_readXIMGFromHeaders"
/**
 * Extract metadata from SIGNA (XIMG) headers and adds to BXH document.
 *
 * @param docp BXH document pointer
 * @param examdata "exam" portion of IMG header
 * @param seriesdata "series" portion of IMG header
 * @param mrimagedata "mr" portion of IMG header
 * @param numslices number of slices in this dataset
 * @param numtimepoints number of timepoints (t) in this dataset
 * @param opt_dimorder string of form "x,y,z,t", "x,y,t,z", etc.
 * @return 0 on success, or 1 on error.
 */
int
bxh_SIGNA_readXIMGFromHeaders(BXHDocPtr docp, const void * examdata, const void * seriesdata, const void * mrimagedata, unsigned int numslices, unsigned int numtimepoints, const char * opt_dimorder, int hdrbigend)
{
    return bxh_SIGNA_addHeaders(docp, 0, examdata, seriesdata, mrimagedata, numslices, numtimepoints, opt_dimorder, 1, hdrbigend, -1);
}

static int
grabAndSwapVal(const void * source, off_t off, size_t size, void * data, int valismsbfirst)
{
    unsigned int tmpint;
    memcpy(data, (char *)source + off, size);
    tmpint = 1;
    if ((((char *)&tmpint)[0] != 0 && valismsbfirst) ||
	(((char *)&tmpint)[0] == 0 && !valismsbfirst)) {
	char * chrdata = (char *)data;
	if (size == 2) {
	    char swap = chrdata[0];
	    chrdata[0] = chrdata[1];
	    chrdata[1] = swap;
	} else if (size == 4) {
	    char swap = chrdata[0];
	    chrdata[0] = chrdata[3];
	    chrdata[3] = swap;
	    swap = chrdata[1];
	    chrdata[1] = chrdata[2];
	    chrdata[2] = swap;
	} else if (size != 1) {
	    fprintf(stderr, "grabAndSwapValue: Can't swap sizes other than 1, 2 or 4");
	    return -1;
	}
    }
    return 0;
}

#undef FUNC
#define FUNC "bxh_SIGNA_addHeaders"
/**
 * Extract metadata from SIGNA headers and adds to BXH document.
 * This is the generic function that implements the main
 * functionality of all the above SIGNA converting functions.
 *
 * @param docp BXH document pointer
 * @param maxalignsize size of byte boundaries on which data fields are
 *        aligned.  0 means always align on multiples of field size.
 * @param examdata "exam" portion of IMG header
 * @param seriesdata "series" portion of IMG header
 * @param mrimagedata "mr" portion of IMG header
 * @param numslices number of slices (z) in this dataset
 * @param numtimepoints number of timepoints (t) in this dataset
 * @param opt_dimorder string of form "x,y,z", "x,y,z,t", "x,y,t,z", etc.
 * @param datamsbfirst 1 if data is big-endian, 0 if little-endian
 * @return 0 on success, or 1 on error.
 */
int
bxh_SIGNA_addHeaders(BXHDocPtr docp, unsigned int maxalignsize, const void * examdata, const void * seriesdata, const void * mrimagedata, unsigned int numslices, unsigned int numtimepoints, const char * opt_dimorder, int datamsbfirst, int hdrbigend, float hdrver)
{
    int result = 0;

    BXHElementPtr bxh = NULL;
    BXHElementPtr imagedata = NULL;
    BXHElementPtr acquisitiondata = NULL;
    BXHElementPtr subject = NULL;
    bxhdimension * dimx = NULL;
    bxhdimension * dimy = NULL;
    bxhdimension * dimz = NULL;
    bxhdimension * dimt = NULL;
    bxhdimension * dimr = NULL;
    bxhdimension * dima = NULL;
    bxhdimension * dims = NULL;

    char tmpbuf[34];
    char numbuf[32];

    struct signaexmap * exmap = NULL;
    struct signasemap * semap = NULL;
    struct signamrmap * mrmap = NULL;
    
    struct signamaps ** curmapptr = NULL;

    int mapind = 0;
    if (maxalignsize == 0) {
	mapind = 0;
    } else if (maxalignsize == 2) {
	mapind = 1;
    } else if (maxalignsize == 4) {
	mapind = 2;
    }
    if (hdrver >= 27.0) {
	curmapptr = &signamap27list[0];
    } else if (hdrver >= 24.0) {
	curmapptr = &signamap24list[mapind];
    } else if (hdrver >= 20.0) {
	curmapptr = &signamap20list[mapind];
    } else if (hdrver >= 11.0) {
	curmapptr = &signamap11list[mapind];
    } else {
	curmapptr = &signamap3list[mapind];
    }

    for (/*null*/; *curmapptr; curmapptr++) {
	if ((*curmapptr)->alignsize == maxalignsize)
	    break;
    }
    if (*curmapptr == NULL) {
	fprintf(stderr, FUNC ": don't have a map supporting alignsize of %u\n", maxalignsize);
	goto FAIL;
    }
    exmap = (*curmapptr)->exmap;
    semap = (*curmapptr)->semap;
    mrmap = (*curmapptr)->mrmap;

    if ((bxh = bxh_getRootElement(docp)) == NULL)
	goto FAIL;
    
    if ((imagedata = bxh_getDatarec(docp, "image", NULL)) == NULL)
	goto FAIL;
    if ((acquisitiondata = bxh_getAcquisitionData(docp)) == NULL)
	goto FAIL;

    if (opt_dimorder == NULL) {
	opt_dimorder = "x,y,z,t";
    }

    /*** IMAGEDATA ***/

    {
	short xsize, ysize, zsize, tsize;
	const char * pos;
	int tmpint;
	float tmpfloat;
	float tlr, tla, tls; /* top left */
	float trr, tra, trs; /* top right */
	float brr, bra, brs; /* bottom right */
	float fsz, lsz; /* first/last scan z-coord */
	char fsras, lsras; /* first/last scan RAS letter */
	double Xr, Xa, Xs; /* rows */
	double Yr, Ya, Ys; /* columns */
	double Xlen, Ylen;
	float Zr, Za, Zs; /* normal */
	bxhrawdatarec * rawdatarec = NULL;
	int dimnum;
	
	grabAndSwapVal(mrimagedata, mrmap->imatrix_X, 2, &xsize, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->imatrix_Y, 2, &ysize, hdrbigend);
	/*grabAndSwapVal(mrimagedata, mrmap->dim_X, 2, &xsize, hdrbigend);*/
	/*grabAndSwapVal(mrimagedata, mrmap->dim_Y, 2, &ysize, hdrbigend);*/
	zsize = numslices;
	tsize = numtimepoints;

	rawdatarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
	memset(rawdatarec, '\0', sizeof(bxhrawdatarec));

	rawdatarec->numdims = 0;
	pos = opt_dimorder;
	while (*pos) {
	    char * dimname;
	    char * commapos;
	    bxhdimension * dimp = NULL;
	    dimname = strdup(pos);
	    if ((commapos = strchr(dimname, ',')) != NULL) {
		*commapos = '\0';
		pos += (commapos - dimname) + 1;
	    } else {
		pos += strlen(dimname);
	    }
	    if (rawdatarec->dimensions == NULL)
		rawdatarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension) * 1);
	    else
		rawdatarec->dimensions = (bxhdimension *)realloc(rawdatarec->dimensions, sizeof(bxhdimension) * (rawdatarec->numdims + 1));
	    dimp = &rawdatarec->dimensions[rawdatarec->numdims];
	    memset(dimp, '\0', sizeof(bxhdimension));
	    rawdatarec->numdims++;

	    dimp->type = strdup(dimname);
	    free(dimname);
	}

	dimnum = 0;
	pos = opt_dimorder;
	while (*pos) {
	    char * dimname;
	    char * commapos;
	    bxhdimension * dimp = NULL;
	    dimname = strdup(pos);
	    if ((commapos = strchr(dimname, ',')) != NULL) {
		*commapos = '\0';
		pos += (commapos - dimname) + 1;
	    } else {
		pos += strlen(dimname);
	    }
	    dimp = &rawdatarec->dimensions[dimnum];
	    dimnum++;
	    if (strcmp(dimname, "x") == 0)
		dimx = dimp;
	    else if (strcmp(dimname, "y") == 0)
		dimy = dimp;
	    else if (strcmp(dimname, "z") == 0)
		dimz = dimp;
	    else if (strcmp(dimname, "t") == 0)
		dimt = dimp;
	    free(dimname);
	}

	if (dimx == NULL || dimy == NULL || dimz == NULL) {
	    fprintf(stderr, "Missing x, y, or z dimension!\n");
	    goto FAIL;
	}

	dimx->size = xsize;
	dimy->size = ysize;
	dimz->size = zsize;
	if (dimt)
	    dimt->size = tsize;

	grabAndSwapVal(mrimagedata, mrmap->tlhc_R, 4, &tlr, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->tlhc_A, 4, &tla, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->tlhc_S, 4, &tls, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->trhc_R, 4, &trr, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->trhc_A, 4, &tra, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->trhc_S, 4, &trs, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->brhc_R, 4, &brr, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->brhc_A, 4, &bra, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->brhc_S, 4, &brs, hdrbigend);
	grabAndSwapVal(seriesdata, semap->start_loc, 4, &fsz, hdrbigend);
	grabAndSwapVal(seriesdata, semap->end_loc, 4, &lsz, hdrbigend);
	grabAndSwapVal(seriesdata, semap->start_ras, 1, &fsras, hdrbigend);
	grabAndSwapVal(seriesdata, semap->end_ras, 1, &lsras, hdrbigend);

	rawdatarec->elemtype = strdup("int16");
	rawdatarec->msbfirstfrags = datamsbfirst;
	
	Xr = trr - tlr;
	Xa = tra - tla;
	Xs = trs - tls;
	Yr = brr - trr;
	Ya = bra - tra;
	Ys = brs - trs;
	Xlen = sqrt(Xr*Xr + Xa*Xa + Xs*Xs);
	Ylen = sqrt(Yr*Yr + Ya*Ya + Ys*Ys);
	Xr /= Xlen;
	Xa /= Xlen;
	Xs /= Xlen;
	Yr /= Ylen;
	Ya /= Ylen;
	Ys /= Ylen;
	grabAndSwapVal(mrimagedata, mrmap->norm_R, 4, &Zr, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->norm_A, 4, &Za, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->norm_S, 4, &Zs, hdrbigend);

	if (fabs(Xr) > fabs(Yr) && fabs(Xr) > fabs(Zr)) {
	    dimr = dimx;
	    if (fabs(Ya) > fabs(Xa) && fabs(Ya) > fabs(Za)) {
		dima = dimy; dims = dimz;
	    } else {
		dima = dimz; dims = dimy;
	    }
	} else if (fabs(Yr) > fabs(Xr) && fabs(Yr) > fabs(Zr)) {
	    dimr = dimy;
	    if (fabs(Xa) > fabs(Ya) && fabs(Xa) > fabs(Za)) {
		dima = dimx; dims = dimz;
	    } else {
		dima = dimz; dims = dimx;
	    }
	} else {
	    dimr = dimz;
	    if (fabs(Xa) > fabs(Ya) && fabs(Xa) > fabs(Za)) {
		dima = dimx; dims = dimy;
	    } else {
		dima = dimy; dims = dimx;
	    }
	}

	grabAndSwapVal(mrimagedata, mrmap->pixsize_X, 4, &tmpfloat, hdrbigend);
	dimx->spacing = tmpfloat;
	grabAndSwapVal(mrimagedata, mrmap->pixsize_Y, 4, &tmpfloat, hdrbigend);
	dimy->spacing = tmpfloat;
	{
	    float gap;
	    float spacing;
	    float thickness;
	    grabAndSwapVal(mrimagedata, mrmap->slthick, 4, &thickness, hdrbigend);
	    grabAndSwapVal(mrimagedata, mrmap->scanspacing, 4, &gap, hdrbigend);
	    spacing = thickness + gap;
	    dimz->spacing = spacing;
	    dimz->gap = fabs(spacing) - thickness;
	}

	if (dimt) {
	    grabAndSwapVal(mrimagedata, mrmap->tr, 4, &tmpint, hdrbigend);
	    dimt->spacing = (double)tmpint/1000.0;
	}

	dimx->units = strdup("mm");
	dimy->units = strdup("mm");
	dimz->units = strdup("mm");
	if (dimt)
	    dimt->units = strdup("ms");

	dimx->direction = (double *)malloc(sizeof(double) * 3);
	dimx->direction[0] = Xr;
	dimx->direction[1] = Xa;
	dimx->direction[2] = Xs;
	dimy->direction = (double *)malloc(sizeof(double) * 3);
	dimy->direction[0] = Yr;
	dimy->direction[1] = Ya;
	dimy->direction[2] = Ys;
	{
	    float major = 0.0;
	    if (dimz == dimr)
		major = Zr;
	    else if (dimz == dima)
		major = Za;
	    else if (dimz == dims)
		major = Zs;
	    if ((fsz > lsz && major > 0) ||
		(fsz < lsz && major < 0)) {
		Zr *= -1.0;
		Za *= -1.0;
		Zs *= -1.0;
	    }
	}
	dimz->direction = (double *)malloc(sizeof(double) * 3);
	dimz->direction[0] = Zr;
	dimz->direction[1] = Za;
	dimz->direction[2] = Zs;

	dimr->origin = tlr;
	dima->origin = tla;
	dims->origin = tls;
	/* fix origin to point to center of voxels
	 * (x and y dimensions use edge of bounding box in GE) */
	dimr->origin +=
		(dimx->spacing / 2.0) * dimx->direction[0] +
		(dimy->spacing / 2.0) * dimy->direction[0];
	dima->origin +=
		(dimx->spacing / 2.0) * dimx->direction[1] +
		(dimy->spacing / 2.0) * dimy->direction[1];
	dims->origin +=
		(dimx->spacing / 2.0) * dimx->direction[2] +
		(dimy->spacing / 2.0) * dimy->direction[2];
	
	if (bxh_datarec_writeToElement(imagedata, rawdatarec) != 0)
	    goto FAIL;
	bxh_datarec_free(rawdatarec);
    }

    /*** ACQUISITIONDATA ***/
    {
	short tmpshort;
	unsigned short tmpushort;
	int tmpint;
	float tmpfloat;
	float tmpfloat2;
	struct tm *tmp;
	time_t tt;

	grabAndSwapVal(examdata, exmap->ex_no, 2, &tmpushort, hdrbigend);
	sprintf(&numbuf[0], "%hu", tmpushort);
	if (bxh_appendChildElement(acquisitiondata, "examnumber", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(seriesdata, semap->se_no, 2, &tmpshort, hdrbigend);
	sprintf(&numbuf[0], "%hd", tmpshort);
	if (bxh_appendChildElement(acquisitiondata, "seriesnumber", &numbuf[0]) != 0)
	    goto FAIL;

	{
	    short instnum;
	    int numimages;
	    grabAndSwapVal(mrimagedata, mrmap->im_no, 2, &instnum, hdrbigend);
	    grabAndSwapVal(seriesdata, semap->se_numimages, 4, &numimages, hdrbigend);
	    if (numimages > 0) {
		tmpshort = instnum/numimages + 1;
		sprintf(&numbuf[0], "%hd", tmpshort);
		if (bxh_appendChildElement(acquisitiondata, "runnumber", &numbuf[0]) != 0)
		    goto FAIL;
	    }
	}

#ifdef BIAC
	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	memcpy(&tmpbuf[0], (char *)examdata + exmap->patid, 13);
	if (bxh_appendChildElement(acquisitiondata, "studyid", &tmpbuf[0]) != 0)
	    goto FAIL;
#endif

	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	memcpy(&tmpbuf[0], (char *)examdata + exmap->hospname, 33);
	if (bxh_appendChildElement(acquisitiondata, "institution", &tmpbuf[0]) != 0)
	    goto FAIL;

	{
	    char sseq[32];
	    char svar[32];
	    short pseq;
/* 	    short pseqmode; */

	    grabAndSwapVal(mrimagedata, mrmap->pseq, 2, &pseq, hdrbigend);
/* 	    grabAndSwapVal(mrimagedata, mrmap->pseqmode, 2, &pseqmode, hdrbigend); */
	    sseq[0] = '\0';
	    svar[0] = '\0';
	    
	    switch (pseq) {
	    case 0:			/* SE */
		strcat(sseq, "/SE");
		break;
	    case 1:			/* IR */
		strcat(sseq, "/IR");
		break;
	    case 2:			/* RM:RM */
		strcat(sseq, "/RM");
		break;
	    case 3:			/* RMGE:none */
		strcat(sseq, "/RM");
		break;
	    case 4:			/* GRE:GR */
		strcat(sseq, "/GR");
		break;
	    case 5:			/* MPGR */
		strcat(sseq, "/GR");
		break;
	    case 6:			/* MPIRS:IR/s */
		strcat(sseq, "/IR");
		break;
	    case 7:			/* MPIRI:IR */
		strcat(sseq, "/IR");
		break;
	    case 8:			/* VOGRE:3D/GR */
		strcat(sseq, "/GR");
		break;
	    case 9:			/* CINEGRE:Cine/GRE */
		strcat(sseq, "/GR");
		break;
	    case 10:		/* SPGR */
		strcat(sseq, "/GR");
		strcat(svar, "/SP");
		break;
	    case 11:		/* SSFP */
		strcat(sseq, "/GR");
		strcat(svar, "/TRSS");
		break;
	    case 12:		/* TF:TOF */
		strcat(sseq, "/GR");
		break;
	    case 13:		/* PC */
		strcat(sseq, "/GR");
		break;
	    case 14:		/* CINSPGR:Cine/SPGR */
		strcat(sseq, "/GR");
		strcat(svar, "/SP");
		break;
	    case 15:		/* TOFGR:TOG/GR */
		strcat(sseq, "/GR");
		break;
	    case 16:		/* TOFSPGR:TOF/SPGR */
		strcat(sseq, "/GR");
		strcat(svar, "/SP");
		break;
	    case 17:		/* PCGR:PC/GR */
		strcat(sseq, "/GR");
		break;
	    case 18:		/* PCSPGR:PC/SPGR */
		strcat(sseq, "/GR");
		strcat(svar, "/SP");
		break;
	    case 19:		/* FSE */
		strcat(sseq, "/SE");
		strcat(svar, "/SK");
		break;
	    case 20:		/* FGR */
		strcat(sseq, "/GR");
		strcat(svar, "/SK");
		break;
	    case 21:		/* FMPGR */
		strcat(sseq, "/GR");
		strcat(svar, "/SK");
		break;
	    case 22:		/* FSPGR */
		strcat(sseq, "/GR");
		strcat(svar, "/SP");
		strcat(svar, "/SK");
		break;
	    case 23:		/* FMPSPGR */
		strcat(sseq, "/GR");
		strcat(svar, "/SP");
		strcat(svar, "/SK");
		break;
	    case 24:		/* SPECT */
		break;
	    case 25:		/* PSEQ_MIXED:MIXED */
		break;
	    default:	break;
	    }

	    if (sseq[0] == '/')
		memmove(&sseq[0], &sseq[1], strlen(&sseq[1])+1);
	    if (svar[0] == '/')
		memmove(&svar[0], &svar[1], strlen(&svar[1])+1);
	    if (svar[0] == '\0')
		strcpy(&svar[0], "NONE");
	    if (bxh_appendChildElement(acquisitiondata, "scanningsequence", &sseq[0]) != 0)
		goto FAIL;
	    if (bxh_appendChildElement(acquisitiondata, "sequencevariant", &svar[0]) != 0)
		goto FAIL;
	}

	grabAndSwapVal(examdata, exmap->magstrength, 4, &tmpint, hdrbigend);
#ifdef BIAC
	/* Correct for MR5's incorrect header */
	if (strncmp("mr5c", (char *)examdata + exmap->ex_suid, 4) == 0) {
	    tmpint = 40000;
	}
#endif
	sprintf(&numbuf[0], "%.15g", (double)tmpint/10000.0);
	if (bxh_appendChildElement(acquisitiondata, "magneticfield", &numbuf[0]) != 0)
	    goto FAIL;

	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	memcpy(&tmpbuf[0], (char *)seriesdata + semap->se_desc, 30);
	if (bxh_appendChildElement(acquisitiondata, "description", &tmpbuf[0]) != 0)
	    goto FAIL;

	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	memcpy(&tmpbuf[0], (char *)examdata + exmap->ex_desc, 30);
	if (bxh_appendChildElement(acquisitiondata, "examdescription", &tmpbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->im_actual_dt, 4, &tmpint, hdrbigend);
	if (tmpint == 0) {
	    /* likely to be wrong, so use series time */
	    grabAndSwapVal(seriesdata, semap->se_datetime, 4, &tmpint, hdrbigend);
	}
	tt = tmpint;
	tmp = gmtime(&tt);
	sprintf(&tmpbuf[0], "%d-%02d-%02d", tmp->tm_year + 1900, tmp->tm_mon+1, tmp->tm_mday);
	if (bxh_appendChildElement(acquisitiondata, "scandate", &tmpbuf[0]) != 0)
	    goto FAIL;
	sprintf(&tmpbuf[0], "%02d:%02d:%02d", tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
	if (bxh_appendChildElement(acquisitiondata, "scantime", &tmpbuf[0]) != 0)
	    goto FAIL;

	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	memcpy(&tmpbuf[0], (char *)examdata + exmap->ex_suid, 4);
	if (bxh_appendChildElement(acquisitiondata, "scanner", &tmpbuf[0]) != 0)
	    goto FAIL;

	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	grabAndSwapVal(mrimagedata, mrmap->dfov, 4, &tmpfloat, hdrbigend);
	sprintf(&numbuf[0], "%.7g", (double)tmpfloat);
	strcat(&tmpbuf[0], &numbuf[0]);
	strcat(&tmpbuf[0], " ");
	grabAndSwapVal(mrimagedata, mrmap->dfov_rect, 4, &tmpfloat, hdrbigend);
	if (tmpfloat == 0)
	    grabAndSwapVal(mrimagedata, mrmap->dfov, 4, &tmpfloat, hdrbigend);
	sprintf(&numbuf[0], "%.7g", (double)tmpfloat);
	strcat(&tmpbuf[0], &numbuf[0]);
	if (bxh_appendChildElement(acquisitiondata, "fieldofview", &tmpbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->tr, 4, &tmpint, hdrbigend);
	sprintf(&numbuf[0], "%.15g", (double)tmpint/1000.0);
	if (bxh_appendChildElement(acquisitiondata, "tr", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->ti, 4, &tmpint, hdrbigend);
	sprintf(&numbuf[0], "%.15g", (double)tmpint/1000.0);
	if (bxh_appendChildElement(acquisitiondata, "ti", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->te, 4, &tmpint, hdrbigend);
	sprintf(&numbuf[0], "%.15g", (double)tmpint/1000.0);
	if (bxh_appendChildElement(acquisitiondata, "te", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->mr_flip, 2, &tmpshort, hdrbigend);
	sprintf(&numbuf[0], "%hd", tmpshort);
	if (bxh_appendChildElement(acquisitiondata, "flipangle", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->freq_dir, 2, &tmpshort, hdrbigend);
	sprintf(&numbuf[0], "%hd", tmpshort);
	if (bxh_appendChildElement(acquisitiondata, "frequencydirection", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->numecho, 2, &tmpshort, hdrbigend);
	sprintf(&numbuf[0], "%hd", tmpshort);
	if (bxh_appendChildElement(acquisitiondata, "nechos", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->interleaves, 2, &tmpshort, hdrbigend);
	sprintf(&numbuf[0], "%hd", tmpshort);
	if (bxh_appendChildElement(acquisitiondata, "nshots", &numbuf[0]) != 0)
	    goto FAIL;

	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	memcpy(&tmpbuf[0], (char *)examdata + exmap->op, 4);
	if (bxh_appendChildElement(acquisitiondata, "operator", &tmpbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->vbw, 4, &tmpfloat, hdrbigend);
	tmpfloat *= 2.0; /* GE uses half-bandwidth, in kHz (not Hz) */
	sprintf(&numbuf[0], "%.7g", (double)tmpfloat);
	if (bxh_appendChildElement(acquisitiondata, "bandwidth", &numbuf[0]) != 0)
	    goto FAIL;

	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	memcpy(&tmpbuf[0], (char *)mrimagedata + mrmap->cname, 17);
	if (bxh_appendChildElement(acquisitiondata, "receivecoilname", &tmpbuf[0]) != 0)
	    goto FAIL;

	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	memcpy(&tmpbuf[0], (char *)seriesdata + semap->prtcl, 25);
	if (bxh_appendChildElement(acquisitiondata, "protocolname", &tmpbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->imatrix_X, 2, &tmpushort, hdrbigend);
	sprintf(&numbuf[0], "%hu", tmpushort);
	if (bxh_appendChildElement(acquisitiondata, "columns", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->imatrix_Y, 2, &tmpushort, hdrbigend);
	sprintf(&numbuf[0], "%hu", tmpushort);
	if (bxh_appendChildElement(acquisitiondata, "rows", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(mrimagedata, mrmap->dfov, 4, &tmpfloat, hdrbigend);
	grabAndSwapVal(mrimagedata, mrmap->dfov_rect, 4, &tmpfloat2, hdrbigend);
	sprintf(&tmpbuf[0], "%.7g %.7g", (double)tmpfloat, (double)tmpfloat2);
	if (bxh_appendChildElement(acquisitiondata, "fieldofview", &tmpbuf[0]) != 0)
	    goto FAIL;
    }


    /*** SUBJECT ***/

    if ((subject = bxh_getSubject(docp)) == NULL)
	goto FAIL;
    {
	short tmpshort;
	float tmpfloat;
	int tmpint;
	double age;
	char * sex;

	memset(&tmpbuf[0], '\0', sizeof(tmpbuf));
	memcpy(&tmpbuf[0], (char *)examdata + exmap->patname, 25);
	if (bxh_appendChildElement(subject, "name", &tmpbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(examdata, exmap->patage, 2, &tmpshort, hdrbigend);
	age = (double)tmpshort;
	grabAndSwapVal(examdata, exmap->patian, 2, &tmpshort, hdrbigend);
	switch (tmpshort) {
	case 1: /* months */
	    age /= 12.0;
	case 2: /* days */
	    age /= 365.0;
	case 3: /* weeks */
	    age /= 52.0;
	case 0: /* years, do nothing */
	default:
	    break;
	}
	sprintf(&numbuf[0], "%.15g", age);
	if (bxh_appendChildElement(subject, "age", &numbuf[0]) != 0)
	    goto FAIL;

	grabAndSwapVal(examdata, exmap->patsex, 2, &tmpshort, hdrbigend);
	switch (tmpshort) {
	case 1:
	    sex = "M";
	    break;
	case 2:
	    sex = "F";
	    break;
	case 0:
	default:
	    sex = "O";
	    break;
	}
	if (bxh_appendChildElement(subject, "sex", sex) != 0)
	    goto FAIL;

	grabAndSwapVal(examdata, exmap->patweight, 4, &tmpint, hdrbigend);
	tmpfloat = (float)tmpint / (float)1000.0;
	sprintf(&numbuf[0], "%.7g", (double)tmpfloat);
	if (bxh_appendChildElement(subject, "weight", &numbuf[0]) != 0)
	    goto FAIL;
    }

    goto EXIT;

FAIL:
    result = -1;

EXIT:
    if (bxh)
	bxh_element_unref(bxh);
    if (imagedata)
	bxh_element_unref(imagedata);
    if (acquisitiondata)
	bxh_element_unref(acquisitiondata);
    if (subject)
	bxh_element_unref(subject);

    return result;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.48  2008/05/21 17:54:59  gadde
 * Don't assume time_t is an integer.
 *
 * Revision 1.47  2005/12/02 17:53:04  gadde
 * Go back to using im_actual_dt for scantime, unless it is zero, then
 * use se_datetime.
 *
 * Revision 1.46  2005/09/14 14:49:27  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.45  2005/08/02 13:16:31  gadde
 * Use series date/time stamp instead of mrimage date/time stamp
 * (is 0 in some GE files).
 *
 * Revision 1.44  2005/07/07 15:05:30  gadde
 * Updates for ver11 pfiles.
 *
 * Revision 1.43  2005/05/10 19:47:36  gadde
 * AUTOGEN orientation comment now created in bxh_datarec_writeToElement.
 *
 * Revision 1.42  2005/02/02 17:49:08  gadde
 * Don't set section offsets until we are sure.
 *
 * Revision 1.41  2005/02/02 17:34:13  gadde
 * Make section size calculation a little more clear.
 *
 * Revision 1.40  2005/02/01 21:45:32  gadde
 * Add support for little-endian (i.e. vers>9.0) image headers.
 *
 * Revision 1.39  2005/01/11 21:29:55  gadde
 * Add forceversion option for Pfiles.
 *
 * Revision 1.38  2004/07/07 17:01:19  gadde
 * Memory fixes.
 *
 * Revision 1.37  2004/06/09 20:18:43  gadde
 * Don't bomb if se_numimages is 0.
 *
 * Revision 1.36  2004/02/23 21:32:04  gadde
 * Add examdescription field.
 *
 * Revision 1.35  2004/01/16 22:18:13  gadde
 * Use series description instead of exam description.
 *
 * Revision 1.34  2004/01/08 21:39:03  gadde
 * pointer arithmetic fix
 *
 * Revision 1.33  2004/01/08 16:33:27  gadde
 * Add iowa GE support and other minor fixes
 *
 * Revision 1.32  2004/01/02 18:41:20  gadde
 * Allow more than one timepoint for Signa files
 *
 * Revision 1.31  2003/12/15 20:21:57  gadde
 * add comment
 *
 * Revision 1.30  2003/12/15 20:13:43  gadde
 * Hack to get correct field strength on BIAC 4T (incorrect in headers)
 *
 * Revision 1.29  2003/12/15 20:09:01  gadde
 * Add operator to signa encapsulation
 *
 * Revision 1.28  2003/11/18 15:14:29  gadde
 * Add receive coil name to header.
 *
 * Revision 1.27  2003/10/22 17:09:37  gadde
 * pseq field is 2 bytes, not 4.
 *
 * Revision 1.26  2003/08/25 19:07:43  gadde
 * Use gmtime to parse dates from Signa formats.
 *
 * Revision 1.25  2003/08/22 17:20:46  gadde
 * Obliterate C++ comments
 *
 * Revision 1.24  2003/08/22 17:17:54  gadde
 * Fix runnumber.
 *
 * Revision 1.23  2003/08/22 16:06:37  gadde
 * Add ScanningSequence and ScanningVariant.
 *
 * Revision 1.22  2003/08/05 15:53:14  gadde
 * #ifdef out a BIAC-specific field.
 *
 * Revision 1.21  2003/08/01 17:51:29  gadde
 * On the road to merging in xerces branch
 *
 * Revision 1.20  2003/07/31 21:13:00  gadde
 * Check for newer version of Pfile header.
 *
 * Revision 1.19  2003/07/28 20:56:54  gadde
 * Slight integration of libxml2/Xerces branches
 *
 * Revision 1.18  2003/07/21 16:46:49  gadde
 * Code-prettiness updates, for the most part, to further protect BXH
 * library users from particulars of DOM implementation (esp. C vs. C++).
 *
 * Revision 1.17  2003/06/18 16:06:07  gadde
 * fopen in binary mode
 *
 * Revision 1.16  2003/05/19 17:04:51  gadde
 * Update docs
 *
 * Revision 1.15  2003/05/12 18:43:37  gadde
 * Additional precision in float/double output.
 *
 * Revision 1.14  2003/05/12 00:46:46  gadde
 * Add log.
 *
 */
