/*
 *  DcmFileBufferedStream and related classes, modified by Syam Gadde
 *  (gadde@biac.duke.edu) from dcistrmf.cc in DCMTK 3.5.4.
 *  
 */
/*
 *
 *  Copyright (C) 2002-2005, OFFIS
 *
 *  This software and supporting documentation were developed by
 *
 *    Kuratorium OFFIS e.V.
 *    Healthcare Information and Communication Systems
 *    Escherweg 2
 *    D-26121 Oldenburg, Germany
 *
 *  THIS SOFTWARE IS MADE AVAILABLE,  AS IS,  AND OFFIS MAKES NO  WARRANTY
 *  REGARDING  THE  SOFTWARE,  ITS  PERFORMANCE,  ITS  MERCHANTABILITY  OR
 *  FITNESS FOR ANY PARTICULAR USE, FREEDOM FROM ANY COMPUTER DISEASES  OR
 *  ITS CONFORMITY TO ANY SPECIFICATION. THE ENTIRE RISK AS TO QUALITY AND
 *  PERFORMANCE OF THE SOFTWARE IS WITH THE USER.
 *
 *  Module:  dcmdata
 *
 *  Author:  Marco Eichelberg
 *
 *  Purpose: DcmInputFileStream and related classes,
 *    implements streamed input from files.
 *
 *  Last Update:      $Author: meichel $
 *  Update Date:      $Date: 2005/12/08 15:41:14 $
 *  CVS/RCS Revision: $Revision: 1.5 $
 *  Status:           $State: Exp $
 *
 *  CVS/RCS Log at end of file
 *
 */

#include "osconfig.h"
#include "dcistrmfb.h"
#include "dcerror.h"

#define INCLUDE_CSTDIO
#define INCLUDE_CERRNO
#include "ofstdinc.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef WIN32
#include <io.h>
#endif
#ifndef WIN32
#include <unistd.h>
#endif

#define DEBUG 0

/*
 *  We override all fields and methods because some necessary fields
 *  are private in DcmFileProducer
 */

DcmFileBufferedProducer::DcmFileBufferedProducer(const char *filename, offile_off_t offset, offile_off_t bufsize, void * buf)
: DcmProducer()
, fd_(-1)
, status_(EC_Normal)
, size_((off_t)-1)
, buf_((char *)buf)
, bufsize_(bufsize)
, buflen_(0)
, buffilepos_(0)
, curfilepos_(0)
, freebuf_(0)
{
#if DEBUG
  fprintf(stderr, "DcmFileBufferedProducer::DcmFileBufferedProducer(%s, %u, %u, 0x%lx)\n", filename, (unsigned int)offset, (unsigned int)bufsize, (unsigned long)buf);
#endif
  if (buf_ == NULL) {
    if (bufsize_ == 0) {
      bufsize_ = 32768;
    }
    buf_ = new char [bufsize_];
    freebuf_ = 1;
  } else {
    if (bufsize_ == 0) {
      status_ = makeOFCondition(OFM_dcmdata, 18, OF_error, "buffer provided but specified bufsize is zero!");
      return;
    }
  }
#ifdef WIN32
  if ((fd_ = open(filename, O_RDONLY | O_BINARY)) == -1) {
#else
  if ((fd_ = open(filename, O_RDONLY)) == -1) {
#endif
    const char *text = strerror(errno);
    if (text == NULL) text = "(unknown error code)";
    status_ = makeOFCondition(OFM_dcmdata, 18, OF_error, text);
    return;
  }
  if ((size_ = lseek(fd_, (off_t)offset, SEEK_END)) == (off_t)-1) {
    const char *text = strerror(errno);
    if (text == NULL) text = "(unknown error code)";
    status_ = makeOFCondition(OFM_dcmdata, 18, OF_error, text);
    return;
  }
  if ((buffilepos_ = lseek(fd_, (off_t)offset, SEEK_SET)) == (off_t)-1) {
    const char *text = strerror(errno);
    if (text == NULL) text = "(unknown error code)";
    status_ = makeOFCondition(OFM_dcmdata, 18, OF_error, text);
    return;
  }
  curfilepos_ = offset;
}

DcmFileBufferedProducer::~DcmFileBufferedProducer()
{
  if (fd_) close(fd_);
  if (freebuf_) delete [] buf_;
}

OFBool DcmFileBufferedProducer::good() const
{
#if DEBUG
  fprintf(stderr, "DcmFileBufferedProducer::good()\n");
#endif
  return status_.good();
}

OFCondition DcmFileBufferedProducer::status() const
{
#if DEBUG
  fprintf(stderr, "DcmFileBufferedProducer::status()\n");
  fprintf(stderr, "  return %s\n", status_.text());
#endif
  return status_;
}

OFBool DcmFileBufferedProducer::eos()
{
#if DEBUG
  fprintf(stderr, "DcmFileBufferedProducer::eos()\n");
#endif
  if (fd_ == -1) {
#if DEBUG
    fprintf(stderr, "  return OFTrue\n");
#endif
    return OFTrue;
  }
  if (curfilepos_ == size_) {
#if DEBUG
    fprintf(stderr, "  return OFTrue\n");
#endif
    return OFTrue;
  } else {
#if DEBUG
    fprintf(stderr, "  return OFFalse\n");
#endif
    return OFFalse;
  }
}

offile_off_t DcmFileBufferedProducer::avail()
{
#if DEBUG
  fprintf(stderr, "DcmFileBufferedProducer::avail()\n");
#endif
  if (fd_ == -1 || size_ == (off_t)-1) {
    return 0;
  }
#if DEBUG
  fprintf(stderr, "  size_=%u curfilepos_=%u\n", (unsigned int)size_, (unsigned int)curfilepos_);
  fprintf(stderr, "  return %u\n", (unsigned int)(size_ - curfilepos_));
#endif
  return size_ - curfilepos_;
}

offile_off_t DcmFileBufferedProducer::read(void *userbuf, offile_off_t userbuflen)
{
#if DEBUG
  fprintf(stderr, "DcmFileBufferedProducer::read(PTR, %u)\n", (unsigned int)userbuflen);
#endif
  offile_off_t bytesread = 0;
  if (!status_.good() || fd_ == -1 || buf_ == NULL || bufsize_ == 0)
  {
#if DEBUG
    fprintf(stderr, "  return %u\n", (unsigned int)bytesread);
#endif
    return bytesread;
  }
  while (userbuflen > 0) {
#if DEBUG
    fprintf(stderr, " DcmFileBufferedProducer::read()  userbuflen=%u, size_=%u curfilepos_=%u buffilepos_=%u buflen_=%u bufsize_=%u\n", (unsigned int)userbuflen, (unsigned int)size_, (unsigned int)curfilepos_, (unsigned int)buffilepos_, (unsigned int)buflen_, (unsigned int)bufsize_);
#endif
    if (curfilepos_ == size_) {
      // at end of file
      break;
    }
    if (curfilepos_ < buffilepos_ || curfilepos_ >= buffilepos_ + buflen_) {
      // current file position out of buffer range, so read some more
      offile_off_t bytestoread = bufsize_;
      offile_off_t bytesreused = 0;
      if (curfilepos_ + bytestoread > size_) {
	bytestoread = size_ - curfilepos_;
      }
      if (curfilepos_ + bytestoread < buffilepos_ + buflen_ &&
	  curfilepos_ + bytestoread > buffilepos_) {
	// new range of data overlaps with current, so don't re-read unnecessarily
	bytestoread = buffilepos_ - curfilepos_;
#if DEBUG
	fprintf(stderr, "  moving %u bytes of overlapping data (bytes to read = %u)\n", (unsigned int)(buflen_ - bytestoread), (unsigned int)bytestoread);
#endif
	memmove(&buf_[bytestoread], &buf_[0], buflen_ - bytestoread);
	bytesreused = buflen_ - bytestoread;
      }
      buffilepos_ = curfilepos_;
      if (lseek(fd_, curfilepos_, SEEK_SET) == (off_t)-1) {
	const char *text = strerror(errno);
	if (text == NULL) text = "(unknown error code)";
	status_ = makeOFCondition(OFM_dcmdata, 18, OF_error, text);
	break;
      }
#if DEBUG
      fprintf(stderr, "  ::read(fd, buf, bytestoread=%u);\n", (unsigned int)(size_t)bytestoread);
      fprintf(stderr, "    (current file descriptor position is %u)\n", (unsigned int)lseek(fd_, 0, SEEK_CUR));
#endif
      buflen_ = ::read(fd_, (void *)buf_, (size_t)bytestoread);
#if DEBUG
      fprintf(stderr, "   returned %u\n", (unsigned int)buflen_);
#endif
      if (buflen_ == 0) {
	break;
      } else if (buflen_ == -1) {
	const char *text = strerror(errno);
	if (text == NULL) text = "(unknown error code)";
	status_ = makeOFCondition(OFM_dcmdata, 18, OF_error, text);
	break;
      }
      buflen_ += bytesreused;
    }
    // there is data in buffer to read
#if DEBUG
    fprintf(stderr, "  curfilepos_=%u buffilepos_=%u buflen_=%u\n", (unsigned int)curfilepos_, (unsigned int)buffilepos_, (unsigned int)buflen_);
#endif
    off_t bufpos = curfilepos_ - buffilepos_;
    offile_off_t bytestouse = buflen_ - bufpos;
    if (bytestouse > userbuflen) {
      bytestouse = userbuflen;
    }
#if DEBUG
    fprintf(stderr, "  Reading %u bytes to buffer position %u\n", (unsigned int)bytestouse, (unsigned int)bufpos);
#endif
    memcpy(userbuf, &buf_[bufpos], bytestouse);
    userbuf = (char *)userbuf + bytestouse;
    userbuflen -= bytestouse;
    curfilepos_ += bytestouse;
    bytesread += bytestouse;
  }
#if DEBUG
  fprintf(stderr, "  return %u (buffilepos=%u curfilepos_=%u buflen=%u)\n", (unsigned int)bytesread, (unsigned int)buffilepos_, (unsigned int)curfilepos_, (unsigned int)buflen_);
#endif
  return bytesread;
}

offile_off_t DcmFileBufferedProducer::skip(offile_off_t skiplen)
{
#if DEBUG
  fprintf(stderr, "DcmFileBufferedProducer::skip(%u)\n", (unsigned int)skiplen);
#endif
  offile_off_t result = 0;
  if (status_.good() && fd_ != -1 && skiplen)
  {
    if (curfilepos_ + skiplen <= buffilepos_ + buflen_) {
      curfilepos_ += skiplen;
      result = skiplen;
    } else {
      result = ((curfilepos_ + skiplen) > size_) ? (size_ - curfilepos_) : skiplen;
      curfilepos_ += result;
    }
  }
#if DEBUG
  fprintf(stderr, "  return %u (buffilepos=%u curfilepos=%u buflen=%u)\n", (unsigned int)result, (unsigned int)buffilepos_, (unsigned int)curfilepos_, (unsigned int)buflen_);
#endif
  return result;
}

void DcmFileBufferedProducer::putback(offile_off_t num)
{
#if DEBUG
  fprintf(stderr, "DcmFileBufferedProducer::putback(%u)\n", (unsigned int)num);
#endif
  if (status_.good() && fd_ != -1 && num)
  {
    if (num <= curfilepos_) {
      curfilepos_ -= num;
    } else {
      status_ = EC_PutbackFailed; // tried to putback before start of file
    }
  }
}

offile_off_t DcmFileBufferedProducer::tell() const
{
  return curfilepos_;
}

/* ======================================================================= */

DcmInputFileBufferedStreamFactory::DcmInputFileBufferedStreamFactory(const char *filename, offile_off_t offset, offile_off_t bufsize)
: DcmInputStreamFactory()
, filename_(filename)
, offset_(offset)
, bufsize_(bufsize)
{
  if (filename) filename_ = filename;
}

DcmInputFileBufferedStreamFactory::DcmInputFileBufferedStreamFactory(const DcmInputFileBufferedStreamFactory& arg)
: DcmInputStreamFactory(arg)
, filename_(arg.filename_)
, offset_(arg.offset_)
, bufsize_(arg.bufsize_)
{
}

DcmInputFileBufferedStreamFactory::~DcmInputFileBufferedStreamFactory()
{
}

DcmInputStream *DcmInputFileBufferedStreamFactory::create() const
{
  return new DcmInputFileBufferedStream(filename_.c_str(), offset_, bufsize_);
}

/* ======================================================================= */

DcmInputFileBufferedStream::DcmInputFileBufferedStream(const char *filename, offile_off_t offset, offile_off_t bufsize, void * buf)
: DcmInputStream(&producer_) // safe because DcmInputStream only stores pointer
, producer_(filename, offset, bufsize, buf)
, filename_()
, bufsize_(bufsize)
{
  if (filename) filename_ = filename;
}

DcmInputFileBufferedStream::~DcmInputFileBufferedStream()
{
}

DcmInputStreamFactory *DcmInputFileBufferedStream::newFactory() const
{
  DcmInputStreamFactory *result = NULL;
  if (currentProducer() == &producer_)
  {
    // no filter installed, can create factory object
    result = new DcmInputFileBufferedStreamFactory(filename_.c_str(), tell(), bufsize_);
  }
  return result;
}

offile_off_t DcmInputFileBufferedStream::skip(offile_off_t skiplen)
{
  return producer_.skip(skiplen);
}

void DcmInputFileBufferedStream::putback(offile_off_t num)
{
  producer_.putback(num);
  return;
}

offile_off_t DcmInputFileBufferedStream::tell() const
{
  return producer_.tell();
}

/*
 * CVS/RCS Log:
 * $Log: dcistrmf.cc,v $
 * Revision 1.5  2005/12/08 15:41:14  meichel
 * Changed include path schema for all DCMTK header files
 *
 * Revision 1.4  2004/02/04 16:34:09  joergr
 * Adapted type casts to new-style typecast operators defined in ofcast.h.
 *
 * Revision 1.3  2002/11/27 12:06:48  meichel
 * Adapted module dcmdata to use of new header file ofstdinc.h
 *
 * Revision 1.2  2002/09/19 08:32:29  joergr
 * Added explicit type casts to keep Sun CC 2.0.1 quiet.
 *
 * Revision 1.1  2002/08/27 16:55:49  meichel
 * Initial release of new DICOM I/O stream classes that add support for stream
 *   compression (deflated little endian explicit VR transfer syntax)
 *
 *
 */
