static const char rcsid[] = "$Id: bxh_applymask.c,v 1.6 2009-01-15 20:55:19 gadde Exp $";

/*
 * bxh_applymask.cpp --
 * 
 *  applies a 3-D mask to 4-D data
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif

#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

int
main(int argc, char *argv[])
{
    int retval = 0;
    struct stat statbuf;
    const char * inputfile = NULL;
    const char * maskfile = NULL;
    char * outputbase = NULL;
    char * outputbxh = NULL;
    char * outputfile = NULL;
    char * outputfilegz = NULL;
    int msbfirst = 1;
    FILE * fp = NULL;
    char * extpos = NULL;
    const char * opt_select[4] = {":", ":", ":", ":"};
    int opt_overwrite = 0;
    int opt_version = 0;
    
    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    struct bxhdataread bdr;
    struct bxhdataread maskbdr;

    const int numopts = 8;
    opt_data opts[8] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  fmriqa_tdiff [opts] maskxmlfile dataxmlfile outputxmlfile\n\n"
	  "Given a 3-D mask and a 4-D BXH- or XCEDE-wrapped time series, "
	  "this program writes out the 4-D data masked by the 3-D mask."
	  "The input files must be BXH or XCEDE files, and the output is "
	  "a 4-D image in the same format, written to outputxmlfile."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[3], 1, "timeselect",
	  "Comma-separated list of timepoints to use (first timepoint is 0).  "
	  "Any timepoint can be a contiguous range, specified as two "
	  "numbers separated by a colon, i.e. 'START:END'.  "
	  "An empty END implies the last timepoint.  "
	  "The default step of 1 (one) in ranges can be changed using "
	  "'START:STEP:END', which is equivalent to "
	  "'START,START+STEP,START+(2*STEP),...,END'." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[0], 1, "xselect",
	  "Just like timeselect, but for the 'x' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[1], 1, "yselect",
	  "Just like timeselect, but for the 'y' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[2], 1, "zselect",
	  "Just like timeselect, but for the 'z' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_overwrite, 1, "overwrite",
	  "Overwrite existing output files (otherwise error and exit). " }
    };

    memset(&bdr, '\0', sizeof(bdr));
    memset(&maskbdr, '\0', sizeof(maskbdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc != 4) {
	fprintf(stderr, "Usage: %s [opts] maskxmlfile inputxmlfile outputfile\n", argv[0]);
	fprintf(stderr, "Use the --help option for more help.\n");
	goto FAIL;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    maskfile = argv[1];
    inputfile = argv[2];
    outputbase = (char *)malloc(sizeof(char)*(strlen(argv[3]) + 1));
    outputbxh = (char *)malloc(sizeof(char)*(strlen(argv[3]) + 5));
    outputfile = (char *)malloc(sizeof(char)*(strlen(argv[3]) + 5));
    outputfilegz = (char *)malloc(sizeof(char)*(strlen(argv[3]) + 8));
    strcpy(outputbxh, argv[3]);
    {
	char * extpos = NULL;
	extpos = strrchr(outputbxh, '.');
	if (extpos == NULL) {
	    /* no extension on output */
	    strcpy(outputbase, outputbxh);
	    strcpy(outputfile, outputbxh);
	    strcpy(outputfilegz, outputbxh);
	    strcat(outputfile, ".nii");
	    strcat(outputfilegz, ".nii.gz");
	} else {
	    size_t baselen = (extpos - outputbxh);
	    strncpy(outputbase, outputbxh, baselen);
	    strncpy(outputfile, outputbxh, baselen);
	    strncpy(outputfilegz, outputbxh, baselen);
	    outputbase[baselen] = '\0';
	    strcpy(outputfile + baselen, ".nii");
	    strcpy(outputfilegz + baselen, ".nii.gz");
	}
    }
    if (!opt_overwrite) {
	if (stat(outputfilegz, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputfile);
	    goto FAIL;
	}
	if (stat(outputbxh, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputbxh);
	    goto FAIL;
	}
    }

    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, ordereddimnames, opt_select, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (bdr.datarec->numdims < 3) {
	fprintf(stderr, "Data must be at least 3-dimensional.\n");
	goto FAIL;
    }
    if (bdr.datarec->dimensions[3].size == 0) {
	fprintf(stderr, "Number of time points must be greater than 0!\n");
	goto FAIL;
    }
    if (bxh_dataReadFileStart(maskfile, "image", NULL, 3, ordereddimnames, opt_select, &maskbdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", maskfile);
	goto FAIL;
    }
    if (maskbdr.datarec->numdims != 3) {
	fprintf(stderr, "Mask must be 3-dimensional.\n");
	goto FAIL;
    }
    if (memcmp(maskbdr.dimsizes, bdr.dimsizes, sizeof(maskbdr.dimsizes[0]) * 3) != 0) {
	fprintf(stderr, "Mask spatial dimensions do not match data dimensions.\n");
	goto FAIL;
    }

    if (bxh_dataReadFinish(&bdr, NULL) != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (bxh_dataReadFinish(&maskbdr, "char") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", maskfile);
	goto FAIL;
    }

    /*** Do the dirty work here ***/
    {
	size_t voxelnum = 0; /* within volume */
	size_t t;
	char * dataptr = (char *)bdr.dataptr;
	char * maskdataptr = (char *)maskbdr.dataptr;
	size_t volsize = bdr.pagesizes[2];
	size_t tsize = 0;
	size_t elemsize = 0;
	char * elemtype = bdr.datarec->elemtype;
	tsize = bdr.pagesizes[bdr.datarec->numdims-1] / bdr.pagesizes[2];
	if (strcmp(elemtype, "int8") == 0 ||
	    strcmp(elemtype, "uint18") == 0) {
	    elemsize = 1;
	} else if (strcmp(elemtype, "int16") == 0 ||
		   strcmp(elemtype, "uint16") == 0) {
	    elemsize = 2;
	} else if (strcmp(elemtype, "int32") == 0 ||
		   strcmp(elemtype, "uint32") == 0 ||
		   strcmp(elemtype, "float32") == 0) {
	    elemsize = 4;
	} else if (strcmp(elemtype, "float64") == 0 ||
		   strcmp(elemtype, "double") == 0) {
	    elemsize = 8;
	}
	for (t = 0; t < tsize; t++) {
	    size_t startvind = 0;
	    char maskval = 0;
	    voxelnum = 0;
	    while (voxelnum < volsize) {
		maskval = maskdataptr[voxelnum];
		voxelnum++;
		while (voxelnum < volsize &&
		       maskval == maskdataptr[voxelnum]) {
		    voxelnum++;
		}
		if (maskval == 0) {
		    memset(dataptr + (elemsize * ((t * volsize) + startvind)), '\0', elemsize * (voxelnum - startvind));
		}
		startvind = voxelnum;
	    }
	}
    }

    /* write out results */
    if (bxh_addAutoHistoryEntry(bdr.docp, argv[0], &inputfile, 1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	goto FAIL;
    }
    writeBXHAndNIIGZ(outputbase, &bdr, bdr.dataptr, 0);

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    bxh_datareaddata_free(&bdr);
    bxh_datareaddata_free(&maskbdr);
    if (outputbase) {
	free(outputbase); outputbase = NULL;
    }
    if (outputbxh) {
	free(outputbxh); outputbxh = NULL;
    }
    if (outputfile) {
	free(outputfile); outputfile = NULL;
    }
    if (outputfilegz) {
	free(outputfilegz); outputfilegz = NULL;
    }
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.5  2007/05/11 15:50:18  gadde
 * Add dataptrsize field to reflect size of data (returned by bxh_dataReadFinish) in memory
 *
 * Revision 1.4  2007/02/26 17:07:04  gadde
 * Allow input to be 3-D
 *
 * Revision 1.3  2006/06/01 20:16:50  gadde
 * const fixes
 *
 * Revision 1.2  2006/05/05 17:15:57  gadde
 * Fix C++-ism.
 *
 * Revision 1.1  2006/04/07 14:49:26  gadde
 * Add some more QA programs
 *
 * Revision 1.7  2005/09/20 18:37:52  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.6  2005/09/19 16:31:53  gadde
 * Documentation and help message updates.
 *
 * Revision 1.5  2005/09/14 14:49:23  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.4  2004/06/18 15:21:50  gadde
 * Standardize frag creation (redux)
 *
 * Revision 1.3  2004/06/18 14:09:30  gadde
 * Standardize frag creation
 *
 * Revision 1.2  2004/05/06 20:00:15  gadde
 * Standardize output file checking
 *
 * Revision 1.1  2004/04/07 17:23:01  gadde
 * *** empty log message ***
 *
 * Revision 1.4  2004/03/31 21:05:07  gadde
 * Performance update.
 *
 * Revision 1.3  2004/03/29 21:08:04  gadde
 * Write correct byte order.
 *
 * Revision 1.2  2004/03/26 22:50:32  gadde
 * Use relative path in output.
 *
 * Revision 1.1  2004/03/25 19:23:37  gadde
 * Add ppm conversion, and various other updates
 *
 * Revision 1.1  2004/03/19 15:13:32  gadde
 * Major changes in datarec support, include a new 'prepare' stage
 * that initiates filtering, permutation.
 * bxh_datarec_readData has been changed to bxh_datarec_readRawData
 * to reflect this.
 * Added some fMRI QA measures.
 *
 *
 */
