static const char rcsid[] = "$Id: bxh_binop.c,v 1.10 2009-02-17 18:31:22 gadde Exp $";

/*
 * bxh_binop.c --
 * 
 *  Given two input datasets A and B, or one input dataset A and a scalar
 *  value B (given as a command-line parameter), the following is
 *  calculated:
 *    A op B
 *  where op is any of addition, subtraction, multiplication, division.
 *  If necessary, the smaller of A and B are replicated so that their sizes
 *  match.  Output dimensions are the same as the larger of A and B.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#ifndef WIN32
#include <unistd.h>
#endif

#include "bxh_niftilib.h"
#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

int
main(int argc, char *argv[])
{
    int retval = 0;
    struct stat statbuf;
    const char * inputfile1 = NULL;
    const char * inputfile2 = NULL;
    const char * outputarg = NULL;
    char * outputbase = NULL;
    char * outputfile = NULL;
    char * outputbxh = NULL;
    char * opt_maskfile = NULL;
    int msbfirst = 1;
    FILE * fp = NULL;
    char * extpos = NULL;
    int opt_overwrite = 0;
    int opt_version = 0;
    float opt_scalar = HUGE_VAL;
    int opt_add = 0;
    int opt_sub = 0;
    int opt_mul = 0;
    int opt_div = 0;

    size_t replicate1 = 0;
    size_t replicate2 = 0;
    
    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    const char * mask_select[] = { ":", ":", ":" };
    struct bxhdataread bdr1;
    struct bxhdataread bdr2;
    struct bxhdataread maskbdr;
    float * dataptr1 = NULL;
    float * dataptr2 = NULL;
    char * maskdataptr = NULL;

    const int numopts = 10;
    opt_data opts[10] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  bxh_binop [opts] xmlfileA [xmlfileB] outputfile\n\n"
	  "This program takes two input BXH or XCEDE images A and B, "
	  "or one image A and a scalar B (given by the --scalar option), and "
	  "performs A op B, where op is either addition (--add), "
	  "subtraction (--sub), multiplication (--mul), or division "
	  "(--div).  Output is written to outputfile, and has the same "
	  "format as the input.  Dimensions of the output are the same as "
	  "the larger-sized input."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_maskfile, 1, "mask",
	  "Use this mask (should be an XML file) before doing "
	  "calculations." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_add, 1, "add",
	  "Perform addition on the inputs." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_sub, 1, "sub",
	  "Perform subtraction on the inputs." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_mul, 1, "mul",
	  "Perform multiplication on the inputs." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_div, 1, "div",
	  "Perform division on the inputs." },
	{ OPT_FLAGS_FULL, OPT_VAL_FLOAT, &opt_scalar, 1, "scalar",
	  "Use this scalar instead of a second input image." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_overwrite, 1, "overwrite",
	  "Overwrite output files if they exist." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
    };

    memset(&bdr1, '\0', sizeof(bdr1));
    memset(&bdr2, '\0', sizeof(bdr2));
    memset(&maskbdr, '\0', sizeof(maskbdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if ((opt_scalar == HUGE_VAL && argc != 4) ||
	(opt_scalar != HUGE_VAL && argc != 3)) {
	if (opt_scalar == HUGE_VAL) {
	    fprintf(stderr, "Two input files and one output file are required if not using --scalar.\n");
	} else {
	    fprintf(stderr, "One input file and one output file is required if using --scalar.\n");
	}
	fprintf(stderr, "Usage: %s [opts] xmlfile outputfile\n", argv[0]);
	fprintf(stderr, "Use the --help option for more help.\n");
	goto FAIL;
    }
    if (opt_add + opt_sub + opt_mul + opt_div != 1) {
	fprintf(stderr, "One (and only one) of --add, --sub, --mul, or --div must be specified.\n");
	fprintf(stderr, "Usage: %s [opts] xmlfile outputfile\n", argv[0]);
	fprintf(stderr, "Use the --help option for more help.\n");
	goto FAIL;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    inputfile1 = argv[1];
    if (opt_scalar == HUGE_VAL) {
	inputfile2 = argv[2];
	outputarg = argv[3];
    } else {
	outputarg = argv[2];
    }
    outputbase = (char *)malloc(sizeof(char)*(strlen(outputarg) + 1));
    outputfile = (char *)malloc(sizeof(char)*(strlen(outputarg) + 8));
    outputbxh = (char *)malloc(sizeof(char)*(strlen(outputarg) + 5));
    strcpy(outputbase, outputarg);
    strcpy(outputbxh, outputarg);
    extpos = strrchr(outputbase, '.');
    if (extpos != NULL) {
	*extpos = '\0';
    }
    strcpy(outputfile, outputbase);
    strcat(outputfile, ".nii.gz");
    if (!opt_overwrite) {
	if (stat(outputfile, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputfile);
	    goto FAIL;
	}
	if (stat(outputbxh, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputbxh);
	    goto FAIL;
	}
    }

    if (bxh_dataReadFileStart(inputfile1, "image", NULL, 4, ordereddimnames, NULL, &bdr1) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile1);
	goto FAIL;
    }
    if (opt_scalar == HUGE_VAL) {
	if (bxh_dataReadFileStart(inputfile2, "image", NULL, 4, ordereddimnames, NULL, &bdr2) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile2);
	    goto FAIL;
	}
    } else {
	bdr2.datarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
	memset(bdr2.datarec, '\0', sizeof(bxhrawdatarec));
	bdr2.datarec->numdims = 1;
	bdr2.dimsizes = (size_t *)malloc(sizeof(size_t) * 1);
	bdr2.pagesizes = (size_t *)malloc(sizeof(size_t) * 1);
	bdr2.dimsizes[0] = 1;
	bdr2.pagesizes[0] = 1;
	bdr2.dataptr = (void *)malloc(sizeof(float) * 1);
	((float *)bdr2.dataptr)[0] = opt_scalar;
	bdr2.datarec->datasize = sizeof(float);
	bdr2.dataptrsize = sizeof(float);
    }
    
    {
	int dimnum;
	int numdims1 = bdr1.datarec->numdims;
	int numdims2 = bdr2.datarec->numdims;
	size_t datasize1 = bdr1.pagesizes[numdims1 - 1];
	size_t datasize2 = bdr2.pagesizes[numdims2 - 1];
	double ratio = 0;
	for (dimnum = 0;
	     dimnum < numdims1 && dimnum < numdims2;
	     dimnum++) {
	    if (bdr1.dimsizes[dimnum] != bdr2.dimsizes[dimnum]) {
		break;
	    }
	}
	if (numdims1 - dimnum > 1 &&
	    numdims2 - dimnum > 1) {
	    fprintf(stderr, "Error: The sizes of the two input datasets must differ (if at all) only in the\nlast dimension of the smaller dataset.  Input 1 has %u dimensions and\nInput 2 has %u dimensions.  They both differ in dimension %u (%u vs. %u)\n", bdr1.datarec->numdims, bdr2.datarec->numdims, dimnum + 1, bdr1.dimsizes[dimnum], bdr2.dimsizes[dimnum]);
	    goto FAIL;
	}
	if (datasize1 > datasize2) {
	    ratio = (double)datasize1 / (double)datasize2;
	    replicate2 = (int)ratio;
	} else if (datasize1 < datasize2) {
	    ratio = (double)datasize2 / (double)datasize1;
	    replicate1 = (int)ratio;
	}
	if ((double)(int)ratio != ratio) {
	    fprintf(stderr, "Error: larger dataset size (%u) is not an exact multiple of\nthe smaller data set size (%u).", (unsigned int)datasize1, (unsigned int)datasize2);
	    goto FAIL;
	}
    }

    if (bxh_dataReadFinish(&bdr1, "float") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile1);
	goto FAIL;
    }
    if (replicate1) {
	size_t repind = 0;
	size_t dataptrsize = bdr1.dataptrsize;
	char * dataptr = (char *)bdr1.dataptr;
	bdr1.dataptr = dataptr = (char *)realloc(dataptr, dataptrsize * replicate1);
	for (repind = 0; repind < replicate1; repind++) {
	    memcpy(&dataptr[dataptrsize * repind], dataptr, dataptrsize);
	}
	bxh_datarec_free(bdr1.datarec);
	bdr1.datarec = bxh_datarec_copy(bdr2.datarec);
    }
    if (opt_scalar == HUGE_VAL) {
	if (bxh_dataReadFinish(&bdr2, "float") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile2);
	    goto FAIL;
	}
    }
    if (replicate2) {
	size_t repind = 0;
	size_t dataptrsize = bdr2.dataptrsize;
	char * dataptr = (char *)bdr2.dataptr;
	bdr2.dataptr = dataptr = (char *)realloc(dataptr, dataptrsize * replicate2);
	for (repind = 1; repind < replicate2; repind++) {
	    memcpy(&dataptr[dataptrsize * repind], dataptr, dataptrsize);
	}
	bxh_datarec_free(bdr2.datarec);
	bdr2.datarec = bxh_datarec_copy(bdr1.datarec);
    }

    dataptr1 = (float *)bdr1.dataptr;
    dataptr2 = (float *)bdr2.dataptr;
	
    if (opt_maskfile) {
	if (bxh_dataReadFileStart(opt_maskfile, "image", NULL, 3, ordereddimnames, mask_select, &maskbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims != 3) {
	    fprintf(stderr, "Mask must be at least 3-dimensional.\n");
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims > bdr1.datarec->numdims) {
	    fprintf(stderr, "Number of dimensions in mask (%u) must not be greater than those in input data (%u).\n", (unsigned int)maskbdr.datarec->numdims, (unsigned int)bdr1.datarec->numdims);
	    goto FAIL;
	}
	if (memcmp(maskbdr.dimsizes, bdr1.dimsizes, sizeof(maskbdr.dimsizes[0]) * maskbdr.datarec->numdims) != 0) {
	    fprintf(stderr, "Mask dimensions do not match data dimensions.\n");
	    goto FAIL;
	}
	if (bxh_dataReadFinish(&maskbdr, "char") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
	maskdataptr = (char *)maskbdr.dataptr;
    }

    /*** Do the dirty work here ***/
    {
	/* results go back into dataptr1 */
	size_t ind = 0; /* within volume */
	off_t numelems = bdr1.pagesizes[bdr1.datarec->numdims-1];
	if (opt_add) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr1[ind] += dataptr2[ind];
	    }
	} else if (opt_sub) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr1[ind] -= dataptr2[ind];
	    }
	} else if (opt_mul) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr1[ind] *= dataptr2[ind];
	    }
	} else if (opt_div) {
	    for (ind = 0; ind < numelems; ind++) {
		dataptr1[ind] /= dataptr2[ind];
	    }
	}
	if (maskdataptr != NULL) {
	    for (ind = 0; ind < numelems; ind++) {
	        size_t maskvoxelnum = ind % maskbdr.pagesizes[maskbdr.datarec->numdims-1];
		if (maskdataptr[maskvoxelnum] == 0) {
		    dataptr1[ind] = 0;
		}
	    }
	}
    }

    /* write out results */
    free(bdr1.datarec->elemtype);
    bdr1.datarec->elemtype = strdup("float32");
    if (bxh_datarec_writeToElement(bdr1.imagedatap, bdr1.datarec) != 0) {
	fprintf(stderr, "Failed writing datarec\n");
	goto FAIL;
    }
    if (bxh_addAutoHistoryEntry(bdr1.docp, argv[0], &inputfile1, 1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	goto FAIL;
    }
    writeBXHAndNIIGZ(outputbase, &bdr1, dataptr1, 0);

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    bxh_datareaddata_free(&bdr1);
    bxh_datareaddata_free(&bdr2);
    if (opt_maskfile) {
	bxh_datareaddata_free(&maskbdr);
	free(opt_maskfile); opt_maskfile = NULL;
    }
    free(outputbxh); outputbxh = NULL;
    free(outputfile); outputfile = NULL;
    free(outputbase); outputbase = NULL;
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.9  2009/01/15 20:55:19  gadde
 * New organization of data read functions to allow for reading of non-bxh data directly by most tools
 *
 * Revision 1.8  2008/04/21 16:59:36  gadde
 * Convert more tools to write NIFTI.
 *
 * Revision 1.7  2007/05/11 15:50:18  gadde
 * Add dataptrsize field to reflect size of data (returned by bxh_dataReadFinish) in memory
 *
 * Revision 1.6  2007/03/20 17:41:57  gadde
 * Add --mask option
 *
 * Revision 1.5  2007/02/08 17:49:40  gadde
 * Do the right thing for --mul and --div.
 *
 * Revision 1.4  2007/02/03 20:02:50  gadde
 * Fix "scalar" option -- don't read from disk for second datarec if using scalar,
 * and set datasize for the fake datarec correctly.
 *
 * Revision 1.3  2006/05/05 17:15:36  gadde
 * Fix C++-ism.
 *
 * Revision 1.2  2006/05/05 14:54:22  gadde
 * fixes for non-4D data
 *
 * Revision 1.1  2006/04/13 21:28:09  gadde
 * Initial import
 *
 *
 */
