static const char rcsid[] = "$Id: bxh_ttest.c,v 1.18 2009-01-15 20:55:19 gadde Exp $";

/*
 * bxh_binttest.c --
 * 
 *  For given "avg", "std", and "n" volumes (3-D or 4-D) of two conditions,
 *  compute a t-statistic comparing the two.
 *  Modeled after portions of tstatprofile2.m by Josh Bizzell.
 *  
 */

#include <bxh_config.h>

#include <stdio.h>

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>

#include "bxh_niftilib.h"
#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

int
main(int argc, char *argv[])
{
    int retval = 0;
    struct stat statbuf;
    int argind = 0;
    int oldargc = argc;
    const char * avgfile1 = NULL;
    const char * stdfile1 = NULL;
    const char * nfile1 = NULL;
    const char * avgfile2 = NULL;
    const char * stdfile2 = NULL;
    const char * nfile2 = NULL;
    char * outputbase_t = NULL;
    char * outputbxh_t = NULL;
    char * outputfile_t = NULL;
    char * outputfilegz_t = NULL;
    const char * opt_select[] = { ":", ":", ":", ":" };
    char * opt_maskfile = NULL;
    char * opt_optsfromfile = NULL;
    int opt_overwrite = 0;
    int opt_version = 0;

    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    const char * mask_select[] = { ":", ":", ":" };
    struct bxhdataread avgbdr1;
    struct bxhdataread stdbdr1;
    struct bxhdataread nbdr1;
    struct bxhdataread avgbdr2;
    struct bxhdataread stdbdr2;
    struct bxhdataread nbdr2;
    struct bxhdataread maskbdr;
    size_t * dimsizes = NULL;
    size_t * pagesizes = NULL;
    double * avgdataptr1 = NULL;
    double * stddataptr1 = NULL;
    unsigned short * ndataptr1 = NULL;
    double * avgdataptr2 = NULL;
    double * stddataptr2 = NULL;
    unsigned short * ndataptr2 = NULL;
    char * maskdataptr = NULL;
    BXHDocPtr docp = NULL;
    bxhrawdatarec * outdatarec = NULL;

    const int numopts = 10;
    opt_data opts[10] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  bxh_ttest [opts] avg1.bxh std1.bxh n1.bxh avg2.bxh std2.bxh n2.bxh out_tfile\n\n"
	  "This program computes a per-voxel t-statistic between two datasets "
	  "given their 3-D or 4-D mean, standard deviation, and n images.  "
	  "Output (in out_tfile) is a data set, with the same dimensions as "
          "the input, storing the t-statistic." },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_optsfromfile, 1, "optsfromfile",
	  "Program options (i.e. those starting with '--') will come from "
	  "this file.  "
	  "If this option is specified, then the options in the file "
	  "will be applied after all command-line options.  "
	  "The options (and their arguments) should be specified "
	  "one per line, with the leading '--' omitted." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_overwrite, 1, "overwrite",
	  "Overwrite existing output files (otherwise error and exit). " },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[3], 1, "timeselect",
	  "Comma-separated list of timepoints to use (first timepoint is 0).  "
	  "Any timepoint can be a contiguous range, specified as two "
	  "numbers separated by a colon, i.e. 'START:END'.  "
	  "An empty END implies the last timepoint.  "
	  "The default step of 1 (one) in ranges can be changed using "
	  "'START:STEP:END', which is equivalent to "
	  "'START,START+STEP,START+(2*STEP),...,END'.  "
	  "Default is all timepoints (:)." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[0], 1, "xselect",
	  "Just like timeselect, but for the 'x' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[1], 1, "yselect",
	  "Just like timeselect, but for the 'y' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[2], 1, "zselect",
	  "Just like timeselect, but for the 'z' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_maskfile, 1, "maskfile",
	  "Use this 3-D mask (should be an XML file) before doing "
	  "calculations." }
    };

    memset(&avgbdr1, '\0', sizeof(avgbdr1));
    memset(&stdbdr1, '\0', sizeof(stdbdr1));
    memset(&nbdr1, '\0', sizeof(nbdr1));
    memset(&avgbdr2, '\0', sizeof(avgbdr2));
    memset(&stdbdr2, '\0', sizeof(stdbdr2));
    memset(&nbdr2, '\0', sizeof(nbdr2));
    memset(&maskbdr, '\0', sizeof(maskbdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);
    if (opt_optsfromfile) {
	opt_parsefile(opt_optsfromfile, numopts, &opts[0], 0);
    }

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc != 8) {
	fprintf(stderr, "Not enough/too many arguments!\n");
	fprintf(stderr, "Use the --help option for more help.\n");
	goto FAIL;
    }

    avgfile1 = argv[1];
    stdfile1 = argv[2];
    nfile1 = argv[3];
    avgfile2 = argv[4];
    stdfile2 = argv[5];
    nfile2 = argv[6];

    outputbase_t = (char *)malloc(sizeof(char)*(strlen(argv[7]) + 1));
    outputbxh_t = (char *)malloc(sizeof(char)*(strlen(argv[7]) + 5));
    outputfile_t = (char *)malloc(sizeof(char)*(strlen(argv[7]) + 5));
    outputfilegz_t = (char *)malloc(sizeof(char)*(strlen(argv[7]) + 8));
    strcpy(outputbxh_t, argv[7]);
    {
	char * extpos = NULL;
	extpos = strrchr(outputbxh_t, '.');
	if (extpos == NULL) {
	    /* no extension on output */
	    strcpy(outputbase_t, outputbxh_t);
	    strcpy(outputfile_t, outputbxh_t);
	    strcpy(outputfilegz_t, outputbxh_t);
	    strcat(outputfile_t, ".nii");
	    strcat(outputfilegz_t, ".nii.gz");
	} else {
	    size_t baselen = (extpos - outputbxh_t);
	    strncpy(outputbase_t, outputbxh_t, baselen);
	    strncpy(outputfile_t, outputbxh_t, baselen);
	    strncpy(outputfilegz_t, outputbxh_t, baselen);
	    outputbase_t[baselen] = '\0';
	    strcpy(outputfile_t + baselen, ".nii");
	    strcpy(outputfilegz_t + baselen, ".nii.gz");
	}
    }
    if (!opt_overwrite) {
	if (stat(outputfilegz_t, &statbuf) == 0) {
	    fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], outputfile_t);
	    goto FAIL;
	}
	if (stat(outputbxh_t, &statbuf) == 0) {
	    fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], outputbxh_t);
	    goto FAIL;
	}
    }

    argind = 1;
    if (bxh_dataReadFileStart(avgfile1, "image", NULL, 4, ordereddimnames, opt_select, &avgbdr1) != 0 ||
	argind++ == 0 ||
	bxh_dataReadFileStart(stdfile1, "image", NULL, 4, ordereddimnames, opt_select, &stdbdr1) != 0 ||
	argind++ == 0 ||
	bxh_dataReadFileStart(nfile1, "image", NULL, 4, ordereddimnames, opt_select, &nbdr1) != 0 ||
	argind++ == 0 ||
	bxh_dataReadFileStart(avgfile2, "image", NULL, 4, ordereddimnames, opt_select, &avgbdr2) != 0 ||
	argind++ == 0 ||
	bxh_dataReadFileStart(stdfile2, "image", NULL, 4, ordereddimnames, opt_select, &stdbdr2) != 0 ||
	argind++ == 0 ||
	bxh_dataReadFileStart(nfile2, "image", NULL, 4, ordereddimnames, opt_select, &nbdr2) != 0) {
	fprintf(stderr, "Error preparing data read for %s.\n", argv[argind]);
	goto FAIL;
    }
    argind = 1;
    if (!(avgbdr1.datarec->numdims == 3 || avgbdr1.datarec->numdims == 4) ||
	argind++ == 0 ||
	!(stdbdr1.datarec->numdims == 3 || stdbdr1.datarec->numdims == 4) ||
	argind++ == 0 ||
	!(nbdr1.datarec->numdims == 3 || nbdr1.datarec->numdims == 4) ||
	argind++ == 0 ||
	!(avgbdr2.datarec->numdims == 3 || avgbdr2.datarec->numdims == 4) ||
	argind++ == 0 ||
	!(stdbdr2.datarec->numdims == 3 || stdbdr2.datarec->numdims == 4) ||
	argind++ == 0 ||
	!(nbdr2.datarec->numdims == 3 || nbdr2.datarec->numdims == 4)) {
	fprintf(stderr, "Data in %s must be 3- or 4-dimensional.\n", argv[argind]);
	goto FAIL;
    }
    argind = 1;
    if (avgbdr1.datarec->numdims != avgbdr1.datarec->numdims ||
	argind++ == 0 ||
	stdbdr1.datarec->numdims != avgbdr1.datarec->numdims ||
	argind++ == 0 ||
	nbdr1.datarec->numdims != avgbdr1.datarec->numdims ||
	argind++ == 0 ||
	avgbdr2.datarec->numdims != avgbdr1.datarec->numdims ||
	argind++ == 0 ||
	stdbdr2.datarec->numdims != avgbdr1.datarec->numdims ||
	argind++ == 0 ||
	nbdr2.datarec->numdims != avgbdr1.datarec->numdims) {
	fprintf(stderr, "Data in %s and %s must have the same number of dimensions.\n", argv[1], argv[argind]);
	goto FAIL;
    }
    argind = 2;
    if (memcmp(avgbdr1.dimsizes, stdbdr1.dimsizes, sizeof(avgbdr1.dimsizes[0])*avgbdr1.datarec->numdims) != 0 ||
	argind++ == 0 ||
	memcmp(avgbdr1.dimsizes, nbdr1.dimsizes, sizeof(avgbdr1.dimsizes[0])*avgbdr1.datarec->numdims) != 0 ||
	argind++ == 0 ||
	memcmp(avgbdr1.dimsizes, avgbdr2.dimsizes, sizeof(avgbdr1.dimsizes[0])*avgbdr1.datarec->numdims) != 0 ||
	argind++ == 0 ||
	memcmp(avgbdr1.dimsizes, stdbdr2.dimsizes, sizeof(avgbdr1.dimsizes[0])*avgbdr1.datarec->numdims) != 0 ||
	argind++ == 0 ||
	memcmp(avgbdr1.dimsizes, nbdr2.dimsizes, sizeof(avgbdr1.dimsizes[0])*avgbdr1.datarec->numdims) != 0) {
	if (avgbdr1.datarec->numdims == 3) {
	    fprintf(stderr, "Dimensions of %s don't match those of %s (%ux%ux%u)\n", argv[argind], argv[1], (unsigned int)avgbdr1.dimsizes[0], (unsigned int)avgbdr1.dimsizes[1], (unsigned int)avgbdr1.dimsizes[2]);
	} else {
	    fprintf(stderr, "Dimensions of %s don't match those of %s (%ux%ux%ux%u)\n", argv[argind], argv[1], (unsigned int)avgbdr1.dimsizes[0], (unsigned int)avgbdr1.dimsizes[1], (unsigned int)avgbdr1.dimsizes[2], (unsigned int)avgbdr1.dimsizes[3]);
	}
	goto FAIL;
    }

    if (opt_maskfile) {
	if (bxh_dataReadFileStart(opt_maskfile, "image", NULL, 3, ordereddimnames, mask_select, &maskbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims != 3) {
	    fprintf(stderr, "Mask must be 3-dimensional.\n");
	    goto FAIL;
	}
	if (memcmp(maskbdr.dimsizes, avgbdr1.dimsizes, sizeof(maskbdr.dimsizes[0]) * 3) != 0) {
	    fprintf(stderr, "Mask spatial dimensions do not match data dimensions.\n");
	    goto FAIL;
	}
    }
    
    argind = 1;
    if (bxh_dataReadFinish(&avgbdr1, "double") != 0 ||
	argind++ == 0 ||
	bxh_dataReadFinish(&stdbdr1, "double") != 0 ||
	argind++ == 0 ||
	bxh_dataReadFinish(&nbdr1, "unsigned short") != 0 ||
	argind++ == 0 ||
	bxh_dataReadFinish(&avgbdr2, "double") != 0 ||
	argind++ == 0 ||
	bxh_dataReadFinish(&stdbdr2, "double") != 0 ||
	argind++ == 0 ||
	bxh_dataReadFinish(&nbdr2, "unsigned short") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", argv[argind]);
	goto FAIL;
    }
    if (opt_maskfile) {
	if (bxh_dataReadFinish(&maskbdr, "char") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
    }

    avgdataptr1 = (double *)avgbdr1.dataptr;
    stddataptr1 = (double *)stdbdr1.dataptr;
    ndataptr1 = (unsigned short *)nbdr1.dataptr;
    avgdataptr2 = (double *)avgbdr2.dataptr;
    stddataptr2 = (double *)stdbdr2.dataptr;
    ndataptr2 = (unsigned short *)nbdr2.dataptr;
    if (opt_maskfile)
	maskdataptr = (char *)maskbdr.dataptr;

    dimsizes = (size_t *)malloc(sizeof(size_t)*4);
    pagesizes = (size_t *)malloc(sizeof(size_t)*4);
    memcpy(dimsizes, avgbdr1.dimsizes, sizeof(size_t)*avgbdr1.datarec->numdims);
    memcpy(pagesizes, avgbdr1.pagesizes, sizeof(size_t)*avgbdr1.datarec->numdims);
    if (avgbdr1.datarec->numdims == 3) {
	dimsizes[3] = 1;
	pagesizes[3] = pagesizes[2];
    }
    docp = avgbdr1.docp;

    /*** Do the dirty work here ***/
    {
	/*** Calculate correlation to template for each voxel ***/
	double * results = (double *)malloc(sizeof(double)*pagesizes[3]);
	float * tempresults = NULL;
	size_t voxelnum = 0;

	double * a1 = avgdataptr1;
	double * s1 = stddataptr1;
	unsigned short * n1 = ndataptr1;
	double * a2 = avgdataptr2;
	double * s2 = stddataptr2;
	unsigned short * n2 = ndataptr2;
	double * r = results;
	bxhrawdatarec * tmpdatarec = NULL;

	memset(results, '\0', sizeof(double)*pagesizes[3]);
	for (voxelnum = 0; voxelnum < pagesizes[3]; voxelnum++) {
	    size_t maskvoxelnum = voxelnum % pagesizes[2];
	    if (maskdataptr == NULL || maskdataptr[maskvoxelnum] != 0) {
	      /* std is assumed to be unbiased. */
	      if (*n1 != 0 && *n2 != 0 && (*s1 != 0 || *s2 != 0)) {
		  *r = ((*a1 - *a2) /
			sqrt((((*s1)*(*s1))/((*n1)*1.0)) +
			     (((*s2)*(*s2))/((*n2)*1.0))));
	      }
	    }
	    a1++; a2++;
	    s1++; s2++;
	    n1++; n2++;
	    r++;
	}

	/* write out results */
	tempresults = bxh_convertBufToFloat(results, sizeof(double)*pagesizes[3], "double");
	outdatarec = bxh_datarec_copy(avgbdr1.datarec);
	free(outdatarec->elemtype);
	outdatarec->elemtype = strdup("float32");
	bxh_datarec_frags_free(outdatarec);
	bxh_datarec_addfrag(outdatarec, outputfile_t, 0, sizeof(float) * pagesizes[3], outputbxh_t, 1);
	if (bxh_datarec_writeToElement(avgbdr1.imagedatap, outdatarec) != 0) {
	    fprintf(stderr, "Failed writing datarec\n");
	    goto FAIL;
	}
	tmpdatarec = avgbdr1.datarec;
	avgbdr1.datarec = outdatarec;
	if (bxh_addAutoHistoryEntry(docp, argv[0], (const char **)&argv[1], oldargc-1) != 0) {
	    fprintf(stderr, "Error adding history entry\n");
	    goto FAIL;
	}
	writeBXHAndNIIGZ(outputbase_t, &avgbdr1, tempresults, 0);
	avgbdr1.datarec = tmpdatarec; tmpdatarec = NULL;

	free(tempresults); tempresults = NULL;
	bxh_datarec_free(outdatarec); outdatarec = NULL;

	free(results);
    }

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    bxh_datareaddata_free(&avgbdr1);
    bxh_datareaddata_free(&stdbdr1);
    bxh_datareaddata_free(&nbdr1);
    bxh_datareaddata_free(&avgbdr2);
    bxh_datareaddata_free(&stdbdr2);
    bxh_datareaddata_free(&nbdr2);
    if (opt_maskfile)
	bxh_datareaddata_free(&maskbdr);
    if (outputbase_t)
	free(outputbase_t);
    if (outputbxh_t)
	free(outputbxh_t);
    if (outputfile_t)
	free(outputfile_t);
    if (outputfilegz_t)
	free(outputfilegz_t);
    if (dimsizes)
	free(dimsizes);
    if (pagesizes)
	free(pagesizes);
    if (outdatarec)
	bxh_datarec_free(outdatarec);
    if (opt_maskfile) {
	free(opt_maskfile); opt_maskfile = NULL;
    }
    if (opt_optsfromfile) {
	free(opt_optsfromfile); opt_optsfromfile = NULL;
    }
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.17  2007/12/10 16:40:04  gadde
 * Write out gzipped NIFTI (with BXH/XCEDE headers) as default
 *
 * Revision 1.16  2007/05/09 16:55:48  gadde
 * If inputs would cause a t-value to be NaN, act as if that voxel is masked.
 *
 * Revision 1.15  2007/03/22 19:43:25  gadde
 * Fix help message.
 *
 * Revision 1.14  2007/02/08 22:05:02  gadde
 * Fix mask usage.
 *
 * Revision 1.13  2007/02/08 21:27:27  gadde
 * Fix mask usage.
 *
 * Revision 1.12  2006/06/01 20:16:50  gadde
 * const fixes
 *
 * Revision 1.11  2005/09/20 18:37:56  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.10  2005/09/19 21:25:37  gadde
 * Fix selector
 *
 * Revision 1.9  2005/09/19 16:31:57  gadde
 * Documentation and help message updates.
 *
 * Revision 1.8  2005/09/14 15:19:17  gadde
 * Some -Wall fixes.
 *
 * Revision 1.7  2005/09/14 14:49:31  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.6  2005/04/19 14:34:00  gadde
 * Add option arguments to history too.
 *
 * Revision 1.5  2005/02/25 16:52:53  gadde
 * Add missing element to ordereddimnames.
 *
 * Revision 1.4  2005/01/18 22:17:15  gadde
 * Fix swapped memcpy args.
 *
 * Revision 1.3  2005/01/18 20:47:56  gadde
 * Accept 4-dimensional data.
 *
 * Revision 1.2  2005/01/18 16:32:37  gadde
 * Remove cruft from earlier log.
 *
 * Revision 1.1  2005/01/18 16:32:11  gadde
 * *** empty log message ***
 *
 */
