static const char rcsid[] = "$Id: dumpheader.c,v 1.13 2007-10-11 14:23:26 gadde Exp $";

/* dumpheader.c --
 *
 * Dump some elements in a BIAC XML header to standard output.
 * This is an example of how to use the bxh utils to read the header file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "bxh_utils.h"

#include <math.h>

extern char * domutil_errorbuf;

int
convertStringToDoubleList(const char * str, double ** retp)
{
    const char * tmpcontent;
    char * endptr;
    double value;
    int numvalues;
    int retval;
    double * dlist = NULL;

    tmpcontent = str;
    endptr = NULL;
    numvalues = 0;
    while (!((value = strtod(tmpcontent, &endptr)) == 0 &&
	     (endptr == tmpcontent))) {
	numvalues++;
	tmpcontent = endptr;
    }
    if (numvalues == 0 ||
	strspn(tmpcontent, " \t\r\n\f\v") != strlen(tmpcontent)) {
	sprintf(domutil_errorbuf, "getDoubleListValue: '%s' is not a list of doubles\n", str);
	goto FAIL;
    }
    if ((dlist = (double *)malloc(sizeof(double)*numvalues)) == NULL)
	goto FAIL;
    tmpcontent = str;
    numvalues = 0;
    while (!((value = strtod(tmpcontent, &endptr)) == 0 &&
	     (endptr == tmpcontent))) {
	dlist[numvalues] = value;
	numvalues++;
	tmpcontent = endptr;
    }
    if (strspn(endptr, " \t\r\n\f\v") != strlen(endptr)) {
	sprintf(domutil_errorbuf, "domutil_getDoubleListValue: '%s' is not a valid list of numbers\n", str);
	goto FAIL;
    }
    retval = numvalues;
    *retp = dlist;
    goto EXIT;

FAIL:
    if (dlist)
	free(dlist);
    retval = -1;

EXIT:
    return retval;	
}

int
main(int argc, char *argv[])
{
    BXHDocPtr doc;
    FILE * outfp = stdout;
    int numrecs;
    int recnum;
    struct bxhdataread bdr;
    BXHElementPtr * datarecnodes = NULL;

    if (argc < 2 || strcmp(argv[1], "--help") == 0) {
	fprintf(stderr, "Usage:\n  %s inputfile\n\nThis program prints a simplistic summary of the BXH or XCEDE file given as input.\n", argv[0]);
	exit(-1);
    }
    
    /* read the input file and data */
    if (bxh_dataReadFileStart(argv[1], "image", NULL, 0, NULL, NULL, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", argv[1]);
	exit(-1);
    }

    {
	int dimnum;
	char * rasflag = NULL;
	bxhdimension * dimx = NULL;
	bxhdimension * dimy = NULL;
	bxhdimension * dimz = NULL;
	bxhdimension * dimr = NULL;
	bxhdimension * dima = NULL;
	bxhdimension * dims = NULL;
	double firstcorner[3];
	double lastcorner[3];
	for (dimnum = 0; dimnum < bdr.datarec->numdims; dimnum++) {
	    bxhdimension * dimp = &bdr.datarec->dimensions[dimnum];
	    if (dimp->direction && dimp->type) {
		if (strcmp(dimp->type, "x") == 0) {
		    dimx = dimp;
		} else if (strcmp(dimp->type, "y") == 0) {
		    dimy = dimp;
		} else if (strcmp(dimp->type, "z") == 0) {
		    dimz = dimp;
		}
	    }
	}
	if (dimx && dimy && dimz) {
	    double Xr = dimx->direction[0];
	    double Xa = dimx->direction[1];
	    /*double Xs = dimx->direction[2];*/
	    double Yr = dimy->direction[0];
	    double Ya = dimy->direction[1];
	    /*double Ys = dimy->direction[2];*/
	    double Zr = dimz->direction[0];
	    double Za = dimz->direction[1];
	    /*double Zs = dimz->direction[2];*/
	    if (fabs(Xr) > fabs(Yr) && fabs(Xr) > fabs(Zr)) {
		dimr = dimx;
		if (fabs(Ya) > fabs(Xa) && fabs(Ya) > fabs(Za)) {
		    dima = dimy; dims = dimz;
		} else {
		    dima = dimz; dims = dimy;
		}
	    } else if (fabs(Yr) > fabs(Xr) && fabs(Yr) > fabs(Zr)) {
		dimr = dimy;
		if (fabs(Xa) > fabs(Ya) && fabs(Xa) > fabs(Za)) {
		    dima = dimx; dims = dimz;
		} else {
		    dima = dimz; dims = dimx;
		}
	    } else {
		dimr = dimz;
		if (fabs(Xa) > fabs(Ya) && fabs(Xa) > fabs(Za)) {
		    dima = dimx; dims = dimy;
		} else {
		    dima = dimy; dims = dimx;
		}
	    }
	    firstcorner[0] = dimr->origin;
	    firstcorner[1] = dima->origin;
	    firstcorner[2] = dims->origin;
	    lastcorner[0] = dimr->origin + (dimx->direction[0] * dimx->spacing * (dimx->size - 1)) + (dimy->direction[0] * dimy->spacing * (dimy->size - 1)) + (dimz->direction[0] * dimz->spacing * (dimz->size - 1));
	    lastcorner[1] = dima->origin + (dimx->direction[1] * dimx->spacing * (dimx->size - 1)) + (dimy->direction[1] * dimy->spacing * (dimy->size - 1)) + (dimz->direction[1] * dimz->spacing * (dimz->size - 1));
	    lastcorner[2] = dims->origin + (dimx->direction[2] * dimx->spacing * (dimx->size - 1)) + (dimy->direction[2] * dimy->spacing * (dimy->size - 1)) + (dimz->direction[2] * dimz->spacing * (dimz->size - 1));
	}

	fprintf(outfp, "Image Data:\n");
	for (dimnum = 0; dimnum < bdr.datarec->numdims; dimnum++) {
	    bxhdimension * dimp = &bdr.datarec->dimensions[dimnum];

	    if (dimr && dima && dims &&
		(dimp == dimr || dimp == dima || dimp == dims)) {
		int rasind = -1;
		if (dimp == dimr) { rasind = 0; }
		else if (dimp == dima) { rasind = 1; }
		else if (dimp == dims) { rasind = 2; }
		fprintf(outfp, " Dimension %d (%s): %g%s to %g%s, %u steps", dimnum+1, dimp->type, firstcorner[rasind], dimp->units, lastcorner[rasind], dimp->units, dimp->size);
	    } else {
		fprintf(outfp, " Dimension %d (%s): %g%s to %g%s, %u steps", dimnum+1, dimp->type, dimp->origin, dimp->units, (dimp->origin + dimp->spacing*dimp->size), dimp->units, dimp->size);
	    }

	    if (dimp->direction) {
		int i;
		fprintf(outfp, ", direction (");
		for (i = 0; i < 3; i++) {
		    fprintf(outfp, "%s%g", (i == 0) ? "" : ", ", dimp->direction[i]);
		}
		fprintf(outfp, ")");
		if (rasflag == NULL) {
		    rasflag = (char *)malloc(sizeof(char)*(bdr.datarec->numdims+1));
		    memset(rasflag, '-', bdr.datarec->numdims);
		    rasflag[bdr.datarec->numdims] = '\0';
		}
		{
		    double absr, absa, abss;
		    absr = fabs(dimp->direction[0]);
		    absa = fabs(dimp->direction[1]);
		    abss = fabs(dimp->direction[2]);
		    if (absr > absa && absr > abss) {
			if (dimp->direction[0] < 0) {
			    rasflag[dimnum] = 'L';
			} else {
			    rasflag[dimnum] = 'R';
			}
		    } else if (absa > absr && absa > abss) {
			if (dimp->direction[1] < 0) {
			    rasflag[dimnum] = 'P';
			} else {
			    rasflag[dimnum] = 'A';
			}
		    } else {
			if (dimp->direction[2] < 0) {
			    rasflag[dimnum] = 'I';
			} else {
			    rasflag[dimnum] = 'S';
			}
		    }
		}
	    }
	    fprintf(outfp, "\n");
	}
	
	if (bdr.datarec->elemtype)
	    fprintf(outfp, " Element type: %s\n", bdr.datarec->elemtype);
	if (bdr.datarec->msbfirstfrags)
	    fprintf(outfp, " Byte order: msbfirst\n");
	else
	    fprintf(outfp, " Byte order: lsbfirst\n");
	if (rasflag)
	    fprintf(outfp, " RASflag: %s\n", rasflag);

	free(rasflag);
    }

    if (bdr.subjectp != NULL) {
	char * name = NULL;
	char * sex = NULL;
	char * agestr = NULL;
	unsigned int age;
	char * weightstr = NULL;
	double weight;

	fprintf(outfp, "Subject:\n");
	
	name = bxh_getChildElementStringValue(bdr.subjectp, "name");
	sex = bxh_getChildElementStringValue(bdr.subjectp, "sex");

	if (name) {
	    fprintf(outfp, "Subject name: %s\n", name);
	    free(name);
	}
	if (sex) {
	    fprintf(outfp, "Subject sex: %s\n", sex);
	    free(sex);
	}

	agestr = bxh_getChildElementStringValue(bdr.subjectp, "age");
	if (agestr) {
	    age = atoi(agestr);
	    fprintf(outfp, "Subject age: %u years\n", age);
	    free(agestr);
	}
	weightstr = bxh_getChildElementStringValue(bdr.subjectp, "weight");
	if (weightstr) {
	    weight = atoi(weightstr);
	    fprintf(outfp, "Subject weight: %g kg\n", weight);
	    free(weightstr);
	}
    }

    if (bdr.acqdatap != NULL) {
	fprintf(outfp, "Acquisition Data:\n");

	/* extract some header info and print out */
	{
	    BXHElementPtr * elemarray = NULL;
	    if ((elemarray = bxh_getChildElementArray(bdr.acqdatap, "*")) != NULL) {
		BXHElementPtr * curelempp = elemarray;
		while (*curelempp) {
		    BXHElementPtr curelemp = *curelempp;
		    char * name = NULL;
		    char * value = NULL;
		    name = bxh_getElementName(curelemp);
		    if (name != NULL && strcmp(name, "acqParam") == 0) {
			free(name); name = NULL;
			name = bxh_getAttributeStringValue(curelemp, "name");
		    }
		    if (name != NULL) {
			if (bxh_getElementStringValue(curelemp, &value) == 0) {
			    char * newline = value;
			    while ((newline = strchr(newline, '\n')) != NULL) {
				*newline = '|';
				newline++;
			    }
			    fprintf(stdout, "acqdata: %s = %s\n", name, value);
			}
		    }
		    free(name); name = NULL;
		    free(value); value = NULL;
		    bxh_element_unref(curelemp); curelemp = NULL;
		    curelempp++;
		}
		free(elemarray); elemarray = NULL;
	    }
	}
    }

    datarecnodes = bxh_getDatarecArray(bdr.docp);
    for (numrecs = 0; datarecnodes[numrecs]; numrecs++) { /* null */ }
    for (recnum = 0; recnum < numrecs; recnum++) {
	char * type = NULL;
	char * subtype = NULL;
	char * elementtype = NULL;
	char * filename = NULL;
	char * fileoffsetstr = NULL;
	char * recordsizestr = NULL;
	char * description = NULL;
	BXHElementPtr datarec = datarecnodes[recnum];
	BXHElementPtr * dimensions = NULL;
	int numdims;
	int dimnum;
	
	if ((type = bxh_getAttributeStringValue(datarec, "type")) == NULL) {
	    fprintf(stderr, "datarec element missing type attribute!\n");
	    exit(-1);
	}
	subtype = bxh_getChildElementStringValue(datarec, "subtype");
	elementtype = bxh_getChildElementStringValue(datarec, "elementtype");
	filename = bxh_getChildElementStringValue(datarec, "filename");
	description = bxh_getChildElementStringValue(datarec, "description");

	fprintf(outfp, "Datarec %u (type=%s, subtype=%s, size=", recnum+1, type, subtype ? subtype : "<none>");
	free(type); type = NULL;
	free(subtype); subtype = NULL;
	
	dimensions = bxh_getChildElementArray(datarec, "dimension");
	for (numdims = 0; dimensions[numdims]; numdims++) { /* null */ }
	for (dimnum = 0; dimnum < numdims; dimnum++) {
	    char * sizestr = NULL;
	    unsigned int size;
	    BXHElementPtr dimension = dimensions[dimnum];
	    if ((sizestr = bxh_getChildElementStringValue(dimension, "size")) == NULL) {
		fprintf(stderr, "Required element 'size' missing from dimension!\n");
		exit(-1);
	    }
	    size = atoi(sizestr);
	    fprintf(outfp, "%s%u", (dimnum == 0) ? "" : "x", size);
	    free(sizestr);
	}
	fprintf(outfp, "\n");
	for (dimnum = 0; dimnum < numdims; dimnum++) {
	    bxh_element_unref(dimensions[dimnum]);
	}
	free(dimensions);

	if ((dimensions = bxh_getChildElementArray(bdr.imagedatap, "dimension")) == NULL) {
	    fprintf(stderr, "Could not find dimensions in image data\n");
	    exit(-1);
	}
	for (numdims = 0; dimensions[numdims]; numdims++) { /* null */ }
	for (dimnum = 0; dimnum < numdims; dimnum++) {
	    BXHElementPtr dimension = dimensions[dimnum];
	    char * units = "";
	    char * type = NULL;
	    char * spacingstr = NULL;
	    double spacing = 1;
	    char * originstr = NULL;
	    double origin = 0;
	    char * sizestr = NULL;
	    unsigned int size;
	    int numdircoords;
	    char * dirstr = NULL;
	    double * dir = NULL;

	    units = bxh_getChildElementStringValue(dimension, "units");
	    type = bxh_getAttributeStringValue(dimension, "type");
	    spacingstr = bxh_getChildElementStringValue(dimension, "spacing");
	    if (spacingstr)
		spacing = strtod(spacingstr, NULL);
	    originstr = bxh_getChildElementStringValue(dimension, "origin");
	    if (originstr)
		origin = strtod(originstr, NULL);
	    sizestr = bxh_getChildElementStringValue(dimension, "size");
	    if (!sizestr) {
		fprintf(stderr, "Required element 'size' missing from dimension!\n");
		exit(-1);
	    }
	    if (sizestr)
		size = (unsigned int)strtod(sizestr, NULL);
	    fprintf(outfp, " Dimension %d (%s): %g%s to %g%s, %u steps\n", dimnum+1, type, origin, units, (origin + spacing*size), units, size);

	    dirstr = bxh_getChildElementStringValue(dimension, "direction");
	    numdircoords = convertStringToDoubleList(dirstr, &dir);

	    if (dir) {
		int i;
		fprintf(outfp, "  Direction: (");
		for (i = 0; i < numdircoords; i++) {
		    fprintf(outfp, "%s%g", (i == 0) ? "" : ", ", dir[i]);
		}
		fprintf(outfp, ")\n");
		free(dir);
		free(dirstr);
	    }
	
	    free(units);
	    free(type);
	    free(spacingstr);
	    free(originstr);
	    if (sizestr)
		free(sizestr);
	    bxh_element_unref(dimension);
	}
	free(dimensions);

	fprintf(outfp, "):\n");

	if (elementtype) {
	    fprintf(stdout, " Element type: %s\n", elementtype);
	    free(elementtype);
	}
	if (filename) {
	    fprintf(stdout, " Filename: %s\n", filename);
	    free(filename);
	}
	if ((fileoffsetstr = bxh_getChildElementStringValue(datarec, "fileoffset")) != NULL) {
	    unsigned int fileoffset;
	    fileoffset = atoi(fileoffsetstr);
	    fprintf(stdout, " File offset: %u\n", fileoffset);
	    free(fileoffsetstr);
	}
	if ((recordsizestr = bxh_getChildElementStringValue(datarec, "filerecordsize")) != NULL) {
	    unsigned int recordsize;
	    recordsize = atoi(recordsizestr);
	    fprintf(stdout, " Record size: %u\n", recordsize);
	    free(recordsizestr);
	}
	if (description) {
	    fprintf(stdout, " Description: \"%s\"\n", description);
	    free(description);
	}
    }

    for (recnum = 0; datarecnodes[recnum]; recnum++) {
	bxh_element_unref(datarecnodes[recnum]);
    }
    free(datarecnodes);

    bxh_datareaddata_free(&bdr);

    return 0;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.12  2005/09/19 16:31:58  gadde
 * Documentation and help message updates.
 *
 * Revision 1.11  2004/02/24 18:01:07  gadde
 * Move history generation to bxh_utils.cpp, and add option to create BIRN files
 *
 * Revision 1.10  2003/07/21 16:46:49  gadde
 * Code-prettiness updates, for the most part, to further protect BXH
 * library users from particulars of DOM implementation (esp. C vs. C++).
 *
 * Revision 1.9  2003/06/18 16:12:52  gadde
 * get rid of some warnings
 *
 * Revision 1.8  2002/12/03 20:41:04  gadde
 * Big update --
 *  add new datarec module, convert more programs from domutil_ to bxh_,
 *  add new bxh_getElement* functions, and domutil_prettify.
 *
 * Revision 1.7  2002/11/25 16:21:31  gadde
 * Mega-update to merge in new library functions.
 *
 * Revision 1.6  2002/09/10 04:27:34  gadde
 * /mriheader -> /bxh
 *
 * Revision 1.5  2002/08/20 20:11:41  gadde
 * Added const to rcsid to avoid compiler warning.
 *
 * Revision 1.4  2002/08/16 19:43:48  gadde
 * Minor changes.  This will probably be rewritten in DOM using Gdome.
 *
 * Revision 1.3  2002/07/23 15:45:16  gadde
 * Added history.
 *
 * Revision 1.2  2002/07/08 17:50:10  gadde
 * xmlStrdupFunc -> xmlMemStrdup
 *
 * Revision 1.1  2002/07/08 16:36:04  gadde
 * Initial import.
 *
 */
