#!/usr/bin/perl -w

use strict;
use File::Spec;
use File::Temp qw/ tempfile /;

die "Usage: $0 image_directory output_file\n"
  if ($#ARGV < 1);

my $imagedir = shift;
my $outfile = shift;

my @infiles = glob(File::Spec->catfile($imagedir, "I.*"));

my %fields = ();

foreach my $filenum (0..$#infiles) {
  my $numbrackets = 20;
  my $closestbracket = int($filenum / (scalar(@infiles) / $numbrackets)) * (scalar(@infiles) / $numbrackets);
  if ($filenum - 1 < $closestbracket && $filenum >= $closestbracket) {
    print STDERR "Read ", int($filenum * 100 / scalar(@infiles)), "%\r";
  }
  open(FH, "printsigna5header.pl $infiles[$filenum] |");
  while (<FH>) {
    my ($val, $name) = /.*\)\s+\[(.*)\]\s+(\S+)/;
    $fields{$name}->[$filenum] = $val;
  }
  close(FH);
}

# first sort in scan order by User Variable 39
my @uservar39 = @{$fields{'MR_user25.14'}};
my @sliceorder = sort { $uservar39[$a] <=> $uservar39[$b] } (0..$#uservar39);

# now go through the first few slices to find out where slice numbers
# (in User Variable 43) wrap around
my @uservar43 = @{$fields{'MR_user25.18'}};
my $lastslicenum = -1;
my $firstslicenum = -1;
my $skipind = 0;
for (my $sliceind = 0; $sliceind <= $#sliceorder; $sliceind++) {
  my $curslicenum = $uservar43[$sliceorder[$sliceind]];
  if ($firstslicenum == -1) {
    $firstslicenum = $curslicenum;
  }
  last if ($lastslicenum >= $curslicenum); # wrapped around
  if ($curslicenum - $lastslicenum > 1) {
    $skipind = $sliceind; # skip in slice numbers (end of scout images?)
    $firstslicenum = $curslicenum;
  }
  $lastslicenum = $curslicenum;
}

if ($firstslicenum == -1 || $lastslicenum == -1) {
  die "Error finding first and last slice numbers\n";
}

if ($skipind > 0) {
  print STDERR "Skipping slices ", $infiles[$sliceorder[0]], " to ", $infiles[$sliceorder[$skipind - 1]], "\n";
  splice(@sliceorder, 0, $skipind);
}

my $totalslices = scalar(@sliceorder);
my $numslices = $lastslicenum + 1 - $firstslicenum;
my $numtimepoints = int($totalslices / $numslices);

if ($numslices * $numtimepoints != scalar(@sliceorder)) {
  die "Total slices ($totalslices) does not equal slices ($numslices) * timepoints ($numtimepoints)\n";
}

my ($tempfh, $tempfilename) = tempfile();
print $tempfh join("\n", @infiles[@sliceorder]), "\n";
close $tempfh;
my $cmd = "signafive2bxh --dimzsize $numslices --dimtsize $numtimepoints --inputsfromfile $tempfilename $outfile";
print STDERR "Running $cmd\n";
system($cmd);
unlink $tempfilename;
