static const char rcsid[] = "$Id: fmriqa_ghostiness.c,v 1.6 2009-01-15 20:55:17 gadde Exp $";

/*
 * fmriqa_ghostiness.c --
 * 
 *  Make a "ghost" image from this image and a mask, and calculate
 *  "ghostiness".
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#ifndef WIN32
#include <unistd.h>
#endif

#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

int
main(int argc, char *argv[])
{
    int retval = 0;
    struct stat statbuf;
    const char * inputfile = NULL;
    const char * outputarg = NULL;
    char * outputfile = NULL;
    char * outputbxh = NULL;
    int msbfirst = 1;
    FILE * fp = NULL;
    char * extpos = NULL;
    int opt_overwrite = 0;
    int opt_version = 0;
    char * opt_maskfile = NULL;
    const char * opt_select[] = { ":", ":", ":", ":" };
    int opt_phaseencodedirisrow = 0;

    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    struct bxhdataread bdr;
    struct bxhdataread maskbdr;
    float * dataptr = NULL;
    char * maskdataptr = NULL;
    char * ghostmaskdataptr = NULL;

    const int numopts = 9;
    opt_data opts[9] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  fmriqa_ghostiness [opts] xmlfile outputghostmaskfile\n\n"
	  "This program takes one input BXH or XCEDE image, a mask, "
	  "and creates an EPI pseudo-\"ghost\" image from the mask and "
	  "calculates the mean of exclusively originally-masked "
	  "voxels, the mean of exclusively ghost-masked voxels, "
	  "and the mean of the top 10% of exclusively ghost-masked "
	  "voxels.  "
	  "The ghost mask is written to outputghostmaskfile."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_maskfile, 1, "maskfile",
	  "Use this mask (should be a BXH or XCEDE XML file).  "
	  "This option is required."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_overwrite, 1, "overwrite",
	  "Overwrite output files if they exist." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[3], 1, "timeselect",
	  "Comma-separated list of timepoints to use (first timepoint is 0).  "
	  "Any timepoint can be a contiguous range, specified as two "
	  "numbers separated by a colon, i.e. 'START:END'.  "
	  "An empty END implies the last timepoint.  "
	  "The default step of 1 (one) in ranges can be changed using "
	  "'START:STEP:END', which is equivalent to "
	  "'START,START+STEP,START+(2*STEP),...,END'." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[0], 1, "xselect",
	  "Just like timeselect, but for the 'x' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[1], 1, "yselect",
	  "Just like timeselect, but for the 'y' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[2], 1, "zselect",
	  "Just like timeselect, but for the 'z' dimension." }
    };

    memset(&bdr, '\0', sizeof(bdr));
    memset(&maskbdr, '\0', sizeof(maskbdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (argc < 3) {
	fprintf(stderr, "ERROR: %s: not enough arguments!\n", argv[0]);
	goto FAIL;
    }

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (opt_maskfile == NULL) {
	fprintf(stderr, "ERROR: %s: --maskfile option is required!\n", argv[0]);
	goto FAIL;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    inputfile = argv[1];
    outputarg = argv[2];
    outputfile = (char *)malloc(sizeof(char)*(strlen(outputarg) + 5));
    outputbxh = (char *)malloc(sizeof(char)*(strlen(outputarg) + 5));
    strcpy(outputbxh, outputarg);
    strcpy(outputfile, outputarg);
    extpos = strrchr(outputfile, '.');
    if (extpos == NULL) {
	strcat(outputfile, ".img");
    } else {
	strcpy(extpos, ".img");
    }
    if (!opt_overwrite) {
	if (stat(outputfile, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputfile);
	    goto FAIL;
	}
	if (stat(outputbxh, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputbxh);
	    goto FAIL;
	}
    }

    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, ordereddimnames, opt_select, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    
    if (opt_maskfile) {
	if (bxh_dataReadFileStart(opt_maskfile, "image", NULL, 3, ordereddimnames, opt_select, &maskbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims != 3) {
	    fprintf(stderr, "Mask must be 3-dimensional.\n");
	    goto FAIL;
	}
	if (memcmp(maskbdr.dimsizes, bdr.dimsizes, sizeof(maskbdr.dimsizes[0]) * maskbdr.datarec->numdims) != 0) {
	    fprintf(stderr, "Mask spatial dimensions do not match data dimensions.\n");
	    goto FAIL;
	}
    }
    if (bxh_dataReadFinish(&bdr, "float") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	goto FAIL;
    }

    if (opt_maskfile) {
	if (bxh_dataReadFinish(&maskbdr, "char") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
    }

    {
	BXHElementPtr freqdirelem;
	int freqdir;
	if ((freqdirelem = bxh_getChildElement(bdr.acqdatap, "frequencydirection")) == NULL) {
	    fprintf(stderr, "Can't find frequency direction in header!\n");
	    goto FAIL;
	}
	if (bxh_getElementIntValue(freqdirelem, &freqdir) != 0) {
	    fprintf(stderr, "Can't get value from frequency direction element!\n");
	    goto FAIL;
	}
	bxh_element_unref(freqdirelem);
	if (freqdir == 1) {
	    opt_phaseencodedirisrow = 0;
	} else {
	    opt_phaseencodedirisrow = 1;
	}
    }

    dataptr = (float *)bdr.dataptr;
    maskdataptr = (char *)maskbdr.dataptr;
    ghostmaskdataptr = (char *)malloc(maskbdr.datarec->datasize);
    memcpy(ghostmaskdataptr, maskdataptr, maskbdr.datarec->datasize);
	
    /*** Do the dirty work here ***/
    {
	size_t numcols = bdr.dimsizes[0];
	size_t numrows = bdr.dimsizes[1];
	size_t slicesize = numcols * numrows;
	size_t numslices = bdr.dimsizes[2];
	size_t numvols = bdr.dimsizes[3];
	size_t volsize = numcols * numrows * numslices;
	size_t rownum, colnum, slicenum;
	size_t volnum;
	double * ghostiness = NULL;
	float globalmin;
	float * sortvol = NULL;
	float * sortvoltmp = NULL;
	float swapval;
	ghostiness = (double *)malloc(sizeof(double)*numvols*volsize);
	sortvol = (float *)malloc(sizeof(float)*volsize);
	sortvoltmp = (float *)malloc(sizeof(float)*volsize);
	if (opt_phaseencodedirisrow) {
	    size_t colshift = numcols / 2;
	    size_t newoffset = colshift;
	    for (slicenum = 0; slicenum < numslices; slicenum++) {
		for (rownum = 0; rownum < numrows; rownum++) {
		    for (colnum = 0; colnum < numcols / 2; colnum++) {
			char * oldloc = &ghostmaskdataptr[(slicenum*slicesize) + (rownum*numcols) + colnum];
			char * newloc = oldloc + newoffset;
			swapval = *oldloc;
			*oldloc = *newloc;
			*newloc = swapval;
		    }
		}
	    }
	} else {
	    size_t rowshift = numrows / 2;
	    size_t newoffset = rowshift * numcols;
	    for (slicenum = 0; slicenum < numslices; slicenum++) {
		for (rownum = 0; rownum < numrows / 2; rownum++) {
		    for (colnum = 0; colnum < numcols; colnum++) {
			char * oldloc = &ghostmaskdataptr[(slicenum*slicesize) + (rownum*numcols) + colnum];
			char * newloc = oldloc + newoffset;
			swapval = *oldloc;
			*oldloc = *newloc;
			*newloc = swapval;
		    }
		}
	    }
	}
	globalmin = dataptr[0];
	for (volnum = 0; volnum < numvols; volnum++) {
	    size_t indxyz;
	    float maskmean = 0;
	    size_t maskn = 0;
	    float ghostmean = 0;
	    size_t ghostn = 0;
	    float noisemean = 0;
	    size_t noisen = 0;
	    double noisesum = 0;
	    double noisesqsum = 0;
	    double noisesd = 0;
	    double nonghostsnr = 0;
	    float brightghostmean = 0;
	    size_t brightghostn = 0;
	    float * volptr = &dataptr[volnum*volsize];
	    for (indxyz = 0; indxyz < volsize; indxyz++) {
		float val = volptr[indxyz];
		if (maskdataptr[indxyz] && !ghostmaskdataptr[indxyz]) {
		    maskmean += (val - maskmean) / ((maskn + 1) * 1.0);
		    maskn++;
		}
		if (!maskdataptr[indxyz] && ghostmaskdataptr[indxyz]) {
		    ghostmean += (val - ghostmean) / ((ghostn + 1) * 1.0);
		    ghostn++;
		}
		if (!maskdataptr[indxyz] && !ghostmaskdataptr[indxyz]) {
		    noisemean += (val - noisemean) / ((noisen + 1) * 1.0);
		    noisesum += val;
		    noisesqsum += (val * val);
		    noisen++;
		}
		if (val < globalmin) {
		    globalmin = val;
		}
	    }
	    if (noisen > 0) {
		noisesd = sqrt(noisesqsum - ((noisesum*noisesum)/noisen)) / noisen;
		nonghostsnr = noisemean / noisesd;
	    }
	    /* now sort volume's "ghost only" voxels to find top 10% */
	    {
		size_t numghostonly = 0;
		size_t setsize;
		size_t indxyz;
		float * volptr = &dataptr[volnum*volsize];
		for (indxyz = 0; indxyz < volsize; indxyz++) {
		    if (!maskdataptr[indxyz] && ghostmaskdataptr[indxyz]) {
			numghostonly++;
			sortvol[indxyz] = volptr[indxyz];
		    } else {
			sortvol[indxyz] = globalmin;
		    }
		}
		/* merge sort */
		setsize = 1;
		while (setsize < volsize) {
		    size_t setstart = 0;
		    indxyz = 0;
		    for (setstart = 0; setstart < volsize; setstart += (2 * setsize)) {
			size_t start1, start2, end1, end2;
			start1 = setstart;
			start2 = end1 = start1 + setsize;
			end2 = start2 + setsize;
			if (end2 > volsize) { end2 = volsize; }
			if (start2 >= volsize) { start2 = volsize; }
			if (end1 > volsize) { end1 = volsize; }
			while (start1 < end1 && start2 < end2) {
			    if (sortvol[start1] < sortvol[start2])
				sortvoltmp[indxyz++] = sortvol[start1++];
			    else
				sortvoltmp[indxyz++] = sortvol[start2++];
			}
			while (start1 < end1) {
			    sortvoltmp[indxyz++] = sortvol[start1++];
			}
			while (start2 < end2) {
			    sortvoltmp[indxyz++] = sortvol[start2++];
			}
		    }
		    {
			float * swap = sortvol;
			sortvol = sortvoltmp;
			sortvoltmp = swap;
		    }
		    setsize *= 2;
		}
		for (indxyz = volsize - (0.1 * numghostonly);
		     indxyz < volsize;
		     indxyz++) {
		    brightghostmean += (sortvol[indxyz] - brightghostmean) / (brightghostn + 1);
		    brightghostn++;
		}
	    }
	    fprintf(stdout, "%u %g %g %g %g\n", (unsigned int)volnum, (double)maskmean, (double)ghostmean, (double)brightghostmean, (double)nonghostsnr);
	}
	free(sortvol); sortvol = NULL;
	free(sortvoltmp); sortvoltmp = NULL;
    }

    /* write out results */
    if ((fp = fopen(outputfile, "wb")) == NULL) {
	fprintf(stderr, "Error opening file %s\n", outputfile);
	goto FAIL;
    }
    if (fwrite(ghostmaskdataptr, sizeof(char) * maskbdr.pagesizes[maskbdr.datarec->numdims-1], 1, fp) != 1) {
	fprintf(stderr, "Error writing to file %s\n", outputfile);
	goto FAIL;
    }
    /* create BXH file for output */
    free(maskbdr.datarec->elemtype);
    maskbdr.datarec->elemtype = strdup("int8");
    bxh_datarec_frags_free(maskbdr.datarec);
    bxh_datarec_addfrag(maskbdr.datarec, outputfile, 0, sizeof(char) * (maskbdr.pagesizes[maskbdr.datarec->numdims-1]), outputbxh, 1);
    maskbdr.datarec->numdims = 3;
    if (bxh_datarec_writeToElement(maskbdr.imagedatap, maskbdr.datarec) != 0) {
	fprintf(stderr, "Failed writing datarec\n");
	goto FAIL;
    }
    if (bxh_addAutoHistoryEntry(maskbdr.docp, argv[0], &inputfile, 1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	goto FAIL;
    }
    if (bxh_writeFile(maskbdr.docp, outputbxh) != 0) {
	fprintf(stderr, "Error writing output file %s\n", outputbxh);
	goto FAIL;
    }

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    free(ghostmaskdataptr); ghostmaskdataptr = NULL;
    bxh_datareaddata_free(&maskbdr);
    bxh_datareaddata_free(&bdr);
    free(outputbxh); outputbxh = NULL;
    free(outputfile); outputfile = NULL;
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.5  2008/10/20 17:56:04  gadde
 * Add non-ghost SNR
 *
 * Revision 1.4  2007/01/15 20:24:41  gadde
 * Fix variable declaration.
 *
 * Revision 1.3  2007/01/15 20:12:52  gadde
 * Print out maskmean, ghostmean, brightghostmean, and let others
 * calculate what they need.
 *
 * Revision 1.2  2007/01/12 16:22:29  gadde
 * Fix log file.
 *
 * Revision 1.1  2007/01/12 16:16:07  gadde
 * Add support for selecting dimensions.
 *
 */
