static const char rcsid[] = "$Id: fmriqa_oediff.cpp,v 1.10 2008-03-07 23:05:07 gadde Exp $";

/*
 * fmriqa_oediff.cpp --
 * 
 *  calculates odd-even time points difference image
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>
#include <string.h>

#include <vector>

#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

#define CONVERTTEMPLATE(inbuf, fromtype, bufsize, retbuf, totype) {	\
    fromtype * buf = NULL;						\
    fromtype * endbuf = (fromtype *)((char *)inbuf + (bufsize));	\
    size_t retsize = sizeof(totype)*((bufsize)/sizeof(*buf));		\
    totype * newbuf = NULL;						\
    newbuf = (totype *)malloc(retsize);					\
    (retbuf) = newbuf;							\
    if ((newbuf) == NULL) {						\
	fprintf(stderr, "Error allocating %lld bytes\n", (long long int)retsize);	\
    }									\
    for (buf = (fromtype *)(inbuf); buf < (endbuf); newbuf++, buf++) {	\
	*(newbuf) = (totype)*buf;					\
    }									\
}

static float *
convertBufToFloat(const void * inbuf, size_t bufsize, const char * elemtype)
{
    float * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "float32") == 0) {
	retbuf = (float *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, float);
    }
    return retbuf;
}

int
main(int argc, char *argv[])
{
    struct stat statbuf;
    const char * inputfile = NULL;
    char * outputfile = NULL;
    char * outputbxh = NULL;
    int dimnum;
    int msbfirst = 1;
    FILE * fp = NULL;
    const char * opt_tselect = ":";
    const char * opt_xselect = ":";
    const char * opt_yselect = ":";
    const char * opt_zselect = ":";
    int opt_version = 0;
    
    const int numopts = 7;
    opt_data opts[7] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  fmriqa_oediff [opts] xmlfile outputfile\n\n"
	  "Given a 4-D BXH- or XCEDE-wrapped time series, this program "
	  "calculates the cumulative difference between the even images "
	  "(where the first selected image is 0) and the odd images.  "
	  "The input file must be BXH or XCEDE file, and the output is "
	  "a 3-D image in the same format, written to outputfile."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_tselect, 1, "timeselect",
	  "Comma-separated list of timepoints to use (first timepoint is 0).  "
	  "Any timepoint can be a contiguous range, specified as two "
	  "numbers separated by a colon, i.e. 'START:END'.  "
	  "An empty END implies the last timepoint.  "
	  "The default step of 1 (one) in ranges can be changed using "
	  "'START:STEP:END', which is equivalent to "
	  "'START,START+STEP,START+(2*STEP),...,END'." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_xselect, 1, "xselect",
	  "Just like timeselect, but for the 'x' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_yselect, 1, "yselect",
	  "Just like timeselect, but for the 'y' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_zselect, 1, "zselect",
	  "Just like timeselect, but for the 'z' dimension." }
    };
    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc != 3) {
	fprintf(stderr, "Usage: %s [opts] xmlfile outputfile\n", argv[0]);
	fprintf(stderr, "Use the --help option for more help.\n");
	return -1;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    inputfile = argv[1];
    outputfile = (char *)malloc(sizeof(char)*(strlen(argv[2]) + 5));
    outputbxh = (char *)malloc(sizeof(char)*(strlen(argv[2]) + 5));
    strcpy(outputbxh, argv[2]);
    strcpy(outputfile, argv[2]);
    char * extpos = NULL;
    extpos = strrchr(outputfile, '.');
    if (extpos == NULL) {
	strcat(outputbxh, ".bxh");
    } else if (strcmp(extpos, ".bxh") == 0) {
	/* user specified BXH file as output */
	strcpy(extpos, ".img");
    } else {
	/* user specified data file as output */
	strcpy(outputbxh + (extpos - outputfile), ".bxh");
    }
    if (stat(outputfile, &statbuf) == 0) {
	fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], outputfile);
	return -1;
    }
    if (stat(outputbxh, &statbuf) == 0) {
	fprintf(stderr, "%s: output file '%s' exists.\n", argv[0], outputbxh);
	return -1;
    }

    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    int ** selectors = NULL;
    BXHDocPtr docp;
    BXHElementPtr imagedatap;
    bxhrawdatarec * datarec = NULL;
    bxhrawdatarec * newdatarec = NULL;
    std::vector<size_t> dimsizes;
    std::vector<size_t> pagesizes;
    void * buf = NULL;
    if ((docp = bxh_readFile(inputfile)) == NULL) {
	fprintf(stderr, "Reading XML file failed.\n");
	return -1;
    }
    if ((imagedatap = bxh_getDatarec(docp, "image", NULL)) == NULL) {
	fprintf(stderr, "Getting 'image' datarec failed.\n");
	return -1;
    }
    if ((datarec = bxh_datarec_createFromElement(imagedatap, argv[1])) == NULL) {
	fprintf(stderr, "Creating raw datarec failed.\n");
	return -1;
    }
    bxh_datarec_prepareForReadPermute(datarec, ordereddimnames, 4);
    newdatarec = bxh_datarec_postprocess(datarec);
    if (newdatarec->numdims != 4) {
	fprintf(stderr, "Data must be 4-dimensional.\n");
	return -1;
    }
    selectors = (int **)malloc(sizeof(int *)*4);
    selectors[0] = bxh_datarec_constructSelector(opt_xselect, 0, newdatarec->dimensions[0].size-1, NULL);
    selectors[1] = bxh_datarec_constructSelector(opt_yselect, 0, newdatarec->dimensions[1].size-1, NULL);
    selectors[2] = bxh_datarec_constructSelector(opt_zselect, 0, newdatarec->dimensions[2].size-1, NULL);
    selectors[3] = bxh_datarec_constructSelector(opt_tselect, 0, newdatarec->dimensions[3].size-1, NULL);
    if (selectors[0] == NULL || selectors[1] == NULL || selectors[2] == NULL || selectors[3] == NULL) {
	fprintf(stderr, "Error parsing one or more selectors\n");
	exit(-1);
    }
    bxh_datarec_free(newdatarec); /* we're just going to get another one */
    bxh_datarec_prepareForReadPermuteSelect(datarec, ordereddimnames, 4, selectors);
    free(selectors[0]);
    free(selectors[1]);
    free(selectors[2]);
    free(selectors[3]);
    free(selectors);
    newdatarec = bxh_datarec_postprocess(datarec);
    if (newdatarec->dimensions[3].size % 2 != 0) {
	fprintf(stderr, "Number of timepoints (%u) must be even.\n", (unsigned int)newdatarec->dimensions[3].size);
	return -1;
    }
    bxh_datarec_free(newdatarec); /* we're just going to get another one */
    newdatarec = bxh_datarec_readData(datarec, &buf, msbfirst);
    bxh_datarec_free(datarec); datarec = NULL;
    for (dimnum = 0; dimnum < newdatarec->numdims; dimnum++) {
	dimsizes.push_back(newdatarec->dimensions[dimnum].size);
	if (dimnum == 0)
	    pagesizes.push_back(dimsizes[dimnum]);
	else
	    pagesizes.push_back(pagesizes[dimnum - 1] * dimsizes[dimnum]);
    }
#define C2I(x,y,z,t) ((x) + (y)*pagesizes[0] + (z)*pagesizes[1] + (t)*pagesizes[2])
    float * dataptr =
	convertBufToFloat(buf, newdatarec->datasize, newdatarec->elemtype);
    free(buf);
	
    /*** Calculate sums for each voxel (separately for odd/even time points) ***/
    {
	float * results = (float *)malloc(sizeof(float)*pagesizes[2]);
	double * oddsums = (double *)malloc(sizeof(double)*pagesizes[2]);
	double * evensums = (double *)malloc(sizeof(double)*pagesizes[2]);
	size_t voxelnum = 0; /* within volume */
	size_t t;
	for (t = 0; t < newdatarec->dimensions[3].size; t++) {
	    for (voxelnum = 0; voxelnum < pagesizes[2]; voxelnum++) {
		size_t ind = C2I(voxelnum,0,0,t);
		if (t % 2 == 0)
		    evensums[voxelnum] += dataptr[ind];
		else
		    oddsums[voxelnum] += dataptr[ind];
	    }
	}
	for (voxelnum = 0; voxelnum < pagesizes[2]; voxelnum++) {
	    /* matlab script does odd - even, but of course they index by 1! */
	    results[voxelnum] = (float)(evensums[voxelnum] - oddsums[voxelnum]);
	}

	/* write out results */
	if ((fp = fopen(outputfile, "wb")) == NULL) {
	    fprintf(stderr, "Error opening file %s\n", outputfile);
	    return -1;
	}
	if (fwrite(results, sizeof(float)*pagesizes[2], 1, fp) != 1) {
	    fprintf(stderr, "Error writing to file %s\n", outputfile);
	    return -1;
	}
	free(oddsums);
	free(evensums);
	free(results);
    }

    /* create BXH file for output */
    newdatarec->numdims = 3;
    free(newdatarec->elemtype);
    newdatarec->elemtype = strdup("float32");
    bxh_datarec_frags_free(newdatarec);
    bxh_datarec_addfrag(newdatarec, outputfile, 0, sizeof(float) * pagesizes[2], outputbxh, 1);
    if (bxh_datarec_writeToElement(imagedatap, newdatarec) != 0) {
	fprintf(stderr, "Failed writing datarec\n");
	return -1;
    }
    if (bxh_addAutoHistoryEntry(docp, argv[0], &inputfile, 1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	return -1;
    }
    if (bxh_writeFile(docp, outputbxh) != 0) {
	fprintf(stderr, "Error writing output file %s\n", outputbxh);
	return -1;
    } 
    bxh_datarec_free(newdatarec);
    free(outputbxh); outputbxh = NULL;
    free(outputfile); outputfile = NULL;
    return 0;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.9  2006/06/01 20:16:48  gadde
 * const fixes
 *
 * Revision 1.8  2006/04/07 14:45:39  gadde
 * Make sure pathnames in .bxh file are written out correctly.
 *
 * Revision 1.7  2005/09/20 18:37:52  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.6  2005/09/19 16:31:53  gadde
 * Documentation and help message updates.
 *
 * Revision 1.5  2005/09/14 14:49:23  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.4  2004/06/18 15:21:50  gadde
 * Standardize frag creation (redux)
 *
 * Revision 1.3  2004/06/18 14:09:30  gadde
 * Standardize frag creation
 *
 * Revision 1.2  2004/05/06 20:00:15  gadde
 * Standardize output file checking
 *
 * Revision 1.1  2004/04/07 17:23:01  gadde
 * *** empty log message ***
 *
 * Revision 1.4  2004/03/31 21:05:07  gadde
 * Performance update.
 *
 * Revision 1.3  2004/03/29 21:08:04  gadde
 * Write correct byte order.
 *
 * Revision 1.2  2004/03/26 22:50:32  gadde
 * Use relative path in output.
 *
 * Revision 1.1  2004/03/25 19:23:37  gadde
 * Add ppm conversion, and various other updates
 *
 * Revision 1.1  2004/03/19 15:13:32  gadde
 * Major changes in datarec support, include a new 'prepare' stage
 * that initiates filtering, permutation.
 * bxh_datarec_readData has been changed to bxh_datarec_readRawData
 * to reflect this.
 * Added some fMRI QA measures.
 *
 *
 */
