static const char rcsid[] = "$Id: fmriqa_tdiff.c,v 1.6 2009-01-15 20:55:18 gadde Exp $";

/*
 * fmriqa_tdiff.cpp --
 * 
 *  creates a new time-series containing the differences between consecutive
 *  volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif

#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

int
main(int argc, char *argv[])
{
    int retval = 0;
    struct stat statbuf;
    const char * inputfile = NULL;
    char * outputfile = NULL;
    char * outputbxh = NULL;
    int msbfirst = 1;
    FILE * fp = NULL;
    char * extpos = NULL;
    const char * opt_select[4] = {":", ":", ":", ":"};
    int opt_overwrite = 0;
    int opt_version = 0;
    char * opt_maskfile = NULL;
    
    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    struct bxhdataread bdr;
    struct bxhdataread maskbdr;
    float * dataptr = NULL;
    char * maskdataptr = NULL;

    const int numopts = 9;
    opt_data opts[9] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  fmriqa_tdiff [opts] xmlfile outputxmlfile\n\n"
	  "Given a 4-D BXH- or XCEDE-wrapped time series, this program "
	  "writes out the differences between consecutive volumes.  "
	  "The input file must be BXH or XCEDE file, and the output is "
	  "a 4-D image in the same format, written to outputxmlfile."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_maskfile, 1, "maskfile",
	  "Use this mask (should be a BXH or XCEDE XML file)."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[3], 1, "timeselect",
	  "Comma-separated list of timepoints to use (first timepoint is 0).  "
	  "Any timepoint can be a contiguous range, specified as two "
	  "numbers separated by a colon, i.e. 'START:END'.  "
	  "An empty END implies the last timepoint.  "
	  "The default step of 1 (one) in ranges can be changed using "
	  "'START:STEP:END', which is equivalent to "
	  "'START,START+STEP,START+(2*STEP),...,END'." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[0], 1, "xselect",
	  "Just like timeselect, but for the 'x' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[1], 1, "yselect",
	  "Just like timeselect, but for the 'y' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[2], 1, "zselect",
	  "Just like timeselect, but for the 'z' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_overwrite, 1, "overwrite",
	  "Overwrite existing output files (otherwise error and exit). " }
    };

    memset(&bdr, '\0', sizeof(bdr));
    memset(&maskbdr, '\0', sizeof(maskbdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc != 3) {
	fprintf(stderr, "Usage: %s [opts] xmlfile outputfile\n", argv[0]);
	fprintf(stderr, "Use the --help option for more help.\n");
	goto FAIL;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    inputfile = argv[1];
    outputfile = (char *)malloc(sizeof(char)*(strlen(argv[2]) + 5));
    outputbxh = (char *)malloc(sizeof(char)*(strlen(argv[2]) + 5));
    strcpy(outputbxh, argv[2]);
    strcpy(outputfile, argv[2]);
    extpos = strrchr(outputfile, '.');
    if (extpos == NULL) {
	strcat(outputfile, ".img");
    } else {
	strcpy(extpos, ".img");
    }
    if (!opt_overwrite) {
	if (stat(outputfile, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputfile);
	    goto FAIL;
	}
	if (stat(outputbxh, &statbuf) == 0) {
	    fprintf(stderr, "ERROR: %s: output file '%s' exists.\n", argv[0], outputbxh);
	    goto FAIL;
	}
    }

    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, ordereddimnames, opt_select, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (bdr.datarec->numdims != 4) {
	fprintf(stderr, "Data must be 4-dimensional.\n");
	goto FAIL;
    }
    if (bdr.datarec->dimensions[3].size == 0) {
	fprintf(stderr, "Number of time points must be greater than 0!\n");
	goto FAIL;
    }

    if (opt_maskfile) {
	if (bxh_dataReadFileStart(opt_maskfile, "image", NULL, 3, ordereddimnames, opt_select, &maskbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims != 3) {
	    fprintf(stderr, "Mask must be 3-dimensional.\n");
	    goto FAIL;
	}
	if (memcmp(maskbdr.dimsizes, bdr.dimsizes, sizeof(maskbdr.dimsizes[0]) * maskbdr.datarec->numdims) != 0) {
	    fprintf(stderr, "Mask spatial dimensions do not match data dimensions.\n");
	    goto FAIL;
	}
    }

    if (bxh_dataReadFinish(&bdr, "float") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (opt_maskfile) {
	if (bxh_dataReadFinish(&maskbdr, "char") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
    }

    dataptr = (float *)bdr.dataptr;
    maskdataptr = (char *)maskbdr.dataptr;
	
    /*** Do the dirty work here ***/
    {
	/* results go back into dataptr */
	size_t voxelnum = 0; /* within volume */
	size_t t;
	off_t volsize = bdr.pagesizes[2];
	for (t = 0; t < bdr.dimsizes[3] - 1; t++) {
	    for (voxelnum = 0; voxelnum < bdr.pagesizes[2]; voxelnum++) {
		size_t ind1 = (t * volsize) + voxelnum;
		size_t ind2 = ind1 + volsize;
		if (opt_maskfile == NULL || maskdataptr[voxelnum]) {
		    dataptr[ind1] -= dataptr[ind2];
		    /* make it x(t+1) - x(t)  rather than  x(t) - x(t+1) */
		    dataptr[ind1] *= -1;
		} else {
		    dataptr[ind1] = 0;
		}
	    }
	}
	/* write out results */
	if ((fp = fopen(outputfile, "wb")) == NULL) {
	    fprintf(stderr, "Error opening file %s\n", outputfile);
	    goto FAIL;
	}
	if (fwrite(dataptr, sizeof(float) * (bdr.pagesizes[3] - bdr.pagesizes[2]), 1, fp) != 1) {
	    fprintf(stderr, "Error writing to file %s\n", outputfile);
	    goto FAIL;
	}
    }

    /* create BXH file for output */
    free(bdr.datarec->elemtype);
    bdr.datarec->elemtype = strdup("float32");
    bxh_datarec_frags_free(bdr.datarec);
    bxh_datarec_addfrag(bdr.datarec, outputfile, 0, sizeof(float) * (bdr.pagesizes[3] - bdr.pagesizes[2]), outputbxh, 1);
    bdr.datarec->dimensions[3].size--; /* one less volume */
    if (bxh_datarec_writeToElement(bdr.imagedatap, bdr.datarec) != 0) {
	fprintf(stderr, "Failed writing datarec\n");
	goto FAIL;
    }
    if (bxh_addAutoHistoryEntry(bdr.docp, argv[0], &inputfile, 1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	goto FAIL;
    }
    if (bxh_writeFile(bdr.docp, outputbxh) != 0) {
	fprintf(stderr, "Error writing output file %s\n", outputbxh);
	goto FAIL;
    }

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    bxh_datareaddata_free(&bdr);
    if (opt_maskfile)
	bxh_datareaddata_free(&maskbdr);
    free(outputbxh); outputbxh = NULL;
    free(outputfile); outputfile = NULL;
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.5  2006/06/01 20:16:49  gadde
 * const fixes
 *
 * Revision 1.4  2006/05/05 17:16:21  gadde
 * Fix C++-ism.
 *
 * Revision 1.3  2006/04/13 21:26:20  gadde
 * Use float instead of double.  Also get rid of extraneous log info.
 *
 * Revision 1.2  2006/04/12 17:58:33  gadde
 * Add mask option.
 *
 * Revision 1.1  2006/04/07 14:49:26  gadde
 * Add some more QA programs
 *
 */
