static const char rcsid[] = "$Id: fmriqa_volmeasures.cpp,v 1.16 2009-02-17 14:19:31 gadde Exp $";

/*
 * fmriqa_volmeasures.cpp --
 * 
 *  calculates some QA measures and writes the output
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>
#include <string.h>

#include <vector>

#include "bxh_datarec.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

#ifdef WIN32
#include <pstdint.h>
#include <float.h>
#ifndef NAN
static const uint32_t nan[1] = {0x7fffffff};
#define NAN (*(const float *) &nan[0])
#endif
#define MYNAN NAN
#ifndef isnan
#define isnan(x) _isnan(x)
#endif
#else /* #ifdef WIN32 */
/* this works on Linux */
#define MYNAN (0.0/0.0)
#endif

int
main(int argc, char *argv[])
{
    int retval = 0;
    const char * inputfile = NULL;
    int msbfirst = 1;
    const char * opt_select[] = { ":", ":", ":", ":" };
    int opt_version = 0;
    char * opt_maskfile = NULL;
    
    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    struct bxhdataread bdr;
    struct bxhdataread maskbdr;
    float * dataptr = NULL;
    char * maskdataptr = NULL;

    const int numopts = 8;
    opt_data opts[8] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  fmriqa_volmeasures [opts] xmlfile\n\n"
	  "This program calculates various measures per volume of the input "
	  "4-D time series, such as mean/minimum/maximum intensity, "
	  "standard deviation, and center-of-mass in "
	  "all three dimensions.  "
	  "The input file must be BXH or XCEDE file, and the output will "
	  "be written to standard output."
	},
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_maskfile, 1, "maskfile",
	  "Use this mask (should be a BXH or XCEDE XML file)."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[3], 1, "timeselect",
	  "Comma-separated list of timepoints to use (first timepoint is 0).  "
	  "Any timepoint can be a contiguous range, specified as two "
	  "numbers separated by a colon, i.e. 'START:END'.  "
	  "An empty END implies the last timepoint.  "
	  "The default step of 1 (one) in ranges can be changed using "
	  "'START:STEP:END', which is equivalent to "
	  "'START,START+STEP,START+(2*STEP),...,END'." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[0], 1, "xselect",
	  "Just like timeselect, but for the 'x' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[1], 1, "yselect",
	  "Just like timeselect, but for the 'y' dimension." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_select[2], 1, "zselect",
	  "Just like timeselect, but for the 'z' dimension." }
    };

    memset(&bdr, '\0', sizeof(bdr));
    memset(&maskbdr, '\0', sizeof(maskbdr));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc != 2) {
	fprintf(stderr, "Usage: %s [opts] xmlfile\n", argv[0]);
	fprintf(stderr, "Use --help for more info.\n");
	return -1;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    inputfile = argv[1];

    if (bxh_dataReadFileStart(inputfile, "image", NULL, 4, ordereddimnames, opt_select, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (bdr.datarec->numdims < 3) {
	fprintf(stderr, "Data must be at least 3-dimensional.\n");
	goto FAIL;
    }
    if (bdr.datarec->numdims >= 4 && bdr.datarec->dimensions[3].size == 0) {
	fprintf(stderr, "Number of time points must be greater than 0!\n");
	goto FAIL;
    }

    if (opt_maskfile) {
	if (bxh_dataReadFileStart(opt_maskfile, "image", NULL, 3, ordereddimnames, opt_select, &maskbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims != 3) {
	    fprintf(stderr, "Mask must be 3-dimensional.\n");
	    goto FAIL;
	}
	if (memcmp(maskbdr.dimsizes, bdr.dimsizes, sizeof(maskbdr.dimsizes[0]) * maskbdr.datarec->numdims) != 0) {
	    fprintf(stderr, "Mask spatial dimensions do not match data dimensions.\n");
	    goto FAIL;
	}
    }

    if (bxh_dataReadFinish(&bdr, "float") != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", inputfile);
	goto FAIL;
    }
    if (opt_maskfile) {
	if (bxh_dataReadFinish(&maskbdr, "char") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
    }

    dataptr = (float *)bdr.dataptr;
    maskdataptr = (char *)maskbdr.dataptr;

    {
	size_t numvolumes = 1;
	size_t t = 0;
	size_t x, y, z = 0;
	for (int i = 3; i < bdr.datarec->numdims; i++) {
	    numvolumes *= bdr.dimsizes[i];
	}
	std::vector<float> means(numvolumes, 0);
	std::vector<float> mins(numvolumes, 0);
	std::vector<float> maxs(numvolumes, 0);
	std::vector<float> stddevs(numvolumes, 0);
	/*** Calculate center of mass for each volume ***/
	std::vector<std::vector<float> > cmass(numvolumes, std::vector<float>(3));
	for (t = 0; t < numvolumes; t++) {
	    float xwsum = 0.0;
	    float ywsum = 0.0;
	    float zwsum = 0.0;
	    float sum = 0.0;
	    float min = MYNAN;
	    float max = MYNAN;
	    size_t num = 0;
	    size_t ind = t * bdr.pagesizes[2];
	    size_t maskind = 0;
	    float ssd = 0.0;

	    for (z = 0; z < bdr.dimsizes[2]; z++) {
		for (y = 0; y < bdr.dimsizes[1]; y++) {
		    for (x = 0; x < bdr.dimsizes[0]; x++) {
			if (opt_maskfile == NULL || maskdataptr[maskind]) {
			    float val = dataptr[ind];
			    sum += val;
			    xwsum += x * val;
			    ywsum += y * val;
			    zwsum += z * val;
			    if (isnan(min) || val < min) {
				min = val;
			    }
			    if (isnan(max) || val > max) {
				max = val;
			    }
			    num++;
			}
			ind++;
			maskind++;
		    }
		}
	    }
	    if (sum == 0) {
		cmass[t][0] = bdr.dimsizes[0] / 2;
		cmass[t][1] = bdr.dimsizes[1] / 2;
		cmass[t][2] = bdr.dimsizes[2] / 2;
	    } else {
		cmass[t][0] = xwsum / sum;
		cmass[t][1] = ywsum / sum;
		cmass[t][2] = zwsum / sum;
	    }
	    means[t] = sum / (float)num;
	    mins[t] = min;
	    maxs[t] = max;

	    ind = t * bdr.pagesizes[2];
	    maskind = 0;
	    num = 0;
	    for (z = 0; z < bdr.dimsizes[2]; z++) {
		for (y = 0; y < bdr.dimsizes[1]; y++) {
		    for (x = 0; x < bdr.dimsizes[0]; x++) {
			if (opt_maskfile == NULL || maskdataptr[maskind]) {
			    float val = dataptr[ind];
			    ssd += (val - means[t]) * (val - means[t]);
			    num++;
			}
			ind++;
			maskind++;
		    }
		}
	    }
	    if (num <= 1) {
	        stddevs[t] = 0;
	    } else {
	        stddevs[t] = sqrt(ssd / (num - 1));
	    }
	}

	/* write out results */
	bxhdimension * dimx = &bdr.datarec->dimensions[0];
	bxhdimension * dimy = &bdr.datarec->dimensions[1];
	bxhdimension * dimz = &bdr.datarec->dimensions[2];
	bxhdimension * dimt = (bdr.datarec->numdims > 3) ? &bdr.datarec->dimensions[3] : NULL;
	fprintf(stdout, "#VOLNUM\tVOLMEAN\tCMASSX(%s)\tCMASSY(%s)\tCMASSZ(%s)\tVOLSTDDEV\tVOLMIN\tVOLMAX\tAXISTICK(%s)\n",
		(dimx->units && *dimx->units) ? dimx->units : "mm",
		(dimy->units && *dimy->units) ? dimy->units : "mm",
		(dimz->units && *dimz->units) ? dimz->units : "mm",
		(dimt && dimt->units && *dimt->units) ? dimt->units : "");
	bxhdatapoints * dpstructp = NULL;
	if (dimt != NULL) {
	    for (int dpstructind = 0; dpstructind < dimt->numdpstructs; dpstructind++) {
		if (dimt->dpstructs[dpstructind].label == NULL) {
		    dpstructp = &dimt->dpstructs[dpstructind];
		}
	    }
	}
	for (t = 0; t < numvolumes; t++) {
	    fprintf(stdout, "%u\t%g\t%g\t%g\t%g\t%g\t%g\t%g\t",
		    (unsigned int)t,
		    means[t],
		    (dimx->units && *dimx->units) ? cmass[t][0] * dimx->spacing : cmass[t][0],
		    (dimy->units && *dimy->units) ? cmass[t][1] * dimy->spacing : cmass[t][1],
		    (dimz->units && *dimz->units) ? cmass[t][2] * dimz->spacing : cmass[t][2],
		    stddevs[t],
		    mins[t],
		    maxs[t]);
	    if (dimt) {
		if (dpstructp) {
		    fprintf(stdout, "%s", dpstructp->values[t]);
		} else {
		    fprintf(stdout, "%g", dimt->origin + (dimt->spacing * t));
		}
	    }
	    fprintf(stdout, "\n");
	}
    }

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:
    bxh_datareaddata_free(&bdr);
    if (opt_maskfile)
	bxh_datareaddata_free(&maskbdr);
    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.15  2009/01/15 20:55:18  gadde
 * New organization of data read functions to allow for reading of non-bxh data directly by most tools
 *
 * Revision 1.14  2008/02/15 22:38:28  gadde
 * Just assume "mm" for spatial units if not present.
 *
 * Revision 1.13  2006/05/31 19:11:10  gadde
 * Remove unnecessary consts.
 *
 * Revision 1.12  2006/05/31 14:44:08  gadde
 * Remove unnecessary consts.
 *
 * Revision 1.11  2006/04/26 15:22:30  gadde
 * Divide mean by number of voxels actually used (not full volume, when
 * using mask).
 *
 * Revision 1.10  2006/04/13 21:26:35  gadde
 * Use float instead of double.
 *
 * Revision 1.9  2006/04/12 18:00:37  gadde
 * Move to simpler bxh_dataReadStart/Finish interface.
 *
 * Revision 1.8  2005/09/20 18:37:53  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.7  2005/09/19 16:31:54  gadde
 * Documentation and help message updates.
 *
 * Revision 1.6  2005/09/14 14:49:25  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.5  2005/04/19 21:01:30  gadde
 * Fix cmass output for y/z dimensions if units aren't specified.
 *
 * Revision 1.4  2004/06/18 13:54:24  gadde
 * Add to usage message
 *
 * Revision 1.3  2004/05/03 19:41:53  gadde
 * Make sure non-data output is treated as a comment.
 *
 * Revision 1.2  2004/04/07 17:19:34  gadde
 * Add options for selecting portions of data.
 *
 * Revision 1.1  2004/03/31 16:24:11  gadde
 * Coalesce volume measures into one program for efficiency.
 *
 *
 */
